/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */
/* 
 * applymap.c - Apply the color map in an RLE file to the pixel data.
 * 
 * Author:	Spencer W. Thomas
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Tue Jul  8 1986
 * Copyright (c) 1986, Spencer W. Thomas
 */

#include <stdio.h>
#include <svfb_global.h>
#include <rle_getraw.h>

#define	map_pixel( pix, cmaplen, cmap )	((pix) > cmaplen ? (pix) : \
					 (cmap[pix]) >> 8)

/*****************************************************************
 * TAG( main )
 * 
 * Usage:
 *  	applymap [-l] [-o outfile] [rlefile]
 * Inputs:
 *	-l:		If specified, a linear map will be placed in the
 *			output file.
 * 	rlefile:    	Input file to have its map applied to its pixels.
 *  	    	    	Defaults to stdin.
 * Outputs:
 * 	outfile:    	Result of applying the map in the input file.
 *  	    	    	Defaults to stdout.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
main( argc, argv )
char **argv;
{
    register int i, c, j;
    char * infname = NULL, * outfname = NULL;
    FILE * infile = stdin, * outfile = stdout;
    int oflag = 0, y, nskip, nrow, one_to_many = 0, lflag = 0;
    struct sv_globals in_glob, out_glob;
    rle_op ** scan, ** outscan;
    int * nraw;
    int cmaplen, ncmap;
    rle_map ** cmap, *linmap, *mapp;

    if ( scanargs( argc, argv, "% l%- o%-outfile!s infile%s",
		   &lflag, &oflag, &outfname, &infname ) == 0 )
	exit( 1 );
    /* Open input file and read RLE header */
    if ( infname != NULL && (infile = fopen( infname, "r" )) == NULL )
    {
	fprintf( stderr, "applymap: Can't open %s: ", infname );
	perror("");
	exit(-1);
    }

    in_glob.svfb_fd = infile;
    rle_get_setup_ok( &in_glob, "applymap", infname );

    /* Open the output file */
    if ( outfname != NULL && (outfile = fopen( outfname, "w" )) == NULL )
    {
	perror( outfname );
	exit( 1 );
    }
    /* Copy global data from the input file */
    out_glob = in_glob;
    out_glob.svfb_fd = outfile;

    nrow = in_glob.sv_xmax - in_glob.sv_xmin + 1;	/* how wide is it? */

    if ( in_glob.sv_ncmap > 0 )
    {
	cmaplen = 1 << in_glob.sv_cmaplen;	/* length of the color map */
	/* Get pointers to the individual rows of the color map */
	cmap = (rle_map **) malloc( in_glob.sv_ncmap * sizeof(rle_map *) );
	for ( c = 0; c < in_glob.sv_ncmap; c++ )
	    cmap[c] = &in_glob.sv_cmap[c * cmaplen];
    }
    else
	cmaplen = 0;

    /* If the input file has only one channel, and has more than one
     * color map channel, then do a one to many mapping.
     */
    if ( in_glob.sv_ncolors == 1 && in_glob.sv_ncmap > 1 )
    {
	one_to_many = 1;
	out_glob.sv_ncolors = in_glob.sv_ncmap;
	for ( i = 1; i < out_glob.sv_ncolors; i++ )
	    SV_SET_BIT( out_glob, i );	/* save all these colors */
    }

    /* If -l, create a linear color mapfor the output file, otherwise,
     * it gets no map.
     */
    if ( lflag )
    {
	linmap = (rle_map *)malloc( cmaplen * out_glob.sv_ncolors *
				  sizeof(rle_map) );
	out_glob.sv_ncmap = out_glob.sv_ncolors;
	out_glob.sv_cmap = linmap;
	for ( c = 0, mapp = linmap; c < out_glob.sv_ncolors; c++ )
	    for ( i = 0; i < cmaplen; i++ )
		*mapp++ = i << (16 - in_glob.sv_cmaplen);	/* right justify */
    }
    else
    {
	out_glob.sv_ncmap = 0;	/* output file won't have a map */
	out_glob.sv_cmap = NULL;
    }

    /* Allocate space for the rle opcode information */
    scan = (rle_op **) malloc( (in_glob.sv_ncolors + in_glob.sv_alpha) *
			       sizeof( rle_op * ) );
    for ( i = in_glob.sv_ncolors + in_glob.sv_alpha - 1;
	  i >= 0;
	  i-- )
	scan[i] = (rle_op *)malloc( (nrow / 3 + 1) * sizeof( rle_op ) );

    outscan = (rle_op **) malloc( (out_glob.sv_ncolors + out_glob.sv_alpha) *
				  sizeof( rle_op * ) );

    if ( one_to_many )
    {
	for ( i = out_glob.sv_ncolors + out_glob.sv_alpha - 1;
	      i >= out_glob.sv_alpha;
	      i-- )
	    outscan[i] = (rle_op *)malloc( (nrow / 3 + 1) *
					   sizeof( rle_op ) );
	if ( out_glob.sv_alpha )
	    outscan[0] = scan[0];
	/* Map background color */
	if ( in_glob.sv_ncmap > 0 && in_glob.sv_background )
	{
	    out_glob.sv_bg_color = (int *)malloc( out_glob.sv_ncolors *
						  sizeof(int) );
	    for ( i = 0; i < out_glob.sv_ncolors; i++ )
		out_glob.sv_bg_color[i] = map_pixel( in_glob.sv_bg_color[0],
						     cmaplen, cmap[i] );
	}
    }
    else
    {
	for ( i = out_glob.sv_ncolors + out_glob.sv_alpha - 1;
	      i >= 0;
	      i-- )
	    outscan[i] = scan[i];
	if ( in_glob.sv_ncmap > 0 && in_glob.sv_background )
	    for ( i = 0; i < out_glob.sv_ncolors; i++ )
		in_glob.sv_bg_color[i] = map_pixel( in_glob.sv_bg_color[i],
						    cmaplen, cmap[i] );
    }

    nraw = (int *) malloc( (out_glob.sv_ncolors + out_glob.sv_alpha) *
			       sizeof( int ) );

    if ( in_glob.sv_alpha )
    {
	scan++;			/* [-1] points to the alpha channel */
	outscan++;
	nraw++;
    }

    /* Start the output file */
    sv_setup( RUN_DISPATCH, &out_glob );

    y = in_glob.sv_ymin - 1;
    while ( (nskip = rle_getraw( &in_glob, scan, nraw )) != 32768 )
    {
	nskip -= y;		/* figure out difference from previous line */
	y += nskip;
	if ( nskip > 1 )
	    sv_skiprow( &out_glob, nskip - 1 );
	if ( in_glob.sv_ncmap > 0 )
	    if ( one_to_many )
	    {
		for ( c = 1; c < out_glob.sv_ncolors; c++ )
		    nraw[c] = nraw[0];	/* all the same length */

		for ( i = 0; i < nraw[0]; i++ )
		    switch( scan[0][i].opcode )
		    {
		    case RRunDataOp:
			for ( c = 0; c < out_glob.sv_ncolors;
			      c++ )
			{
			    outscan[c][i] = scan[0][i];
			    outscan[c][i].u.run_val =
				map_pixel( scan[0][i].u.run_val,
					   cmaplen, cmap[c] );
			}
			break;
		    case RByteDataOp:
			for ( c = 0; c < out_glob.sv_ncolors;
			      c++ )
			{
			    outscan[c][i] = scan[0][i];
			    outscan[c][i].u.pixels =
				(rle_pixel *)malloc( outscan[c][i].length *
						     sizeof (rle_pixel) );
			    for ( j = 0; j < outscan[c][i].length;
				  j++ )
				outscan[c][i].u.pixels[j] = 
				    map_pixel( scan[0][i].u.pixels[j],
					       cmaplen, cmap[c] );
			}
			break;
		    }
	    }
	    else
	    {
		for ( c = 0; c < in_glob.sv_ncolors; c++ )
		    if ( c < in_glob.sv_ncmap )
			for ( i = 0; i < nraw[c]; i++ )
			    switch( scan[c][i].opcode )
			    {
			    case RRunDataOp:
				scan[c][i].u.run_val =
				    map_pixel( scan[c][i].u.run_val,
					       cmaplen, cmap[c] );
				break;
			    case RByteDataOp:
				for ( j = 0; j < scan[c][i].length;
				      j++ )
				    scan[c][i].u.pixels[j] = 
					map_pixel( scan[c][i].u.pixels[j],
						   cmaplen, cmap[c]);
				break;
			    }
	    }
	sv_putraw( outscan, nraw, &out_glob );
    }
    sv_puteof( &out_glob );
    exit( 0 );
}


