/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 *  Modified at BRL 16-May-88 by Mike Muuss to avoid Alliant STDC desire
 *  to have all "void" functions so declared.
 */
/* 
 * rle_getraw.c - 
 * 
 * Author:	Spencer W. Thomas
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Mon Nov 10 1986
 * Copyright (c) 1986, University of Utah
 */
#ifndef lint
static char rcs_ident[] = "$Header: rle_getraw.c,v 1.3 86/11/10 12:30:34 thomas Exp $";
#endif

#include <stdio.h>
#include <svfb_global.h>
#include <rle_getraw.h>

extern char * malloc();

struct inst {
	short opcode:8,
	    datum:8;
};

#define BREAD(type, var, len)\
	if ( infile->_cnt < len )\
	    fread( (char *)&var, 1, len, infile );\
	else\
	{\
	    var = *(type *)infile->_ptr;\
	    infile->_ptr += len;\
	    infile->_cnt -= len;\
	}

#define OPCODE(inst) (inst.opcode & ~LONG)
#define LONGP(inst) (inst.opcode & LONG)
#define DATUM(inst) (0x00ff & inst.datum)

/*****************************************************************
 * TAG( rle_getraw )
 * 
 * Get a raw scanline from the input file.
 * Inputs:
 *	globals:    sv_globals structure containing information about 
 *		    the input file.
 * Outputs:
 * 	scanraw:    an array of pointers to the individual color
 *		    scanlines.  Scanraw is assumed to have
 *		    globals->sv_ncolors pointers to arrays of rle_op,
 *		    each of which with enough elements, at least
 *		    1 + (globals->sv_xmax - globals->sv_xmin) / 3.
 *	nraw:	    an array of integers giving the number of rle_ops for
 *		    each color channel.
 *	Returns the current scanline number.  Returns 32768 at EOF.
 * Assumptions:
 * 	rle_get_setup has already been called.
 * Algorithm:
 *	Read input until a vertical skip is encountered,
 *	decoding the instructions into the scanraw array.
 *	Vertical skips that separate scanlines with no data do not
 *	cause a return.  In other words, the only reason for returning
 *	with an empty scanline is end of file.
 */
rle_getraw( globals, scanraw, nraw )
struct sv_globals *globals;
rle_op *scanraw[];
int nraw[];
{
    register int channel = 0;
    register rle_op * rawp = NULL;
    FILE *infile = globals->svfb_fd;
    struct inst inst;
    int scan_x = globals->sv_xmin;
    short word, long_data, nc, been_some = 0;

    /* Add in vertical skip from last scanline */
    if ( globals->sv_private.get.vert_skip > 0 )
	globals->sv_private.get.scan_y += globals->sv_private.get.vert_skip;

    /* Set run lengths to 0 */
    for ( channel = (globals->sv_alpha ? -1 : 0);
	  channel < globals->sv_ncolors;
	  channel++ )
	if ( SV_BIT( *globals, channel ) )
	     nraw[channel] = 0;

    if ( globals->sv_private.get.is_eof )
	return 32768;		/* too big for 16 bits, signal EOF */

    /* Otherwise, read and interpret instructions until a skipLines
     * instruction is encountered.
     */
    for (;;)
    {
        BREAD(struct inst, inst, 2 );
	if ( feof(infile) )
	{
	    globals->sv_private.get.is_eof = 1;
	    break;		/* <--- one of the exits */
	}

	switch( OPCODE(inst) )
	{
	case RSkipLinesOp:
	    if ( LONGP(inst) )
	    {
	        BREAD( short, long_data, sizeof long_data );
		SWAB( long_data );
		globals->sv_private.get.vert_skip = long_data;
	    }
	    else
		globals->sv_private.get.vert_skip = DATUM(inst);
	    break;			/* need to break for() here, too */

	case RSetColorOp:
	    channel = DATUM(inst);	/* select color channel */
	    if ( channel == 255 )
		channel = -1;
	    scan_x = globals->sv_xmin;
	    if ( SV_BIT( *globals, channel ) )
		rawp = scanraw[channel];
	    else
		rawp = NULL;
	    break;

	case RSkipPixelsOp:
	    if ( LONGP(inst) )
	    {
	        BREAD( short, long_data, sizeof long_data );
		SWAB( long_data );
		scan_x += long_data;
	    }
	    else
	    {
		scan_x += DATUM(inst);
	    }
	    break;

	case RByteDataOp:
	    if ( LONGP(inst) )
	    {
	        BREAD( short, long_data, sizeof long_data );
		SWAB( long_data );
		nc = (int)long_data;
	    }
	    else
		nc = DATUM(inst);
	    nc++;
	    if ( rawp != NULL )
	    {
		rawp->opcode = RByteDataOp;
		rawp->xloc = scan_x;
		rawp->length = nc;
		rawp->u.pixels = (rle_pixel *)malloc( (unsigned)nc );
		fread( (char *)rawp->u.pixels, 1, nc, infile );
		rawp++;
		nraw[channel]++;
	    }
	    else
	    {
		if ( fseek( infile, nc, 1 ) < 0 )
		{
		    register int ii;
		    for ( ii = nc; ii > 0; ii-- )
			(void) getc( infile );	/* discard it */
		}
	    }
 	    if ( nc & 1 )
		(void)getc( infile );	/* throw away odd byte */
	    scan_x += nc;
	    been_some = 1;
	    break;

	case RRunDataOp:
	    if ( LONGP(inst) )
	    {
	        BREAD( short, long_data, sizeof long_data );
		SWAB( long_data );
		nc = long_data;
	    }
	    else
		nc = DATUM(inst);

	    nc++;
	    BREAD( short, word, sizeof(short) );
	    SWAB( word );
	    if ( rawp != NULL )
	    {
		rawp->opcode = RRunDataOp;
		rawp->xloc = scan_x;
		rawp->length = nc;
		rawp->u.run_val = word;
		rawp++;
		nraw[channel]++;
	    }
	    scan_x += nc;
	    been_some = 1;
	    break;

	case REOFOp:
	    globals->sv_private.get.is_eof = 1;
	    break;

	default:
	    fprintf( stderr,
		     "rle_getraw: Unrecognized opcode: %d\n", inst.opcode );
	    exit(1);
	}
	if ( OPCODE(inst) == REOFOp )
	    break;			/* <--- the other loop exit */
	if ( OPCODE(inst) == RSkipLinesOp )
	{
	    if ( been_some )
		break;			/* <--- the other loop exit */
	    else
		/* No data on that scanline, so move up to this scanline */
		globals->sv_private.get.scan_y +=
		    globals->sv_private.get.vert_skip;
	}
    }

    /* Return current Y value */
    return globals->sv_private.get.scan_y;
}

/*****************************************************************
 * TAG( rle_freeraw )
 * 
 * Free all the pixel arrays in the raw scan struct.
 * Inputs:
 *  	globals:    Globals struct corresponding to this RLE data.
 *  	scanraw:    Array of pointers to array of rle_op, as above.
 * 	nraw:	    Array of lengths (as above)
 * Outputs:
 * 	Frees the areas pointed to by the pixels elements of any
 *  	RByteDataOp type rle_op structs.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
void
rle_freeraw( globals, scanraw, nraw )
struct sv_globals * globals;
int nraw[];
rle_op *scanraw[] ;
{
    int c, i;
    register rle_op * raw_p;

    for ( c = -globals->sv_alpha; c < globals->sv_ncolors; c++ )
	if ( SV_BIT( *globals, c ) )
	    for ( i = nraw[c], raw_p = scanraw[c]; i > 0; i--, raw_p++ )
		if ( raw_p->opcode == RByteDataOp )
		    free( raw_p->u.pixels );
}
