/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 *  Modified at BRL 16-May-88 by Mike Muuss to avoid Alliant STDC desire
 *  to have all "void" functions so declared.
 */
/* 
 * Runsv.c - General purpose Run Length Encoding for svfb.
 * 
 * Author:	Spencer W. Thomas
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Mon Aug  9 1982
 * Copyright (c) 1982,1986 Spencer W. Thomas
 * 
 * $Header: Runsv.c,v 2.9 86/02/27 10:01:13 thomas Locked $
 * 
 * Modified by:	Todd W. Fuqua
 * 	Date:	Jul 22 1984
 * convert to new RLE format to make room for larger frame buffers
 */

/* THIS IS WAY OUT OF DATE.  See rle.5.
 * The output file format is:
 * 
 * Word 0:	A "magic" number.  The top byte of the word contains
 *		the letter 'R' or the letter 'W'.  'W' indicates that
 *		only black and white information was saved.  The bottom
 *		byte is one of the following:
 *	' ':	Means a straight "box" save, -S flag was given.
 *	'B':	Image saved with background color, clear screen to
 *		background before restoring image.
 *	'O':	Image saved in overlay mode.
 * 
 * Words 1-6:	The structure
 * { *     short   xpos,			/* Lower left corner
 *             ypos,
 *             xsize,			/* Size of saved box
 *             ysize;
 *     char    rgb[3];			/* Background color
 *     char    map;			/* flag for map presence
 * }
 * 
 * If the map flag is non-zero, then the color map will follow as 
 * 3*256 16 bit words, first the red map, then the green map, and
 * finally the blue map.
 * 
 * Following the setup information is the Run Length Encoded image.
 * Each instruction consists of a 4-bit opcode, a 12-bit datum and
 * possibly one or more following words (all words are 16 bits).  The
 * instruction opcodes are:
 * 
 * SkipLines (1):   The bottom 10 bits are an unsigned number to be added to
 *		    current Y position.
 * 
 * SetColor (2):    The datum indicates which color is to be loaded with
 * 		    the data described by the following ByteData and
 * 		    RunData instructions.  0->red, 1->green, 2->blue.  The
 * 		    operation also resets the X position to the initial
 * 		    X (i.e. a carriage return operation is performed).
 * 
 * SkipPixels (3):  The bottom 10 bits are an unsigned number to be
 * 		    added to the current X position.
 * 
 * ByteData (5):    The datum is one less than the number of bytes of
 * 		    color data following.  If the number of bytes is
 * 		    odd, a filler byte will be appended to the end of
 * 		    the byte string to make an integral number of 16-bit
 * 		    words.  The bytes are in PDP-11 order.  The X
 * 		    position is incremented to follow the last byte of
 * 		    data.
 * 
 * RunData (6):	    The datum is one less than the run length.  The
 * 		    following word contains (in its lower 8 bits) the
 * 		    color of the run.  The X position is incremented to
 * 		    follow the last byte in the run.
 */

#include	"stdio.h"
#include	"svfb.h"
#include	"svfb_global.h"
#include	"XtndRunsv.h"
extern char * malloc();

#define UPPER 255			/* anything bigger ain't a byte */

/* Predefine LITTLE_ENDIAN for vax and pdp11 machines */
#if defined(vax) || defined(pdp11)
#define LITTLE_ENDIAN
#endif

#ifndef LITTLE_ENDIAN
static rle_pixel t;
static rle_pixel *a1,*a2;
static short tm;
#endif

#ifdef LITTLE_ENDIAN
#define	ASN(a,b)	a = b
#else
#define	ASN(a,b)	(tm=(int)(b)),(a1=(rle_pixel*)(&(a))),\
			(a2=(rle_pixel*)(&(tm))),(*a1=(*(a2+1))),(*(a1+1)=(*a2))
#endif

/*
 * Macros to make writing instructions with correct byte order easier.
 */
union { short s; char c[2]; } arg;
#ifdef LITTLE_ENDIAN
#define	put16(a)    arg.s=a,putc(arg.c[0],sv_fd), putc(arg.c[1],sv_fd)
#else
#define	put16(a)    arg.s=a,putc(arg.c[1],sv_fd), putc(arg.c[0],sv_fd)
#endif

/* short instructions */
#define mk_short_1(oper,a1)		/* one argument short */ \
    putc(oper,sv_fd), putc((char)a1,sv_fd)

#define mk_short_2(oper,a1,a2)		/* two argument short */ \
    putc(oper,sv_fd), putc((char)a1,sv_fd), put16(a2)

/* long instructions */
#define mk_long_1(oper,a1)		/* one argument long */ \
    putc((char)(LONG|oper),sv_fd), putc('\0', sv_fd), put16(a1)

#define mk_long_2(oper,a1,a2)		/* two argument long */ \
    putc((char)(LONG|oper),sv_fd), putc('\0', sv_fd), \
    put16(a1), put16(a2)

/* choose between long and short format instructions */
/* NOTE: these macros can only be used where a STATEMENT is legal */

#define mk_inst_1(oper,a1)		/* one argument inst */ \
    if (a1>UPPER) (mk_long_1(oper,a1)); else (mk_short_1(oper,a1))

#define mk_inst_2(oper,a1,a2)		/* two argument inst */ \
    if (a1>UPPER) (mk_long_2(oper,a1,a2)); else (mk_short_2(oper,a1,a2))

/* 
 * Opcode definitions
 */
#define	    RSkipLines(n)   	    mk_inst_1(RSkipLinesOp,(n))

#define	    RSetColor(c)	    mk_short_1(RSetColorOp,(c))
				    /* has side effect of performing */
				    /* "carriage return" action */

#define	    RSkipPixels(n)	    mk_inst_1(RSkipPixelsOp,(n))

#define	    RNewLine		    RSkipLines(1)

#define	    RByteData(n)	    mk_inst_1(RByteDataOp,n)
					/* followed by ((n+1)/2)*2 bytes */
					/* of data.  If n is odd, last */
					/* byte will be ignored */
					/* "cursor" is left at pixel */
					/* following last pixel written */

#define	    RRunData(n,c)	    mk_inst_2(RRunDataOp,(n),(c))
					/* next word contains color data */
					/* "cursor" is left at pixel after */
					/* end of run */

#define     REOF		    mk_inst_1(REOFOp,0)
					/* Really opcode only */

/*****************************************************************
 * TAG( RunSetup )
 * Put out initial setup data for RLE svfb files.
 */
void
RunSetup(globals)
register struct sv_globals * globals;
{
    struct XtndRsetup setup;
    short magik;
    register FILE * sv_fd = globals->svfb_fd;

    ASN(magik,XtndRMAGIC);
    fwrite((char *)&magik, 2, 1, sv_fd);
    if ( globals->sv_background == 2 )
	setup.h_flags = H_CLEARFIRST;
    else if ( globals->sv_background == 0 )
	setup.h_flags = H_NO_BACKGROUND;
    else
	setup.h_flags = 0;
    if ( globals->sv_alpha )
	setup.h_flags |= H_ALPHA;
    if ( globals->sv_comments != NULL && *globals->sv_comments != NULL )
	setup.h_flags |= H_COMMENT;

    setup.h_ncolors = globals->sv_ncolors;
    setup.h_pixelbits = 8;		/* Grinnell dependent */
    if ( globals->sv_ncmap > 0 && globals->sv_cmap == NULL )
    {
	fprintf( stderr,
		 "Color map of size %d*%d specified, but not supplied\n" );
	globals->sv_ncmap = 0;
    }
    setup.h_cmaplen = globals->sv_cmaplen;	/* log2 of color map size */
    setup.h_ncmap = globals->sv_ncmap;	/* no of color channels */
    ASN(setup.h_xpos,globals->sv_xmin);
    ASN(setup.h_ypos,globals->sv_ymin);
    ASN(setup.h_xlen,globals->sv_xmax - globals->sv_xmin + 1);
    ASN(setup.h_ylen,globals->sv_ymax - globals->sv_ymin + 1);
    fwrite((char *)&setup, SETUPSIZE, 1, sv_fd);
    if ( globals->sv_background != 0 )
    {
	register int i;
	rle_pixel * background =
		(rle_pixel *)malloc( (unsigned)(globals->sv_ncolors + 1) );
	/* 
	 * If even number of bg color bytes, put out one more to get to 
	 * 16 bit boundary.
	 */
	for ( i = 0; i < globals->sv_ncolors; i++ )
	    background[i] =  globals->sv_bg_color[i];
	fwrite((char *)background, (globals->sv_ncolors / 2) * 2 + 1, 1,
	       sv_fd);
	free( background );
    }
    else
	putc( '\0', sv_fd );
    if (globals->sv_ncmap > 0)
    {
#ifdef	LITTLE_ENDIAN
	/* Code for little-endian machines */
	fwrite( (char *)globals->sv_cmap,
		globals->sv_ncmap * (1 << globals->sv_cmaplen) * 2,
		1, sv_fd ); 
#else
	/* Big-endian machines are harder */
	register int i, nmap = (1 << globals->sv_cmaplen) *
			       globals->sv_ncmap;
	register rle_map * h_cmap = (rle_map *)malloc( nmap * sizeof(rle_map) );
	if ( h_cmap == NULL )
	{
	    fprintf( stderr, "Malloc failed for color map of size %d\n",
		     nmap );
	    exit( 1 );
	}
	for ( i = 0; i < nmap; i++ )
		h_cmap[i] = ((globals->sv_cmap[i] >> 8) & 0xff)
			  | ((globals->sv_cmap[i] & 0xff) << 8);
	fwrite( (char *)h_cmap, nmap, 2, sv_fd );
	free( (char *)h_cmap );
#endif
    }

    /* Now write out comments if given */
    if ( setup.h_flags & H_COMMENT )
    {
	int comlen;
	short wrlen;

#ifdef	macintosh_os
#define	comp	compa
#endif	macintosh_os

	register char ** comp;

	/* Get the total length of comments */
	comlen = 0;
	for ( comp = globals->sv_comments; *comp != NULL; comp++ )
	    comlen += 1 + strlen( *comp );

	ASN( wrlen, comlen );
	fwrite( (char *)&wrlen, 2, 1, sv_fd );
	for ( comp = globals->sv_comments; *comp != NULL; comp++ )
	    fwrite( *comp, 1, strlen( *comp ) + 1, sv_fd );

	if ( comlen & 1 )	/* if odd length, round up */
	    putc( '\0', sv_fd );
    }
}

/*****************************************************************
 * TAG( RunSkipBlankLines )
 * Skip one or more blank lines in the RLE file.
 */
void
RunSkipBlankLines(nblank, globals)
register struct sv_globals * globals;
{
    register FILE * sv_fd = globals->svfb_fd;
    RSkipLines(nblank);
}

/*****************************************************************
 * TAG( RunSetColor )
 * Select a color and do carriage return.
 * color: 0 = Red, 1 = Green, 2 = Blue.
 */
void
RunSetColor(c, globals)
register struct sv_globals * globals;
{
    register FILE * sv_fd = globals->svfb_fd;
    RSetColor(c);
}

/*****************************************************************
 * TAG( RunSkipPixels )
 * Skip a run of background.
 */

/* ARGSUSED */
void
RunSkipPixels(nskip, last, wasrun, globals)
register struct sv_globals * globals;
{
    register FILE * sv_fd = globals->svfb_fd;
    if (! last && nskip > 0)
    {
	RSkipPixels(nskip);
    }
}

/*****************************************************************
 * TAG( RunNewScanLine )
 * Perform a newline action.  Since CR is implied by the Set Color
 * operation, only generate code if the newline flag is true.
 */
void
RunNewScanLine(flag, globals)
register struct sv_globals * globals;
{
    register FILE * sv_fd = globals->svfb_fd;
    if (flag)
    {
	RNewLine;
    }
}

/*****************************************************************
 * TAG( Runputdata )
 * Put one or more pixels of byte data into the output file.
 */
void
Runputdata(buf, n, globals)
rle_pixel * buf;
register struct sv_globals * globals;
{
    register FILE * sv_fd = globals->svfb_fd;
    if (n == 0)
	return;

    RByteData(n-1);
    fwrite((char *)buf, ((n+1)/2), 2, sv_fd);
}

/*****************************************************************
 * TAG( Runputrun )
 * Output a single color run.
 */

/* ARGSUSED */
void
Runputrun(color, n, last, globals)
register struct sv_globals * globals;
{
    register FILE * sv_fd = globals->svfb_fd;
    RRunData(n-1,color);
}


/*****************************************************************
 * TAG( RunputEof )
 * Output an EOF opcode
 */
void
RunputEof( globals )
register struct sv_globals * globals;
{
    register FILE * sv_fd = globals->svfb_fd;
    REOF;
}
