/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */
/* 
 * rletops.c - Convert RLE to postscript file.
 * 
 * Author:	Rod Bogart & John W. Peterson
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Tue Nov  4 1986
 * Copyright (c) 1986 Rod Bogart
 * 
 * Based on "tobw.c" by Spencer Thomas, and 
 * "rps" by Marc Majka (UBC Vision lab)
 *
 * Options: 
 * -h X - Image height in inches (default 3", width is found via aspect & size)
 * -s	- Scribe mode: don't generate "showpage", and default center is 3.25
 * -c X - Center image about X inches on the page (or margin if -s is on).
 * -a   - Aspect ratio of image (default 1.0)
 * 
 * If an input file isn't specified, it reads from stdin.  An output file
 * can be specified with -o (otherwise it writes to stdout).
 */
#ifndef lint
static char rcs_ident[] = "$Header:$";
#endif

#include <stdio.h>
#include <math.h>
#include "svfb_global.h"


main( argc, argv )
char **argv;
{
    char * infname = NULL, * outfname = NULL;
    FILE * infile = stdin, * outfile = stdout;
    int sflag =  0, oflag = 0, cflag = 0, dummy;
    int y, nrow, nscan, i, pix, add_extra_white_line = 0;
    float heightinch = 3.0, center = 3.25, widthinch, aspect = 1.0;
    float x1,y1,x2,y2;
    unsigned char ** scan;
    unsigned char * buffer;

    if ( scanargs( argc, argv,
	    "% s%- h%-height!f c%-center!f a%-aspect!f o%-outfile!s infile%s",
	    &sflag, &dummy, &heightinch, &cflag, &center, &dummy, &aspect,
	    &oflag, &outfname, &infname ) == 0 )
	exit( 1 );

    /* Open RLE file and get header information */

    if ( infname != NULL && (infile = fopen( infname, "r" )) == NULL )
    {
	perror( infname );
	exit( 1 );
    }

    sv_globals.svfb_fd = infile;
    rle_get_setup_ok( &sv_globals, "rletops", infname );

    if ( ( sv_globals.sv_ncolors != 3 ) && ( sv_globals.sv_ncolors != 1))
    {
	fprintf( stderr, "%s is not RGB or b&w image",
		    infname ? infname : "stdin" );
	exit( 0 );
    }

    if ( outfname != NULL && (outfile = fopen( outfname, "w" )) == NULL )
    {
	perror( outfname );
	exit( 1 );
    }

    /* 
     * Spencer trick: save space by sliding the input image over to the
     * left margin.
     */
    sv_globals.sv_xmax -= sv_globals.sv_xmin;
    sv_globals.sv_xmin = 0;
    nrow = sv_globals.sv_xmax + 1;
    nscan = (sv_globals.sv_ymax - sv_globals.sv_ymin + 1);

    /* The laserwriters throw out files with an odd number of scanlines! */

    if (nscan % 2)
    {
	nscan++;
	add_extra_white_line = 1;
    }

    /* Allocate scanline memory */

    buffer = (unsigned char *)malloc( nrow );
    scan = (unsigned char **) malloc( (sv_globals.sv_ncolors +
				       sv_globals.sv_alpha) *
				      sizeof( unsigned char * ) );

    for ( i = 0;
	  i < sv_globals.sv_ncolors + sv_globals.sv_alpha;
	  i++ )
	scan[i] = (unsigned char *)malloc( nrow );

    if ( sv_globals.sv_alpha )
    {
	scan++;
    }

    /* Calculate image size and placement */

    widthinch = (float) nrow * heightinch * aspect / (float) nscan;
    if (sflag)
    {
	x1 = center - widthinch / 2.0;
	y1 = 0.0;
    }
    else
    {
	if (cflag)
	    x1 = center - widthinch / 2.0;
	else
	    /* center on whole page */
	    x1 = 4.25 - widthinch / 2.0;
	y1 = 0.5;
    }
    x2 = x1 + widthinch;
    y2 = y1 + heightinch;
    prologue(outfile,sflag,nscan,nrow,x1,y1,x2,y2);    

    while ( (y = rle_getrow( &sv_globals, scan )) <=
	    sv_globals.sv_ymax )
    {
	if (sv_globals.sv_ncolors == 1)
	    buffer = scan[0];
	else
	    rgb_to_bw( scan[0], scan[1], scan[2], buffer, nrow );

	for(pix = 0; pix < nrow; pix ++)
	    puthexpix(outfile,buffer[pix]);
    }
    if (add_extra_white_line)
	for(pix = 0; pix < nrow; pix ++)
	    puthexpix(outfile,255);

    epilogue(outfile, sflag);

    exit( 0 );
}

prologue(outfile,scribe_mode,nr,nc,x1,y1,x2,y2)
FILE *outfile;
int scribe_mode;
int nr,nc;
float x1,y1,x2,y2;
{
    fprintf(outfile,"%%!\n");
    fprintf(outfile,"gsave\n");
    if (! scribe_mode)
	fprintf(outfile,"initgraphics\n");
    fprintf(outfile,"72 72 scale\n");
    fprintf(outfile,"/imline %d string def\n",nc*2);
    fprintf(outfile,"/drawimage {\n");
    fprintf(outfile,"    %d %d 8\n",nc,nr);
    fprintf(outfile,"    [%d 0 0 %d 0 %d]\n",nc,-1*nr,nr);
    fprintf(outfile,"    { currentfile imline readhexstring pop } image\n");
    fprintf(outfile,"} def\n");
    fprintf(outfile,"%f %f translate\n",x1,y2);
    fprintf(outfile,"%f %f scale\n",x2-x1,y1-y2);
    fprintf(outfile,"drawimage\n");
}

epilogue(outfile, scribemode)
FILE *outfile;
int scribemode;
{
    fprintf(outfile,"\n");
    if (!scribemode)
	fprintf(outfile,"showpage\n");
    fprintf(outfile,"grestore\n");
}

puthexpix(outfile,p)
FILE *outfile;
unsigned char p;
{
    char hi, lo;
    static npixo = 0;
    static char *tohex = "0123456789ABCDEF";

    hi = (p >> 4) & 15;
    lo = p & 15;
    
    fprintf(outfile,"%c%c",tohex[hi],tohex[lo]);
    npixo += 1;
    if (npixo >= 32) {
        fprintf(outfile,"\n");
        npixo = 0;
    }
}
