/* 
 * unslice.c - Build a finished frame from a series of slices
 * 
 * Author:	John W. Peterson
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Tue May 19 1987
 * Copyright (c) 1987, University of Utah
 *
 * Lots of this code is swiped from comp.c
 */

/*
 * The "control file" is a text file with two numbers per line, each
 * line giving the starting and ending lines (inclusive) that are to
 * be taken from each input file to the output file.  This allows
 * potentially ragged portions of slices to be cropped away.
 *
 * If no control file is given, then the extents given in the slices' headers
 * are used.  If two files overlap, the first lines from the second file
 * are thrown away.  The maximum y of the image must be specified this way.
 */

#include <stdio.h>
#include <svfb_global.h>
#include <rle_getraw.h>

struct sv_globals in_globals, out_globals;

/* 
 * Global raw data structures for copy_scanline.
 */
rle_op ** out_raw;
int * out_nraw;

int max_y = 0;			/* Maximum Y value */
int num_skip = 0;		/* Counter for keeping track of raw lines */

int ctlflag = 0;		/* If true then we are using a ctl file */
int ctl_lines;			/* Number of lines in control file */
int *start_line,
    *stop_line;			/* Start and ending lines for the slices */

char default_ctl_file[] = "unslice.ctl";

main (argc, argv)
int argc;
char **argv;
{
    int nfiles, i;
    char **slicefiles;
    FILE *ctlfile;
    int y_flag;

    char * ctlfilename = NULL;

    if (! scanargs( argc, argv, "% y%-ymax!d f%-ctlfile!s files!*s",
		   &y_flag, &max_y,
		   &ctlflag, &ctlfilename, &nfiles, &slicefiles ))
	exit(1);

    if (! (start_line = (int *) malloc( nfiles * sizeof(int) )))
    {
    	fprintf(stderr, "unslice: too many slices\n");
    	exit(-1);
    }
    if (! (stop_line = (int *) malloc( nfiles * sizeof(int) )))
    {
    	fprintf(stderr, "unslice: too many slices\n");
    	exit(-1);
    }

    if (ctlflag)		/* Read in and verify the control file */
    {
	if (! ctlfilename)
	    ctlfilename = default_ctl_file;

	/* Read in the control file */

	if (! (ctlfile = fopen( ctlfilename, "r" )))
	{
	    perror("unslice (opening control file)");
	    exit(-1);
	}

	ctl_lines = 0;
	while (! feof( ctlfile ))
	{
	    fscanf( ctlfile, "%d %d",
		   &(start_line[ctl_lines]), &(stop_line[ctl_lines]) );

	    if (start_line[ctl_lines] >= stop_line[ctl_lines])
		fprintf(stderr, "unslice: ctl file garbled? (start %d, stop %d)\n",
			start_line[ctl_lines], stop_line[ctl_lines] );
	    ctl_lines++;
	}
	ctl_lines--;

	if (ctl_lines > nfiles)
	    fprintf(stderr,
		    "unslice: Warning: not enough slices for ctl file\n");
	if (ctl_lines < nfiles)
	    fprintf(stderr,
		    "unslice: Warning: too many slices for ctl file\n");

    }
    else
	if (! max_y)
	{
	    fprintf(stderr,
	       "unslice: max y (-y ymax) must be given if no ctl file used\n");
	    exit(-1);
	}

    /* Process the slices */

    for (i = 0; i < nfiles; i++)
	do_slice( i, slicefiles[i] );

    sv_puteof( &out_globals );

    exit( 0 );
}

/*****************************************************************
 * TAG( do_slice )
 * 
 * Read one slice from the given file and write it to the output.
 * Also generate the output globals if it's the first file.
 */
do_slice( num, filename )
int num;
char *filename;
{
    register int y;
    static int current_y = 0;

    if (! (in_globals.svfb_fd = fopen( filename, "r" )))
    {
	fprintf(stderr, "unslice: can't open %s", filename);
	perror("");
	exit(-1);
    }
    
    /*
     * Many sanity checks.  Code must be 3am-proof!
     */

    rle_get_setup_ok( &in_globals, "unslice", filename );
    
    if (ctlflag &&
       ((start_line[num] > in_globals.sv_ymax) ||
	(stop_line[num] < in_globals.sv_ymin)))
    {
	fprintf(stderr, "unslice: %s is out of slice range (%d %d)\n",
		filename, start_line[num], stop_line[num]);
	exit(-1);
    }

    /* 
     * If this is the first slice, generate the output header.
     */
    
    if (num == 0)
    {
	out_globals = in_globals;
	if (ctlflag)
	{
	    out_globals.sv_ymin = start_line[0];
	    out_globals.sv_ymax = stop_line[ctl_lines-1];
	}
	else
	{
	    out_globals.sv_ymin = in_globals.sv_ymin;
	    out_globals.sv_ymax = max_y;
	}
	current_y = out_globals.sv_ymin;
	
	out_globals.svfb_fd = stdout;

	sv_setup( RUN_DISPATCH, &out_globals );

	if (rle_raw_alloc( &out_globals, &out_raw, &out_nraw ))
	{
	    fprintf(stderr,"unslice: not enough heap\n");
	    exit(-1);
	}
    }

    if ((! ctlflag) && (in_globals.sv_ymax < current_y))
    {
	fprintf(stderr,
	       "unslice: warning: slice %s completly ignored (wrong order?)\n",
		filename);
	fclose( in_globals.svfb_fd );
	return;
    }

    /*
     * Copy the file to the output.
     */
    num_skip = 0;

    if (ctlflag)
    {
	for (y = in_globals.sv_ymin; y <= in_globals.sv_ymax; y++ )
	    if ((y >= start_line[num]) && (y <= stop_line[num]))
		copy_scanline( y, 1 );
	    else
		copy_scanline( y, 0 ); /* Data out of range, just toss it. */
    }
    else
    {
	for (y = in_globals.sv_ymin; y <= in_globals.sv_ymax; y++ )
	    if (y >= current_y)
		copy_scanline( y, 1 );
	    else
		copy_scanline( y, 0 );

	current_y = in_globals.sv_ymax + 1;
    }

    fclose( in_globals.svfb_fd );
}

/*
 * The "skip counter" is stolen from comp.  It works like this:
 * if num_skip == 0, then we read the next line normally.  If it's
 * positive, then it tells us how many blank lines before the
 * next available real data.  If it's -1, then it means that the
 * output raw data should be used before calling sv_getraw again.
 */

/*****************************************************************
 * TAG( copy_scanline )
 * 
 * Copy the scanlines using the raw format, if the copy_flag is on.  If
 * copy_flag is false, scanlines are just eaten away from the input file.
 */

copy_scanline( ypos, copy_flag )
int ypos;
int copy_flag;			/* If true, write the output */
{

 SKIP_ROW:

    if (num_skip > 0)		/* Must skip blank rows */
    {
	if (copy_flag)
	    sv_skiprow( &out_globals, 1 );
	num_skip--;
	if (num_skip == 0)
	    num_skip = -1;	/* Flag raw data available */
	return;
    }

    if (num_skip == 0)		/* ...Must read new data */
	num_skip = rle_getraw( &in_globals, out_raw, out_nraw );
    else
	num_skip = ypos;	/* num_skip < 0, data was already there */

    if (num_skip == 32768)	/* Hit the EOF */
    {
	if (copy_flag)
	    sv_skiprow( &out_globals, 1 );
	return;
    }

    num_skip -= ypos;		/* Find how many blank lines left */

    if (num_skip > 0)
	goto SKIP_ROW;

    if (copy_flag)
	sv_putraw( out_raw, out_nraw, &out_globals );
    rle_freeraw( &out_globals, out_raw, out_nraw );
}
