/* 
 * smush.c - Blur an image.  A level 1 smush is just a convolution with
 * a gaussian mask (or given mask).  Higher levels will be more blurry.
 * 
 * Author:	Rod Bogart
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Wed Oct 15 1986
 * Copyright (c) 1986, University of Utah
 * 
 */
#ifndef lint
static char rcs_ident[] = "$Header:$";
#endif

#include <stdio.h>
#include "svfb_global.h"

#define and &&
#define or ||

main(argc, argv)
int  argc;
char *argv[];
{
    char * infname = NULL, * outfname = NULL, * maskfname = NULL;
    FILE * infile = stdin, * outfile = stdout, * maskfile = NULL;
    int oflag = 0, mflag = 0, no_norm = 0;
    int xlinewidth, xlen, ylen, x, y;
    int i, j, k, chan, nchan, level, levels = 1;
    int maskskip, origmasksize;
    int maxmaskskip, maxmasksize, maxmaskbelow, maxmaskabove;
    int masksize, maskbelow, maskabove; /*number of maskels below and above */
					/* the center maskel. */
    float *maskmult;
    float *mask_mult_table;
    struct sv_globals outglobals;
    float total, *mask, *gauss_mask(), *read_mask();
    int newvalue;
    rle_pixel *rastptr, *rasterbase, *centerpxl, *calcpxl;
    rle_pixel *outrastptr, *outrasterbase, *outpxl, *tempbase;
    rle_pixel *bot_edge_pxl, *bot_data_pxl, *bot_extrap_pxl;
    rle_pixel *top_edge_pxl, *top_data_pxl, *top_extrap_pxl;
    rle_pixel *left_edge_pxl, *left_data_pxl, *left_extrap_pxl;
    rle_pixel *right_edge_pxl, *right_data_pxl, *right_extrap_pxl;
    rle_pixel **scanout;
    rle_pixel **rows;
    unsigned char *malloc();

    if ( scanargs( argc, argv, "% level%d m%-maskfile!s n%- o%-outfile!s infile%s",
		  &levels, &mflag, &maskfname, &no_norm,
		  &oflag, &outfname, &infname ) == 0 )
	exit( 1 );

    if ( infname != NULL && (infile = fopen( infname, "r" )) == NULL )
    {
	perror( infname );
	exit( 1 );
    }
    sv_globals.svfb_fd = infile;

    rle_get_setup_ok( &sv_globals, "smush", infname );

    if ( sv_globals.sv_alpha )
	SV_SET_BIT( sv_globals, SV_ALPHA );

    if ( outfname != NULL && (outfile = fopen( outfname, "w" )) == NULL )
    {
	perror( outfname );
	exit( 1 );
    }

    outglobals = sv_globals;
    outglobals.svfb_fd = outfile;

    if (mflag)
    {
	mask = read_mask(&origmasksize, maskfname, no_norm);
    }
    else
    {
	origmasksize = 5;
	mask = gauss_mask(origmasksize);
    }

    /* initialize mask_mult_table */
    mask_mult_table = (float *) malloc(sizeof(float) * origmasksize * 
				       origmasksize * 256);
    for (i=0; i < origmasksize; i++)
    {
	maskmult = &(mask_mult_table[(i*origmasksize) << 8]);
	for (j=0; j < origmasksize; j++)
	{
	    for (k=0;k < 256; k++)
		maskmult[ k ] = (float) k * mask[i*origmasksize+j];
	    maskmult += 256;
	}
    }

    if (levels < 1)
    {
	fprintf(stderr, "Level must be greater than 0.\n");
	exit(-1);
    }

    /* initial size of array has a border large enough for the biggest mask */
    maxmaskskip = 1 << (levels - 1);
    maxmasksize = (origmasksize - 1) * maxmaskskip + 1;
    maxmaskbelow = (maxmasksize-1) / 2;
    maxmaskabove = maxmasksize / 2;
    sv_globals.sv_xmax -= sv_globals.sv_xmin;
    sv_globals.sv_xmin = 0;
    xlen = sv_globals.sv_xmax - sv_globals.sv_xmin + 1;
    ylen = sv_globals.sv_ymax - sv_globals.sv_ymin + 1;
    xlinewidth = xlen + maxmasksize - 1;

    /* assumes that sv_alpha is 1 or 0 */
    nchan = sv_globals.sv_ncolors + sv_globals.sv_alpha;

    /* Note:
     * When you read in a row of pixels with rle_getrow, it places blank
     * pixels between 0 and sv_xmin of your buffer.  However, when you
     * use sv_putrow to write them out, the buffer must be pointing at
     * where the data actually starts (i.e., at [sv_xmin] of the getrow
     * buffer.  */

    /* WARNING: Heavy-duty pointer munging ahead */

    rows = (rle_pixel **) malloc( nchan * sizeof( rle_pixel * ) );
    if (! rows)
	fprintf(stderr, "rlepixel: malloc failed\n");

    rasterbase = malloc( xlinewidth * (ylen + maxmasksize - 1) * nchan );
    outrasterbase = malloc( xlinewidth * (ylen + maxmasksize - 1) * nchan );

    if ((!rasterbase) or (!outrasterbase))
    {
	fprintf(stderr, "smush: No memory for raster\n");
	exit(-1);
    }
    rastptr = &(rasterbase[maxmaskbelow * xlinewidth * nchan + maxmaskbelow]);

    /****************************************************
     * Read in all of the pixels
     ****************************************************/

    for (i = sv_globals.sv_ymin; i <= sv_globals.sv_ymax; i++)
    {
        for (chan=0; chan < nchan; chan++)
        {
            rows[chan] = rastptr;
	    /* Increment pointer by xlinewidth */
	    rastptr = &(rastptr[xlinewidth]);
	}
	/* assumes 1 or 0 for sv_alpha */
	rle_getrow( &sv_globals, &rows[sv_globals.sv_alpha] );
    }

    for(level = 1; level <= levels; level++)
    {
	/* actual mask is always size by size of data with extra zeros */
	maskskip = 1 << (level - 1);
	masksize = (origmasksize - 1) * maskskip + 1;
	maskbelow = (masksize-1) / 2;
	maskabove = masksize / 2;

	/****************************************************
	 * Extrapolate boundary pixels
	 ****************************************************/
	if ((maskbelow > 0) && (maskabove > 0))
	{
	    rastptr = &(rasterbase[maxmaskbelow +
		maxmaskbelow * xlinewidth * nchan]);

	    for (chan = 0; chan < nchan; chan++)
	    {
		bot_edge_pxl = &(rastptr[chan * xlinewidth]);
		top_edge_pxl = &(rastptr[(ylen-1) * nchan * xlinewidth
					    + chan * xlinewidth]);
		for(x=0; x < xlen; x++)
		{
		    for (i=1; i <= maskbelow; i++)
		    {
			bot_data_pxl = bot_edge_pxl + i * xlinewidth * nchan;
			bot_extrap_pxl = bot_edge_pxl - i * xlinewidth * nchan;
			newvalue = 2 * (*bot_edge_pxl) - (*bot_data_pxl);
			*bot_extrap_pxl = (newvalue < 0) ? 0 :
				((newvalue > 255) ? 255 : newvalue);
		    }
		    for (i=1; i <= maskabove; i++)
		    {
			top_data_pxl = top_edge_pxl - i * xlinewidth * nchan;
			top_extrap_pxl = top_edge_pxl + i * xlinewidth * nchan;
			newvalue = 2 * (*top_edge_pxl) - (*top_data_pxl);
			*top_extrap_pxl = (newvalue < 0) ? 0 :
				((newvalue > 255) ? 255 : newvalue);
		    }
		    bot_edge_pxl++;
		    top_edge_pxl++;
		}
	    }

	    left_edge_pxl = &(rastptr[(-maskbelow) * nchan * xlinewidth]);
	    right_edge_pxl = &(rastptr[(-maskbelow) * nchan * xlinewidth
					    + xlinewidth - masksize]);
	    for (chan = 0; chan < nchan; chan++)
	    {
		for(y=0; y < ylen + masksize - 1; y++)
		{
		    for (i=1; i <= maskbelow; i++)
		    {
			left_data_pxl = left_edge_pxl + i;
			left_extrap_pxl = left_edge_pxl - i;
			newvalue = 2 * (*left_edge_pxl) - (*left_data_pxl);
			*left_extrap_pxl = (newvalue < 0) ? 0 :
				((newvalue > 255) ? 255 : newvalue);
		    }
		    for (i=1; i <= maskabove; i++)
		    {
			right_data_pxl = right_edge_pxl - i;
			right_extrap_pxl = right_edge_pxl + i;
			newvalue = 2 * (*right_edge_pxl) - (*right_data_pxl);
			*right_extrap_pxl = (newvalue < 0) ? 0 :
				((newvalue > 255) ? 255 : newvalue);
		    }
		    left_edge_pxl += xlinewidth;
		    right_edge_pxl += xlinewidth;
		}
	    }
	}

	for (y = 0; y < ylen; y++)
	{
	    rastptr = &(rasterbase[maxmaskbelow + (y+maxmaskbelow) *
				    xlinewidth * nchan]);
	    outrastptr = &(outrasterbase[maxmaskbelow + (y+maxmaskbelow) *
				    xlinewidth * nchan]);

	    for (chan = 0; chan < nchan; chan++)
	    {
		centerpxl = &(rastptr[chan * xlinewidth]);
		outpxl = &(outrastptr[chan * xlinewidth]);
		for(x=0; x < xlen; x++)
		{
		    total = 0.0;
		    for (i=0; i < origmasksize; i++)
		    {
			calcpxl = centerpxl + (i * maskskip - maskbelow)
				    * xlinewidth * nchan - maskbelow;
			maskmult = &(mask_mult_table[(i*origmasksize) << 8]);

			for (j=0; j < origmasksize; j++)
			{
			    /* look up multiply in maskmult table */
			    total += maskmult[ calcpxl[j*maskskip] ];
			    maskmult += 256;
			}
		    }

		    /* bound result to 0 to 255 */
		    *outpxl = (total < 0.0) ? 0 :
			    ((total > 255.0) ? 255 : (unsigned char) total );
		    centerpxl++;
		    outpxl++;
		}
	    }
	}
	/* swap inbase and out base pointers for next pass */
	tempbase = rasterbase;
	rasterbase = outrasterbase;
	outrasterbase = tempbase;
    }

    /****************************************************
     * Set up output scanline
     ****************************************************/
    if (rle_row_alloc( &outglobals, &scanout ))
    	fprintf(stderr, "smush: Ran out of malloc space\n");
    if (outglobals.sv_alpha)
	scanout--;	/* Use zero based (vs. -1 based) addressing */
    for ( chan=0; chan < nchan; chan++)
    {
	rows[chan] = scanout[chan];
    }

    sv_setup( RUN_DISPATCH, &outglobals );
    for (y = 0; y < ylen; y++)
    {
	rastptr = &(rasterbase[maxmaskbelow + (y+maxmaskbelow) *
				xlinewidth * nchan]);

	for (chan = 0; chan < nchan; chan++)
	{
	    outpxl = &(rastptr[chan * xlinewidth]);
	    for(x=0; x < xlen; x++)
	    {
		scanout[chan][x] = *outpxl;
		outpxl++;
	    }
	}
	sv_putrow( &rows[outglobals.sv_alpha], xlen, &outglobals );
    }

}

float *
gauss_mask(siz)
int siz;
{
    int i,j;
    float w[5];
    float *mptr;

    w[0] = 0.05;
    w[1] = 0.25;
    w[2] = 0.4;
    w[3] = 0.25;
    w[4] = 0.05;

    mptr = (float *) malloc( sizeof(float) * siz * siz );

    if (siz != 5)
    {
	fprintf(stderr,"mask size not 5\n");
	exit(-1);
    }

    for(i=0; i< 5; i++)
    {
	for(j=0;j<5;j++)
	{
	    mptr[j+i*siz] = w[j]*w[i];
	}
    }
    return mptr;
}

float *
read_mask(size, fname, no_norm)
int * size;
char * fname;
int no_norm;
{
    FILE * maskfile;
    int i,j,masksize;
    float *mptr, total;

    if ( fname != NULL && (maskfile = fopen( fname, "r" )) == NULL )
    {
	perror( fname );
	exit( 1 );
    }

    fscanf(maskfile, "%d", &masksize);

    mptr = (float *) malloc( sizeof(float) * masksize * masksize);

    total = 0.0;
    for (i=0; i< masksize; i++)
	for (j=0; j< masksize; j++)
	{
	    fscanf(maskfile, "%f", &(mptr[j+i*masksize]));
	    total += mptr[j+i*masksize];
	}

    if (!no_norm)
    {
	if (total <= 0.0)
	{
	    fprintf(stderr,"smush: Input mask total not valid\n");
	    exit(1);
	}
	/* normalize the mask */
	for (i=0; i< masksize; i++)
	    for (j=0; j< masksize; j++)
	    {
		mptr[j+i*masksize] /= total;
	    }
    }

    *size = masksize;

    return mptr;
}
