/* 
 * rlenoise.c - Add random (uniform) noise to an image.
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 * Spencer W. Thomas
 *
 * Usage:
 * 	rlenoise [-a amount] [file]
 *
 * Adds random noise to an image.  May be useful when the image will
 * be output to a quantized display, and the display program doesn't
 * understand how to compensate.  (That's why I wrote it, anyway.)
 * Optionally, can specify the amount of noise (peak-to-peak) to add.
 * Default is 4 (because that's what I needed.)
 */

#include <stdio.h>
#include <svfb_global.h>

/* Change this according the value on your system.  This is a crock. */
#define USE_RANDOM
#define	RANDOM_MAX	((double)(int)(((unsigned int)~0)>>1))

#define MALLOC_ERR {fprintf(stderr, "rlenoise: ran out of heap space\n");exit(-2);}

main( argc, argv )
char ** argv;
{
    int x, y, i;
    int noise_amount = 4;
    char * infname = NULL;
    FILE * infile = stdin;
    struct sv_globals inglobals, outglobals;
    rle_pixel **inrows, **outrows;
    double rand_mult, newpix;

#ifdef USE_RANDOM
    /* Use the BSD random() function */
    {
	long seed;
	(void)time( &seed );
	srandom( *(int *)&seed );
    }
#else
    /* use the standard Unix rand function if nothing better */
    {
	long seed;
	(void)time( &seed );
	srand( *(int *)&seed );
    }
#define random rand
#endif

    if ( scanargs( argc, argv, "% n%-amount!d file%s",
		   &noise_amount, &noise_amount, &infname ) == 0 )
	exit( 1 );

    rand_mult = noise_amount / RANDOM_MAX;
    noise_amount /= 2;

    /* Get input file */
    if ( infname != NULL && (infile = fopen( infname, "r" )) == NULL )
    {
	fprintf( stderr, "rlenoise: Can't open %s: ", infname );
	perror("");
	exit(-1);
    }
    inglobals.svfb_fd = infile;
   
    /* Try to open the RLE file */
    rle_get_setup_ok( &inglobals, "rlenoise", infname );

    /* Always write to stdout */
    outglobals = inglobals;
    outglobals.svfb_fd = stdout;

    /* Init output file */
    sv_setup( RUN_DISPATCH, &outglobals );

    /* Get some memory */
    if ( rle_row_alloc( &inglobals, &inrows ) )
	MALLOC_ERR;
    /* Set up the output pointers.  This is a pain. */
    if ( (outrows = ((rle_pixel **) malloc(sizeof(rle_pixel *)
					   * outglobals.sv_ncolors + 1 ) + 1))
	 == NULL )
	MALLOC_ERR;
    for ( i = -outglobals.sv_alpha; i < outglobals.sv_ncolors; i++ )
	if ( inrows[i] )
	    outrows[i] = inrows[i] + outglobals.sv_xmin;

    for ( y = inglobals.sv_ymin; y <= inglobals.sv_ymax; y++ )
    {
	rle_getrow( &inglobals, inrows );
	for ( i = 0; i < inglobals.sv_ncolors; i++ )
	    if ( inrows[i] != NULL )
		for ( x = inglobals.sv_xmin; x <= inglobals.sv_xmax; x++ )
		{
		    newpix = inrows[i][x] + random() * rand_mult -
			noise_amount + 0.5;
		    if ( newpix < 0 )
			newpix = 0;
		    if ( newpix > 255 )
			newpix = 255;
		    inrows[i][x] = newpix;
		}
	sv_putrow( outrows, outglobals.sv_xmax - outglobals.sv_xmin + 1,
		   &outglobals );
    }

    sv_puteof( &outglobals );
}
