/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */
/* 
 * comp.c - Digitial image compositor (The Poor Man's Pixar)
 * 
 * Author:	Rod Bogart and John W. Peterson
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Tue Feb 25 1986
 * Copyright (c) 1986 Rod Bogart and John W. Peterson
 * 
 *    For an explanation of the compositing algorithms, see 
 *    "Compositing Digital Images", by Porter and Duff,
 *    SIGGRAPH 84, p.255.
 */

#include <stdio.h>
#include <svfb_global.h>
#include <rle_getraw.h>

#define MAX(i,j)   ( (i) > (j) ? (i) : (j) )
#define MIN(i,j)   ( (i) < (j) ? (i) : (j) )
#define MALLOC_ERR {fprintf(stderr, "comp: ran out of heap space\n");exit(-2);}

#define NUM_OPS		9

#define CLEAR_OP 	0
#define OVER_OP		1	/* Operations */
#define IN_OP		2	/* (these must match the comp_ops table) */
#define OUT_OP		3
#define ATOP_OP		4
#define XOR_OP		5
#define PLUS_OP		6
#define MINUS_OP	7
#define DIFF_OP		8

#define IN_WINDOW(y,wind) ((y >= wind.sv_ymin) && (y <= wind.sv_ymax))

/* 
 * Global raw data structures for copy_scanline.
 */
rle_op ** Araw, **Braw;
int * Anraw, *Bnraw;
rle_pixel * non_zero_pixels;

main(argc, argv)
int	argc;
char	*argv[];
{
    FILE	*rleout;
    int         row, alph, i, j, k, temp;
    char	outname[255];
    int		xlen;
    struct	sv_globals Aglobals, Bglobals;
    struct 	sv_globals outglobals;
    rle_pixel	**Ascanline, **Bscanline;
    rle_pixel	**rows;
    rle_pixel	**scanout;
    register
      rle_pixel	*Ascan, *Bscan, *scan;
    register
      rle_pixel	Aalph, Balph;
    int		int_result;
    int 	op_flag = 0, op_code;
    char 	*Afilename, *Bfilename, *op_name;
    int 	Askip, Bskip;	/* Blank space counters for copy_scanline */
      
    static char	*comp_ops[NUM_OPS] =
	 { "clear",
	   "over",
	   "in",
	   "out",
	   "atop",
	   "xor",
	   "plus",
	   "minus",
	   "diff" };

    /*
     * Parse arguments and set up input and output files.
     *   -o : select operator other than OVER
     */

    if (scanargs(argc, argv, "% o%-op!s Afile!s Bfile!s",
                 &op_flag, &op_name, &Afilename, &Bfilename ) == 0)
    {
	exit(-1);
    }

    if (op_flag)
    {
	op_code = -1;
	for (i=0; i < NUM_OPS; i++)
	    if (strcmp(op_name, comp_ops[i]) == 0) op_code = i;

	if (op_code == -1)
	{
	    fprintf(stderr, "%s: Invalid compositor operation\n", op_name);
	    exit(-2);
	}
    }
    else
	op_code = OVER_OP;

    /* Look for special case filenames "-" */

    Aglobals = sv_globals; Bglobals = sv_globals;
    if (strcmp(Afilename, "-") == 0)
	Aglobals.svfb_fd = stdin;
    else
    {
	Aglobals.svfb_fd = fopen(Afilename, "r");
	if (!Aglobals.svfb_fd)
	{
	    fprintf(stderr, "File %s not found \n", Afilename);
	    exit( -2 );
	}
    }

    if (strcmp(Bfilename, "-") == 0)
    {
	if ( Aglobals.svfb_fd == stdin )
	{
	    fprintf( stderr, "Can't read both inputs from stdin!\n" );
	    exit( -2 );
	}
	Bglobals.svfb_fd = stdin;
    }
    else
    {
	Bglobals.svfb_fd = fopen(Bfilename, "r");
	if (!Bglobals.svfb_fd)
	{
	    fprintf(stderr, "File %s not found \n", Bfilename);
	    exit(-2);
	}
    }

    rle_get_setup_ok( &Aglobals, "comp", Afilename );
    rle_get_setup_ok( &Bglobals, "comp", Bfilename );

    outglobals = Aglobals;
    outglobals.sv_xmin = MIN( Aglobals.sv_xmin, Bglobals.sv_xmin );
    outglobals.sv_ymin = MIN( Aglobals.sv_ymin, Bglobals.sv_ymin );
    outglobals.sv_xmax = MAX( Aglobals.sv_xmax, Bglobals.sv_xmax );
    outglobals.sv_ymax = MAX( Aglobals.sv_ymax, Bglobals.sv_ymax );

    outglobals.sv_alpha = 1;
    outglobals.sv_ncolors = MAX(Aglobals.sv_ncolors, Bglobals.sv_ncolors);

    outglobals.svfb_fd = stdout;
    xlen = outglobals.sv_xmax - outglobals.sv_xmin + 1;

    /* Enable all channels in output file */

    for (i = -outglobals.sv_alpha; i < outglobals.sv_ncolors; i++)
        SV_SET_BIT( outglobals, i );

    sv_setup( RUN_DISPATCH, &outglobals );

    /*
     * Allocate row storage
     */

    if (rle_row_alloc( &outglobals, &Ascanline ))
	MALLOC_ERR;
    bzero( Ascanline[SV_ALPHA], outglobals.sv_xmax + 1 );

    if (rle_row_alloc( &outglobals, &Bscanline ))
	MALLOC_ERR;
    bzero( Bscanline[SV_ALPHA], outglobals.sv_xmax + 1 );

    if (rle_row_alloc( &outglobals, &scanout ))
	MALLOC_ERR;

    if (!(rows = (rle_pixel **) malloc(sizeof(rle_pixel *)
        * outglobals.sv_ncolors+outglobals.sv_alpha)))
	MALLOC_ERR;

    /*
     * Allocate raw storage
     */
    if (rle_raw_alloc( &outglobals, &Araw, &Anraw ))
	MALLOC_ERR;

    if (rle_raw_alloc( &outglobals, &Braw, &Bnraw ))
	MALLOC_ERR;

    if (!(non_zero_pixels = (rle_pixel *)malloc( xlen * sizeof( rle_pixel ))))
	MALLOC_ERR;

    Askip = 0;			/* Initialize counters for copy_scanline */
    Bskip = 0;

    /*
     * Loop through all (possible) scanlines in the output file,
     * compositing each one.
     */

    for ( j = outglobals.sv_ymin; j <= outglobals.sv_ymax ; j++)
    {
	/* 
	 * Special case - if this scanline is in picture A but not
	 * B, don't do the compositing arithmaticly - just copy (or
	 * don't copy) the picture information, depending on the
	 * operation.
	 */

	if (IN_WINDOW(j, Aglobals) && !IN_WINDOW(j, Bglobals))
	{
	    switch (op_code) {
	    case OVER_OP:
	    case OUT_OP:
	    case XOR_OP:
	    case PLUS_OP:
	    case MINUS_OP:
	    case DIFF_OP:
		/**** Read the A channel and dump it ****/
		copy_scanline( &Aglobals, &outglobals, j,
			       &Askip, Araw, Anraw, 0 );
		break;

	    case ATOP_OP:
	    case IN_OP:
		/* Read the A channel, but dump one blank scanline */
		copy_scanline( &Aglobals, &outglobals, j,
			       &Askip, Araw, Anraw, 1 );
		break;
	    }
	}
	else
	if ((!IN_WINDOW(j, Aglobals)) && IN_WINDOW(j, Bglobals))

	/* As above - special case */

	{
	    switch (op_code) {
	    case OVER_OP:
	    case ATOP_OP:
	    case XOR_OP:
	    case PLUS_OP:
	    case MINUS_OP:
	    case DIFF_OP:
		/**** Read the B channel and dump it ****/
		copy_scanline( &Bglobals, &outglobals, j,
			       &Bskip, Braw, Bnraw, 0 );
		break;

	    case OUT_OP:
	    case IN_OP:
		/* Read the B channel, but dump one blank scanline */
		copy_scanline( &Bglobals, &outglobals, j,
			       &Bskip, Braw, Bnraw, 1 );
		break;
	    }
	}
	else if (!IN_WINDOW(j, Aglobals) && !IN_WINDOW(j, Bglobals))
	{
	    sv_putrow( NULL, 1, &outglobals );
	}
	else
	{
	    /**** Read the A channel  ****/
	    get_scanline( &Aglobals, Ascanline, &Askip, Araw, Anraw );

	    /**** Read the B channel  ****/
	    get_scanline( &Bglobals, Bscanline, &Bskip, Braw, Bnraw );

	    /* For each channel... */
	    for( k = SV_ALPHA; k < outglobals.sv_ncolors; k++)
	    {
		Ascan = &Ascanline[k][outglobals.sv_xmin];
		Bscan = &Bscanline[k][outglobals.sv_xmin];
		scan = &scanout[k][outglobals.sv_xmin];

		for( i = outglobals.sv_xmin; i <= outglobals.sv_xmax;
		    i++, Ascan++, Bscan++, scan++)
		{
		    Aalph = Ascanline[SV_ALPHA][i];
		    Balph = Bscanline[SV_ALPHA][i];

		    switch (op_code) {

		    /* Note OVER has been optimized for special cases */
		    case OVER_OP:	/* cA * 1.0 + cB * (1-alphaA) */
			if (Aalph == 0)
			    int_result = *Bscan;
			else if (Aalph == 255)
			    int_result = *Ascan;
			else
			    int_result = ( *Ascan * 255 +
				    *Bscan * (255 - Aalph))/255;
			break;

		    case IN_OP:		/* cA * alphaB + cB * 0.0 */
			int_result = ( *Ascan * Balph ) /255;
			break;

		    case OUT_OP:	/* cA * (1-alphaB) + cB * 0.0 */
			int_result = ( *Ascan * (255 - Balph) ) /255;
			break;

		    case ATOP_OP:	/* cA * alphaB + cB * (1-alphaA) */
			int_result = ( *Ascan * Balph +
				*Bscan * (255 - Aalph) )/255;
			break;

		    case XOR_OP:       /* cA * (1-alphaB) + cB * (1-alphaA) */
			int_result = (*Ascan * (255 - Balph) + *Bscan *
				(255 - Aalph) )/255;
			break;

		    case PLUS_OP:
			int_result = ((temp = ((int)*Ascan + (int)*Bscan))
				 > 255) ? 255 : temp;
			break;

		    /* minus is intended for subtracting images only, so
		     * the alpha channel is explicitly set to 255.
		     */
		    case MINUS_OP:
			if (k == SV_ALPHA)
			    int_result = 255;
			else
			    int_result = ((temp = ((int)*Ascan - (int)*Bscan))
				      < 0) ? 0 : temp;
			break;

		    case DIFF_OP:
			int_result = abs((int)*Ascan - (int)*Bscan);
			break;
		    }

		    *scan = (rle_pixel) ((int_result > 255) ? 255 : 
					 ((int_result < 0) ? 0 : int_result));
		}
	    }

	    /* Write out the composited data */

	    for( i = 0; i < outglobals.sv_ncolors+outglobals.sv_alpha; i++ )
		rows[i] = &scanout[i-1][outglobals.sv_xmin];
	    sv_putrow( &rows[1], xlen, &outglobals );
	}
    }
    sv_puteof( &outglobals );
    exit( 0 );
}

/*
 * Read a scanline from an RLE file.  Fake up an alpha channel (from non-
 * background pixels) if alpha isn't present.
 */
get_scanline( globals, scanline, num_skip, tmp_raw, tmp_nraw )
struct sv_globals * globals;
rle_pixel **scanline;
int * num_skip;
rle_op ** tmp_raw;		/* Raw pointers for data left behind by */
int * tmp_nraw;			/* copy_scanline. */
{
    int i,j,no_backgr;

    if (*num_skip > 0)		/* Generate a blank (skipped) scanline */
    {
	for( i = SV_ALPHA; i < globals->sv_ncolors; i++ )
	    bzero( scanline[i], globals->sv_xmax );
	(*num_skip)--;
	if (*num_skip == 0)
	    *num_skip = -1;	/* Flag raw data available */
	return;
    }

    if (*num_skip < 0)		/* num_skip < 0, use stashed raw data */
    {
	rle_rawtorow( globals, tmp_raw, tmp_nraw, scanline );
	*num_skip = 0;
    }
    else
	rle_getrow(globals, scanline );

    /* If no alpha channel, then fake one up from non-background pixels */

    if (!globals->sv_alpha)
	for( i = globals->sv_xmin; i <= globals->sv_xmax; i++)
	{
	    no_backgr = 0;
	    for( j = 0; j < globals->sv_ncolors; j++)
	    {
		no_backgr = no_backgr || (scanline[j][i] != 0);
	    }
	    if (no_backgr)
	    {
		scanline[SV_ALPHA][i] = 255;
	    }
	    else
	    {
		scanline[SV_ALPHA][i] = 0;
	    }
	}
}

/*
 * Read a scanline from an RLE file in raw mode, because we are about to dump
 * it to the outfile.  Fake up an alpha channel (from non-background pixels)
 * if alpha isn't present.
 */
copy_scanline( inglobals, outglobals, ypos, num_skip, out_raw, out_nraw, 
	       blank_output )
struct sv_globals * inglobals, * outglobals;
int ypos;
int *num_skip;			/* Number of scanlines to be skipped. */
rle_op ** out_raw;
int * out_nraw;
int blank_output;		/* if non-zero, just eat input & blank output */
{
    register int i,j;
    register rle_pixel * ptr;
    int chan, fakeruns, xlen;

    xlen = inglobals->sv_xmax - inglobals->sv_xmin + 1;

    /*
     * If the skip counter == 0, then we read the next line normally.
     * If it's positive, then it tells us how many blank lines before
     * the next available data.  If it's negative, it flags that
     * out_raw contains data to be used.
     */

 SKIP_ROW:
    if (*num_skip > 0)		/* We're in a blank space, output blanks */
    {
	sv_skiprow( outglobals, 1 );
	(*num_skip)--;
	if (! *num_skip)
	    *num_skip = -1;	/* Flag data available. */
	return;
    }
    
    if (! *num_skip)		/* num_skip == 0, must read data... */
	*num_skip = rle_getraw( inglobals, out_raw, out_nraw );
    else
	*num_skip = ypos;	/* num_skip < 0, data was already there */

    if (*num_skip == 32768)	/* EOF, just quit. */
    {
	sv_skiprow( outglobals, 1 );
	return;
    }

    *num_skip -= ypos;
    if ( *num_skip > 0 )
	goto SKIP_ROW;		/* It happens to the best of us... */

    if (!blank_output)
    {
	/*
	 * If no alpha channel, then fake one up from non-background pixels.
	 *
	 * This is not the most intelligent way to do this.  It's
	 * possible to look at the rle_ops directly, and do a set
	 * union of them to produce the fake alpha channel rle_ops.
	 * The algorithm to do this is not obvious to the casual
	 * observer.  If you want to take a crack at it, look at
	 * lib/sv_putrow.c (the findruns routine) for hints.
	 */

	if (! inglobals->sv_alpha )
	{
	    /*
	     * Create a "bytemask" of the non-zero pixels.
	     */
	    bzero( non_zero_pixels, xlen );
	    for (chan = 0; chan < inglobals->sv_ncolors; chan++ )
		for (i = 0; i < out_nraw[chan]; i++)
		    if (out_raw[chan][i].opcode == RByteDataOp ||
			out_raw[chan][i].u.run_val != 0)
		    {
			for (ptr = &(non_zero_pixels[out_raw[chan][i].xloc]),
			     j = out_raw[chan][i].length;
			     j > 0;
			     j--)
			    *(ptr++) = (rle_pixel) 255;
		    }
	    /*
	     * Collect the bytemask into real opcodes.  Assume that this won't
	     * be fragmented into byte data (it doesn't check).
	     */
	    fakeruns = 0;
	    i = 0;
	
	    while ( i < xlen )
	    {
		j = 0;
		out_raw[SV_ALPHA][fakeruns].opcode = RRunDataOp;
		out_raw[SV_ALPHA][fakeruns].u.run_val = (rle_pixel) 255;
	    
		while ( (non_zero_pixels[i] == (rle_pixel) 0) && (i < xlen) )
		    i++;
		out_raw[SV_ALPHA][fakeruns].xloc = i;

		while( (non_zero_pixels[i] != (rle_pixel) 0) && (i < xlen) )
		    j++, i++;
		out_raw[SV_ALPHA][fakeruns].length = j;
		if (j) fakeruns++;
	    }
	    out_nraw[SV_ALPHA] = fakeruns;
	}
		
	/* dump the raw stuff to the output file */
	sv_putraw( out_raw, out_nraw, outglobals );
    }
    else
	sv_putrow( NULL, 1, outglobals );

    rle_freeraw( outglobals, out_raw, out_nraw );
}

/*****************************************************************
 * TAG( rle_rawtorow )
 * 
 * Convert a "raw" scanline to a row format.  It is not particularly
 * clever about it, since comp does this infrequently (i.e., only when
 * a data stream skips into the region occupied by another.)  If this
 * were to be a generalized routine, it should take into account
 * channel masks and backgrounds.
 */
    
rle_rawtorow(globals, raw, nraw, outrows)
struct sv_globals * globals;
rle_op ** raw;
int * nraw;
rle_pixel ** outrows;
{
    register int i, j;
    register rle_pixel * outptr;
    int chan;
    
    for (chan = -globals->sv_alpha; chan < globals->sv_ncolors; chan++)
    {
	bzero( outrows[chan], globals->sv_xmax );
	for( i = 0; i < nraw[chan]; i++ )
	{
	    outptr = &(outrows[chan][raw[chan][i].xloc]);
	    switch (raw[chan][i].opcode)
	    {
	    case RByteDataOp:
		bcopy( raw[chan][i].u.pixels, outptr, raw[chan][i].length );
		break;

	    case RRunDataOp:
		for ( j = raw[chan][i].length; j > 0; j--)
		    *(outptr++) = (rle_pixel) raw[chan][i].u.run_val;
		break;
	    }
	}
    }
}
