/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 *  Modified at BRL 16-May-88 by Mike Muuss to avoid Alliant STDC desire
 *  to have all "void" functions so declared.
 */
/* 
 * rle_getrow.c - Read an RLE file in.
 * 
 * Author:	Spencer W. Thomas
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Wed Apr 10 1985
 * Copyright (c) 1985 Spencer W. Thomas
 * 
 * $Header: rle_getrow.c,v 1.12 86/11/10 12:30:34 thomas Exp $
 * $Log:	rle_getrow.c,v $
 * Revision 1.12  86/11/10  12:30:34  thomas
 * NULL comments field if no comment in input file.
 * 
 * Revision 1.11  86/11/10  11:27:01  thomas
 * Remove infile arguments.
 * 
 * Revision 1.10  86/11/10  11:19:04  thomas
 * Fix to work with pipe input.
 * 
 * Revision 1.9  86/10/08  13:06:32  thomas
 * Add header comments.
 * Add EOF opcode.
 * 
 * Revision 1.6  86/02/27  10:05:55  thomas
 * Change for new sv_globals interface.  Put "static" vars in structure.
 * 
 * Revision 1.5  86/02/25  17:32:28  thomas
 * Add debugging.
 * 
 * Revision 1.4  86/02/11  11:38:09  thomas
 * Oops: Only did clear to background right for ncolors = 3.
 * Some lint fixes.
 * 
 * Revision 1.3  86/02/11  11:27:52  thomas
 * Handle alpha channel.  Use bfill routine for efficiency.
 * 
 * Revision 1.2  85/05/22  15:09:23  thomas
 * Swap magic number on BIGENDIAN machines.
 * 
 * Revision 1.1  85/05/22  12:36:21  thomas
 * Initial revision
 * 
 */
#ifndef lint
static char rcs_ident[] = "$Header: rle_getrow.c,v 1.12 86/11/10 12:30:34 thomas Exp $";
#endif

#include "stdio.h"
#include "svfb_global.h"
#include "XtndRunsv.h"

/*
 * Automatically define LITTLE_ENDIAN on vax and pdp11 machines
 */
#if defined(vax) || defined(pdp11)
#define	LITTLE_ENDIAN
#endif

extern char * malloc();

struct inst {
	short opcode:8,
	    datum:8;
};

#define BREAD(type, var, len)\
	if ( infile->_cnt < len )\
	    fread( (char *)&var, 1, len, infile );\
	else\
	{\
	    var = *(type *)infile->_ptr;\
	    infile->_ptr += len;\
	    infile->_cnt -= len;\
	}

#define OPCODE(inst) (inst.opcode & ~LONG)
#define LONGP(inst) (inst.opcode & LONG)
#define DATUM(inst) (0x00ff & inst.datum)

static int	   debug_f;		/* if non-zero, print debug info */
static void	bfill();

/*****************************************************************
 * TAG( rle_get_setup )
 * 
 * Read the initialization information from an RLE file.
 * Inputs:
 * 	globals:    Contains pointer to the input file.
 * Outputs:
 * 	globals:    Initialized with information from the
 *		    input file.
 *	Returns 0 on success, -1 if the file is not an RLE file,
 *	-2 if malloc of the color map failed, -3 if an immediate EOF
 *	is hit (empty input file), and -4 if an EOF is encountered reading
 *	the setup information.
 * Assumptions:
 * 	infile points to the "magic" number in an RLE file (usually
 * byte 0 in the file).
 * Algorithm:
 * 	Read in the setup info and fill in sv_globals.
 */
rle_get_setup( globals )
struct sv_globals * globals;
{
    struct XtndRsetup setup;
    short magic;			/* assume 16 bits */
    register FILE *infile = globals->svfb_fd;
    rle_pixel * bg_color;
    register int i;
    char * comment_buf;

    BREAD( short, magic, sizeof magic );
    SWAB(magic);
    if ( feof( infile ) )
	return -3;
    if ( magic != XtndRMAGIC )
	return -1;
    BREAD( struct XtndRsetup, setup, SETUPSIZE );  /* assume VAX packing */
    if ( feof( infile ) )
	return -4;
    SWAB( setup.h_xpos );
    SWAB( setup.h_ypos );
    SWAB( setup.h_xlen );
    SWAB( setup.h_ylen );

    /* Extract information from setup */
    globals->sv_ncolors = setup.h_ncolors;
    for ( i = 0; i < globals->sv_ncolors; i++ )
	SV_SET_BIT( *globals, i );

    if ( !(setup.h_flags & H_NO_BACKGROUND) )
    {
	globals->sv_bg_color = (int *)malloc(
	    (unsigned)(sizeof(int) * setup.h_ncolors) );
	bg_color = (rle_pixel *)malloc(
	    (unsigned)(1 + (setup.h_ncolors / 2) * 2) );
	fread( (char *)bg_color, 1, 1 + (setup.h_ncolors / 2) * 2, infile );
	for ( i = 0; i < setup.h_ncolors; i++ )
	    globals->sv_bg_color[i] = bg_color[i];
	free( bg_color );
    }
    else
	getc( infile );			/* skip filler byte */

    if ( setup.h_flags & H_NO_BACKGROUND )
	globals->sv_background = 0;
    else if ( setup.h_flags & H_CLEARFIRST )
	globals->sv_background = 2;
    else
	globals->sv_background = 1;
    if ( setup.h_flags & H_ALPHA )
    {
	globals->sv_alpha = 1;
	SV_SET_BIT( *globals, SV_ALPHA );
    }
    else
	globals->sv_alpha = 0;

    globals->sv_xmin = setup.h_xpos;
    globals->sv_ymin = setup.h_ypos;
    globals->sv_xmax = globals->sv_xmin + setup.h_xlen - 1;
    globals->sv_ymax = globals->sv_ymin + setup.h_ylen - 1;

    globals->sv_ncmap = setup.h_ncmap;
    globals->sv_cmaplen = setup.h_cmaplen;
    if ( globals->sv_ncmap > 0 )
    {
	register int maplen =
		     globals->sv_ncmap * (1 << globals->sv_cmaplen);
	globals->sv_cmap = (rle_map *)malloc(
	    (unsigned)(sizeof(rle_map) * maplen) );
	if ( globals->sv_cmap == NULL )
	{
	    fprintf( stderr,
		"Malloc failed for color map of size %d*%d in rle_get_setup\n",
		globals->sv_ncmap, (1 << globals->sv_cmaplen) );
	    return -2;
	}
	fread( (char *)globals->sv_cmap, sizeof(short), maplen, infile );
#ifndef LITTLE_ENDIAN
    	/* Swap bytes on bigendian machines
	 */
    	for ( i = 0; i < maplen; i++ )
    	    SWAB( globals->sv_cmap[i] );
#endif
    }

    /* Check for comments */
    if ( setup.h_flags & H_COMMENT )
    {
	short comlen, evenlen;
	register char * cp;

	BREAD( short, comlen, sizeof comlen );	/* get comment length */
	SWAB( comlen );
	evenlen = (comlen + 1) & ~1;	/* make it even */
	comment_buf = malloc( (unsigned) evenlen );
	if ( comment_buf == NULL )
	{
	    fprintf( stderr,
		     "Malloc failed for comment buffer of size %d in rle_get_setup\n",
		     comlen );
	    return -2;
	}
	fread( comment_buf, 1, evenlen, infile );
	/* Count the comments */
	for ( i = 0, cp = comment_buf; cp < comment_buf + comlen; cp++ )
	    if ( *cp == 0 )
		i++;
	i++;			/* extra for NULL pointer at end */
	/* Get space to put pointers to comments */
	globals->sv_comments =
	    (char **)malloc( (unsigned)(i * sizeof(char *)) );
	if ( globals->sv_comments == NULL )
	{
	    fprintf( stderr,
		    "Malloc failed for %d comment pointers in rle_get_setup\n",
		     i );
	    return -2;
	}
	/* Get pointers to the comments */
	*globals->sv_comments = comment_buf;
	for ( i = 1, cp = comment_buf + 1; cp < comment_buf + comlen; cp++ )
	    if ( *(cp - 1) == 0 )
		globals->sv_comments[i++] = cp;
	globals->sv_comments[i] = NULL;
    }
    else
	globals->sv_comments = NULL;

    /* Initialize state for rle_getrow */
    globals->sv_private.get.scan_y = globals->sv_ymin;
    globals->sv_private.get.vert_skip = 0;
    globals->sv_private.get.is_eof = 0;
    globals->sv_private.get.is_seek = ftell( infile ) > 0;
    debug_f = 0;

    if ( !feof( infile ) )
	return 0;			/* success! */
    else
    {
	globals->sv_private.get.is_eof = 1;
	return -4;
    }
}


/*****************************************************************
 * TAG( rle_get_error )
 * 
 * Print an error message for the return code from rle_get_setup
 * Inputs:
 * 	code:		The return code from rle_get_setup.
 *	pgmname:	Name of this program (argv[0]).
 *	fname:		Name of the input file.
 * Outputs:
 * 	Prints an error message on standard output.
 *	Returns code.
 */

rle_get_error( code, pgmname, fname )
char *pgmname;
char *fname;
{
    switch( code )
    {
    case 0:			/* success */
	break;

    case -1:			/* Not an RLE file */
	fprintf( stderr, "%s: %s is not an RLE file\n",
		 pgmname, fname );
	break;

    case -2:			/* malloc failed */
	fprintf( stderr,
		 "%s: Malloc failed reading header of file %s\n",
		 pgmname, fname );
	break;

    case -3:
	fprintf( stderr, "%s: %s is an empty file\n", pgmname, fname );
	break;

    case -4:
	fprintf( stderr,
		 "%s: RLE header of %s is incomplete (premature EOF)\n",
		 pgmname, fname );
	break;

    default:
	fprintf( stderr, "%s: Error encountered reading header of %s\n",
		 pgmname, fname );
	break;
    }
    return code;
}


/*****************************************************************
 * TAG( rle_get_setup_ok )
 * 
 * Read the initialization information from an RLE file.
 * Inputs:
 * 	globals:    Contains pointer to the input file.
 *	prog_name:  Program name to be printed in the error message.
 *      file_name:  File name to be printed in the error message.
 *                  If NULL, the string "stdin" is generated.
 * Outputs:
 * 	globals:    Initialized with information from the
 *		    input file.
 *      If reading the globals fails, it prints an error message
 *	and exits with the appropriate status code.
 * Algorithm:
 * 	sv_get_setup does all the work.
 */
void
rle_get_setup_ok( globals, prog_name, file_name )
struct sv_globals * globals;
char *prog_name;
char *file_name;
{
    int code;

    if (! file_name)
	file_name = "stdin";

    code = rle_get_error( rle_get_setup( globals ), prog_name, file_name );
    if (code)
	exit( code );
}


/*****************************************************************
 * TAG( rle_debug )
 * 
 * Turn RLE debugging on or off.
 * Inputs:
 * 	on_off:		if 0, stop debugging, else start.
 * Outputs:
 * 	Sets internal debug flag.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
void
rle_debug( on_off )
int on_off;
{
    debug_f = on_off;
}


/*****************************************************************
 * TAG( rle_getrow )
 * 
 * Get a scanline from the input file.
 * Inputs:
 *	globals:    sv_globals structure containing information about 
 *		    the input file.
 * Outputs:
 * 	scanline:   an array of pointers to the individual color
 *		    scanlines.  Scanline is assumed to have
 *		    globals->sv_ncolors pointers to arrays of rle_pixel,
 *		    each of which is at least globals->sv_xmax+1 long.
 *	Returns the current scanline number.
 * Assumptions:
 * 	rle_get_setup has already been called.
 * Algorithm:
 * 	If a vertical skip is being executed, and clear-to-background is
 *	specified (globals->sv_background is true), just set the
 *	scanlines to the background color.  If clear-to-background is
 *	not set, just increment the scanline number and return.
 * 
 *	Otherwise, read input until a vertical skip is encountered,
 *	decoding the instructions into scanline data.
 */

rle_getrow( globals, scanline )
struct sv_globals * globals;
rle_pixel *scanline[];
{
    register rle_pixel * scanc;
    register int nc;
    register FILE *infile = globals->svfb_fd;
    int scan_x = globals->sv_xmin,	/* current X position */
	   channel = 0;			/* current color channel */
    short word, long_data;
    struct inst inst;

    /* Clear to background if specified */
    if ( globals->sv_background == 2 )
    {
	if ( globals->sv_alpha && SV_BIT( *globals, -1 ) )
	    bfill( (char *)scanline[-1], globals->sv_xmax + 1, 0 );
	for ( nc = 0; nc < globals->sv_ncolors; nc++ )
	    if ( SV_BIT( *globals, nc ) )
		bfill( (char *)scanline[nc], globals->sv_xmax+1,
			globals->sv_bg_color[nc] );
    }

    /* If skipping, then just return */
    if ( globals->sv_private.get.vert_skip > 0 )
    {
	globals->sv_private.get.vert_skip--;
	globals->sv_private.get.scan_y++;
	if ( globals->sv_private.get.vert_skip > 0 )
	    return globals->sv_private.get.scan_y;
    }

    /* If EOF has been encountered, return also */
    if ( globals->sv_private.get.is_eof )
	return ++globals->sv_private.get.scan_y;

    /* Otherwise, read and interpret instructions until a skipLines
     * instruction is encountered.
     */
    if ( SV_BIT( *globals, channel ) )
	scanc = scanline[channel] + scan_x;
    else
	scanc = NULL;
    for (;;)
    {
        BREAD(struct inst, inst, 2 );
	if ( feof(infile) )
	{
	    globals->sv_private.get.is_eof = 1;
	    break;		/* <--- one of the exits */
	}

	switch( OPCODE(inst) )
	{
	case RSkipLinesOp:
	    if ( LONGP(inst) )
	    {
	        BREAD( short, long_data, sizeof long_data );
		SWAB( long_data );
		globals->sv_private.get.vert_skip = long_data;
	    }
	    else
		globals->sv_private.get.vert_skip = DATUM(inst);
	    if (debug_f)
		fprintf(stderr, "Skip %d Lines (to %d)\n",
			globals->sv_private.get.vert_skip,
			globals->sv_private.get.scan_y +
			    globals->sv_private.get.vert_skip );

	    break;			/* need to break for() here, too */

	case RSetColorOp:
	    channel = DATUM(inst);	/* select color channel */
	    if ( channel == 255 )
		channel = -1;
	    scan_x = globals->sv_xmin;
	    if ( SV_BIT( *globals, channel ) )
		scanc = scanline[channel]+scan_x;
	    if ( debug_f )
		fprintf( stderr, "Set color to %d (reset x to %d)\n",
			 channel, scan_x );
	    break;

	case RSkipPixelsOp:
	    if ( LONGP(inst) )
	    {
	        BREAD( short, long_data, sizeof long_data );
		SWAB( long_data );
		scan_x += long_data;
		scanc += long_data;
		if ( debug_f )
		    fprintf( stderr, "Skip %d pixels (to %d)\n",
			    long_data, scan_x );
			 
	    }
	    else
	    {
		scan_x += DATUM(inst);
		scanc += DATUM(inst);
		if ( debug_f )
		    fprintf( stderr, "Skip %d pixels (to %d)\n",
			    DATUM(inst), scan_x );
	    }
	    break;

	case RByteDataOp:
	    if ( LONGP(inst) )
	    {
	        BREAD( short, long_data, sizeof long_data );
		SWAB( long_data );
		nc = (int)long_data;
	    }
	    else
		nc = DATUM(inst);
	    nc++;
	    if ( SV_BIT( *globals, channel ) )
	    {
		fread( (char *)scanc, 1, nc, infile );
		if ( nc & 1 )
		    (void)getc( infile );	/* throw away odd byte */
	    }
	    else
		if ( globals->sv_private.get.is_seek )
		    fseek( infile, ((nc + 1) / 2) * 2, 1 );
		else
		{
		    register int ii;
		    for ( ii = ((nc + 1) / 2) * 2; ii > 0; ii-- )
			(void) getc( infile );	/* discard it */
		}

	    scanc += nc;
	    scan_x += nc;
	    if ( debug_f )
		if ( SV_BIT( *globals, channel ) )
		{
		    rle_pixel * cp = scanc - nc;
		    fprintf( stderr, "Pixel data %d (to %d):", nc, scan_x );
		    for ( ; nc > 0; nc-- )
			fprintf( stderr, "%02x", *cp++ );
		    putc( '\n', stderr );
		}
	    else
		fprintf( stderr, "Pixel data %d (to %d)\n", nc, scan_x );
	    break;

	case RRunDataOp:
	    if ( LONGP(inst) )
	    {
	        BREAD( short, long_data, sizeof long_data );
		SWAB( long_data );
		nc = long_data;
	    }
	    else
		nc = DATUM(inst);
	    scan_x += nc + 1;

	    BREAD( short, word, sizeof(short) );
	    SWAB( word );
	    if ( debug_f )
		fprintf( stderr, "Run length %d (to %d), data %02x\n",
			    nc + 1, scan_x, word );
	    if ( SV_BIT( *globals, channel ) )
	    {
		if ( nc >= 10 )		/* break point for 785, anyway */
		{
		    bfill( (char *)scanc, nc + 1, word );
		    scanc += nc + 1;
		}
		else
		    for ( ; nc >= 0; nc--, scanc++ )
			*scanc = word;
	    }
	    break;

	case REOFOp:
	    globals->sv_private.get.is_eof = 1;
	    break;

	default:
	    fprintf( stderr,
		     "rle_getrow: Unrecognized opcode: %d\n", inst.opcode );
	    exit(1);
	}
	if ( OPCODE(inst) == RSkipLinesOp || OPCODE(inst) == REOFOp )
	    break;			/* <--- the other loop exit */
    }

    return globals->sv_private.get.scan_y;
}


/* Fill buffer at s with n copies of character c.  N must be <= 65535*/
/* ARGSUSED */
static void bfill( s, n, c )
char *s;
int n, c;
{
#ifdef vax
    asm("   movc5   $0,*4(ap),12(ap),8(ap),*4(ap)");
#else
    while ( n-- > 0 )
	*s++ = c;
#endif
}
