/* 
 * flip_book.c - Display movies on Apollo DN660 and 560 displays.
 * 
 * Author:	John W. Peterson
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Thu Apr 16 1987
 * Copyright (c) 1987, University of Utah
 *
 * This clever chunk of code turns the apollo's 1Kx1Kx8 (or 1Kx2K!)
 * frame buffer into a gigantic "flip-book".  The screen is first
 * divided into evenly sized (power of 2) sub-pictures.  The frames
 * are sequentially written into these blocks right-to-left, top to
 * bottom.  As each page is filled, it starts onto the next.  The
 * screen is zoomed up so one of these fills the screen.  The
 * 0,0 frame is NOT written with any data; when the flip-book is
 * flipped, data from the other frames is BLTed into this one.
 */

#include "/sys/ins/base.ins.c"
#include "/sys/ins/gpr.ins.c"
#include "/sys/ins/error.ins.c"
#include "/sys/ins/pgm.ins.c"
#include "/sys/ins/time.ins.c"
#include "/sys/ins/cal.ins.c"
#include "/sys/ins/kbd.ins.c"

#include <stdio.h>
#include "svfb_global.h"

#define raster_width	1024
#define raster_height	2048
#define bit_planes	8

/*
 * derived constants
 */

#define frames_across  (raster_width / frame_size)
#define frames_down  (raster_height / frame_size)
#define frames_per_plane  (frames_across * frames_down)

#define film_speed 10417  /*  close to: ( (1/24) * 1000000 ) / 4 */
#define video_speed 8333  /*  close to: ( (1/30) * 1000000 ) / 4 */
#define very_fast 4000    /* Approx 1/60 sec. */

/*
 * Globals
 */
int ticks_per_frame = film_speed;
int bidirection_mode = false;
int forward = true;
int step_mode = false;

status_$t status;
int	scale_factor;		/* amount to divide file data */
rle_pixel ** rows = NULL;	/* Allocated once. */
int	frame_size;		/* Size image is on screen. */

gpr_$pixel_array_t dest_pixels;

/*
 * Check GPR return status codes.
 */
#define check(msg) if (status.all != status_$ok) \
    { error_$std_format( status, msg ); pgm_$exit; }

/*
 * Wake up the graphics.
 */
init()
{
    gpr_$offset_t size;
    gpr_$bitmap_desc_t bitmap;
    int i;
    gpr_$color_vector_t colmap;

    /*  nice error msgs */
    error_$init_std_format( stream_$errout, '?', "flip_book", 9 );

    size.x_size=raster_width; size.y_size=raster_height;

    gpr_$init(gpr_$borrow,
	      1, size, bit_planes - 1, bitmap, status);  /* Borrow the screen*/

    check("Getting screen %$");

    /*  Strange GPR quirk:  Full size of screen must bet snatched by hand */

    gpr_$set_bitmap_dimensions( bitmap, size, bit_planes - 1,  status);
    check("Re-setting size %$");
    gpr_$clear( (gpr_$pixel_value_t) 0, status );
    check("Erasing... %$");	/*  Clear the new chunk we just got */

    /* Assume black and white linear map. */

    for (i = 0; i < 256; i++)
	colmap[i] = (i << 16) | (i << 8) | i;
    
    gpr_$set_color_map((gpr_$pixel_value_t) 0,
		       (short) 256, colmap, status );
    check("Set zero map%$");

    gpr_$set_clipping_active( true, status );
    check("Set clipping on%$");
}

/*
 * Event handling goodies - wait for key presses.
 */
event_wait()
{
    gpr_$event_t  event;
    char  eventdat;
    boolean  temp;
    gpr_$offset_t  size;	/* Size of allocated bitmap*/
    gpr_$position_t  pos;
    gpr_$keyset_t keys;
    short i;

    lib_$init_set( keys, 256 );

    /* Enable everything */
    for (i = 0; i <= (short) 255; i++) 
	lib_$add_to_set( keys, 256, i );
 
    gpr_$enable_input(gpr_$keystroke, keys, status);
    check("gpr enable input%$");
    temp=gpr_$event_wait(event, eventdat, pos, status);
    check("gpr event wait%$");

    step_mode = false;
    switch (eventdat)
    {
    case ' ':
	forward = true;
	step_mode = true;
	break;

    case 'q':
	exit(0);

    case 'f':
	ticks_per_frame = film_speed;
	break;

    case 'v':
	ticks_per_frame = video_speed;
	break;

    case 'F':
	ticks_per_frame = very_fast;
	break;

    case KBD_$CR:
	forward = false;
	step_mode = true;
	break;

    case 'b':
	bidirection_mode = ! bidirection_mode;
	break;
    }
}


/*
 * Initialize parameters and allocate scanlines based on the size and shape
 * of the first frame.
 * Note Well!  This assumes all the frames are the same size!
 */
init_first_frame()
{
    int i, j, c;
    int maplen, ncmap;
    rle_map * cmap;
    gpr_$color_vector_t colmap;

    rle_row_alloc( &sv_globals, &rows );

    /*
     * Find size and scale factors.
     */

    frame_size = raster_width;	/* Max allowable */
    if ( (sv_globals.sv_xmax + 1) > frame_size )
    {
	fprintf( stderr, "flip_book: images are too big!\n" );
	exit( -1 ) ;
    }
    
    while ( frame_size > (sv_globals.sv_xmax + 1) )
	frame_size = frame_size >> 1;

    scale_factor = raster_width / frame_size;

    /*
     * If the first frame has a color map, load it.
     */
    if ( sv_globals.sv_ncmap )
    {
	maplen = (1 << sv_globals.sv_cmaplen);
	ncmap = sv_globals.sv_ncmap;
	cmap = sv_globals.sv_cmap;

	/* 
	 * Assumes only top eight bits of RLE color map is significant
	 */

	if (ncmap == 3)
	    for ( i = 0; i < 256; i++ )
		colmap[i] = (cmap[i] << 8) | cmap[i+maplen]
		            | (cmap[i+2*maplen] >> 8);
	else
	    /* Just take the first channel, ignore the rest. */

	    for ( i = 0; i < 256; i++ )
		colmap[i] = (cmap[i] << 8) | cmap[i]
		            | (cmap[i] >> 8);

	gpr_$set_color_map((gpr_$pixel_value_t) 0,
			   (short) 256, colmap, status );
	check("Set zero map%$");
    }
}
    
/*
 * Load the files into the frame buffer.
 */
load_frames( files, nfiles )
char ** files;
int nfiles;
{
    int frame_no = 1;
    gpr_$window_t dest_box;
    register gpr_$pixel_value_t * dest_ptr;
    register rle_pixel * src_ptr;
    register int i;
    int y;

    while( nfiles )
    {
	/*
	 * Open the file.
	 */
	if ((sv_globals.svfb_fd = fopen(*files, "r")) == NULL)
	{
	    fprintf(stderr, "getap: ");
	    perror(*files);
	    exit(1);
	}
	switch( (int) rle_get_setup( &sv_globals ) )
	{
	case 0:
	    break;		/* successful open */
	case -1:		/* not an RLE file */
	case -4:		/* or header is short */
	    fprintf( stderr, "getap: %s is not an RLE file\n", *files );
	    exit(-4);
	    break;
	case -2:
	    fprintf( stderr, "getap: malloc failed\n" );
	    exit(-2);
	    break;
	case -3:
	    fprintf( stderr, "getap: input file is empty\n" );
	    exit(-3);
	    break;
	}

	if (! rows)
	    init_first_frame();

	/*
	 * We only want the red channel.
	 */

#ifdef REDONLY  /* W/old versions of librle this kills rle_getrow on DN660's */
	SV_CLR_BIT( sv_globals, SV_ALPHA );
	for ( i = 1; i < sv_globals.sv_ncolors; i++)
	    SV_CLR_BIT( sv_globals, i );
#endif
	
	select_frame( frame_no );

	/*
	 * Copy the scanlines to the frame buffer.
	 */

	dest_box.window_base.x_coord = sv_globals.sv_xmin;
	dest_box.window_size.y_size = 1;

	for ( y = sv_globals.sv_ymin; y <= sv_globals.sv_ymax; y++ )
	{
	    rle_getrow( &sv_globals, rows );
	    dest_ptr = dest_pixels;
	    src_ptr = rows[0];

	    for ( i = 0; i <= sv_globals.sv_xmax; i++ )
		*dest_ptr++ = (gpr_$pixel_value_t) *src_ptr++;
		
	    dest_box.window_base.y_coord = sv_globals.sv_ymax - y;
	    dest_box.window_size.x_size =
		sv_globals.sv_xmin + sv_globals.sv_xmax + 1;

	    gpr_$write_pixels( dest_pixels, dest_box, status );
	}

	/*
	 * Go to the next file.
	 */

	fclose( sv_globals.svfb_fd ); /* ..or else we run out. */
	files++;
	nfiles--;
	frame_no++;
    }
}
	 
	
/*
 * Select, via masks & origin, where frame_no will be.  This is for writing
 * the frames.  Never write a frame 0, that's the viewing frame.
 */

select_frame( frame_no )
int frame_no;
{
    short plane, row, col;
    gpr_$position_t  place;
    gpr_$window_t  clip_wind;

    row = frame_no / frames_across;
    col = frame_no % frames_across;

    place.x_coord = col * frame_size;
    place.y_coord = row * frame_size;

    gpr_$set_coordinate_origin( place, status );

    clip_wind.window_size.x_size = frame_size;
    clip_wind.window_size.y_size = frame_size;
    clip_wind.window_base = place;

    gpr_$set_clip_window( clip_wind, status );
}

/* 
 * Run through the flip book 
 */
see_frames( num_frames )
int num_frames;
{
    int i;
    gpr_$position_t  org;
    gpr_$bitmap_desc_t  bitmap;
    gpr_$window_t  frame_wind;
    time_$clock_t  delay_time, current_time;

    delay_time.low = ticks_per_frame; /* Set up 1/24 sec timing */
    delay_time.high = 0;

    org.x_coord = 0;
    org.y_coord = 0;

    gpr_$set_coordinate_origin( org, status );

    gpr_$inq_bitmap( bitmap, status ); /* source=dest bitmap */

    frame_wind.window_size.x_size = frame_size;
    frame_wind.window_size.y_size = frame_size;

    if (forward)
	i = 1;
    else
	i = num_frames;

    while ((i >= 1) && (i <= num_frames))
    {
	time_$clock( current_time );
	cal_$add_clock( current_time, delay_time );

	frame_wind.window_base.x_coord = ( i % frames_across) * frame_size;
	frame_wind.window_base.y_coord = ( i / frames_across) * frame_size;

	gpr_$pixel_blt( bitmap, frame_wind, org, status );
	check("Doing blt %$");

	time_$wait( time_$absolute, current_time, status );

	if (step_mode) event_wait();

	if (forward)
	    i++;
	else
	    i--;
    }
    if (bidirection_mode) forward = ! forward;
}

main(argc, argv)
int argc;
char **argv;
{
    int nfiles;
    char **filenames;

    init();			/* start graphics */

    if (! scanargs( argc, argv, "% files%*s", &nfiles, &filenames))
    {
	fprintf(stderr,
		"\nKeys: space = step forward, ret = step backward, q = quit\n\
f = film speed, v = video speed, F = way fast, b = bi-directional\n");
	exit( 1 );		/* Bad usage */
    }

    load_frames(filenames, nfiles);
    gpr_$color_zoom( (short) scale_factor, (short) scale_factor, status );
    gpr_$set_clipping_active( false, status );

    while( 1 )
    {
	see_frames( nfiles );
	event_wait();
    }
}
