/*
 * cone.c
 *
 * Copyright (C) 1989, Craig E. Kolb
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely .  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Id: cone.c,v 3.0 89/10/27 02:05:47 craig Exp $
 *
 * $Log:	cone.c,v $
 * Revision 3.0  89/10/27  02:05:47  craig
 * Baseline for first official release.
 * 
 */
#include <stdio.h>
#include <math.h>
#include "typedefs.h"
#include "funcdefs.h"
#include "constants.h"

Object *
makcone(surf, cent, ax, br, ar, trans)
char *surf;
Vector *cent, *ax;
double br, ar;
TransInfo *trans;
{
	Cone *cone;
	Primitive *prim;
	Object *newobj;
	extern int yylineno, Quiet;
	double len, dtmp;
	Vector axis, base, tmp;

	prim = mallocprim();
	prim->surf = find_surface(surf);
	prim->type = CONE;
	newobj = new_object(NULL, CONE, (char *)prim, (Trans *)NULL);
	cone = (Cone *)Malloc(sizeof(Cone));
	prim->objpnt.p_cone = cone;

	/*
	 * Cones are defined by a basepoint, an apex point, and
	 * base and apex radii.  The cone is stored as
	 * the origin of the cone, the change in radius per
	 * unit distance from the origin of the cone, the maximum z-value
	 * of the cone, and "start_pos",
	 * the distance along the axis from the cone origin where
	 * the first endcap appears (where the passed "basepoint"
	 * appears).
	 *
	 * The intcone() routine intersects a ray with a cone aligned
	 * along the Z axis.  Thus, we must define a transformation
	 * matrix which will transform an axis-aligned cone to the desired.
	 */

	/*
	 * The passed basepoint must be closer to the origin of the
	 * cone than the apex point, implying that the base radius
	 * must be smaller than the apex radius.  If the values passed
	 * reflect the opposite, we switch everything.
	 */
	if(ar < br) {
		tmp = *cent;
		*cent = *ax;
		*ax = tmp;
		dtmp = br;
		br = ar;
		ar = dtmp;
	} else if (equal(ar, br)) {
		/*
		 * If the base and apex radii are equal, then we
		 * can treat the cone as a cylinder.
		 */
		return makcyl(surf, cent, ax, br, trans);
	}
	/*
	 * Find the axis and axis length.
	 */
	vecsub(*ax, *cent, &axis);
	len = normalize(&axis);
	if (len < EPSILON) {
		if (!Quiet)
			fprintf(stderr,"Degenerate cone (line %d).\n",
							yylineno);
		free((char *)cone);
		free((char *)prim);
		free((char *)newobj);
		return (Object *)0;
	}
	cone->apex_rad = ar;
	/*
	 * "tantheta" is the change in radius per unit length along
	 * the cone axis.
	 */
	cone->tantheta = (ar - br) / len;
	/*
	 * Start pos defines the point along the axis where the first
	 * endcap should be placed.
	 */
	cone->start_pos = br / cone->tantheta;
	/*
	 * Find the true base (origin) of the cone.
	 */
	scalar_prod(-cone->start_pos, axis, &base);
	vecadd(base, *cent, &base);
	/*
	 * The apex endcap is placed cone->len units from the cone
	 * origin.
	 */
	cone->end_pos = cone->start_pos + len;
	/*
	 * Calculate rotation matrix to map from world space to cone space.
	 */
/*	if (equal(axis.z*axis.z, 1.)) {
		tmp.x = 0.;
		tmp.y = -axis.z;
		tmp.z = 0.;
	} else { */
		tmp.x = axis.y;
		tmp.y = -axis.x;
		tmp.z = 0.;
	/*} */
	rotate(trans, &tmp, acos(axis.z));
	translate(trans, &base);
	cone->tantheta *= cone->tantheta;

	return newobj;
}

/*
 * Ray-cone intersection test.  This routine is far from optimal, but
 * it's straight-forward and it works...
 */
double
intcone(pos, ray, obj)
Vector           *pos, *ray;
Primitive       *obj;
{
	double t1, t2, a, b, c, disc, zpos, et1, et2;
	double x, y;
	extern unsigned long primtests[];
	Cone *cone;

	primtests[CONE]++;
	cone = obj->objpnt.p_cone;

	/*
	 * Recall that 'tantheta' is really tantheta^2.
	 */
	a = ray->x * ray->x + ray->y * ray->y - ray->z*ray->z*cone->tantheta;
	b = ray->x * pos->x + ray->y * pos->y - cone->tantheta*ray->z*pos->z;
	c = pos->x*pos->x + pos->y*pos->y - cone->tantheta*pos->z*pos->z;

	if (equal(a, 0.)) {
		/*
		 * Only one intersection point...
		 */
		t1 = -c / b;
		zpos = pos->z + t1 * ray->z;
		if (t1 < EPSILON || zpos < cone->start_pos ||
		    zpos > cone->end_pos)
			t1 = FAR_AWAY;
		t2 = FAR_AWAY;
	} else {
		disc = b*b - a*c;
		if(disc < 0.)
			return 0.;		/* No possible intersection */
		disc = sqrt(disc);
		t1 = (-b + disc) / a;
		t2 = (-b - disc) / a;
		/*
		 * Clip intersection points.
		 */
		zpos = pos->z + t1 * ray->z;
		if (t1 < EPSILON || zpos < cone->start_pos ||
		    zpos > cone->end_pos)
			t1 = FAR_AWAY;
		zpos = pos->z + t2 * ray->z;
		if (t2 < EPSILON || zpos < cone->start_pos ||
		    zpos > cone->end_pos)
			t2 = FAR_AWAY;
	}
	/*
	 * Find t for both endcaps.
	 */
	et1 = (cone->start_pos - pos->z) / ray->z;
	x = pos->x + et1 * ray->x;
	y = pos->y + et1 * ray->y;
	if (x*x + y*y > cone->start_pos*cone->start_pos*cone->tantheta)
		et1 = FAR_AWAY;
	et2 = (cone->end_pos - pos->z) / ray->z;
	x = pos->x + et2 * ray->x;
	y = pos->y + et2 * ray->y;
	if (x*x + y*y > cone->end_pos*cone->end_pos*cone->tantheta)
		et2 = FAR_AWAY;

	t1 = min(t1, min(t2, min(et1, et2)));
	return (t1 == FAR_AWAY ? 0. : t1);
}

/*
 * Compute the normal to a cone at a given location on its surface.
 */
nrmcone(pos, obj, nrm)
Vector *pos, *nrm;
Primitive *obj;
{
	Cone *cone;

	cone = obj->objpnt.p_cone;

	if (equal(pos->z, cone->start_pos)) {
		nrm->x = nrm->y = 0.;
		nrm->z = -1.;
	} else if (equal(pos->z, cone->end_pos)) {
		nrm->x = nrm->y = 0.;
		nrm->z = 1.;
	} else {
		/*
		 * The following is equal to
		 * (pos X (0, 0, 1)) X pos
		 */
		nrm->x = pos->x * pos->z;
		nrm->y = pos->y * pos->z;
		nrm->z = -pos->x * pos->x - pos->y * pos->y;
	}
}

/*
 * Return the extent of a cone.
 */
coneextent(o, bounds)
Primitive *o;
double bounds[2][3];
{
	Cone *cone;

	cone = o->objpnt.p_cone;

	bounds[LOW][X] = bounds[LOW][Y] = -cone->apex_rad;
	bounds[HIGH][X] = bounds[HIGH][Y] = cone->apex_rad;
	bounds[LOW][Z] = cone->start_pos;
	bounds[HIGH][Z] = cone->end_pos;
}
