#include <stdio.h>
#include <quadtree.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>

extern char    *getenv(), *optarg, *malloc();
extern int      optind;
extern FILE    *fopen();

Display        *QTDisplay;
Window          QTWindow;
GC              WhiteRegion, BlackRegion;
int             Verbose;

struct Qentry {
    QT_TreeNode_t  *node;
    struct Qentry  *next, *prev;
    short           x, y, width, height, parent;
}              *Head, *Tail, *Freelist;

#define ColorsAgree(c1,c2) ((c1)?(c2):(!(c2)))

struct Qentry  *NewQentry()
{
    struct Qentry  *result, *ptr;
    int             i;

    if (result = Freelist) {
	Freelist = result->next;
	return (result);
    }
    result = (struct Qentry *) malloc(64 *
				      (sizeof(struct Qentry)));
    for (i = 0, ptr = result; i < 63; ++ptr, ++i) {
	ptr->next = ptr + 1;
    }
    ptr->next = Freelist;
    Freelist = result->next;
    return (result);
}

void            ReleaseQentry(qe)
struct Qentry  *qe;
{
    qe->next = Freelist;
    Freelist = qe;
}

main(argc, argv)
int             argc;
char          **argv;
{
    int             c, breadth = 0, screen;
    short           width, height;
    QT_TreeNode_t  *Tree;
    FILE           *fp;
    char           *displayname;
    Window          rwindow;
    XGCValues       gcv;

    Freelist = 0;
    Verbose = 0;
    while ((c = getopt(argc, argv, "bdv")) != EOF) {
	switch (c) {
	    case 'b':
		breadth = 1;
		break;
	    case 'd':
		breadth = 0;
		break;
	    case 'v':
		++Verbose;
		break;
	}
    }
    if (!(fp = fopen(argv[optind], "r"))) {
	exit(1);
    }
    Tree = QT_Tree_Read(fp, &width, &height);
    if (!(displayname = getenv("DISPLAY")))
	displayname = ":0";
    QTDisplay = XOpenDisplay(displayname);
    screen = DefaultScreen(QTDisplay);
    rwindow = RootWindow(QTDisplay, screen);
    QTWindow = XCreateSimpleWindow(QTDisplay, rwindow,
				   0, 0, width, height,
				   1, WhitePixel(QTDisplay, screen),
				   BlackPixel(QTDisplay, screen));
    XMapWindow(QTDisplay, QTWindow);
    XSync(QTDisplay, 0);
    gcv.foreground = WhitePixel(QTDisplay, screen);
    gcv.function = GXset;
    WhiteRegion = XCreateGC(QTDisplay, QTWindow,
			    GCFunction | GCForeground, &gcv);
    gcv.function = GXclear;
    BlackRegion = XCreateGC(QTDisplay, QTWindow,
			    GCFunction | GCForeground, &gcv);
    puts("Press ENTER to begin");
    while ('\n' != getchar());
    if (breadth)
	BreadthDraw(Tree, width, height);
    else
	DepthDraw(Tree, width, height);
    XSync(QTDisplay, 0);
    puts("Press ENTER to terminate");
    while ('\n' != getchar());
    exit(0);
}

depthdraw(node, x, y, w, h, parent)
QT_TreeNode_t  *node;
int             x, y, w, h, parent;
{
    int             color = QT_TreeNode_Color(node), w2, h2;
    QT_TreeNode_t  *child;

    if (!(ColorsAgree(color, parent))) {
	if (Verbose) {
	    fprintf(stderr, "Filling (%d,%d) (%dx%d) [%s]\n",
		    x, y, w, h,
		    color ? "black" : "white");
	}
	XFillRectangle(QTDisplay, QTWindow,
		       color ? BlackRegion : WhiteRegion,
		       x, y, w, h);
    }
    w2 = w >> 1;
    h2 = h >> 1;
    if (child = QT_TreeNode_UpperLeft(node)) {
	depthdraw(child, x, y, w2, h2, color);
    }
    if (child = QT_TreeNode_UpperRight(node)) {
	depthdraw(child, x + w2, y, w - w2, h2, color);
    }
    if (child = QT_TreeNode_LowerLeft(node)) {
	depthdraw(child, x, y + h2, w2, h - h2, color);
    }
    if (child = QT_TreeNode_LowerRight(node)) {
	depthdraw(child, x + w2, y + h2, w - w2, h - h2, color);
    }
}

DepthDraw(node, width, height)
QT_TreeNode_t  *node;
int             width, height;
{
    depthdraw(node, 0, 0, width, height, 1);
}

QueueInit()
{
    Head = Tail = 0;
}

QueueAdd(node, x, y, w, h, parent)
QT_TreeNode_t  *node;
int             x, y, w, h, parent;
{
    struct Qentry  *result = NewQentry();

    result->node = node;
    result->x = x;
    result->y = y;
    result->width = w;
    result->height = h;
    result->parent = parent;
    result->next = 0;
    result->prev = Tail;
    if (Tail)
	Tail->next = result;
    Tail = result;
    if (!Head)
	Head = result;
}

QueueDo()
{
    struct Qentry  *entry;
    QT_TreeNode_t  *node, *child;
    int             x, y, w, h, color, parent, w2, h2;

    for (entry = Head; entry; entry = entry->next) {
	if (entry->prev)
	    ReleaseQentry(entry->prev);
	node = entry->node;
	x = entry->x;
	y = entry->y;
	w = entry->width;
	h = entry->height;
	color = QT_TreeNode_Color(node);
	parent = entry->parent;
	if (!(ColorsAgree(color, parent))) {
	    if (Verbose) {
		fprintf(stderr, "Filling (%d,%d) (%dx%d) [%s]\n",
			x, y, w, h,
			color ? "black" : "white");
	    }
	    XFillRectangle(QTDisplay, QTWindow,
			   color ? BlackRegion : WhiteRegion,
			   x, y, w, h);
	}
	w2 = w >> 1;
	h2 = h >> 1;
	if (child = QT_TreeNode_UpperLeft(node)) {
	    QueueAdd(child, x, y, w2, h2, color);
	}
	if (child = QT_TreeNode_UpperRight(node)) {
	    QueueAdd(child, x + w2, y, w - w2, h2, color);
	}
	if (child = QT_TreeNode_LowerLeft(node)) {
	    QueueAdd(child, x, y + h2, w2, h - h2, color);
	}
	if (child = QT_TreeNode_LowerRight(node)) {
	    QueueAdd(child, x + w2, y + h2, w - w2, h - h2, color);
	}
    }
}

BreadthDraw(node, width, height)
QT_TreeNode_t  *node;
int             width, height;
{
    QueueInit();
    QueueAdd(node, 0, 0, width, height, 1);
    QueueDo();
}
