/*LINTLIBRARY*/

/*  @(#)atariterm.c 1.9 89/12/11
 *
 *  Popi graphics driver for atari TERM windows.
 *  Written by Stephen Frede, Softway Pty Ltd.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc. 
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */


#include <stdio.h>
#include <sys/types.h>
#include "popi.h"

thresh[BITSPERPIXEL][BITSPERPIXEL] =
{        /* Array containing threshold values. */
    {   0, 128,  32, 160,   8, 136,  40, 168, },
    { 192,  64, 224,  96, 200,  72, 232, 104, },
    {  48, 176,  16, 144,  56, 184,  24, 152, },
    { 240, 112, 208,  80, 248, 120, 216,  88, },
    {  12, 140,  44, 172,   4, 132,  36, 164, },
    { 204,  76, 236, 108, 196,  68, 228, 100, },
    {  60, 188,  28, 156,  52, 180,  20, 148, },
    { 252, 124, 220,  92, 244, 116, 212,  84, },
};

int	ImgInProgress = 0;

/*  There are ten exportable routines used by the popi program.
 *
 *  These are:
 *
 *  disp_init(argc, argv)    - called from main at the start.
 *  disp_finish()            - called from main prior to exit.
 *  disp_imgstart()          - called prior to drawing an image.
 *  disp_imgend()            - called after drawing an image.
 *  disp_putline(line, y)    - to draw an image scanline.
 *  disp_getchar()           - to get the next character typed.
 *  disp_ungetc(c)           - put back the last character typed.
 *  disp_prompt()            - display popi prompt and clear input buffer.
 *  disp_error(errtype)      - display error message.
 *  disp_percentdone(n)      - display percentage value of conversion.
 */

/*ARGSUSED*/
void
disp_init(argc,argv)
int argc;
char *argv[];
{
}

void
disp_finish()
{
}

void
disp_imgstart()
{
    ImgInProgress = 1;

    putchar('\033');
    putchar('[');
    putchar('?');
    putchar('3');
    putchar('2');
    putchar('h');
    putchar('\033');
    putchar('[');
    putchar('2');
    putchar('J');
    putchar('\033');
    putchar('[');
    putchar(';');
    putchar('H');
}

void
disp_imgend()
{
    ImgInProgress = 0;

    putchar('\033');
    putchar('[');
    putchar('?');
    putchar('3');
    putchar('2');
    putchar('l');
}

void
disp_putline(line,y)                     /* Output scanline y. */
pixel_t *line;
int y;
{
    int x;
    short wd;
    int bit;

    for (x = 0; x < Xsize;)
    {
	wd = 0;
	for (bit = 15; bit >= 0; --bit, ++x)
	    if (*line++ < thresh[y % BITSPERPIXEL][x % BITSPERPIXEL])
		wd |= 1 << bit;
	putchar('0' + ((wd >> 10) & 0x3f));
	putchar('0' + ((wd >> 5) & 0x1f));
	putchar('0' + (wd & 0x1f));
    }
    putchar('\n');
}


disp_getchar()          /* Get next user typed character. */
{
    return(getchar());
}


void
disp_ungetc(c)         /* Put back the last character typed. */
char c ;
{
    UNGETC(c, stdin);
}



disp_prompt()         /* Display popi prompt and clear input line. */
{
    PRINTF("-> ");
    return 3;
}


void
disp_error(errtype, pos)         /* Display error message. */
int	errtype,
	pos;
{
    extern int  errno;
    extern char *sys_errlist[];

    if (errtype & ERR_PARSE)
    {
        int     i;
 
        for (i=1; i < pos; ++i)
            PUTC('-', stderr);
        PUTC('^', stderr);
        PUTC('\n', stderr);
    }
 
    FPRINTF(stderr, "%s\n", ErrBuf);
    /* we assume errno hasn't been reset by the preceding output */
    if (errtype & ERR_SYS)
        FPRINTF(stderr, "\t(%s)\n", sys_errlist[errno]);
}

void
disp_percentdone(percent)
int	percent;
{
    static int	lastpercent = 100;

    if (!Verbose || ImgInProgress)
	return;

    if (percent == 100)
    {
	printf("\r    \r");
	return;
    }

    if
    (
	percent != lastpercent
	/* && percent % 5 == 0 */
    )
    {
	printf("\r%2d%% ", percent);
	fflush(stdout);
	lastpercent = percent;
    }
}
