/* pbmtomacp.c - read a portable bitmap and produce a MacPaint bitmap file
**
** Copyright (C) 1988 by Douwe vand der Schaaf.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
*/

#include <stdio.h>
#include "pbm.h"
#include "macp.h"

#define TRUE		1
#define FALSE		0
#define EQUAL		1
#define UNEQUAL		0

FILE *fdout;
char *Program;
char *usage =
    "usage:  %s [-l left] [-r right] [-b bottom] [-t top] [pbmfile]\n";

extern int optind;
extern char *optarg;

main(argc, argv)
int argc;
char *argv[];
{ int c;
  FILE *ifd;
  bit **bits, **bitsr;
  int rows, cols;
  int left,bottom,right,top;
  int lflg, rflg, tflg, bflg, errflg;
  char name[100];
int i,j;

  fdout = stdout;
  Program = argv[0];
  errflg = lflg = rflg = tflg = bflg = 0;
  if( argc > 1 )
  { while( ( c = getopt( argc, argv, "l:r:t:b:" ) ) != EOF )
    switch(c)
    {
case 'l':
      lflg++;
      left = atoi( optarg );
      break;

case 'r':
      rflg++;
      right = atoi( optarg );
      break;

case 't':
      tflg++;
      top = atoi( optarg );
      break;

case 'b':
      bflg++;
      bottom = atoi( optarg );
      break;

case '?':
default:
       errflg++;
    }
    if (errflg)
    { fprintf( stderr, "%s: ERROR. incorrect flag\n", Program );
      fprintf( stderr, usage, Program );
      exit(1);
  } }

  if ( argc - optind > 1 )
  { fprintf( stderr, "%s: ERROR. more than 1 inputfile given.\n", Program );
    fprintf( stderr, usage, Program );
    exit( 2 );
  }

  if ( optind == argc - 1 )
  { ifd = fopen( argv[optind], "r" );
    if ( ifd == NULL )
    { fprintf( stderr, "%s: can't open.\n", argv[optind] );
      exit( 3 );
    }
    strcpy( name, argv[optind] );
  }
  else
  { ifd = stdin;
    strcpy( name, "noname" );
  }

  bitsr = pbm_readpbm( ifd, &cols, &rows );

  if ( ifd != stdin )
    fclose( ifd );
    
  bits = pbm_allocarray( MAX_COLS, MAX_LINES );

  if( !lflg )
    left = 0;

  if( rflg )
  { if( right - left >= MAX_COLS )
      right = left + MAX_COLS - 1;
  }
  else
    right = ( left + MAX_COLS > cols ) ? ( cols - 1 ) : ( left + MAX_COLS - 1 );

  if( !tflg )
    top = 0;

  if( bflg )
  { if( bottom - top >= MAX_LINES )
      bottom = top + MAX_LINES - 1;
  }
  else
    bottom = ( top + MAX_LINES > rows ) ?
		   ( rows - 1 ) : ( top + MAX_LINES - 1 );
  
  if( !checkborders( top, left, bottom, right ) )
    exit( 4 );

  fillbits( bits, bitsr, top, left, bottom, right );

  writemacp( bits );

} /* main */

/* - - - - - - - - - - - - - - - - - - - - - - - - - - */

checkborders( top, left, bottom, right )
int top, left, bottom, right;
{ if( right <= left || left < 0 || right - left + 1 > MAX_COLS )
  { fprintf(stderr,"%s: error in right (= %d) and/or left (=%d)\n",
	      Program,right,left );
    return( FALSE );
  }
  if( bottom <= top || top < 0 || bottom - top + 1 > MAX_LINES )
  { fprintf(stderr,"%s: error in bottom (= %d) and/or top (=%d)\n",
	      Program,bottom,top );
    return( FALSE );
  }
  return( TRUE );
} /* checkborders */

/* - - - - - - - - - - - - - - - - - - - - - - - - - - */

/* centreer het over te zenden plaatje in het MacPaint document
 *
 * Het plaatje wordt vanaf al of niet opgegeven (left, bottom)
 * in een pbm bitmap van de juist macpaint afmetingen gezet,
 * en eventueel afgekapt.
 */
fillbits( bits, bitsr, top, left, bottom, right )
bit **bits, **bitsr;
int top, left, bottom, right;
{ register bit *bi, *bir;
  register int i, j;
  register int bottomr, leftr, topr, rightr;
  int width, height;

  width = right - left + 1;
  leftr = (MAX_COLS - width) / 2;
  rightr = leftr + width - 1;

  height = bottom - top + 1;
  topr = ( MAX_LINES - height ) / 2;
  bottomr = topr + height - 1;

  for( i = 0; i < topr; i++ )
  { bi = bits[i];
    for( j = 0; j < MAX_COLS; j++ )
      *bi++ = 0;
  }

  for( i = topr; i <= bottomr; i++ )
  { bi = bits[i];
    { for( j = 0; j < leftr; j++ )
	*bi++ = 0;
      bir = bitsr[ i - topr + top ];
      for( j = leftr; j <= rightr; j++ )
	*bi++ = bir[j - leftr + left];
      for( j = rightr + 1; j < MAX_COLS; j++ )
	*bi++ = 0;
  } }

  for( i = bottomr + 1; i < MAX_LINES; i++ )
  { bi = bits[i];
    for( j = 0; j < MAX_COLS; j++ )
      *bi++ = 0;
  }
} /* fillbits */
      
/* - - - - - - - - - - - - - - - - - - - - - - - - - - */

writemacp( bits )
bit **bits;
{ register int i;
  bit pb[MAX_COLS * 2];
  int npb;

  header();
  for( i=0; i < MAX_LINES; i++ )
  { npb = packit( pb, bits[i] );
    sendbytes( pb, npb );
  }
} /* writemacp */

/* - - - - - - - - - - - - - - - - - - - - - - - - - - */

/* pack regel van MacPaint doc in Apple's format
 * return value = # of bytes in pb 
 */
int
packit( pb, bits )
bit *pb, *bits;
{ register int i, charcount, npb, newcount, flg;
  bit temp[72];
  bit *count, *srcb, *destb, save;

  srcb = bits; destb = temp;
  filltemp( destb, srcb );
  srcb = temp;
  destb = pb;
  npb = 0;
  charcount = BYTES_WIDE;
  flg = EQUAL;
  while( charcount )
  { save = *srcb++;
    charcount--;
    newcount = 1;
    while( (*srcb == save) && charcount )
    { srcb++;
      newcount++;
      charcount--;
    }
    if( newcount > 2 )
    { count = destb++;
      *count = 257 - newcount;
      *destb++ = save;
      npb += 2;
      flg = EQUAL;
    }
    else
    { if( flg == EQUAL )
      { count = destb++;
	*count = newcount - 1;
	npb++;
      }
      else
	*count += newcount;
      while( newcount-- )
      { *destb++ = save;
        npb++;
      }
      flg = UNEQUAL;
  } }
  return npb;
} /* packit */

/* - - - - - - - - - - - - - - - - - - - - - - - - - - */

filltemp( dest, src )
bit *src, *dest;
{ register unsigned char ch, zero, acht;
  register int i, j;

  zero = '\0';
  acht = 8;
  i = BYTES_WIDE;
  while( i-- )
  { ch = zero; 
    j = acht;
    while( j-- )
    { ch <<= 1;
      if( *src++ )
	ch++;
    }
    *dest++ = ch;
  }
} /* filltemp */

/* - - - - - - - - - - - - - - - - - - - - - - - - - - */

sendbytes( pb, npb )
bit *pb;
register int npb;
{ register bit *b;

  b = pb;
  while( npb-- )
    putc( *b++, fdout );
} /* sendbytes */

/* - - - - - - - - - - - - - - - - - - - - - - - - - - */

header()
{ register int i;
  register char ch;

  /* header contains nothing ... */
  ch = '\0';
  for(i = 0; i < HEADER_LENGTH; i++ )
    putc( ch, fdout );
} /* header */
