#ifndef lint
static char sccsid[] = "@(#) n_curves.c 5.1 89/02/20";
#endif

/*
 *	Copyright (c) David T. Lewis 1988
 *	All rights reserved.
 *
 *	Permission is granted to use this for any personal noncommercial use.
 *	You may not distribute source or executable code for profit, nor
 *	may you distribute it with a commercial product without the written
 *	consent of the author.  Please send modifications to the author for
 *	inclusion in updates to the program.  Thanks.
 */

/*	Sat Aug  6 20:38:14 EDT 1988
**
** n_arc():
**
** Routine to draw an arc as a partial ellipse.  This routine uses
** parts of the ellipse() routine to generate the curve.
** The arc is drawn around an ellipse located at (x,y) with axes a and b.
** The arc begins at the angle angle1 and traverses counterclockwise to
** angle2.  "Counterclockwise" is in the sense of the normalized coordinate
** system, with an inverted Y axis, so the arcs are actually drawn clockwise
** on the display screen.
** The values of angle1 and angle2 are given in radians, with 0 being to the
** right of the center of the ellipse.
**
** n_ellipse():
**
** Routine to draw an ellipse in normalized 2-d coordinates.  Uses a line
** segment approximation to the ellipse.  The trig is done with hard coded
** tables of cos and sin values for speed.  The number of vertices on the
** ellipse varies according to the size of the ellipse.  The eliminates
** unnecessary calculations of endpoints for small images.
** Another method for ellipe drawing would be to do a modified Bresenham
** type algorithm, but this would be less adaptable for drawing partial arcs.
**
** NOTE:  This is an ellipse routine, and not a circle, because the PC display
** adapters do not generally have square pixels.  A circle routine should be
** done at a higher level, and should take the screen aspect ration into
** account (see graphics->aspect_ratio in the graphics.h file).
**
** The basic idea is:
**
**	Equation of an ellipse:
**
**	x^2 / a^2 + y^2 / b^2 = 1
**
**	Therefore the X and Y values are:
**
**		X = a * cos(theta)
**		Y = b * sin(theta)
**
**		For some angle theta.
**
**	By stepping through values of theta, we can generate a list of 
**	line segment endpoints on the ellipse.  Then call n_movepen() and
**	n_draw() to display the ellipse.
**
**	Note:  The macros PTS_PER_QUADRANT and TRIG_SCALE are defined 
**	in graphics.h.
*/

#include "config.h"
#if MIX_C
#else
#include <math.h>
#endif /* MIX_C */
#include "bitmaps.h"
#include "graphics.h"

#ifndef M_PI
#define M_PI	3.1415926536 
#endif /* M_PI */
#ifndef M_PI_2
#define M_PI_2	1.5707963268 
#endif /* M_PI_2 */

#define XINDEX 0		/* For specifying array indices.	*/
#define YINDEX 1

/* The following two arrays are tables of values for cos and sin	*/
/* for one quadrant of a circle, divided into PTS_PER_QUADRANT slices.	*/
/* They are found in source file trig.c.				*/

extern long cos_table[PTS_PER_QUADRANT];
extern long sin_table[PTS_PER_QUADRANT];
extern long longsin(), longcos();
extern int n_movepen(), n_draw();

static void make_endpoints(x,y,a,b,endpoints,step)
int x,y,a,b,step;
int endpoints[4][PTS_PER_QUADRANT][2];
{
	int idx;		/* Loop index.			*/
	int value;		/* Temporary storage of values	*/

	/* Set the values for points on the ellipse.  While we're	*/
	/* at it, scale the values back down using TRIG_SCALE, and do	*/
	/* the x and y translation to the center point of the ellipse.	*/
	/* Calculations are done with long ints to preserve precision,	*/
	/* then cast back to short ints in the endpoints array.		*/
	/* Use the value of step to skip unnecessary points (for small	*/
	/* ellipes and arcs).						*/

	for(idx=step-1; idx<PTS_PER_QUADRANT; idx+= step)  {
		value = (cos_table[idx] * (long)a) / TRIG_SCALE;
		endpoints[0][idx][XINDEX] = value + x;
		value = -value;
		endpoints[2][idx][XINDEX] = value + x;
		value = (sin_table[idx] * (long)a) / TRIG_SCALE;
		endpoints[3][idx][XINDEX] = value + x;
		value = -value;
		endpoints[1][idx][XINDEX] = value + x;
		value = (cos_table[idx] * (long)b) / TRIG_SCALE;
		endpoints[1][idx][YINDEX] = value + y;
		value = -value;
		endpoints[3][idx][YINDEX] = value + y;
		value = (sin_table[idx] * (long)b) / TRIG_SCALE;
		endpoints[0][idx][YINDEX] = value + y;
		value = -value;
		endpoints[2][idx][YINDEX] = value + y;
	}
	return;
}

static int make_step(a,b)
int a,b;
{
	int size;
	size = a + b;	/* an indication of the "size" of the ellipse.	*/
	if (size <= 0) return(1);	/* Check for integer overflow.	*/
	else if (size < 2000) return(16);
	else if (size < 8000) return(8);
	else if (size < 16000) return(4);
	else if (size < 32000) return(2);
	else return(1);
}

int n_arc(x,y,a,b,angle1,angle2)
int x,y,a,b;
float angle1,angle2;
{
	int endpoints[4][PTS_PER_QUADRANT][2];
	int x_1,y_1,x_2,y_2;	/* The start and end points of the arc.	*/
	int quadrant;		/* Quadrant of first point.		*/
	int endquadrant;	/* Quadrant of second point.		*/
	int index;		/* First index into endpoints array.	*/
	int endindex;		/* Last index into endpoints array.	*/
	int istat = 0;		/* Function call return status.		*/
	int step;	/* Amount by which to step through array.	*/

	/* Size of a circle, in radians.				*/
	float cir_size = M_PI * 2;

	/* Don't feed huge numbers to the trig functions.		*/
	if (fabs(angle1) > cir_size)
		angle1 = fmod((double)angle1,(double)cir_size);
	if (fabs(angle2) > cir_size)
		angle2 = fmod((double)angle2,(double)cir_size);

	/* Make the angles positive.					*/
	while (angle1 < 0) angle1 += cir_size;
	while (angle2 < 0) angle2 += cir_size;

	/* Calculate a value for step, so we can use fewer endpoints	*/
	/* for small arcs.						*/
	step = make_step(a,b);

#ifdef DEBUG
	printf("step is %d\n",step);
#endif /* DEBUG */

	/* Calculate the end points.	*/

#if NOFLOAT
	x_1 = x + (int)((longcos(angle1) * a) / TRIG_SCALE); 
	y_1 = y + (int)((longsin(angle1) * b) / TRIG_SCALE);
	x_2 = x + (int)((longcos(angle2) * a) / TRIG_SCALE);
	y_2 = y + (int)((longsin(angle2) * b) / TRIG_SCALE);
#else
	x_1 = x + (int)(a * cos((double)angle1));
	y_1 = y + (int)(b * sin((double)angle1));
	x_2 = x + (int)(a * cos((double)angle2));
	y_2 = y + (int)(b * sin((double)angle2));
#endif /* NOFLOAT */

	/* Generate the array of arc endpoints. */
	make_endpoints(x,y,a,b,endpoints,step);

	/* Deduce the first and last element of interest in the 	*/
	/* endpoint array.	*/

	/* First calculate the initial index value.	*/
	index = (int)(angle1 / M_PI_2 * (float)PTS_PER_QUADRANT);
	/* Now figure the quadrant.	*/
	quadrant = (index / PTS_PER_QUADRANT) % 4;
	/* Wrap around so we point at the array.	*/
	index = (index % PTS_PER_QUADRANT);
	index = index + step - (index % step) - 1;

#ifdef DEBUG
	printf("In n_arc(), index is %d\n",index);
#endif /* DEBUG */

	/* Calculate the final index value.	*/
	endindex = (int)(angle2 / M_PI_2 * (float)PTS_PER_QUADRANT);
	/* Now figure the quadrant.	*/
	endquadrant = (endindex / PTS_PER_QUADRANT) % 4;
	/* Wrap around so we point at the array.	*/
	endindex = (endindex % PTS_PER_QUADRANT);
	endindex = endindex + step - (endindex % step) - 1;

#ifdef DEBUG
	printf("In n_arc(), endindex is %d\n",endindex);
#endif /* DEBUG */

	/* Move pen to start point. */

#ifdef DEBUG
	printf("Move pen to %d %d\n",x_1,y_1);
#endif /* DEBUG */

	if(n_movepen(x_1,y_1)) istat = 1;

	/* Start drawing, beginning with the first point of interest. 	*/
	/* Wrap around the array if needed.				*/

	while ((index != endindex) || (quadrant != endquadrant))  {

#ifdef DEBUG
	printf("Draw to %d %d -- quadrant %d index %d\n", 
		endpoints[quadrant][index][XINDEX], 
		endpoints[quadrant][index][YINDEX],quadrant,index);
#endif /* DEBUG */

		if(n_draw(endpoints[quadrant][index][XINDEX], 
			endpoints[quadrant][index][YINDEX]))  {
			istat = 1;
			/* This is just for cleaner error recovery: */
			n_movepen(endpoints[quadrant][index][XINDEX], 
				endpoints[quadrant][index][YINDEX]); 
		}
		/* Increment the indices.	*/
		if ((index+=step) >= PTS_PER_QUADRANT)  {
			index = step - 1;
			if ((++quadrant) >= 4)  {
				quadrant = 0;
			}
		}
	};

	/* Finish at last point of interest.	*/

#ifdef DEBUG
	printf("Finish by drawing to %d %d\n",x_2,y_2);
#endif /* DEBUG */

	if(n_draw(x_2,y_2))  {
		istat = 1;
		/* This is just for cleaner error recovery: */
		n_movepen(x_2,y_2);
	} 
	return(istat);
}

int n_ellipse(x,y,a,b)
int x,y,a,b;
{
	/* The endpoints array will hold line segment (x,y) end points.	*/
	/* There are four quadrants, each with PTS_PER_QUADRANT points,	*/
	/* with two values (X and Y) each.				*/
	int endpoints[4][PTS_PER_QUADRANT][2];
	int pointindex, quadrant;	/* Loop index.			*/
	int istat = 0;			/* Function call return status.	*/
	int step;	/* Amount by which to step through array.	*/

	/* Calculate a value for step, so we can use fewer endpoints	*/
	/* for small ellipses.						*/
	step = make_step(a,b);

	/* Build the array of endpoints.				*/
	make_endpoints(x,y,a,b,endpoints,step);

	/* Display the result, drawing all four quadrants.		*/

	/* Start the pen at the location of the last endpoint.  This	*/
	/* is the point we want to end up on.				*/
	if(n_movepen(endpoints[3][PTS_PER_QUADRANT-1][XINDEX],
		endpoints[3][PTS_PER_QUADRANT-1][YINDEX]))
		istat = 1;

	for (quadrant=0; quadrant<=3; quadrant++)  {

		for (pointindex = step -1; pointindex < PTS_PER_QUADRANT; 
			pointindex += step)  {
			if(n_draw(endpoints[quadrant][pointindex][XINDEX],
				endpoints[quadrant][pointindex][YINDEX]))  {
				istat = 1;
				/* This is just for cleaner error recovery: */
				n_movepen(endpoints[quadrant][pointindex]
						[XINDEX],
					endpoints[quadrant][pointindex]
						[YINDEX]);
			}
		}
	}
	return(istat);
}

