/*
   xc - execute commands contained in a file
   xcb - same, but in the background

   Usage: xc  [-c key_character] [file]
          xcb [-c key_character] [file]

   xc -c x file	key: 'x'	file: "file"
   xc file	key: ' '	file: "file"
   xc -c x	key: 'x'	file: ~/.xc
   xc		key: ~/.xc	file: ~/.xc

   Searches a file for a sequence of lines (probably in a comment)
   with the form
   %c command_line
   or
   $c command_line
   where 'c' (any single character) in "%c" or "$c" is the same as
   the key character given with the "-c" option on the command line,
   and executes the command_lines.
   If the "-c" option is omitted, lines with the form
   % command_line
   or
   $ command_line
   are executed.
   The '%' form echos the command_line; the '$' form does not.

   When invoked as xcb, it automatically runs in the background, redirects
   stdout and stderr to the file ./ERRS, and announces when it finishes.

   It saves the last filename and key character which were used
   into $HOME/.xc, and uses that file's contents as the defaults if the
   filename argument is omitted.
   $HOME/.xc contains one line of the form:
   c-filename
   where 'c' is the key character, or a space if none was given.

   Latest modification: 12/30/88 */

#include <stdio.h>

#define XCFILE      "/.xc"	/* Where default file, key are read from. */
#define ERRFILE     "ERRS"	/* Where output goes if in background.  */

void perror (), exit ();
char *getenv (), *strcpy (), *strcat (), *strchr (), *strrchr (), *fgets ();

void pfatal (), usage ();
void read_xc_file (), write_xc_file ();
void detatch (), notify (), execute ();
char *make_xc_name (), *basename ();

char *program_name;		/* Base of program name. */

main (argc, argv)
  int argc;
  char **argv;
{
  extern int optind;		/* Option index. */
  extern char *optarg;		/* Option argument. */
  char *xcfile;			/* Expansion of "$HOME/.xc". */
  char *file_name;		/* File to get commands from. */
  char key;			/* Command line selector. */
  int background;		/* Run in the background? */
  int c;			/* Option character. */

  program_name = basename (argv[0]);
  file_name = NULL;
  key = 0;
  background = !strcmp (program_name, "xcb");
  xcfile = make_xc_name ();

  while ((c = getopt(argc, argv, "c:")) != EOF)
    switch (c)
      {
      case 'c':
	if (strlen (optarg) != 1)
	  {
	    fprintf (stderr, "%s: %s: Key must be one character long\n",
	      argv[0], optarg);
	    usage ();
	  }
	key = *optarg;
	break;
      default:
	usage ();
      }

  if (optind < argc - 1)
    usage ();
  else if (optind == argc - 1)
    file_name = argv[optind];
  else
    /* Read filename, and key if not set yet, from XCFILE. */
    read_xc_file (xcfile, &file_name, &key);

  if (key == 0 || key == '\t')
    key = ' ';

  if (background)
    detatch ();

  execute (file_name, key);

  write_xc_file (xcfile, file_name, key);

  if (background)
    notify ();

  exit (0);
}

/* Return the path of the file "$HOME/.xc".  */

char *make_xc_name ()
{
  static char xcbuf[BUFSIZ];	/* "$HOME/XCFILE". */
  char *home;

  home = getenv ("HOME");
  if (!home)
    {
      fprintf (stderr,
	"%s: HOME environment variable is not defined\n", program_name);
      exit (1);
    }
  (void) strcpy (xcbuf, home);
  (void) strcat (xcbuf, XCFILE);
  return xcbuf;
}

/* Read from xcfile the filename, and the key if it's not already set.  */

void read_xc_file (xcfile, file_namep, keyp)
  char *xcfile;
  char **file_namep;
  char *keyp;
{
  FILE *fp;
  char *newline;
  static char line[BUFSIZ];

  fp = fopen (xcfile, "r");
  if (!fp)
    pfatal (xcfile);
  (void) fgets (line, BUFSIZ, fp);
  newline = strchr (line, '\n');
  if (newline)
    *newline = 0;
  (void) fclose (fp);

  *file_namep = &line[2];
  if (*keyp == 0)
    *keyp = *line;
}

/* Update the xcfile.  */

void write_xc_file (xcfile, file_name, key)
  char *xcfile;
  char *file_name;
  char key;
{
  FILE *fp;

  fp = fopen (xcfile, "w");
  if (!fp)
    return;
  fprintf (fp, "%c-%s\n", key, file_name);
  (void) fclose (fp);
}

static int save_stdout;		/* Save file descriptor for notification. */

/* Redirect the standard streams and fade into the background.  */

void detatch ()
{
  save_stdout = dup (1);
  if (save_stdout == -1)
    pfatal ("Can't dup standard output");
  if (!freopen (ERRFILE, "w", stdout))
    pfatal (ERRFILE);
  (void) close (2);
  if (dup (1) != 2)		/* Could be 0 if stdin was closed. */
    if (dup (1) == -1)
      pfatal ("Can't dup standard output");
  switch (fork ())
    {
    case -1:
      pfatal ("Can't fork");
    case 0:
      break;
    default:
      exit (0);
    }
}

void notify ()
{
  (void) write (save_stdout, "\r\n\007[xcb: done]\r\n", 16);
  printf ("Done\n");
  (void) fflush (stdout);
}

/* Read the file and execute command lines that match the key.  */

void execute (file_name, key)
  char *file_name;
  char key;
{
  FILE *fp;			/* Pointer to file stream. */
  char line[BUFSIZ];		/* One line of file. */
  char *command;		/* Command to execute. */
  int key_matched = 0;		/* Has command line matching key been found? */
  int status;			/* Return status from system. */

  fp = fopen (file_name, "r");
  if (!fp)
    pfatal (file_name);

  while (fgets (line, BUFSIZ, fp))
    {
      if (line[1] == '\t')
	line[1] = ' ';
      if ((*line != '%' && *line != '$') || line[1] != key)
	{
	  if (key_matched)
	    break;
	  else
	    continue;
	}
      key_matched = 1;
      /* Skip leading whitespace in the command for aesthetic reasons. */
      for (command = &line[2];
	*command && (*command == ' ' || *command == '\t'); ++command)
	 /* Do nothing. */ ;
      if (*line == '%')
	{
	  printf ("  %s", command);
	  (void) fflush (stdout);
	}
      status = system (command);
      if (status)
	{
	  fprintf (stderr, "%s: ** Exit status 0%o **\n", program_name, status);
	  break;
	}
    }
  (void) fclose (fp);

  if (!key_matched)
    {
      fprintf (stderr,
	"%s: %s: No commands for key '%c'\n", program_name, file_name, key);
      exit (1);
    }
}

/* Return name with any leading path stripped off.  */

char *basename (name)
  char *name;
{
  char *base;

  base = strrchr (name, '/');
  return base ? base + 1 : name;
}

void pfatal (message)
  char *message;
{
  fprintf (stderr, "%s: ", program_name);
  perror (message);
  exit (1);
}

void usage ()
{
  fprintf (stderr, "Usage: %s [-c key_character] [file]\n", program_name);
  exit (1);
}
