/********************************************
 VFILER by Richard Conn
  
	VFILER is an implementation (in concept) of a tool similar
to ZCPR3's VFILER for UNIX.  This tool allows the user to point
to files and directory names and perform operations on them.
The multi-windowed display of VFILER consists of a main screen
which may be overlayed by any of a number of windows as commands
are issued.  The main screen looks like this:
  
---------------------------------------------------------------------------
Type h for Help                    VFILER 2.0                      16 Files
                             /usr/usr2/xanadu/work 
     .vfrc              averyverylongf*    commands           demo.a
     fifteencharname    makefile           sak                tape.list
     vf                 vf.1           --> vf.c               vf19.c
     vfrc              /.                 /..                /dtest
                  
File:  vf.c
          Permissions: rw-    Size: 45842
------------------ Sample Main Screen (blank lines omitted) ---------------
  
	The Help Window, which overlays the main screen when the user
strikes the h key, gives some insight into the available commands.
Read vf.1 for a complete description of this program.
  
Acknowledgments:
	Directory lookup routines derived from D 2.0 by Richard Conn.
	Special thanks to Frank Wancho for providing the SYSTEM V
code specifics and checking the program out under several SYSTEM V
machines.  Thanks also to Frank for his review effort and constructive
comments.
 ********************************************/

#ifdef SYSVR2
#define SYSV
#endif

#ifdef SYSVR3
#define SYSV
#endif

#include <stdio.h>
#include <curses.h>
#include <ctype.h>

/* Global Parameters */

#define VERSION "VFILER 2.0"

#define COPYCMD "cp %s %s"
#define EDITCMD "ed %s"
#define PRINTCMD "pr %s | lpr"
#define VIEWCMD "more %s"

#define ENABLESIBSHELL 0	/* enable sibling shell facility? */

#define CURDIR "."		/* name of current directory */

#define SPECIAL1 "."		/* directory to always include in
				 * display */
#define SPECIAL2 ".."		/* directory to always include in
				 * display */

#define RETAGCHAR '#'		/* retag character in display */
#define TAGCHAR '@'		/* tag character in display */
#define DIRCHAR '/'		/* directory character in display */

#define CMTCHAR '#'		/* comments in VFRC files */
#define PTRCHAR '$'		/* current data into command line */

#define FSIZETLIMIT 8000	/* info messages appear if file sizes
				 * exceed this limit */

#define VFRC "/.vfrc"

#define VF_OK 0			/* OK status */
#define VF_NOTDIR 1		/* item is not a directory */

#define CMDLEN 80		/* maximum length of a built-in
				 * command */
#define DIRNAMELEN 400		/* maximum length of a dir name */
#define MAXNAMELEN 600		/* maximum length of a file name */

/* Supporting data structures for shells */
#ifndef TURBOC
#if	ENABLESIBSHELL
int             pid;		/* process id */
FILE           *Shell;
char           *Shelle;
char            Cmd[420];	/* Shell points to a pipe to the
				 * shell, cmd holds the commands sent
				 * to it. Shelle points to the
				 * environment var SHELL. [odj] */
#endif				/* ENABLESIBSHELL */
#endif				/* TURBOC */

/*
 * The following constants are used in parsing and evaluating the
 * VFRC file 
 */
#define O_NOTDEF 0
#define O_CMDCOP 1
#define O_CMDEDI 2
#define O_CMDPRI 3
#define O_CMDVIE 4
#define O_COMMEN 11
#define O_CMNDSYS 12
#define O_CMNDPIPE 13
#define O_CONFIR 21
#define O_EXITCH 22
#define O_INCHID 23

/*
 * The following global variables may be set under Options or from
 * the VFRC file 
 */
int             inchidden = 0;	/* Include hidden files? */
int             confirmdel = 1;	/* Confirm single (vs group) deletes? */
int             exitcheck = 0;	/* Confirm on exit */
char            cmdcopy[CMDLEN];/* Copy Command */
char            cmdedit[CMDLEN];/* Edit Command */
char            cmdprint[CMDLEN];	/* Print Command */
char            cmdview[CMDLEN];/* View Command */
typedef struct cmdentry {
    char            cmdletter;	/* letter of command */
    int             cmdtype;	/* shell or subshell */
    char           *text;	/* text of command */
    struct cmdentry *next;	/* ptr to next command */
}               CMDELT, *CMDELTPTR;
CMDELTPTR       cfirst;		/* ptr to first command in list */

/* The following command table is used for VFRC command parsing */
typedef struct vfrccmnd {
    int             code;
    char           *cname;
}               VFRCMD;
#define NVFCMNDS 11
VFRCMD          cmdset[NVFCMNDS] = {
				    {O_CMNDSYS, "alias"},
				    {O_CMNDPIPE, "basealias"},
				    {O_CMDCOP, "copy"},
				    {O_CMDEDI, "edit"},
				    {O_CMDPRI, "print"},
				    {O_CMDVIE, "view"},
				    {O_CONFIR, "delete"},
				    {O_EXITCH, "exit"},
				    {O_INCHID, "include"}};

/* The following is set on startup */
char            homedir[DIRNAMELEN];
char            vfrcfile[MAXNAMELEN];

typedef struct entry {
    char           *name;
    char            code[4];
    long            size;
    int             number;
    struct entry   *next, *last;
}               ENT, *ENTPTR;

ENTPTR          efirst = NULL;

int             ldcnt;		/* number of files displayed in
				 * dirwin */
int             more;		/* more files than those shown on
				 * screen? */
int             curframe = 0;	/* current frame (screen) number */
int             curstart = 1;	/* number of first file in current
				 * frame */
int             filecount = 0;	/* total number of files loaded */
int             taggedsize = 0;	/* total size of all tagged files */
ENTPTR          cfptr;		/* ptr to current file in linked list */
int             cfnum;		/* number of current file */
char            dname[DIRNAMELEN];	/* name of current directory */
char            savefile[MAXNAMELEN];	/* holding place for current
					 * file */

/* Main Window */
#define TITLE VERSION
#define TITLEY 0
#define TITLEX ((80-strlen(TITLE))/2)
#define TITLESIZELOC (COLS - 18)
#define TITLEHELP "Type h for Help"
#define DIRNAMEY 1

/* Directory Display SubWindow */
WINDOW         *dirwin;

#define DIRWINY 2
#define DIRWINX 0
#define DIRWINLINES 20
#define DIRWINCOLS 80
#define DIRWINMAXENT (DIRWINLINES*4)
#define DIRENTSIZE 15
char            entformat[6];

/* Current File Info Window */
WINDOW         *curfilewin;

#define CURFWINY 22
#define CURFWINX 0
#define CURFWINLINES 2
#define CURFWINCOLS 80
#define CURFENTSIZE 70
#define CFNTEXT "File:"		/* at 0,0 */
#define CURFNAME 6
#define CFCTEXT "Permissions:"	/* at 1,0 */
#define CURFCODE 23
#define CFSTEXT "Size:"		/* at 1,30 */
#define CURFSIZE 36
#define TAGGEDMSG 50
#define CFTTEXT "Tagged:"
#define TAGGEDBYTES 61
#define GROUPLOC 2
char            curfentfmt[6];

/* Command and Information/Status Message Windows */
WINDOW         *cmdmsgwin;
WINDOW         *infomsgwin;

#define CMDMSGWINY (DIRWINY+8)
#define CMDMSGWINX (DIRWINX)
#define CMDMSGWINLINES 4
#define CMDMSGWINCOLS 78
char            cmdmsgfmt[6];

/* Confirmation Message Window */
WINDOW         *confirmwin;

#define CONFIRMWINY (DIRWINY+8)
#define CONFIRMWINX (DIRWINX+10)
#define CONFIRMWINLINES 3
#define CONFIRMWINCOLS 40
#define CONFIRMWINBOTMSG " Strike y to Confirm "
#define CONFIRMWINBOTCOL ((CONFIRMWINCOLS-strlen(CONFIRMWINBOTMSG))/2)
char            confirmentfmt[6];

/* Error Message Window */
WINDOW         *errorwin;

#define ERRORWINY (DIRWINY+8)
#define ERRORWINX (DIRWINX+10)
#define ERRORWINLINES 3
#define ERRORWINCOLS 40
#define ERRORWINBOTMSG " Strike x to Abort "
#define ERRORWINBOTCOL ((ERRORWINCOLS-strlen(ERRORWINBOTMSG))/2)
char            errentfmt[6];

/* Help Window */
WINDOW         *helpwin;

#define HELPWINY (DIRWINY)
#define HELPWINX (DIRWINX+18)
#define HELPWINLINES 19
#define HELPWINCOLS 42
#define HELPWINBOTMSG " Strike Any Command to Continue "
#define HELPWINBOTCOL ((HELPWINCOLS-strlen(HELPWINBOTMSG))/2)
char           *helpmsg[HELPWINLINES - 2] = {
			  "+  Arrow/Screen Movement  File Commands",
			  "-        ^P               c = copy     ",
			  "-         ^               d = delete   ",
			  "-    ^B <-+-> ^F          e = edit     ",
			  "-         v               f = find     ",
			  "-        ^N               p = print    ",
			  "-  + = next screen        t = tag/untag",
			  "-  - = last screen        v = view     ",
			  "-                                      ",
			  "+  Directory Movement/Reload Commands  ",
			  "- l = login        s = switch directory",
			  "- r = reload       u = parent directory",
			  "-                                      ",
			  "+            Other Commands            ",
			  "-^L = refresh      = = system command  ",
#ifdef TURBOC
			  "- g = group                            ",
#else
#if ENABLESIBSHELL
			  "- g = group        ~ = base command    ",
#else
			  "- g = group                            ",
#endif				/* ENABLESIBSHELL */
#endif				/* TURBOC */
			 "- x = exit         / = .vfrc command   "};
char           *ghelpmsg[HELPWINLINES - 2] = {
			  "+      Group Command Help Message      ",
			  "-                                      ",
			  "+  Group Commands Apply Only to Tagged ",
			  "+                Files                 ",
			  "-  c = Group Copy                      ",
			  "-  d = Group Delete                    ",
			  "-  p = Group Print                     ",
			  "-                                      ",
			  "+      Manipulation of Tagged Flags    ",
			  "-  r = Group Retag                     ",
			  "-  t = Group Tag                       ",
			  "-  u = Group Untag                     ",
			  "-                                      ",
			  "+             Help Message             ",
			  "-  h = Group Help (this display)       ",
			  "-                                      ",
			 "-                                      "};

/* Words to appear in lower corner (for command selection indicators) */
#define GROUPWORD "Group"
#define VFRCWORD  "/Cmnd"

/* Control Character Definitions */
#define CTRLB '\02'
#define CTRLF '\06'
#define CTRLH '\010'
#define CTRLI '\011'
#define CTRLL '\014'
#define CTRLN '\016'
#define CTRLP '\020'
#define CTRLU '\025'
#define DELETE 127

/********************************************
 Begin BSD 4.x UNIX Implementation Specifics
 ********************************************/
#ifdef BSD

#include <sys/types.h>
#include <sys/dir.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/wait.h>

union wait      shdummy;	/* dummy variable used by wait () */

dirsize(dirp)
    struct direct  *dirp;
{
    return DIRSIZ(dirp);
}

/* Return current working directory into a buffer of MAXNAMELEN size */
getcurwd(name)
    char           *name;
{
    getwd(name);
}

/********************************************
 End   BSD 4.x UNIX Implementation Specifics
 ********************************************/
#endif				/* BSD */

/********************************************
 Begin POSIX Implementation Specifics
 ********************************************/
#ifdef POSIX

#include <sys/types.h>
#include <dirent.h>
#include <stat.h>

int             shdummy;	/* dummy variable used by wait () */

dirsize(dirp)
    struct dirent  *dirp;
{
    return strlen(dirp->d_name) + 1;
}

/* Return current working directory into a buffer of MAXNAMELEN size */
getcurwd(name)
    char           *name;
{
    getwd(name);
}

/********************************************
 End   POSIX Implementation Specifics
 ********************************************/
#endif				/* POSIX */

/********************************************
 Begin UNIX SYSTEM V Implementation Specifics
 ********************************************/
#ifdef SYSV

#ifdef SYSVR3
#include <sys/types.h>
#endif
#include <sys/file.h>
#include <sys/dir.h>
#include <sys/stat.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>

int             shdummy;	/* dummy variable used by wait () */

typedef struct {
    int             size;
    struct direct  *files;
    struct direct  *current;
}               DIR;

/* Return current working directory into a buffer of MAXNAMELEN size */
getcurwd(name)
    char           *name;
{
    getcwd(name, MAXNAMELEN);
}

/* Open directory */
DIR            *
opendir(name)
{
    DIR            *dir;
    int             fd, open();
    struct stat     sbuf;
    int             x;

    if (stat(name, &sbuf) != 0)
	return (NULL);
    if (!(sbuf.st_mode & S_IFDIR)) {
	errno = ENOTDIR;
	return (NULL);
    }
    fd = open(name, O_RDONLY, 0);
    if (fd == -1)
	perror("open");
    if (fd == -1)
	return (NULL);
    if ((dir = (DIR *) malloc(sizeof(DIR))) == NULL) {
	close(fd);
	return (NULL);
    }
    dir->size = sbuf.st_size;
    if ((dir->files = (struct direct *) malloc(dir->size)) == 0) {
	close(fd);
	free((char *) dir);
	return (NULL);
    }
    x = read(fd, dir->files, dir->size);
    if (x == -1)
	perror("read");
    dir->current = dir->files;
    close(fd);
    return (dir);
}

/* Read a record from the directory file */
struct direct  *
readdir(dirp)
    DIR            *dirp;
{
    int             s = dirp->size / sizeof(struct direct);
    struct direct  *temp;
    if (dirp->current >= dirp->files + s)
	return (NULL);
    while (dirp->current->d_ino == 0)
	if (dirp->current >= dirp->files + s)
	    return (NULL);
	else
	    dirp->current++;
    if (dirp->current >= dirp->files + s)
	return (NULL);
    temp = dirp->current++;
    return (temp);
}

/* Rewind the directory */
rewinddir(dirp)
    DIR            *dirp;
{
    dirp->current = dirp->files;
}

/* Close the directory */
closedir(dirp)
    DIR            *dirp;
{
    free((char *) dirp->files);
    free((char *) dirp);
}

/* Compute the size of the directory entry */
dirsize(dirp)
    struct direct  *dirp;
{
    return DIRSIZ;
}

/********************************************
 End   UNIX SYSTEM V Implementation Specifics
 ********************************************/
#endif				/* SYSV */

/********************************************
 Begin TURBO-C Implementation Specifics
 ********************************************/
#ifdef TURBOC

#include <alloc.h>
#include <dir.h>
#include <dos.h>
#include <sys\stat.h>

#define GETDISK 0x19
#define SETDISK 0x0e

typedef struct direct {
    long            d_size;
    char            d_name[14];
    struct direct  *next;
}               DIRECT;

typedef struct {
    DIRECT         *files;
    DIRECT         *current;
}               DIR;

gotodir(name)
    char           *name;
{
    unsigned        newdisk;

    if (name[1] == ':') {
	if (name[0] >= 'a')
	    newdisk = name[0] - 'a';
	else
	    newdisk = name[0] - 'A';
	bdos(SETDISK, newdisk, 0);	/* set new disk */
    }
    chdir(name);
}

DIR            *
opendir(name)
    char           *name;
{
    DIR            *dir;
    DIRECT         *current;
    struct stat     sbuf;
    int             done, i, pass1;
    struct ffblk    curent;
    char           *home;
    unsigned        curdisk, newdisk;

    home = getcwd(NULL, 200);
    curdisk = bdos(GETDISK, 0, 0);	/* get current disk */
    if (name[1] == ':') {
	if (name[0] >= 'a')
	    newdisk = name[0] - 'a';
	else
	    newdisk = name[0] - 'A';
	bdos(SETDISK, newdisk, 0);	/* set new disk */
    }
    if (chdir(name) != 0) {
	free(home);
	bdos(SETDISK, curdisk, 0);	/* return home */
	return (NULL);
    }
    if ((dir = (DIR *) malloc(sizeof(DIR))) == NULL) {
	free(home);
	bdos(SETDISK, curdisk, 0);	/* return home */
	return (NULL);
    }
    done = findfirst("*.*", &curent,
		     FA_RDONLY | FA_DIREC | FA_ARCH);
    pass1 = 1;
    while (!done) {
	if (pass1) {
	    if ((dir->files = (DIRECT *) malloc(sizeof(DIRECT))) == 0) {
		free(home);
		bdos(SETDISK, curdisk, 0);	/* return home */
		return (NULL);
	    }
	    current = dir->files;
	    pass1 = 0;
	} else {
	    if ((current->next = (DIRECT *) malloc(sizeof(DIRECT))) == 0) {
		free(home);
		bdos(SETDISK, curdisk, 0);	/* return home */
		return (NULL);
	    }
	    current = current->next;
	}
	current->next = NULL;
	for (i = 0; i < 13; i++)
	    current->d_name[i] = curent.ff_name[i];
	current->d_name[i] = '\0';
	current->d_size = curent.ff_fsize;
	done = findnext(&curent);
    }
    dir->current = dir->files;
    bdos(SETDISK, curdisk, 0);	/* return home */
    chdir(home);
    free(home);
    return (dir);
}

struct direct  *
readdir(dirp)
    DIR            *dirp;
{
    DIRECT         *temp;

    temp = dirp->current;
    if (temp != NULL)
	dirp->current = dirp->current->next;
    return (temp);
}

rewinddir(dirp)
    DIR            *dirp;
{
    dirp->current = dirp->files;
}

closedir(dirp)
    DIR            *dirp;
{
    free(dirp->files);
    free(dirp);
}

dirsize(dirp)
    struct direct  *dirp;
{
    return 14;
}

/* Return current working directory into a buffer of MAXNAMELEN size */
getcurwd(name)
    char           *name;
{
    getcwd(name, MAXNAMELEN);
}

/********************************************
 End   TURBO-C Implementation Specifics
 ********************************************/
#endif				/* TURBOC */

/* Define gotodir() as chdir() if not TURBOC */
#ifndef TURBOC
gotodir(name)
    char           *name;
{
    chdir(name);
}
#endif				/* TURBOC */

main(argc, argv)
    int             argc;
    char           *argv[];
{
    char           *getenv(), *rover;

#ifndef TURBOC
#if	ENABLESIBSHELL
    /* Open pipe to shell */
    Shelle = getenv("SHELL");
    if (Shelle != NULL && Shelle[0] != '\0')
	strcpy(Cmd, Shelle);
    else
	strcpy(Cmd, "/bin/sh");
    Shell = popen(Cmd, "w");
    if (Shell == NULL) {
	fprintf(stderr, "Can't open pipe to %s\n", Cmd);
	exit(5);
    }
    pid = getpid();
#endif				/* ENABLESIBSHELL */
#endif				/* TURBOC */

    /* Set up home directory, VFRC file, and initialize from VFRC */
    strcpy(homedir, getenv("HOME"));
    rover = VFRC;
    strcpy(vfrcfile, ++rover);
    strcpy(cmdcopy, COPYCMD);
    strcpy(cmdedit, EDITCMD);
    strcpy(cmdprint, PRINTCMD);
    strcpy(cmdview, VIEWCMD);
    getvfrc();

    /* Set up for Curses */
    initscr();

    /* Clear I/O */
    raw();
    noecho();

    /* Set up display formats */
    sprintf(entformat, "%%-%ds", DIRENTSIZE);
    sprintf(cmdmsgfmt, "%%-%ds", CMDMSGWINCOLS - 2);
    sprintf(confirmentfmt, "%%-%ds", CONFIRMWINCOLS - 2);
    sprintf(errentfmt, "%%-%ds", ERRORWINCOLS - 2);
    sprintf(curfentfmt, "%%-%ds", CURFENTSIZE);

    /* Set up all windows */
    dirwin = subwin(stdscr, DIRWINLINES, DIRWINCOLS, DIRWINY, DIRWINX);
    curfilewin = subwin(stdscr, CURFWINLINES, CURFWINCOLS,
			CURFWINY, CURFWINX);
    helpwin = newwin(HELPWINLINES, HELPWINCOLS, HELPWINY, HELPWINX);
    cmdmsgwin = newwin(CMDMSGWINLINES, CMDMSGWINCOLS,
		       CMDMSGWINY, CMDMSGWINX);
    infomsgwin = newwin(CMDMSGWINLINES, CMDMSGWINCOLS,
			CMDMSGWINY, CMDMSGWINX);
    errorwin = newwin(ERRORWINLINES, ERRORWINCOLS, ERRORWINY, ERRORWINX);
    confirmwin = newwin(CONFIRMWINLINES, CONFIRMWINCOLS,
			CONFIRMWINY, CONFIRMWINX);
    sethelp();			/* Load help window with text */

    /*
     * Start VF procedure with either current directory or passed
     * directory 
     */
    if (argc == 1)
	vf(".");
    else
	vf(argv[1]);

    /* Restore communications parameters, end windows, and exit */
    echo();
    noraw();
    endwin();
    puts("\n");
}

/* Main VF Processor */
vf(dirname)
    char           *dirname;
{
    int             target;
    char            answer[200];
    int             ch;
    char            message[80];
    int             looping;
    int             retcode;
    DIR            *dirp, *opendir();
    ENTPTR          rover;
    CMDELTPTR       erover;
    int             carryover;

    looping = 1;
    if (strcmp(dirname, CURDIR) == 0)
	getcurwd(dname);
    else
	strcpy(dname, dirname);
    while (looping) {
	if (dir(dname) == VF_NOTDIR) {
	    errormsg("Invalid Directory Selection");
	    cmdmsg("Enter name of directory", answer);
	    if (strcmp(answer, CURDIR) == 0)
		getcurwd(dname);
	    else
		strcpy(dname, answer);
	} else
	    looping = 0;
    }
    ncscrefresh();
    carryover = 0;
    while (1) {
	if (!carryover)
	    ch = command();	/* get command */
	carryover = 0;
	switch (ch) {
	case CTRLB:		/* Backward one file */
	case CTRLH:
	    if (cfnum == curstart)
		target = curstart + ldcnt - 1;
	    else
		target = cfnum - 1;
	    moveupdate(target);
	    break;
	case CTRLF:		/* Forward one file */
	case ' ':
	    advance(ldcnt);
	    break;
	case CTRLL:		/* Refresh screen for current dir */
	    screfresh();
	    break;
	case CTRLN:		/* Down one line */
	    if (cfnum + 4 >= curstart + ldcnt) {
		target = cfnum % 4;
		if (target == 0)
		    target = 4;
		target += curstart - 1;
	    } else
		target = cfnum + 4;
	    moveupdate(target);
	    break;
	case CTRLP:		/* Up one line */
	    if (cfnum - 4 < curstart) {
		target = cfnum;
		while (target + 4 < curstart + ldcnt)
		    target += 4;
	    } else
		target = cfnum - 4;
	    moveupdate(target);
	    break;
	case '~':		/* Subshell accessed via pipe */
	    cmdmsg("Enter command line text", answer);
	    if (*answer > ' ')
		dosubshell(answer);
	    break;
	case '=':		/* Command invocation via system */
	    cmdmsg("Enter command line text", answer);
	    if (*answer > ' ')
		dosystem(answer);
	    break;
	case '+':		/* Advance to next screen */
	    if (more) {
		curframe++;
		curstart = (curframe * DIRWINMAXENT) + 1;
	    } else {
		curframe = 0;
		curstart = 1;
	    }
	    cfnum = curstart;
	    screfresh();
	    break;
	case '-':		/* Back up to previous screen */
	    if (curframe) {
		curframe--;
	    } else {
		curframe = (filecount - 1) / DIRWINMAXENT;
	    }
	    curstart = (curframe * DIRWINMAXENT) + 1;
	    cfnum = curstart;
	    screfresh();
	    break;
	case '/':		/* VFRC Command */
	    setgroup(VFRCWORD);
	    ch = command();
	    setgroup("");
	    if (ch > ' ') {
		erover = cfirst;
		while (erover) {
		    if (erover->cmdletter == ch)
			break;
		    erover = erover->next;
		}
		if (erover) {
		    if (erover->cmdtype == O_CMNDSYS)
			dosystem(erover->text);
		    else
			dosubshell(erover->text);
		} else
		    errormsg("System/Subshell command not found");
	    }
	    break;
	case 'c':		/* Copy file */
	    if (*cmdcopy) {
		if (*cfptr->name == DIRCHAR)
		    errormsg("You may not copy a dir");
		else {
		    cmdmsg
			    ("Enter name of target directory",
			     answer);
		    if (*answer > ' ') {
			if ((dirp = opendir(answer)) == NULL) {
			    closedir(dirp);
			    sprintf(message, "Not a dir: %s", answer);
			    errormsg(message);
			} else {
			    closedir(dirp);
			    if (strcmp(answer, dname) == 0)
				errormsg
					("Copy into current directory not allowed");
			    else {
				sprintf(message, cmdcopy, &(cfptr->name[1]),
					answer);
				if (cfptr->size > FSIZETLIMIT)
				    infomsg("Copy in Progress", 1);
				noraw();
				echo();
				retcode = system(message);
				raw();
				noecho();
				if (cfptr->size > FSIZETLIMIT)
				    clrinfomsg();
				if (retcode)
				    errormsg("Error in copy operation");
			    }
			}
		    }
		}
	    } else
		errormsg("Copy command not defined");
	    break;
	case 'd':		/* Delete file (note: this routine
				 * takes advantage of the knowledge
				 * that the last entry in the linked
				 * list will never be deleted because
				 * it is a directory */
	    if (*cfptr->name == DIRCHAR)
		errormsg("You may not delete a directory");
	    else {
		if (confirmdel) {
		    if (!confirmmsg("Delete File?"))
			break;
		}
		if (unlink(&(cfptr->name[1])) != 0)
		    errormsg("Delete not possible");
		else {
		    rover = cfptr;
		    if (cfptr->last != NULL)
			(cfptr->last)->next = cfptr->next;
		    else
			efirst = cfptr->next;
		    (cfptr->next)->last = cfptr->last;
		    cfptr = cfptr->next;
		    if (*rover->name == TAGCHAR) {
			taggedsize -= rover->size;
		    }
		    filecount--;
		    free((char *) rover);
		    screfresh();
		}
	    }
	    break;
	case 'e':		/* Edit file */
	    if (*cmdedit) {
		if (*cfptr->name == DIRCHAR)
		    errormsg("You may not edit a directory");
		else {
		    sprintf(message, cmdedit, &(cfptr->name[1]));
		    strcpy(savefile, &(cfptr->name[1]));
		    noraw();
		    echo();
		    system(message);
		    raw();
		    noecho();
		    xreload();
		    findf(savefile);
		    screfresh();
		}
	    } else
		errormsg("Edit command not defined");
	    break;
	case 'f':		/* Find file */
	    cmdmsg("Enter name or partial name of file to find", answer);
	    if (*answer > ' ')
		findf(answer);
	    screfresh();
	    break;
	case 'g':		/* Group subcommand */
	    dogroup();
	    break;
	case 'h':		/* Help Window */
	case '?':
	    helpme();
	    ch = command();
	    if (ch != ' ')
		carryover = 1;
	    touchwin(dirwin);
	    wrefresh(dirwin);
	    break;
	case 'l':		/* Log into directory pointed-to */
	    if (*(cfptr->name) == DIRCHAR) {
		looping = 1;
		if (strcmp(&(cfptr->name[1]), CURDIR) == 0)
		    getcurwd(dname);
		else
		    strcpy(dname, &(cfptr->name[1]));
		while (looping) {
		    if (dir(dname) == VF_NOTDIR) {
			errormsg("Invalid Directory Selection");
			cmdmsg("Enter name of directory", answer);
			if ((strcmp(answer, CURDIR) == 0) ||
			    (*answer == '\0'))
			    getcurwd(dname);
			else
			    strcpy(dname, answer);
		    } else
			looping = 0;
		}
		getcurwd(dname);
		screfresh();
	    } else {
		errormsg("Not a Valid Directory");
	    }
	    break;
	case 'p':		/* Print current file */
	    if (*cmdprint) {
		if (*cfptr->name == DIRCHAR)
		    errormsg("Cannot print a directory");
		else {
		    if (*cfptr->code != 'r')
			errormsg
				("You do not have read permission on this file");
		    else {
			sprintf(answer, cmdprint, &(cfptr->name[1]));
			if (cfptr->size > FSIZETLIMIT)
			    infomsg
				    ("Print Formatting and Spooling in Process", 1);
			noraw();
			echo();
			system(answer);
			raw();
			noecho();
			if (cfptr->size > FSIZETLIMIT)
			    clrinfomsg();
		    }
		}
	    } else
		errormsg("Print command not defined");
	    break;
	case 'r':		/* Reload current directory */
	    reload();
	    break;
	case 's':		/* Switch to directory named by user */
	    looping = 1;
	    while (looping) {
		cmdmsg
			("Enter name of directory", answer);
		if (*answer <= ' ')
		    break;
		if (strcmp(answer, CURDIR) == 0)
		    getcurwd(dname);
		else
		    strcpy(dname, answer);
		if (dir(dname) == VF_NOTDIR)
		    errormsg("Invalid Directory Selection");
		else
		    looping = 0;
	    }
	    curframe = 0;
	    curstart = 1;
	    getcurwd(dname);
	    screfresh();
	    break;
	case 't':		/* Tag current file and advance to
				 * next */
	    if (*cfptr->name == DIRCHAR)
		errormsg("You may not tag a directory");
	    else {
		if (*cfptr->name != TAGCHAR) {
		    *cfptr->name = TAGCHAR;
		    displaytag(TAGCHAR);
		    taggedsize += cfptr->size;
		} else {
		    *cfptr->name = ' ';
		    displaytag(' ');
		    taggedsize -= cfptr->size;
		}
		advance(ldcnt);
	    }
	    break;
	case 'u':		/* Move up and log into parent
				 * directory */
	    looping = 1;
	    strcpy(answer, "..");
	    while (looping) {
		if (strcmp(answer, CURDIR) == 0)
		    getcurwd(dname);
		else
		    strcpy(dname, answer);
		if (dir(dname) == VF_NOTDIR) {
		    cmdmsg
			    ("Enter name of directory",
			     answer);
		    if (*answer <= ' ')
			break;
		} else
		    looping = 0;
	    }
	    curframe = 0;
	    curstart = 1;
	    getcurwd(dname);
	    screfresh();
	    break;
	case 'v':		/* View current file */
	    if (*cmdview) {
		if (*cfptr->name == DIRCHAR)
		    errormsg("Cannot view a directory");
		else {
		    if (*cfptr->code != 'r')
			errormsg
				("You do not have read permission on this file");
		    else {
			sprintf(answer, cmdview, &(cfptr->name[1]));
			noraw();
			echo();
			clear();
			refresh();
			system(answer);
			printf("Strike RETURN to Continue -- ");
			raw();
			noecho();
			*answer = getchar();
			screfresh();
		    }
		}
	    } else
		errormsg("View command not defined");
	    break;
	case 'x':
	    if (exitcheck) {
		if (confirmmsg("Exit VFILER?"))
		    ch = 'z';	/* Exit code */
		move(23, 0);
		refresh();
	    } else
		ch = 'z';
	    break;
	case 'z':
	    ch = 'x';		/* Convert to something harmless */
	    break;
	default:
	    sprintf(message, "Invalid Command: %c", ch);
	    errormsg(message);
	    break;
	}
	if (ch == 'z')
	    break;
    }
}

/* Group Subcommand Processor */
dogroup()
{
    char            answer[200];
    int             ch;
    char            message[80];
    int             retcode;
    DIR            *dirp, *opendir();
    ENTPTR          rover;

    setgroup(GROUPWORD);
    ch = command();
    setgroup("");
    *answer = '\0';
    if (ch > ' ') {
	switch (ch) {
	case 'h':
	    setghelp();
	    helpme();
	    sethelp();
	    ch = command();
	    touchwin(dirwin);
	    wrefresh(dirwin);
	    ch = 'x';
	    break;
	case 'c':
	    if (*cmdcopy) {
		cmdmsg
			("Enter name of target directory",
			 answer);
		if (*answer > ' ') {
		    if ((dirp = opendir(answer)) == NULL) {
			closedir(dirp);
			errormsg("Invalid Directory Selection");
			ch = 'x';
		    } else {
			closedir(dirp);
			if (strcmp(answer, dname) == 0) {
			    errormsg
				    ("Copy into current directory not allowed");
			    ch = 'x';
			} else {
			    infomsg("Copy in Progress", 1);
			    noraw();
			    echo();
			    retcode = 0;
			}
		    }
		    rover = efirst;
		} else
		    ch = 'x';
	    } else {
		errormsg("Copy command not defined");
		ch = 'x';
	    }
	    break;
	case 'd':
	    if (!(confirmmsg("Delete the tagged files?")))
		ch = 'x';
	    rover = efirst;
	    break;
	case 'p':
	    if (*cmdprint) {
		infomsg("Print Formatting and Spooling in Process", 1);
		noraw();
		echo();
		retcode = 0;
		rover = efirst;
	    } else {
		errormsg("Print command not defined");
		ch = 'x';
	    }
	    break;
	case 'r':
	    rover = efirst;
	    break;
	case 't':
	case 'u':
	    rover = cfptr;
	    break;
	default:
	    errormsg("Not a Group Subcommand (c,d,h,p,r,t,u)");
	    ch = 'x';
	    break;
	}
	if (ch != 'x') {
	    while (rover != NULL) {
		switch (ch) {
		case 'c':
		    if (*rover->name == TAGCHAR) {
			*rover->name = RETAGCHAR;
			sprintf(message, cmdcopy, &(rover->name[1]),
				answer);
			if (system(message))
			    retcode = 1;
		    }
		    break;
		case 'd':
		    if (*rover->name == TAGCHAR) {
			if (unlink(&(rover->name[1])) != 0) {
			    strcpy(answer,
				   "Not all deletes were possible");
			}
		    }
		    break;
		case 'p':
		    if (*rover->name == TAGCHAR) {
			if (*rover->code != 'r')
			    retcode = 1;
			else {
			    sprintf(answer, cmdprint, &(rover->name[1]));
			    if (system(answer))
				retcode = 1;
			    else
				*rover->name = RETAGCHAR;
			}
		    }
		    break;
		case 'r':
		    if (*rover->name == RETAGCHAR) {
			if (*rover->name != TAGCHAR) {
			    *rover->name = TAGCHAR;
			    taggedsize += rover->size;
			}
		    }
		    break;
		case 't':
		    if (*rover->name != DIRCHAR) {
			if (*rover->name != TAGCHAR) {
			    *rover->name = TAGCHAR;
			    taggedsize += rover->size;
			}
		    }
		    break;
		case 'u':
		    if (*rover->name != DIRCHAR) {
			if (*rover->name == TAGCHAR) {
			    taggedsize -= rover->size;
			}
			*rover->name = ' ';
		    }
		    break;
		default:
		    break;
		}
		rover = rover->next;
	    }
	    if ((ch == 'c') || (ch == 'p')) {
		raw();
		noecho();
		clrinfomsg();
		if (retcode)
		    errormsg("Error in operation");
		*answer = '\0';
	    }
	    if (ch == 'd')
		reload();
	    else
		screfresh();
	    if (*answer)
		errormsg(answer);
	}
    }
}

/* Get options and other information from VFRC file */
getvfrc()
{
    FILE           *fp, *fopen();
    char            target[MAXNAMELEN];
    char           *strcat();
    int             error = 0;
    int             cmdcode;
    char            argument[MAXNAMELEN], *rover;
    CMDELTPTR       erover, elast;

    cfirst = NULL;		/* set empty command list */
    elast = NULL;
    if ((fp = fopen(vfrcfile, "r")) == NULL) {
	strcpy(target, homedir);
	strcat(target, VFRC);
	if ((fp = fopen(target, "r")) == NULL)
	    return;
    }
    while (fgets(target, MAXNAMELEN, fp)) {
	target[strlen(target) - 1] = '\0';
	parse(target, &cmdcode, argument);
	switch (cmdcode) {
	case O_CONFIR:
	    confirmdel = 0;
	    break;
	case O_EXITCH:
	    exitcheck = 1;
	    break;
	case O_INCHID:
	    inchidden = 1;
	    break;
	case O_CMDCOP:
	    csubst(argument, cmdcopy);
	    break;
	case O_CMDEDI:
	    csubst(argument, cmdedit);
	    break;
	case O_CMDPRI:
	    csubst(argument, cmdprint);
	    break;
	case O_CMDVIE:
	    csubst(argument, cmdview);
	    break;
	case O_COMMEN:
	    break;
	case O_CMNDSYS:
	case O_CMNDPIPE:
	    if (strlen(argument) > 1) {
		if ((erover = (CMDELTPTR) (malloc(sizeof(CMDELT)))) == NULL) {
		    fprintf(stderr, " Dynamic Memory Overflow\n");
		    exit(1);
		}
		if ((erover->text = (char *) (malloc(strlen(argument) + 1)))
		    == NULL) {
		    fprintf(stderr, " Dynamic Memory Overflow\n");
		    exit(1);
		}
		erover->cmdletter = *argument;
		erover->cmdtype = cmdcode;
		rover = &argument[1];
		while (*rover) {
		    if (*rover > ' ')
			break;
		    rover++;
		}
		strcpy(erover->text, rover);
		if (elast == NULL) {
		    cfirst = erover;
		    elast = erover;
		} else {
		    elast->next = erover;
		    elast = erover;
		}
		elast->next = NULL;
	    } else {
		fprintf(stderr, " Command error on %s\n", argument);
		exit(1);
	    }
	    break;
	default:
	    error = 1;
	    break;
	}
    }
    fclose(fp);
    if (error) {
	printf("Error in processing %s - strike RETURN to continue\n",
	       vfrcfile);
	gets(target);
    }
}

/* Command substitutor */
csubst(source, dest)
    char           *source, *dest;
{
    while (*source) {
	if (*source == PTRCHAR) {
	    source++;
	    if (*source != PTRCHAR) {
		switch (*source) {
		case 'd':
		case 'f':
		case 'n':
		    *dest++ = '%';
		    *dest++ = 's';
		    break;
		default:
		    *dest++ = PTRCHAR;
		    *dest++ = *source;
		    if (*source == '\0')
			source--;	/* for following ++ */
		    break;
		}
		source++;
	    } else
		*dest++ = *source++;
	} else
	    *dest++ = *source++;
    }
    *dest = '\0';
}

/* VFRC line parser */
parse(target, cmdcode, argument)
    char           *target, *argument;
    int            *cmdcode;
{
    char            verb[40], *rover;
    int             i;

    if ((*target == '\0') || (*target == ' '))
	*cmdcode = O_COMMEN;
    else {
	if (*target == CMTCHAR)
	    *cmdcode = O_COMMEN;
	else {
	    rover = verb;
	    while (*target > ' ')
		*rover++ = *target++;
	    *rover = '\0';
	    while (*target) {
		if (*target > ' ')
		    break;
		target++;
	    }
	    rover = argument;
	    while (*target) {
		*rover++ = *target++;
	    }
	    *rover = '\0';
	    rover = verb;
	    while (*rover) {
		*rover = isupper(*rover) ? tolower(*rover) : *rover;
		rover++;
	    }
	    *cmdcode = O_NOTDEF;
	    for (i = 0; i < NVFCMNDS; i++) {
		if (strcmp(verb, cmdset[i].cname) == 0) {
		    *cmdcode = cmdset[i].code;
		    break;
		}
	    }
	}
    }
}

/* Reload current directory */
reload()
{
    xreload();
    screfresh();
}

xreload()
{
    int             looping;

    looping = 1;
    getcurwd(dname);
    while (looping) {
	if (dir(dname) == VF_NOTDIR) {
	    errormsg("Current Directory Not Valid (Odd!)");
	    cmdmsg("Enter name of directory", dname);
	    if (*dname <= ' ')
		getcurwd(dname);
	} else
	    looping = 0;
    }
    curstart = 1;
    curframe = 0;
}

/* Set up Help Window */
sethelp()
{
    int             row;

    box(helpwin, '|', '-');
    wmove(helpwin, 0, 2);
    waddstr(helpwin, " Help ");
    for (row = 0; row < HELPWINLINES - 2; row++) {
	if (*helpmsg[row] == '+')
	    wstandout(helpwin);
	wmove(helpwin, row + 1, 2);
	waddstr(helpwin, &helpmsg[row][1]);
	if (*helpmsg[row] == '+')
	    wstandend(helpwin);
    }
    wmove(helpwin, HELPWINLINES - 1, HELPWINBOTCOL);
    waddstr(helpwin, HELPWINBOTMSG);
}

/* Load Group Help Message */
setghelp()
{
    int             row;

    box(helpwin, '|', '-');
    wmove(helpwin, 0, 2);
    waddstr(helpwin, " Help ");
    for (row = 0; row < HELPWINLINES - 2; row++) {
	if (*ghelpmsg[row] == '+')
	    wstandout(helpwin);
	wmove(helpwin, row + 1, 2);
	waddstr(helpwin, &ghelpmsg[row][1]);
	if (*ghelpmsg[row] == '+')
	    wstandend(helpwin);
    }
    wmove(helpwin, HELPWINLINES - 1, HELPWINBOTCOL);
    waddstr(helpwin, HELPWINBOTMSG);
}

/* Set the numbers of the file entries in the linked list */
setfilenumbers()
{
    ENTPTR          rover;
    int             count = 0;

    rover = efirst;
    while (rover != NULL) {
	rover->number = ++count;
	rover = rover->next;
    }
}

/* Set CFPTR to CFNUM */
setcfptr()
{
    int             i;

    cfptr = efirst;
    for (i = 1; i < cfnum; i++)
	cfptr = cfptr->next;
}

/* Enable/Disable GROUP Command Message */
setgroup(text)
    char           *text;
{
    wmove(curfilewin, 1, GROUPLOC);
    wprintw(curfilewin, "%5s", text);
    wrefresh(curfilewin);
}

/* Process help request (? command) */
helpme()
{
    touchwin(helpwin);
    wrefresh(helpwin);
}

/* Execute command line via the subshell */
dosubshell(cmnd)
    char           *cmnd;
{
#ifdef TURBOC
    dosystem(cmnd);
#else
#if	ENABLESIBSHELL
    char            pcommand[MAXNAMELEN];

    cprocess(cmnd, pcommand);	/* Preprocess command line */
    strcpy(savefile, &(cfptr->name[1]));	/* Save current file
						 * name */
    clear();
    refresh();
    /*
     * execute command; the wait will notify the parent process when
     * the shell (child) is done processing this line 
     */
    if (fork() == 0) {
	noraw();
	echo();
	pid = getpid();
	sprintf(Cmd, "%s;kill %d", pcommand, pid);
	fprintf(Shell, "%s\n", Cmd);
	fflush(Shell);
	pause();		/* infinite wait ... child is killed
				 * by shell */
    } else
	wait(&shdummy);		/* parent waits for child to be
				 * killed */
    cmdret();
#else
    dosystem(cmnd);
#endif				/* ENABLESIBSHELL */
#endif				/* TURBOC */
}

/* Execute command line via system */
dosystem(cmnd)
    char           *cmnd;
{
    char            pcommand[MAXNAMELEN];

    cprocess(cmnd, pcommand);	/* Preprocess command line */
    strcpy(savefile, &(cfptr->name[1]));	/* Save current file
						 * name */
    clear();
    refresh();
    noraw();
    echo();
    system(pcommand);
    cmdret();
}

/* Cleanup after returning from remote command */
cmdret()
{
    raw();
    noecho();
    xreload();
    findf(savefile);		/* Locate current file */
    screfresh();
}

/* Preprocess command line */
cprocess(source, target)
    char           *source, *target;
{
    char           *rover;

    while (*source) {
	if (*source == PTRCHAR) {
	    source++;
	    if (*source != PTRCHAR) {
		switch (*source) {
		case 'd':
		    rover = dname;
		    rover++;
		    *target++ = '/';
		    while (*rover)
			*target++ = *rover++;
		    break;
		case 'f':
		    rover = cfptr->name;
		    rover++;
		    while (*rover)
			*target++ = *rover++;
		    break;
		case 'n':
		    rover = cfptr->name;
		    rover++;
		    while (*rover) {
			if (*rover == '.')
			    break;
			*target++ = *rover++;
		    }
		    break;
		default:
		    *target++ = PTRCHAR;
		    *target++ = *source;
		    if (*source == '\0')
			source--;	/* for following ++ */
		    break;
		}
		source++;
	    } else
		*target++ = *source++;
	} else
	    *target++ = *source++;
    }
    *target = '\0';
}

/*
 * Find text in first part of a file name (set CURFRAME, CURSTART,
 * CFNUM, and CFPTR) 
 */
findf(text)
    char           *text;
{
    if (*text) {
	cfnum = 2;
	cfptr = efirst->next;
	while (cfptr->next != NULL) {
	    if (strcmp(text, &(cfptr->name[1])) == 0)
		break;
	    if (strcmp(text, &(cfptr->name[1])) < 0) {
		cfnum--;
		cfptr = cfptr->last;
		break;
	    }
	    cfnum++;
	    cfptr = cfptr->next;
	}
	curframe = (cfnum - 1) / DIRWINMAXENT;
	curstart = (curframe * DIRWINMAXENT) + 1;
    }
}

/* Advance to next position on screen */
advance(ldcnt)
    int             ldcnt;
{
    int             target;

    if (cfnum == curstart + ldcnt - 1)
	target = curstart;
    else
	target = cfnum + 1;
    moveupdate(target);
}

/* Move arrow and update display */
moveupdate(target)
    int             target;
{
    displaycur(target);
    displaycf();
    refresh();
}

/* Refresh the screen with a leading clear */
screfresh()
{
    clear();
    ncscrefresh();
}

/* Refresh the screen at the current position */
ncscrefresh()
{
    displaymain(dname);
    refresh();
    displaydir(curstart);
    displaycur(cfnum);
    displaycf();
}

/* Center string into CENTERSTR and return pointer to it */
char           *
center(text, size)
    char           *text;
    int             size;
{
    int             offset, i;
    char           *rover;
    static char     centerstr[80];

    offset = (size - strlen(text)) / 2;
    rover = centerstr;
    for (i = 0; i < offset; i++)
	*rover++ = ' ';
    strcpy(rover, text);
    return centerstr;
}

/* Position cursor and get command */
command()
{
    move(23, 0);
    refresh();
    return getch() & 0x7f;
}

/* Display main window */
displaymain(dirname)
    char           *dirname;
{
    char           *rover;
    int             i;

    move(TITLEY, 0);
    addstr(TITLEHELP);
    move(TITLEY, TITLEX);
    addstr(TITLE);
    move(TITLEY, TITLESIZELOC);
    printw("%10d Files", filecount);
    if (strlen(dirname) > 78) {
	rover = dirname;
	for (i = strlen(dirname); i > 78; i--)
	    rover++;
	move(DIRNAMEY, 1);
	printw("%s", rover);
    } else {
	move(DIRNAMEY, ((80 - strlen(dirname)) / 2));
	printw("%s", dirname);
    }
}

/* Display directory starting at first (sets LDCNT and MORE) */
displaydir(first)
    int             first;
{
    ENTPTR          rover;

    rover = efirst;
    for (; --first > 0;)
	if (rover != NULL)
	    rover = rover->next;
    for (ldcnt = 0; ldcnt < DIRWINMAXENT; ldcnt++)
	if (rover == NULL)
	    break;
	else {
	    displayent(rover, ldcnt);
	    rover = rover->next;
	}
    if ((ldcnt == DIRWINMAXENT) && (rover != NULL))
	more = 1;
    else
	more = 0;
}

/* Display entry at ith position */
displayent(ptr, position)
    ENTPTR          ptr;
    int             position;
{
    int             y, x, i;
    char            nbuf[DIRENTSIZE + 1];
    char           *rover1, *rover2;

    rover1 = ptr->name;
    rover2 = nbuf;
    for (i = 0; (i < DIRENTSIZE) && *rover1; i++)
	*rover2++ = *rover1++;
    *rover2++ = *rover1;
    if (*rover1) {
	if (*(++rover1)) {
	    rover2--;
	    *rover2++ = '*';
	    *rover2 = '\0';
	} else
	    *rover2 = '\0';
    }
    y = (position / 4);
    x = (position % 4) * (DIRENTSIZE + 4) + 4;
    wmove(dirwin, y, x);
    wprintw(dirwin, entformat, nbuf);
}

/* Clear cursor from old position and display it at new position */
displaycur(target)
    int             target;
{
    int             y, x;

    y = (cfnum - curstart) / 4;
    x = ((cfnum - curstart) % 4) * (DIRENTSIZE + 4) + 1;
    mvwaddstr(dirwin, y, x, "   ");
    cfnum = target;
    setcfptr();
    y = (cfnum - curstart) / 4;
    x = ((cfnum - curstart) % 4) * (DIRENTSIZE + 4) + 1;
    mvwaddstr(dirwin, y, x, "-->");
    wrefresh(dirwin);
}

/* Display the tag character of the current file */
displaytag(tagch)
    char            tagch;
{
    int             y, x;

    y = (cfnum - curstart) / 4;
    x = ((cfnum - curstart) % 4) * (DIRENTSIZE + 4) + 4;
    wmove(dirwin, y, x);
    waddch(dirwin, tagch);
    wrefresh(dirwin);
}

/* Display information on current file */
displaycf()
{
    char            dfname[CURFENTSIZE + 1], *r1, *r2;
    int             i;

    r1 = dfname;
    r2 = cfptr->name;
    for (i = 0; i < CURFENTSIZE; i++)
	if (*r2)
	    *r1++ = *r2++;
	else
	    break;
    *r1 = '\0';
    mvwaddstr(curfilewin, 0, 0, CFNTEXT);
    mvwaddstr(curfilewin, 1, 10, CFCTEXT);
    mvwaddstr(curfilewin, 1, 30, CFSTEXT);
    wmove(curfilewin, 0, CURFNAME);
    wprintw(curfilewin, curfentfmt, dfname);
    mvwaddstr(curfilewin, 1, CURFCODE, cfptr->code);
    wmove(curfilewin, 1, CURFSIZE);
    wprintw(curfilewin, "%-10d", cfptr->size);
    if (taggedsize) {
	wmove(curfilewin, 1, TAGGEDMSG);
	wprintw(curfilewin, "%-7s", CFTTEXT);
	wmove(curfilewin, 1, TAGGEDBYTES);
	wprintw(curfilewin, "%-10d", taggedsize);
    } else {
	wmove(curfilewin, 1, TAGGEDMSG);
	wprintw(curfilewin, "%-7s", "");
	wmove(curfilewin, 1, TAGGEDBYTES);
	wprintw(curfilewin, "%10s", "");
    }
    wrefresh(curfilewin);
}

/* Process command message */
cmdmsg(message, answer)
    char           *message, *answer;
{
    char           *center();

    box(cmdmsgwin, '|', '-');
    mvwaddstr(cmdmsgwin, 0, 2, " Command Message ");
    wstandout(cmdmsgwin);
    wmove(cmdmsgwin, 1, 1);
    wprintw(cmdmsgwin, cmdmsgfmt, center(message, CMDMSGWINCOLS - 2));
    wstandend(cmdmsgwin);
    wmove(cmdmsgwin, 2, 1);
    wprintw(cmdmsgwin, cmdmsgfmt, "");
    wrefresh(cmdmsgwin);
    getline(cmdmsgwin, 2, 1, answer);
    touchwin(dirwin);
    wrefresh(dirwin);
}

/* Input a line with editing from the console */
getline(win, line, col, buffer)
    WINDOW         *win;
    int             line, col;
    char           *buffer;
{
    int             ccnt = 0;
    char            ch;

    wmove(win, line, col);
    wrefresh(win);
    while (((ch = wgetch(win) & 0x7f) != '\r') && (ch != '\n')) {
	if (ch == erasechar()) {
	    if (ccnt) {
		col--;
		ccnt--;
		buffer--;
		wmove(win, line, col);
		waddch(win, ' ');
		wmove(win, line, col);
		wrefresh(win);
	    }
	} else {
	    switch (ch) {
	    case CTRLH:
	    case DELETE:
		if (ccnt) {
		    col--;
		    ccnt--;
		    buffer--;
		    wmove(win, line, col);
		    waddch(win, ' ');
		    wmove(win, line, col);
		    wrefresh(win);
		}
		break;
	    case CTRLU:
		if (ccnt) {
		    while (ccnt) {
			col--;
			ccnt--;
			buffer--;
			wmove(win, line, col);
			waddch(win, ' ');
			wmove(win, line, col);
			wrefresh(win);
		    }
		}
		break;
	    case CTRLI:
		*buffer++ = ' ';
		ccnt++;
		col++;
		waddch(win, ' ');
		wrefresh(win);
		break;
	    default:
		if (ch >= ' ') {
		    *buffer++ = ch;
		    ccnt++;
		    col++;
		    waddch(win, ch);
		    wrefresh(win);
		}
		break;
	    }
	}
    }
    *buffer = '\0';
}

/* Process information message */
infomsg(message, standby)
    char           *message;
    int             standby;
{
    char           *center();

    box(infomsgwin, '|', '-');
    mvwaddstr(infomsgwin, 0, 2, " Information Message ");
    wmove(infomsgwin, 1, 1);
    wstandout(infomsgwin);
    wprintw(infomsgwin, cmdmsgfmt, center(message, CMDMSGWINCOLS - 2));
    wstandend(infomsgwin);
    if (standby) {
	wmove(infomsgwin, 2, 1);
	wprintw(infomsgwin, cmdmsgfmt, center("<< Please Stand By >>",
					      CMDMSGWINCOLS - 2));
    }
    wrefresh(infomsgwin);
    move(23, 0);
    refresh();
}

/* Clear information message */
clrinfomsg()
{
    touchwin(dirwin);
    wrefresh(dirwin);
}

/* Process confirm message */
confirmmsg(message)
    char           *message;
{
    char            ch, *center();

    box(confirmwin, '|', '-');
    mvwaddstr(confirmwin, 0, 2, " Confirmation Required ");
    mvwaddstr(confirmwin, 2, CONFIRMWINBOTCOL, CONFIRMWINBOTMSG);
    wstandout(confirmwin);
    wmove(confirmwin, 1, 1);
    wprintw(confirmwin, confirmentfmt, center(message, CONFIRMWINCOLS - 2));
    wstandend(confirmwin);
    wrefresh(confirmwin);
    ch = command();
    touchwin(dirwin);
    wrefresh(dirwin);
    return (ch == 'y');
}

/* Process error message */
errormsg(message)
    char           *message;
{
    char           *center();

    box(errorwin, '|', '-');
    mvwaddstr(errorwin, 0, 2, " Error ");
    mvwaddstr(errorwin, 2, ERRORWINBOTCOL, ERRORWINBOTMSG);
    wstandout(errorwin);
    wmove(errorwin, 1, 1);
    wprintw(errorwin, errentfmt, center(message, ERRORWINCOLS - 2));
    wstandend(errorwin);
    wrefresh(errorwin);
    if (command() == 'x') {
	echo();
	noraw();
	endwin();
	puts("\n");
	exit(1);
    }
    touchwin(dirwin);
    wrefresh(dirwin);
}

/* Load directory */
dir(dirname)
    char           *dirname;
{
    DIR            *opendir(), *dirp;
#ifdef POSIX
    struct dirent  *current_entry, *readdir();
#else
    struct direct  *current_entry, *readdir();
#endif				/* POSIX */
    char            current_name[256];
    struct stat     statbuf;
    int             stat();
    char           *strcat(), *strncpy();
    ENTPTR          rover;

    /* Free up space in current linked list */
    while (efirst != NULL) {
	rover = efirst;
	efirst = efirst->next;
	free(rover->name);
	free((char *) rover);
    } filecount = 0;
    taggedsize = 0;

    /* Try to open dirname as a directory */
    if ((dirp = opendir(dirname)) == NULL) {
	/* dirname was not a directory, so error */
	return VF_NOTDIR;
    } else {
	/*
	 * dirname is a directory, so read and store all entries in
	 * it 
	 */
	gotodir(dirname);
	for (current_entry = readdir(dirp); current_entry != NULL;
	     current_entry = readdir(dirp)) {
	    strncpy(current_name, current_entry->d_name,
		    dirsize(current_entry));
	    /*
	     * Generate full file name (with directory prefix) and
	     * get status information of it and store 
	     */
	    stat(current_name, &statbuf);
	    store_entry(current_name, &statbuf);
	}
	closedir(dirp);
    }
    setfilenumbers();
    cfnum = 1;
    setcfptr();
    return VF_OK;
}

/* Store a line of the directory display in the linked list */
store_entry(filename, statbuf)
    char           *filename;
    struct stat    *statbuf;
{
    ENTPTR          output, erover, elast;
    int             shift;
    int             str, stw, stx;

    /* Check for hidden file and inchidden option */
    if ((strcmp(SPECIAL1, filename) != 0) &&
	(strcmp(SPECIAL2, filename) != 0)) {
	if (!inchidden && (*filename == '.'))
	    return;
    }
    /*
     * Increment file count and total size and allocate memory for
     * new entry 
     */
    filecount++;
    if ((output = (ENTPTR) (malloc(sizeof(ENT)))) == NULL) {
	fprintf(stderr, " Dynamic Memory Overflow\n");
	exit(1);
    }
    if ((output->name = (char *) (malloc(strlen(filename) + 2))) == NULL) {
	fprintf(stderr, " Dynamic Memory Overflow\n");
	exit(1);
    }
    /*
     * Determine if we are same user, group, or world as target file
     * and set shift count accordingly 
     */
#ifdef TURBOC
    str = statbuf->st_mode & S_IREAD;
    stw = statbuf->st_mode & S_IWRITE;
    stx = statbuf->st_mode & S_IEXEC;
#else
    if (statbuf->st_uid == getuid()) {
#ifdef POSIX
	str = statbuf->st_mode & S_IRUSR;
	stw = statbuf->st_mode & S_IWUSR;
	stx = statbuf->st_mode & S_IXUSR;
#else
	shift = 0;
#endif				/* POSIX */
    } else {
	if (statbuf->st_gid == getgid()) {
#ifdef POSIX
	    str = statbuf->st_mode & S_IRGRP;
	    stw = statbuf->st_mode & S_IWGRP;
	    stx = statbuf->st_mode & S_IXGRP;
#else
	    shift = 3;
#endif				/* POSIX */
	} else {
#ifdef POSIX
	    str = statbuf->st_mode & S_IROTH;
	    stw = statbuf->st_mode & S_IWOTH;
	    stx = statbuf->st_mode & S_IXOTH;
#else
	    shift = 6;
#endif				/* POSIX */
	}
    }
#ifndef POSIX
    str = statbuf->st_mode & (S_IREAD >> shift);
    stw = statbuf->st_mode & (S_IWRITE >> shift);
    stx = statbuf->st_mode & (S_IEXEC >> shift);
#endif				/* POSIX */
#endif				/* TURBOC */

    /* Generate output string */
    sprintf(output->name, "%c%s",
#ifdef POSIX
	    ((S_ISDIR(statbuf->st_mode)) ? DIRCHAR : ' '),
#else
	    ((statbuf->st_mode & S_IFDIR) ? DIRCHAR : ' '),
#endif				/* POSIX */
	    filename);
    output->size = statbuf->st_size;
    sprintf(output->code, "%c%c%c",
	    (str ? 'r' : '-'),
	    (stw ? 'w' : '-'),
	    (stx ? 'x' : '-'));


    /* Place output record in linked list */
    if (efirst == NULL) {
	efirst = output;
	output->next = NULL;
	output->last = NULL;
    } else {
	erover = efirst;
	elast = NULL;
	while ((erover != NULL) && (strcmp(erover->name,
					   output->name) < 0)) {
	    elast = erover;
	    erover = erover->next;
	}
	if (erover != NULL) {
	    output->next = erover;
	    output->last = erover->last;
	    erover->last = output;
	} else {
	    output->next = NULL;
	    output->last = elast;
	}
	if (elast == NULL)
	    efirst = output;
	else
	    elast->next = output;
    }
}
