/*
 * rmi - remove files by inode
 *
 * Rmi takes arguments of the form "directory/inode"; where the "directory/"
 * is an optional pathname defaulting to ".", and "inode" is a numeric inode
 * number of a file in that directory.
 *
 * For each file in the directory with inode as its inode number, the user
 * is prompted with "Remove %s?" where %s corresponds to the path and name
 * of the file.  The name is made printable by converting control characters
 * to "^X".
 *
 * David A. Curry, 8/15/83
 */
#include <sys/param.h>
#include <sys/dir.h>
#include <stdio.h>

char *pname;

main(argc, argv)
int argc;
char **argv;
{
	ino_t inode;
	char *dir, *split();

	pname = *argv;

	if (argc < 2) {
		fprintf(stderr, "Usage: %s [directory]/inode ...\n", pname);
		exit(1);
	}

	/*
	 * Go through the arguments, splitting each one into a pathname
	 * and an inode.  Then remove each file.
	 */
	while (--argc) {
		if ((dir = split(*++argv, &inode)) == NULL) {
			fprintf(stderr, "%s: cannot decipher \"%s\".\n", pname, *argv);
			continue;
		}

		remove(dir, inode);
	}

	exit(0);
}

/*
 * split - takes a directory/inode argument and splits it into directory and
 *	   inode.
 */
char *split(s, ino)
char *s;
ino_t *ino;
{
	long atol();
	register char *t;
	char *p, *rindex();

	/*
	 * If there is no "/" in the string, then a directory
	 * wasn't given, and we default to the current directory (".").
	 * If there is a "/", and it's the first character of the
	 * string, then we set the directory to "/", otherwise, we
	 * stick a null in between the directory and the inode.
	 */
	if ((p = rindex(s, '/')) == NULL) {
		p = s;
		s = ".";
	}
	else {
		if (p == s)
			s = "/";
			
		*p++ = NULL;
	}

	/*
	 * If there is no inode, or if the inode
	 * has non-digits in it, send back an error.
	 */
	if (*p == NULL)
		return(NULL);

	for (t=p; *t; t++) {
		if ((*t < '0') || (*t > '9'))
			return(NULL);
	}

	/*
	 * Convert the inode to a number, and return
	 * the directory.
	 */
	*ino = (ino_t) atol(p);
	return(s);
}

/*
 * remove - reads directory dir, and for each file with inode number ino,
 *	    prompts the user to ask whether or not to remove it.
 */
remove(dir, ino)
char *dir;
ino_t ino;
{
	int n;
	char *file;
	char *printable();
	DIR *dirp, *opendir();
	char tmp[MAXNAMLEN], buf[8];
	struct direct *dbuf, *readdir();

	/*
	 * Read the directory.
	 */
	if ((dirp = opendir(dir)) == NULL) {
		fprintf(stderr, "%s: cannot open directory \"%s\".\n", pname, dir);
		return;
	}

	/*
	 * For each file in the directory....
	 */
	n = 0;
	while ((dbuf = readdir(dirp)) != NULL) {
		/*
		 * If the inodes don't match, skip it.
		 */
		if (dbuf->d_ino != ino)
			continue;

		/*
		 * Count the number of times we
		 * find the right inode.
		 */
		n++;

		/*
		 * Put the name into tmp so it's null terminated,
		 * and set file to the printable version of the
		 * name.
		 */
		strncpy(tmp, dbuf->d_name, MAXNAMLEN);
		file = printable(tmp);

		/*
		 * Prompt the user and read his response.
		 */
		printf("Remove \"%s/%s\"? ", dir, file);
		gets(buf);

		if ((*buf == 'n') || (*buf == 'N'))
			continue;

		/*
		 * Get rid of the file.
		 */
		if (unlink(tmp) < 0) {
			fprintf(stderr, "%s: ", pname);
			perror("unlink");
		}
	}
}

/*
 * printable - goes through s, copying characters to buf.  If the character
 *	       is non-printable, it is converted to "^X".
 */
char *printable(s)
char *s;
{
	char c, *t;
	static char buf[64];

	t = buf;

	while (*s) {
		c = *s;

		if (((c &= 0177) < 040) || (c == 0177)) {
			c |= 0100;
			*t++ = '^';
			*t++ = c;
		}
		else {
			*t++ = c;
		}

		s++;
	}

	*t = 0;
	return(buf);
}
