/* Routines for manipulating a "property list" file:
 *
 *------
 * name-value-pair
 * name-value-pair
 *
 * Anything after the blank line is ignored by getprop, maintained by
 * putprop/remprop.
 *------
 * The name/value pairs can be in various formats:
 *
 * RFC: This is in RFC format (default). Space after the colon, and
 * 	property continued on sucessive lines by indenting it.
 *
 * Makefile: This is in Makfile format. This is similar to RFC format,\
 * but continued lines have a backslash at the end.
 *
 * Shell='This is in shell format: no space after equal sign, property
 * in single quotes on multiple lines. Quotes inside the line are escaped
 * like so: \'. Will accept "..." quotes and mixed quotes, but will turn
 * it back into the standard form if the file is modified.'
 *
 * For future work: a c-shell format:
 *
 * set Cshell='This is what the c-shell format would be like: the keyword \
 * "set" would be required, and lines would be continued with a space and a \
 * backslash. I don't want to bother with this one.'
 */
#include <stdio.h>
#include <ctype.h>

#include "prop.h"

char TagChar = NULL;	/* What character we're using to tag variables */
char DefaultTag = ':';	/* what character to use for a new proplist file */
unsigned long Flags = NOFLAG;	/* various flags affecting our behaviour */
unsigned char Style = RFC;	/* Style of variables */

#ifdef RENAME
static rename(from, to)
char *from, *to;
{
	if(link(from, to) == 0)
		unlink(from);
}
#endif

/* Open a file and read a particular property out of it */

/* Arguments: name of file, name of property, and buffer to return value */

/* Returns: ERROR if file doesn't exist or is malformed, FAILURE if the
 * property is not found. SUCCESS if the property is found.
 */
getprop(file, prop, val)
char *file;
char *prop;
char val[BUFSIZ];
{
	FILE *fp;
	char *propname, *propval;
	char buffer[BUFSIZ];
	int state;

	state = ERROR;

	if(!(fp = fopen(file, "r"))) {
		perror(file);
		return state;
	}

	state = FAILURE;

	while(getline(fp, buffer, &propname, &propval)) {
		if(match(prop, propname)) {
			if(state != SUCCESS) {
				state = SUCCESS;
				val[0] = 0;
			} else {
				if(strlen(val) + strlen(propval) + 1 > BUFSIZ) {
					state = ERROR;
					break;
				}
				strcat(val, "\n");
			}
			strcat(val, propval);
			if( (Flags&ALLFLAG) == NOFLAG )
				break;
		}
	}
	fclose(fp);
	return state;
}

/* Open a file and insert a given property in it. */

/* Arguments: name of file, name of property, and value */

/* Results: ERROR if file can't be opened or other problem occurs.
 * SUCCESS if property sucessfully saved.
 */
putprop(file, prop, val)
char *file;
char *prop;
char *val;
{
	FILE *fp, *tfp;
	char tempfile[BUFSIZ];
	char buffer[BUFSIZ], *propname, *propval;

	fp = fopen(file, "r");

	buildtemp(tempfile, file);

	if(!(tfp = fopen(tempfile, "w"))) {
		perror(tempfile);
		if(fp) fclose(fp);
		return ERROR;
	}

	if(fp) {
		while(getline(fp, buffer, &propname, &propval)) {
			if(!match(prop, propname)) {
				if(!putline(tfp, propname, propval)) {
					fclose(fp);
					fclose(tfp);
					perror(tempfile);
					unlink(tempfile);
					return ERROR;
				}
			} else {
				if(!putline(tfp, prop, val)) {
					if(fp) fclose(fp);
					fclose(tfp);
					perror(tempfile);
					unlink(tempfile);
					return ERROR;
				}
				*prop = 0;
			}
		}
	}

	if(*prop && !putline(tfp, prop, val)) {
		if(fp) fclose(fp);
		fclose(tfp);
		perror(tempfile);
		unlink(tempfile);
		return ERROR;
	}

	if(fp) {
		if(!copyfile(fp, tfp)) {
			fclose(fp);
			fclose(tfp);
			perror(tempfile);
			unlink(tempfile);
			return ERROR;
		}
	}
	if(fp) fclose(fp);
	fclose(tfp);
	if(fp) unlink(file);
	rename(tempfile, file);
	return SUCCESS;
}

/* Removes an entry from a file. */

/* Arguments: file name and property. */

/* Returns: SUCCESS or ERROR. It sould probably be rewritten to avoid
 * rewriting the file if the property isn't found.
 */
remprop(file, prop)
char *file;
char *prop;
{
	FILE *fp, *tfp;
	char tempfile[BUFSIZ];
	char buffer[BUFSIZ], *propname, *propval;

	if(!(fp = fopen(file, "r"))) {
		perror(file);
		return ERROR;
	}

	buildtemp(tempfile, file);

	if(!(tfp = fopen(tempfile, "w"))) {
		perror(tempfile);
		fclose(fp);
		return ERROR;
	}

	while(getline(fp, buffer, &propname, &propval))
		if(!match(prop, propname))
			if(!putline(tfp, propname, propval)) {
				fclose(fp);
				fclose(tfp);
				perror(tempfile);
				unlink(tempfile);
				return ERROR;
			}

	if(!copyfile(fp, tfp)) {
		fclose(fp);
		fclose(tfp);
		perror(tempfile);
		unlink(tempfile);
		return ERROR;
	}
	fclose(fp);
	fclose(tfp);
	unlink(file);
	rename(tempfile, file);
	return SUCCESS;
}

/* Read a line in RFC format, with possible continuations */
rfcreadline(fp, buffer)
FILE *fp;
char buffer[BUFSIZ];
{
	int c;
	char *p;

	p = buffer;
	while((c = getc(fp)) != EOF) {
		if(c=='\n') {
			c = getc(fp);
			if(c==EOF) break;
			if(c!='\t' && c!=' ') {
				ungetc(c, fp);
				break;
			}
			c = '\n';
		}
		if(p - buffer < BUFSIZ-1)
			*p++ = c;
	}
	*p = 0;
}

/* Read a line in Makefile format, with possible continuations */
makereadline(fp, buffer)
FILE *fp;
char buffer[BUFSIZ];
{
	int c;
	char *p;

	p = buffer;
	while((c = getc(fp)) != EOF) {
		if(c == '\\') {
			c = getc(fp);
			if(c==EOF) break;
			if(c!='\n') {
				ungetc(c, fp);
				c = '\\';
			}
		} else if(c=='\n') {
			break;
		}
		if(p - buffer < BUFSIZ-1)
			*p++ = c;
	}
	*p = 0;
}

/* read a line in /bin/sh format, with possible continuations */

/* Tries to interpret quotes, backslash, semicolons, and comments.
 * Semicolons will be replaced by linefeeds on output. Comments will
 * be eaten.
 */

/* Backslash handling is rather simple: backslash is kept before
 * quotes to simplify regeneration of the string. Other escapes
 * are simply replaced by the character, so will be hidden by the
 * quoted string when the line is re-output.
 */
shreadline(fp, buffer)
FILE *fp;
char buffer[BUFSIZ];
{
	int c;
	char *p;

	p = buffer;
	while((c = getc(fp)) != EOF) {
		if(c=='\\') { /* handle backslash */
			c = getc(fp);
			if(c!='\'')
				if(p - buffer < BUFSIZ-1)
					*p++ = '\\';
			if(c==EOF) break;
			if(p - buffer < BUFSIZ-1)
				*p++ = c;
		} else if(c=='"' || c=='\'') {
			char quote = c;
			while((c = getc(fp)) != EOF) {
				if(c=='\\') {
					c = getc(fp);
					if(p - buffer < BUFSIZ-1)
						*p++ = '\\';
					if(c==EOF) break;
					if(p - buffer < BUFSIZ-1)
						*p++ = c;
				} else {
					if(c==quote)
						break;
					if(p - buffer < BUFSIZ-1)
						*p++ = c;
				}
			}
			if(c==EOF) break;
		} else if(strchr(" \t\n;#", c)) {
			break;
		} else {
			if(p - buffer < BUFSIZ-1)
				*p++ = c;
		}
	}
	*p = 0;
}

/* Read a line... see what the format is and call the correct routine. Then
 * convert the line into name-value pair.
 */
getline(fp, buffer, nameptr, valptr)
FILE *fp;
char buffer[BUFSIZ];
char **nameptr;
char **valptr;
{
	int c;
	char *p;

	if(feof(fp))
		return 0;

	if(Style == RFC)
		rfcreadline(fp, buffer);
	else if(Style == MAKEFILE)
		makereadline(fp, buffer);
	else if(Style == BINSH)
		shreadline(fp, buffer);

	p = buffer;

	while(isspace(*p))
		p++;

	if(!*p)
		return 0;

	*nameptr = p;
	while(*p && *p != TagChar && *p != ':' && *p != '=') {
		p++;
	}

	if(*p) {
		if(TagChar == NULL)
			TagChar = *p;
		*p++ = 0;
		while(isspace(*p))
			p++;
	}

	*valptr = p;
}

/* Disctionary comparison of two strings */
match(s1, s2)
char *s1, *s2;
{
	register char c1, c2;

	while(*s1 && *s2) {
		c1 = *s1++;
		c2 = *s2++;

		if(isupper(c1))
			c1 = tolower(c1);
		if(isupper(c2))
			c2 = tolower(c2);
		
		if(c1 != c2)
			return 0;
	}

	return !(*s1 || *s2);
}

/* Putline: puts the name, the tag, and the value. Does various aesthetic
 * stuff and quotes continuations and so on.
 */
putline(fp, name, val)
FILE *fp;
char *name, *val;
{
	while(*name) {
		putc(*name, fp);
		name++;
	}
	if(*val || Style == BINSH) {
		if(!TagChar)
			TagChar = DefaultTag;
		putc(TagChar, fp);
	}
	if(*val) {
		if(Style == RFC || Style == MAKEFILE) { /* EZ, just continue */
			putc(' ', fp);
			while(*val) {
				if(*val == '\n' && Style == MAKEFILE)
					putc('\\', fp);
				putc(*val, fp);
				if(*val=='\n' && Style == RFC)
					putc('\t', fp);
				val++;
			}
		} else { /* More complex, have to quote ' and \ as well */
			putc('\'', fp);
			while(*val) {
				if(*val=='\\') {
					putc(*val, fp);
					val++;
					if(!*val) break;
				} else if(*val=='\'') {
					putc('\\', fp);
				}
				putc(*val, fp);
				val++;
			}
			putc('\'', fp);
		}
	}
	putc('\n', fp);
}

/* Copy everything from one stream to another. Used to copy rest of file
 * after the header.
 */
copyfile(fp1, fp2)
FILE *fp1, *fp2;
{
	char buffer[BUFSIZ];
	if(feof(fp1))
		return;

	putc('\n', fp2);
	while(fgets(buffer, BUFSIZ, fp1))
		if(!fputs(buffer, fp2))
			return 0;

	return 1;
}

/* Creates a temp file in the same directory (thus same file system) as
 * the original.
 */
buildtemp(temp, file)
char temp[BUFSIZ];
char *file;
{
	char *p, *s, *strrchr();

	s = strrchr(file, '/');

	p = temp;

	if(s)
		s++;
	else
		*p++ = '@';

	while(*file) {
		if(file==s)
			*p++ = '@';
		*p++ = *file++;
	}

	*p++ = 0;
}
