/*
 * name - find a file's real name
 */

#include <stdio.h>
#include <errno.h>
#include <strings.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/dir.h>

char usage[] = "Usage: name [-ainv] pathname...\n";

int absolute;				/* print absolute pathnames */
int normalize;				/* normalize pathnames */
int ignore;				/* ignore symbolic links */
int verbose;				/* describe each symbolic link */

main (argc, argv)
char **argv;
{
    int option;
    extern int optind;
    int errors = 0;
    char realname[MAXPATHLEN + MAXNAMLEN];
    extern char *name();

    while ((option = getopt (argc, argv, "ainv")) != EOF)
	switch (option)
	{
	  case 'a':
	    absolute += 1;
	    break;
	    
	  case 'i':
	    ignore += 1;
	    break;
	    
	  case 'n':
	    normalize += 1;
	    break;
	    
	  case 'v':
	    verbose += 1;
	    break;
	    
	  default:
	    fputs (usage, stderr);
	    exit (1);
	}
    
    argc -= optind;
    argv += optind;
    
    if (argc == 0)			/* have to have a path... */
    {
	fputs (usage, stderr);
	exit (1);
    }

    while (argc-- > 0)
    {
	if (name (*argv++, realname) == 0)
	{				/* print error returned by name() */
	    fputs (realname, stderr);
	    (void) putc ('\n', stderr);
	    errors++;
	}
	else				/* print the real name itself */
	    puts (realname);
    }

    exit (errors);
}

static int links;			/* how many symbolic links in path */

char *name (path, truename)
char *path;
char *truename;
{
    static char prefix[MAXPATHLEN + MAXNAMLEN + 1];
    extern char *getwd();

    if (*path != '/' && absolute)	/* get absolute pathname of relative */
    {
	if (getwd (prefix) == NULL)	/* system five doesn't have this... */
	{
	    strcpy (truename, prefix);	/* contains error message from getwd */
	    return (NULL);
	}
	strcat (prefix, "/");		/* add trailing '/' */
    }    
    else
	prefix[0] = '\0';

    links = 0;

    if (name1 (prefix, path) == 0)	/* an error occurred */
    {
	strcpy (truename, prefix);	/* copy back the error message */
	return (NULL);
    }    
    else
    {
	strcpy (truename, prefix);	/* copy back the real name */
	return (truename);
    }
}


#define rootdir(name) ((name)[0]=='/' && (name)[1]=='\0')

#define dotdir(name) \
((name)[0]=='.' && ((name)[1]=='\0' || ((name)[1]=='.' && (name)[2]=='\0')))

#define dotdotdir(name) \
((name)[0]=='.' && (name)[1]=='.' && (name)[2]=='\0')


/*
 * Recursively add suffix to prefix, canonicalizing as we go...
 */

static name1 (prefix, suffix)
char *prefix;
register char *suffix;
{
    extern char *sys_errlist[];
    char link[MAXPATHLEN];
    struct stat status;
    register char *splice;
    register char *cut;
    register int cc;
    int result;

    splice = prefix + strlen (prefix);

    do
    {
	if (!normalize)
	{
	    if (*suffix == '/')		/* skip one leading "/" */
	    {
		*splice++ = '/';
		*splice = '\0';
		suffix++;
	    }
	}
	else
	{
	    if (*suffix == '/')
	    {
		while (*suffix == '/')
		    suffix++;		/* treat "//" as "/" */

		if (splice == prefix ||
		    (*suffix != '\0' && splice[-1] != '/'))
		{
		    *splice++ = '/';
		    *splice = '\0';
		}
	    }

	    if (*suffix == '.')
	    {
		if (suffix[1] == '\0')
		{			/* treat "." as "." (not "") */
		    if (splice == prefix)
		    {
			*splice++ = '.';
			*splice = '\0';
		    }			/* treat "x/." as "x" */
		    else if (!rootdir (prefix))
			*--splice = '\0';
		    return (1);		/* treat "/." as "/" */
		}

		if (suffix[1] == '/')
		{
		    suffix += 2;	/* treat "/./" as "/" */
		    continue;
		}

		if (suffix[1] == '.')
		{
		    if (suffix[2] == '\0')
		    {			/* treat ".." as ".." */
			if (prefix == splice)
			    strcpy (prefix, "..");

			else if (!rootdir (prefix))
			{
			    *--splice = '\0';
			
			    if ((splice = rindex (prefix, '/')) == 0)
			    {
				if (dotdotdir (prefix))
				{
				    strcat (prefix, "/..");
				    return (1);
				}
				else
				    splice = prefix;
			    }
			    else if (dotdotdir (splice + 1))
			    {
				strcat (splice, "/..");
				return (1);
			    }

			    *splice= '\0';
			}			

			return (1);
		    }

		    if (suffix[2] == '/')
		    {
			if (splice == prefix)
			{
			    strcpy (prefix, "..");
			    splice += 2;
			    suffix += 2;
			}

			else if (!rootdir (prefix))
			{		/* don't back up "/" prefix */
			    *--splice = '\0';
			
			    if ((splice = rindex (prefix, '/')) == 0)
			    {
				if (dotdotdir (prefix))
				{
				    strcat (prefix, "/..");
				    splice = prefix + strlen (prefix);
				    suffix += 2;
				}
				else
				{
				    splice = prefix;
				    *splice = '\0';
				    suffix += 3;
				}
			    }
			    else
			    {
				if (dotdotdir (splice))
				{
				    strcat (splice, "..");
				    splice += strlen (splice);
				    suffix += 2;
				}
				else
				{
				    *splice = '\0';
				    suffix += 2;
				}
			    }
			}			
			else
			    suffix += 3;

			continue;
		    }
		}		    
	    }		    
	}

	if (!*suffix)
	    break;			/* empty suffix string */

	if ((cut = index (suffix, '/')) == 0)
	{
	    cc = strlen (suffix);
	    cut = suffix + cc;
	}
	else
	    cc = cut - suffix;

	if (cc >= MAXNAMLEN)
	{
	    (void) sprintf (prefix, "%s: %s",
			    suffix, sys_errlist[ENAMETOOLONG]);
	    return (0);
	}

	if (cc == 0)			/* suffix has leading '/' */
	{
	    cut++;
	    cc = 1;			/* so force it to be copied */
	}
	
	strncpy (splice, suffix, cc);
	splice[cc] = '\0';

	if (!ignore)
	{
	    if ((result = lstat (prefix, &status)) == -1)
	    {
		(void) sprintf (splice + cc, ": %s", sys_errlist[errno]);
		return (0);
	    }

	    if ((status.st_mode & S_IFMT) == S_IFLNK)
	    {
		if ((result = readlink (prefix, link, MAXPATHLEN)) == -1)
		{
		    (void) sprintf (splice + cc, ": %s", sys_errlist[errno]);
		    return (0);
		}
		link[result] = '\0';

		if (links++ == MAXSYMLINKS &&
		    (result = stat (prefix, &status)) == -1)
		{
		    (void) sprintf (splice + cc, ": %s", sys_errlist[errno]);
		    return (0);
		}

		if (verbose)
		{
		    (void) printf ("%s -> %s\n", prefix, link);
		}

		if (*link == '/')
		    *prefix = '\0';	/* chop prefix if link is absolute */
		else
		    *splice = '\0';	/* chop just the link name */

		if (name1 (prefix, link) == 0)
		    return (0);		/* recurse */

		splice = prefix + strlen (prefix);
	    }
	}
	else if ((result = stat (prefix, &status)) == -1)
	{
	    (void) sprintf (splice + cc, ": %s", sys_errlist[errno]);
	    return (0);
	}

	suffix = cut;			/* advance suffix past cut */

	splice += strlen (splice);	/* advance splice to end of prefix */
    }
    while (*suffix);

    return (1);
}
