
/*
**
**	Copyright (c) 1988, Robert L. McQueer
**		All Rights Reserved
**
** Permission granted for use, modification and redistribution of this
** software provided that no use is made for commercial gain without the
** written consent of the author, that all copyright notices remain intact,
** and that all changes are clearly documented.  No warranty of any kind
** concerning any use which may be made of this software is offered or implied.
**
*/

#include <stdio.h>
#include <sys/param.h>
#include "node.h"
#include "config.h"

extern char *htab_find();
extern char *str_store();

extern NODE *RatList[];
extern char *RatTab;

extern int Verbosity;

NODE *tab_enter();

static NODE *Curfile;
static NODE *Avail = NULL;
static int Avnum = 0;

/*
** enter a symbol definition into the table.  If already there, enter
** it as a DDEF.
*/
rat_def_enter(s)
char *s;
{
	KEY key;

	if (s == NULL)
		return;

	key.name = s;
	key.type = FORGET;

	if (htab_find(RatTab,(char *) &key) != NULL)
		return;

	key.type = DEF;

	if (htab_find(RatTab,(char *) &key) != NULL)
		key.type = DDEF;
	if (Verbosity > 1)
	{
		if (key.type == DDEF)
			fprintf(stderr,"enter DOUBLEDEF: %s\n",s);
		else
			fprintf(stderr,"enter DEF: %s\n",s);
	}
	(tab_enter(s,key.type))->d.file = Curfile;
}

/*
** enter a symbol reference into the table.
**
** NOTE: this routine depends on the way htab_find handles multiple
**	entries - MOST RECENT entry will be returned.  This is used
**	to prevent multiple entries for one file.
*/
rat_ref_enter(s)
char *s;
{
	NODE *ptr;
	KEY key;

	if (s == NULL)
		return;

	key.name = s;
	key.type = FORGET;

	if (htab_find(RatTab,(char *) &key) != NULL)
		return;

	key.type = REF;
	ptr = (NODE *) htab_find(RatTab,(char *) &key);

	if (ptr == NULL || ptr->d.file != Curfile)
	{
		if (Verbosity > 1)
		{
			if (ptr == NULL)
				fprintf(stderr,"enter NEWREF: %s\n",s);
			else
				fprintf(stderr,"enter OLDREF(%s): %s\n",
					(ptr->d.file)->key.name,s);
		}
		ptr = tab_enter(s,REF);
		ptr->d.file = Curfile;
	}
	else
	{
		if (Verbosity > 2)
			fprintf(stderr,"MULTREF: %s\n",s);
	}
}

/*
** new file.  Sets Curfile as well as entering new file into table.
** until another call to this routine, all Ref's & Def's will be from
** this file.
*/
rat_file(s)
char *s;
{
	Curfile = tab_enter(s,FNAME);
	if (Verbosity > 0)
		fprintf(stderr,"FILE: %s\n",s);
}

/*
** rat_dep_scan is called after all files are entered, and runs down the list
** of symbol references, finding their definitions.  If a definition is found,
** a file dependency is noted.  If not, an undef is noted.
**
** Sets up the file dependancy / refcount / circle information
** for later use by topological sort.
*/

rat_dep_scan()
{
	register NODE *ref,*def,*ptr;
	KEY key;
	int i;
	char *str;
	char name[CATBUFFER < 2*MAXPATHLEN+3 ? 2*MAXPATHLEN+3 : CATBUFFER];

	for (ptr = RatList[FNAME]; ptr != NULL; ptr = ptr->next)
	{
		ptr->d.fname.dep = NULL;
		ptr->d.fname.refcount = 0;
	}

	for (ref = RatList[REF]; ref != NULL; ref = ref->next)
	{
		key.name = ref->key.name;
		key.type = DEF;

		/*
		** if DEF node exists, we have a dependency
		*/
		if ((def = (NODE *) htab_find(RatTab,(char *) &key)) != NULL)
		{
			if (Verbosity > 2)
				fprintf(stderr,"CHECKREF_DEF: %s(%s,%s)\n",
					key.name, (ref->d.file)->key.name,
					(def->d.file)->key.name);
			/* forget about depending on ourselves */
			if (def->d.file == ref->d.file)
				continue;

			/*
			** make a key from the two names, and continue
			** if the dependency already exists.  The key
			** contains "illegal" characters to avoid problems
			** arising from different names concatenating into
			** the same string.   Since the first part is a
			** filename which can contain anything, this can
			** still break in theory, but only truly perverse
			** filenames would cause problems.
			*/
			sprintf(name,")%s$%s",(ref->d.file)->key.name,
						(def->d.file)->key.name);
			key.name = name;
			key.type = DEP;
			if (htab_find(RatTab,(char *) &key) != NULL)
				continue;

			/*
			** new dependency.  Chain the dep node onto the
			** referring files dep list, bump the reffering
			** file's reference count.
			*/
			if (Verbosity > 1)
				fprintf(stderr,"DEPEND: %s\n",name);
			ptr = tab_enter(name,DEP);
			ptr->d.dep.sym = ref->key.name;
			ptr->d.dep.dfile = def->d.file;
			ptr->d.dep.rfile = ref->d.file;
			ptr->d.dep.next = (ref->d.file)->d.fname.dep;
			(ref->d.file)->d.fname.dep = ptr;
			++((ref->d.file)->d.fname.refcount);
			continue;
		}

		/*
		** not defined.  If undef node already exists, concatenate
		** new file onto descriptor string, otherwise make new node.
		** This wastes string storage somewhat, but WHOGAS.
		*/
		key.type = UDEF;
		if ((ptr = (NODE *) htab_find(RatTab,(char *) &key)) != NULL)
		{
			if (Verbosity > 2)
				fprintf(stderr,"OLD_UNDEF: %s(%s)\n",
								key.name,name);
			++(ptr->d.ud.count);
			str = (ref->d.file)->key.name;
			i = strlen(ptr->d.ud.files) + strlen(str) + 1;
			if (i < CATBUFFER)
			{
				sprintf(name,"%s %s",ptr->d.ud.files,str);
				ptr->d.ud.files = str_store(name);
			}
			continue;
		}

		if (Verbosity > 1)
			fprintf(stderr,"NEW_UNDEF: %s(%s)\n",key.name,
					(ref->d.file)->key.name);
		ptr = tab_enter(key.name,UDEF);
		ptr->d.ud.files = (ref->d.file)->key.name;
		ptr->d.ud.count = 1;
	}
}

static int Ckey = 0;

/*
** enter a cycle into the hash table - we really only need the
** list of the things, but this avoids bothering with a different
** mechanism.  We don't use the cycle itself as the key to avoid
** hashing huge strings needlessly.  Entry is made by topological
** sort, which detects cycles.
*/
new_cycle(s)
char *s;
{
	char kbuf[12];

	sprintf(kbuf,"%x",Ckey++);
	(tab_enter(kbuf,CYCLE))->d.cycle = str_store(s);
}

/*
** Forget symbol.  we really don't need the list of these - only
** the fact that a "forget" has been entered.
*/
rat_forget(s)
char *s;
{
	tab_enter(s,FORGET);
}

/*
** new entry into hash table, allocating a node and a permanent copy of
** the key string.  Also builds the RatList[] array of entries of the various
** types as it goes.  Returns node pointer, so caller may fill in type
** specific information.
*/
static NODE *
tab_enter(str,type)
char *str;
unsigned type;
{
	NODE *ptr;

	if (type > MAXTYPE)
		fatal("prog. err - table entry with type = %d",type);

	if (Avnum <= 0)
	{
		Avnum = NODE_BLOCK;
		if ((Avail = (NODE *) malloc(NODE_BLOCK*sizeof(NODE))) == NULL)
				fatal("can't allocate memory for table nodes");
	}

	ptr = Avail;
	++Avail;
	--Avnum;

	ptr->next = RatList[type];
	RatList[type] = ptr;

	ptr->key.name = str_store(str);
	ptr->key.type = type;

	htab_enter(RatTab, (char *) &(ptr->key), (char *) ptr);

	return(ptr);
}
