/*************************************************************************
                     vms.hc -- fm routines for VAX/VMS
		      Copyright (c) Tony Field 1990

  In principle this code should be maintained directly in-line in fm.c
  However, there is a BIG gotcha:

  1.  All VMS entry points and most types, fields, etc. contain the character
   "$" (dollar sign).  This is a noble attempt to avoid the possibility of
   user symbols conflicting with the predefined ones in the operating system.

   Of course, all DEC-supported languages regard "$" as being yet another
   admissible character in an identifier.

  2.  ANSI C preprocessors are permitted to barf if they come across a
   character that's not a legal character in C.  So even if the VMS code
   is not used on the "foobar" architecture, it is possible that the mere
   presence of its implementation in fm.c could prevent it from compiling.

SO.  We put all this junk in a separate file so that ports to other hosts
will not be influenced by the presence of a lot of "$"s in weird places.

N.B. -- VMS DEBUG is not perturbed by include files containing C code, as
is dbx; thus there is no need to move this code back into fm.c if you need to
debug this RMS junk.

--D. Jason Penney, June 1990
****************************************************************************/

#include <fab.h>
#include <rab.h>

typedef void VmsAstFuncType(long arg);

long SYS$CLOSE(struct FAB *fab, VmsAstFuncType *err, VmsAstFuncType *SUC);
long SYS$CONNECT(struct RAB *rab, VmsAstFuncType *err, VmsAstFuncType *SUC);
long SYS$OPEN(struct FAB *fab, VmsAstFuncType *err, VmsAstFuncType *SUC);
long SYS$READ(struct RAB *aRab, VmsAstFuncType *err, VmsAstFuncType *SUC);
long SYS$WRITE(struct RAB *aRab, VmsAstFuncType *err, VmsAstFuncType *SUC);

static struct FAB theFab;
static struct RAB theRab;

/* =======================================================================
 * Name - doRead
 *
 * Purpose - Abstraction for file read
 *
 * Arguments:  fp -- file from which to read
 *	       fpos -- location from which to read in file
 *	       loc -- address to place result
 *             maxsize -- max number of bytes that may be read into loc
 *
 * Returns     function return -- -1 on error, otherwise success
 *
 *========================================================================
 */
static int doRead(int fp, long fpos, char *loc, int maxsize)
{
long retVal;

/* reads are done at arbitrary position and length, so we have to
   fake it at this level */
theRab.rab$l_bkt = 1 + fpos / 512; /* where on disk (one-based) */
if (fpos % 512 == 0) { /* aligned read, easy special case */
  theRab.rab$w_usz = maxsize;	   /* how long */
  theRab.rab$l_ubf = loc;	   /* where in memory */
  theRab.rab$l_stv = 0;

  retVal = SYS$READ(&theRab, NULL, NULL);
  if (retVal % 2 != 1)
    return -1;
  /* VMS 5.1 doesn't set rab$l_stv, documentation notwithstanding */
  if (theRab.rab$l_stv != 0 && (theRab.rab$l_stv % 2 != 1)
    return -1;
  }
else { /* transfer required */
  char aBlock[MAX_RETURNED_FROM_DISK + 512];

  theRab.rab$w_usz = maxsize + 512; /* get extra block */
  theRab.rab$l_ubf = &aBlock[0];
  theRab.rab$l_stv = 0;

  retVal = SYS$READ(&theRab, NULL, NULL);
  if (retVal % 2 != 1)
    return -1;
  /* VMS 5.1 doesn't set rab$l_stv, documentation notwithstanding */
  if (theRab.rab$l_stv != 0 && (theRab.rab$l_stv % 2 != 1)
    return -1;
  /* return part that we want */
  memcpy(loc, &aBlock[fpos % 512], maxsize);
  }

/* return lesser of two lengths */
if (theRab.rab$w_rsz < maxsize)
  return theRab.rab$w_rsz;
else
  return maxsize;
}

/* =======================================================================
 * Name - doWrite
 *
 * Purpose - Abstraction to perform write on file
 *
 * Arguments:  fp -- file to which to write
 *             fpos -- position at which to write
 *             loc -- address of bytes to be written
 *             nbytes -- number of bytes to be written
 *
 * Returns     function return -- -1L on error, otherwise success
 *
 *========================================================================
 */
static int doWrite(int fp, long fpos, char *loc, int nbytes)
{
/* writes are done in half-block chunks, so we have to normalize if
   requested address is not on a block boundary */
char scratch[512];
long retVal;

/* Get original block */
if (doRead(fp, fpos / 512 * 512, &scratch[0], 512) == -1)
  return -1;
if (fpos % 512 == 0)
  memcpy(&scratch[0], loc, 256);
else
  memcpy(&scratch[256], loc, 256);
theRab.rab$w_usz = 512;			/* how large */
theRab.rab$l_ubf = &scratch[0];		/* where in memory */
theRab.rab$l_bkt = 1 + fpos / 512;	/* where on disk (1-based) */
theRab.rab$l_stv = 0;

retVal = SYS$WRITE(&theRab, NULL, NULL);
if (retVal % 2 != 1)
  return -1;
/* VMS 5.1 doesn't set rab$l_stv, documentation notwithstanding */
if (theRab.rab$l_stv != 0 && (theRab.rab$l_stv % 2 != 1)
  return -1;
return theRab.rab$w_rsz;		/* return how many actually written */
}

/* =======================================================================
 * Name - doOpen
 *
 * Purpose - Abstraction to open file
 *
 * Arguments:  name -- null-terminated name of file to open
 *             readOnly -- FALSE if you are allowed to write to file
 *
 * Returns     function return -- -1L on error, otherwise success
 *
 *========================================================================
 */
static int doOpen(char name[], int /*BoolType*/ readOnly)
/* Open the file using the RMS utilities */
{
long retStatus;

/* zero all of the fields in the fab */
memset( (char *)&theFab, 0, sizeof(theFab) );

/* identify the Block as a fab */
theFab.fab$b_bid = FAB$C_BID;
theFab.fab$b_bln = FAB$C_BLN;

/* fill in the fab fields needed for the open function */

theFab.fab$l_fna = name;
theFab.fab$b_fns = strlen(name);
theFab.fab$b_fac = FAB$M_BIO + FAB$M_GET; /* block i/o, read allowed */
if (!readOnly)
  theFab.fab$b_fac |= FAB$M_PUT; /* write allowed */
theFab.fab$b_shr = FAB$M_NIL; /* no sharing, required for block i/o */
theFab.fab$w_mrs = 512; /* physical block size */
theFab.fab$b_org = FAB$C_SEQ;
theFab.fab$b_rfm = FAB$C_FIX;

retStatus = SYS$OPEN(&theFab, (VmsAstFuncType *)(NULL), 
    (VmsAstFuncType *)(NULL));

if (retStatus % 2 != 1) { /* error on open */
  return -1;
  }
/* rab$l_stv not set by this call... */

/* create record stream */
memset(&theRab, 0, sizeof(struct RAB));
theRab.rab$b_bid = RAB$C_BID;
theRab.rab$b_bln = RAB$C_BLN;
theRab.rab$l_fab = &theFab;
theRab.rab$w_isi = 0L;
theRab.rab$l_rop = 0L;
theRab.rab$l_stv = 0;
retStatus = SYS$CONNECT(&theRab, NULL, NULL);

if (retStatus % 2 != 1) { /* error on connect */
  return -1;
  }
/* VMS 5.1 doesn't set rab$l_stv, documentation notwithstanding */
if (theRab.rab$l_stv != 0 && (theRab.rab$l_stv % 2 != 1)
  return -1;

return 0; /* junk */
}

/* =======================================================================
 * Name - doClose
 *
 * Purpose - Abstraction to close file (when done)
 *
 * Arguments:  fp -- file to be closed
 *
 * Returns     function return -- 
 *
 *========================================================================
 */
static void doClose(int fp)
{
int retStatus = SYS$CLOSE(&theFab, NULL, NULL);
}
