/**********************************************************************
				f m . c
		      Copyright (c) Tony Field 1990

Please examine the associated Readme file for details on the lack of
warranty and explicit licensing information.

No responsibility is taken for any errors on inaccuracies inherent either to
the comments or the code of this program, but, if reported to me, an attempt
will be made to fix them.

  Authors:  Tony Field      (uunet!ajfcal!tony)
            D. Jason Penney (penneyj@slc.com)

------------------------------
  - curses-based binary file modifier
  - default is BSD curses.
  - compile with -DSYSV_CURSES for ATT termcap curses.
  - seems to work on files, devices, and directories(read only)
  - developed for AT&T Unix 3.2.2 and  Xenix 2.3.2  (System V/386)

  - Major hacks by D. Jason Penney (penneyj@slc.com) to run under SunOS,
    Ultrix, MS-DOS, NEWS-OS, and VAX/VMS, June 1990.  

Some warnings for people attempting to port this code to other hosts:

1. VAX/VMS curses doesn't have mvwprintw().  Use wmove() followed by wprintw()
   instead.  I considered implementing mvwprintw() within this module, but
   that would just open up another (varargs) portability problem...

2. BSD 4.2 systems (such as Sony NEWS) do not have #elif preprocessor directive.
   Use nested #if/#else/#endif constructs instead.

3. Most curses implementations don't have beep().  It is trivially implemented
   here.  Enable the implementation if you need it.

4. A number of characters do NOT come through well on some hosts, raw mode
   notwithstanding:  ^S/^Q are often used for flow-control (even if stdin
   is in raw mode, the user might have flow control enabled in some
   intermediate machine).  ^C/^Y should be avoided because they are often
   used for panic (SIGINT) handling.  ^T is a VMS meta-character
   that is not usually available to curses.  ^Z is a favorite command in
   BSD unix for job control, and is hard-wired to EOF in VAX/VMS.  If you
   decide to add commands or change the bindings of existing commands, please
   consider the portability impact of your changes.
**********************************************************************

Patchlevels
===========
Patchlevel 1:	May-1990, by Tony Field
	  1. clean up cursor motion while editing hex side of screen.
	  2. KEY_UNDO removed for !SYS_TERMCAP compilation
	  3. fixed file print on last 16 bytes of file

Patchlevel 2:  June, 1990 by D. Jason Penney
	  1. adjust code for portability
	  2. detect zero-length files and exit appropriately
	  3. fix boundary condition editing last byte of file

*/

/*============================================
  System Declarations
============================================*/

#include <stdio.h>

#if defined(VMS)
#include <file.h>
#include <stat.h>
#else
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif

#include <ctype.h>
#include <curses.h>
#include <errno.h>
#include <signal.h>
#include <string.h>

/* Use ANSI C if available... */
#if defined(MSDOS)
#define FLG_PROTOTYPE TRUE
#include <dos.h>
#endif
#if defined(VMS)
#define FLG_PROTOTYPE TRUE
#endif

#if defined(MSDOS) /* prototypes for curses calls */
int	 addch(char ch);	/* put char in stdscr */
void	 attroff(int attr);	/* clear attribute(a) stdscr */
void	 attron(int attr);	/* add attribute(s) stdscr */
void	 beep(void);		/* sound bell */
void	 clear(void);		/* clear stdscr */
int	 clrtobot(void);	/* clear end of stdscr */
int	 clrtoeol(void);	/* clear end of line in stdscr */
void	 echo(void);		/* set terminal echo mode */
int	 endwin(void);		/* cleanup and finitialization */
void	 erase(void);		/* erase stdscr */
int	 initscr(void);		/* curses initialization */
void	 keypad(WINDOW *scr, int enabled);
				/* marks a window for keypad usage */
int	 move(int row, int col);/* move cursor in stdscr */
int	 mvaddstr(int row, int col, char msg[]);
				/* move & put string in stdscr */
int	 mvwgetch(WINDOW *scr, int row, int col);
				/* move & get char to a window */
int	 mvprintw(int row, int col, char pat[], ...);
				/* move & print string in stdscr */
int	 mvwaddstr(WINDOW *scr, int row, int col, char msg[]);
				/* move & put string in stdscr */
int	 mvwprintw(WINDOW *scr,int row, int col, char pat[], ...);
				/* move & print string in a window */
void	 nl(void);			/* set terminal cr-crlf map mode */
void	 noecho(void);		/* unset terminal echo mode */
void	 nonl(void);		/* unset terminal cr-crlf map mode */
void	 noraw(void);		/* unset raw terminal mode */
void	 raw(void);		/* set raw terminal mode */
void	 refresh(void);		/* refresh stdscr */
int	 resetty(void);		/* restore terminal I/O modes */
int	 savetty(void);		/* save terminal I/O modes */
int	 waddch(WINDOW *scr, char ch);
				/* put char in a window */
int	 waddstr(WINDOW *scr, char *buf);
				/* put string in a window */
void	 wattrset(WINDOW *scr, int attr);
				/* set window char attributes */
int	 wclrtobot(WINDOW *scr);/* clear end of a window */
int	 wclrtoeol(WINDOW *scr);/* clear end of line in a window */
int	 wgetch(WINDOW *scr);	/* get char to a window */
int	 wgetstr(WINDOW *scr, char *buf);
				/* get string to window and buffer */
int	 wmove(WINDOW *scr, int row, int col);	/* move cursor in a window */
int	 wprintw(WINDOW *scr, char *pat,...);	/* print string in a window */
void	 wrefresh(WINDOW *scr);	/* refresh screen */
#endif

/* Prototypes for System calls: */

#if defined(FLG_PROTOTYPE)
int close(int fp);
#else
int close();
#endif

#if defined(FLG_PROTOTYPE)
void exit(int code);
#else
void exit();
#endif

#if defined(FLG_PROTOTYPE)
int lseek(int fp, long pos, int mode);
#else
int lseek();
#endif

/* Classic irony that most important syscall is also most host dependent! */
#if defined(FLG_PROTOTYPE)
#if defined(MSDOS)
int open(char name[], int mode, int pmode);
#else
#if defined(VMS)
int open(char name[], int mode,...);
#else
int open(char name[], int mode);
#endif
#endif
#else
int open();
#endif

#if defined(FLG_PROTOTYPE)
int read(int fp, char *buf, int len);
#else
int read();
#endif

#if defined(FLG_PROTOTYPE)
int write(int fp, char *buf, int len);
#else
int write();
#endif

extern int errno;

/*============================================
  Local Includes, Defines, and Types
============================================*/

#include "patchlevel.h"

#if !defined(FALSE)
#define FALSE 0
#define TRUE 1
#endif
typedef unsigned char BoolType;

/* Define your variant of curses.  If you have compilation or link problems,
   you probably should reverse (undef/define) this flag.
*/
#if !defined(SYSV_CURSES)  &&  !defined(M_TERMCAP)
#if !defined(sun) && !defined(ultrix) && !defined(sony_news) && !defined(MSDOS)
#define SYSV_CURSES TRUE
#endif
#endif

#define AS_CTRL(x)      ((x) - '@')

#define BSIZE	256		/*	size of window onto file */
#define MAX_RETURNED_FROM_DISK 1000 /* buffer size limit */
#define	FIRSTL	3		/* 	first screen line */
#define FIRSTH	6		/*	first hex column */
#define FIRSTA	58		/*	first alpha column */
#define PROMPTCOL 22            /*      col for prompts */
#define STAT1   22              /*      first status line */
#define STAT2   23              /*      second status line */

typedef enum {asciiMode, hexMode} ViewModeEType;

/* Key Bindings: */
#define CMD_ASC		AS_CTRL('A')	/*	ascii side */
#define	CMD_BAK		AS_CTRL('B')	/* 	previous page */
#define	CMD_DON		AS_CTRL('D')	/*	done/quit */
#define	CMD_FWD		AS_CTRL('F')	/*	next page */
#define	CMD_JMP		AS_CTRL('G')	/*	goto address */
#define CMD_LEFT        AS_CTRL('H')
#define CMD_DOWN        AS_CTRL('J')
#define CMD_UP          AS_CTRL('K')
#define CMD_REDRAW      AS_CTRL('L')
#define CMD_PRI		AS_CTRL('P')	/*	print to file */
#define	CMD_SEARCH	AS_CTRL('N')	/*	search in current view mode */
#define CMD_UND		AS_CTRL('U')	/*	undo current changes */

/*============================================
  Local Variables
============================================*/

static char	binToHex[] = "0123456789abcdef";
static int		byte_pos; /*	current byte editing in buffer */
static BoolType	maybe_dirty;	  /*	user may have changed data */
static long	max_fpos;	  /*	end of file byte */
static ViewModeEType vmode = hexMode; /* remember hex/ascii side of screen */

static FILE	*printFile = NULL;/*	print file */
static BoolType writeOk;
static char	srcName[200];

/*============================================
  Local forwards and macros
============================================*/

#define IS_VALID(x) ((x) >= ' ' && (x) < 127)
#define T_TO_A(x) (IS_VALID(x) ? (x) : '.')
#define EXIT_SUCCESS exit(0)
#define EXIT_ERR     exit(1)

#if defined(VMS)
#define EXTRA_REFRESH  wrefresh(stdscr)
#define CANONICAL { echo(); noraw(); nocrmode();}
#define NONCANONICAL { noecho(); crmode();}
#else
#if defined(SYSV_CURSES)
#define EXTRA_REFRESH
#define CANONICAL { echo(); }
#define NONCANONICAL { noecho(); }
#else /* default */
#define EXTRA_REFRESH  wrefresh(stdscr)
#define CANONICAL { echo(); noraw(); nl(); }
#define NONCANONICAL { noecho(); raw(); nonl(); }
#endif
#endif

/* The point behind the following is to try to make standout() give
   us reverse video as opposed to bold, which doesn't stand out (heh, heh)
   very well on a lot of displays...
*/
#if defined(SYSV_CURSES)
#if defined(VMS)
#define STANDOUT standout()
#define STANDEND standend()
#else
#define STANDOUT wattrset(stdscr, A_STANDOUT)
#define STANDEND wattrset(stdscr, 0)
#endif
#else
#if defined(MSDOS)
#define STANDOUT attron(A_REVERSE)
#define STANDEND attroff(A_REVERSE)
#else /* default */
#define STANDOUT standout()
#define STANDEND standend()
#endif
#endif

/* Local forwards: */

#if defined(FLG_PROTOTYPE)
static void cleanup(void);
#else
static void cleanup();
#endif

#if defined(FLG_PROTOTYPE)
static int hexval(int c);
#else
static int hexval();
#endif

#if defined(FLG_PROTOTYPE)
static void reflect_change(int byte, ViewModeEType vmode, char *wbuf);
#else
static void reflect_change();
#endif

#if defined(FLG_PROTOTYPE)
static void ttox(char x[], int c);
#else
static void ttox();
#endif

#if defined(FLG_PROTOTYPE)
static void whereis(int pos, ViewModeEType which, int *x, int *y);
#else
static void whereis();
#endif

/*============================================
  Host-dependent Implementations
============================================*/

/* Special routines for file access, host-dependent */
#if defined(VMS)
#include "vms.hc" /* see comment in vms.hc for explanation of purpose */
#else
/* Unix, MS-DOS Implementations */

/* =======================================================================
 * Name - doLseek
 *
 * Purpose - Abstraction in front of lseek() -- checks for errors
 *
 * Arguments:  fp -- file we are seeking into
 *	       ptr -- position to which we wish to move
 *             n -- "mode" of move, as in lseek(2).  This is really just noise.
 *
 * Returns     function return -- 
 *
 *========================================================================
 */
#if defined(FLG_PROTOTYPE)
static void doLseek(int fp, long ptr, int n)
#else
static void doLseek(fp, ptr, n)
int fp;
long ptr;
int n;
#endif
{
if (lseek(fp, ptr, n) == -1) {
  cleanup();
  perror("lseek() error");
  EXIT_ERR;
  }
}

/* =======================================================================
 * Name - doRead
 *
 * Purpose - Abstraction for file read
 *
 * Arguments:  fp -- file from which to read
 *	       fpos -- location from which to read in file
 *	       loc -- address to place result
 *             maxsize -- max number of bytes that may be read into loc
 *
 * Returns     function return -- -1 on error, otherwise success
 *
 *========================================================================
 */
static int doRead(fp, fpos, loc, maxsize)
int fp;
long fpos;
char *loc;
int maxsize;
{
int nn;

doLseek(fp, fpos, 0);	/*	position to current byte */
nn = read(fp, loc, maxsize);
if (nn == -1) {
  if (errno != ERANGE) {
    cleanup();
    perror("error in read()");
    EXIT_ERR;
    }
  }
return nn;
}

/* =======================================================================
 * Name - doWrite
 *
 * Purpose - Abstraction to perform write on file
 *
 * Arguments:  fp -- file to which to write
 *             fpos -- position at which to write
 *             loc -- address of bytes to be written
 *             nbytes -- number of bytes to be written
 *
 * Returns     function return -- -1L on error, otherwise success
 *
 *========================================================================
 */
static int doWrite(fp, fpos, loc, nbytes)
int fp;
long fpos;
char *loc;
int nbytes;
{
int result;

doLseek(fp, fpos, 0);
result = write(fp, loc, nbytes);
if (result == -1) {
  cleanup();
  perror("error in write()");
  EXIT_ERR;
  }
return result;
}

/* =======================================================================
 * Name - doOpen
 *
 * Purpose - Abstraction to open file
 *
 * Arguments:  name -- null-terminated name of file to open
 *             readOnly -- FALSE if you are allowed to write to file
 *
 * Returns     function return -- -1L on error, otherwise success
 *
 *========================================================================
 */
static int doOpen(name, readOnly)
char name[];
BoolType readOnly;
{
int fp;

if (readOnly) {
#if defined(MSDOS)
    fp = open(name, (int)(O_RDONLY | O_BINARY), 0);
#else
    fp = open(name, O_RDONLY);
#endif
  }
else {
#if defined(MSDOS)
  fp = open(name, (int)(O_RDWR | O_BINARY), 0);
#else
  fp = open(name, O_RDWR);
#endif
  }
return fp;
}

/* =======================================================================
 * Name - doClose
 *
 * Purpose - Abstraction to close file (when done)
 *
 * Arguments:  fp -- file to be closed
 *
 * Returns     function return -- 
 *
 *========================================================================
 */
static void doClose(fp)
int fp;
{
close(fp);
}
#endif

/*============================================
  Implementations of Routines
============================================*/

/* =======================================================================
 * Name - cleanup
 *
 * Purpose - Restore display and keyboard to sane (cooked) mode
 *
 * Arguments:  None
 *
 * Returns     function return -- 
 *
 *========================================================================
 */
#if defined(FLG_PROTOTYPE)
static void cleanup(void)
#else
static void cleanup()
#endif
{
  if (printFile != NULL)
    fclose(printFile);
  wmove(stdscr, LINES - 1, 0);
  wclrtobot(stdscr);
  refresh();
  CANONICAL;
  endwin();
}

/* =======================================================================
 * Name - freakout
 *
 * Purpose - Signal handler to trap signal and return control to shell
 *
 * Arguments:  None (whatever signal handler passes, we ignore it)
 *
 * Returns     function return -- 
 *
 *========================================================================
 */
#if defined(FLG_PROTOTYPE)
static void freakout(void)
#else
static void freakout()
#endif
{
  cleanup();
  EXIT_ERR;
}

/* =======================================================================
 * Name - beep
 *
 * Purpose - Ring terminal bell
 *
 * Arguments:  None
 *
 * Returns     function return -- 
 *
 * This is missing from many implementations of curses.  Note that we
 * don't use curses to write the character, because at best it would write
 * a character on the screen, and at worst it will become confused about
 * the cursor position since the bell doesn't display.
 *
 *========================================================================
 */
#if !defined(SYSV_CURSES) || defined(VMS)
#if defined(FLG_PROTOTYPE)
static void beep(void)
#else
static void beep()
#endif
{
fprintf(stderr, "\007");
fflush(stderr);
}
#endif

/* =======================================================================
 * Name - display
 *
 * Purpose - display a 256 byte block of file with hex on left and alpha
 *	     on right side.
 *
 * Arguments:  fpos -- current position in the file
 *	       wbuf -- working data buffer
 *	       nbytes -- size of wbuf
 *
 * Returns     function return -- 
 *
 *========================================================================
 */
#if defined(FLG_PROTOTYPE)
static void display(long fpos, char wbuf[], int nbytes)
#else
static void display(fpos, wbuf, nbytes)
long	fpos;
char	wbuf[];
int	nbytes;
#endif
{
  int		i, j, k, m;
  char	hbuf[2]; /*	working buf to convert byte to hex */

  erase();
  wmove(stdscr, 0, 0);
  wprintw(stdscr, "File: %s    (0x%lx bytes) %s", srcName, max_fpos,
      writeOk ? "" : "(read only)");
  wmove(stdscr, 1, 0);
  wprintw(stdscr, "Byte: 0x%-8lx ", fpos);

  for (i = 0, j = FIRSTL;  i < nbytes;  i += 16, j++) {
    /*display hex side  */
    wmove(stdscr, j, 0);
    wprintw(stdscr, "%02x", i);
    wmove(stdscr, j, FIRSTH);
    for (k = 0, m = i; k < 48  &&  m < nbytes;  k += 3, m++) {
      ttox(hbuf, wbuf[m]);
      waddch(stdscr, hbuf[0]);
      waddch(stdscr, hbuf[1]);
      waddch(stdscr, ' ');
      }

    /*	display ascii side	*/
    wmove(stdscr, j, FIRSTA);
    for (k = 0, m = i;  k < 16 &&  m < nbytes;  k++, m++)
      waddch(stdscr, T_TO_A(wbuf[m]));
    }
  
  /*	bottom annotation	*/
  for (i = 0;  i < 16;  i++) {
    wmove(stdscr, j + 1, i * 3 + FIRSTH);
    waddch(stdscr, binToHex[i]);
    }
  wmove(stdscr, j + 1, FIRSTA);
  waddstr(stdscr, "0123456789abcdef");
  wmove(stdscr, STAT1, 0);
#if defined(KEY_F)
  waddstr(stdscr, 
" ^F/pgdn=fwd  ^A/f1=ascii/hex ^H=left  ^K=up  ^N/f2=srch ^U/f3=undo ^D/f6=done"
      );
  wmove(stdscr, STAT2, 0);
  waddstr(stdscr,
" ^B/pgup=back ^D/f6=done      ^M=right ^J=dwn ^P/f4=prnt ^G/f5=goto ^L=redraw"
      );
#else
  waddstr(stdscr, 
      " ^F/pgdn=fwd  ^A=ascii/hex ^H=left  ^K=up  ^N=srch ^U=undo ^D=done");
  wmove(stdscr, STAT2, 0);
  waddstr(stdscr,
      " ^B/pgup=back ^D=done      ^M=right ^J=dwn ^P=prnt ^G=goto ^L=redraw");
#endif
  wrefresh(stdscr);
}

/* =======================================================================
 * Name - printit
 *
 * Purpose - display a 256 byte block of file with hex on left and alpha
 *	     on right side to a print file.
 *
 * Arguments:  comment -- commentnote
 *	       fpos -- position within file
 *	       wbuf -- working data buffer
 *	       nbytes -- size of wbuf
 *
 * Returns     function return -- 
 *
 *========================================================================
 */
#if defined(FLG_PROTOTYPE)
static void printit(char *comment, long fpos, char wbuf[], int nbytes)
#else
static void printit(comment, fpos, wbuf, nbytes)
char	*comment;
long	fpos;
char	wbuf[];
int	nbytes;
#endif
{
  int		i, j, k, m;
  char	hbuf[2]; /*	working buf to convert byte to hex */

  fprintf(printFile, "File: %s    (%lx bytes) %s\nByte: %lx      %s\n", 
      srcName, max_fpos, writeOk ? "" : "(read only)", fpos, comment);

  for (i = 0, j = FIRSTL;  i < nbytes;  i += 16, j++) {
    /*	display hex side  */
    fprintf(printFile, "%02x    ", i);
    for (k = 0, m = i; k < 48  &&  m < nbytes;  k += 3, m++) {
      ttox(hbuf, wbuf[m]);
      fputc(hbuf[0], printFile);
      fputc(hbuf[1], printFile);
      fputc(' ', printFile);
      }

    for (; (m % 16 != 0);  m++)
      fprintf(printFile, "   ");

    /*	display ascii side	*/
    fprintf(printFile, "   ");
    for (k = 0, m = i;  k < 16 &&  m < nbytes;  k++, m++)
      fputc(T_TO_A(wbuf[m]), printFile);

    fprintf(printFile, "\n");
    }
  /*	bottom annotation	*/
  fprintf(printFile, "      ");	
  for (i = 0;  i < 16;  i++)
    fprintf(printFile, "%c  ", binToHex[i]);
  fprintf(printFile, "   0123456789abcdef\n\n");
}

/* =======================================================================
 * Name - edit
 *
 * Purpose - allow user to modify the 256 byte block of text.
 *
 * Arguments:  wbuf -- working buffer
 *	       nbytes -- size of wbuf
 *	       fpos -- current position in file
 *
 * Returns     function return -- keyboard command if not handled locally
 *
 *========================================================================
 */
#if defined(FLG_PROTOTYPE)
static int edit(char *wbuf, int nbytes, long fpos)
#else
static int edit(wbuf, nbytes, fpos)
char	*wbuf;
int	nbytes;
long	fpos;
#endif
{
  int	key;	/*	user entered this key	*/
  int		x, y;	/*	screen coords.	*/
  int		hexa, hexb;	
  BoolType	reflected; /*	was a change really done */
  BoolType      have_key;
  
  have_key = FALSE;
  while (TRUE) {
    if (byte_pos >= nbytes)		
      byte_pos = nbytes - 1;
    if (byte_pos < 0)
      byte_pos = 0;
	  
    wmove(stdscr, 1, 6);
    wprintw(stdscr, "0x%-8lx",  fpos + byte_pos);
    whereis(byte_pos, vmode, &x, &y);
#if !defined(SYSV_CURSES)
    wmove(stdscr, y, x);
#endif
    EXTRA_REFRESH;
    if (!have_key)
      key = mvwgetch(stdscr, y, x);
    have_key = FALSE;

    switch (key) { /*	any user keys or function keys	*/
      case CMD_FWD:
#if defined(KEY_NPAGE)
      case KEY_NPAGE:
#endif
	return CMD_FWD;
	break;
      
      case CMD_BAK:
#if defined(KEY_PPAGE)
      case KEY_PPAGE:
#endif
	return CMD_BAK;
	break;

      case CMD_UP:
#if defined(KEY_UP)
      case KEY_UP:
#endif
	if (byte_pos - 16 >= 0)
          byte_pos -= 16;
        else
          beep();
	break;

      case CMD_DOWN:
#if defined(KEY_DOWN)
      case KEY_DOWN:
#endif
	if (byte_pos + 16 < nbytes)
          byte_pos += 16;
        else
          beep();
	break;

      case CMD_LEFT:
#if defined(KEY_LEFT)
      case KEY_LEFT:
#endif
	if (byte_pos - 1 >= 0)
          byte_pos -= 1;
        else
          beep();
	break;

      case '\r':
#if defined(KEY_RIGHT)
      case KEY_RIGHT:
#endif
	if (byte_pos + 1 < nbytes)
          byte_pos += 1;
        else
          beep();
	break;

      case CMD_ASC:
#if defined(KEY_F)
      case KEY_F(1):
#endif
        if (vmode == hexMode)
	  vmode = asciiMode;
        else
	  vmode = hexMode;
	break;

      case CMD_SEARCH:
#if defined(KEY_F)
      case KEY_F(2):
#endif
	return CMD_SEARCH;
	break;

      case CMD_UND:
#if defined(KEY_F)
      case KEY_F(3):
#endif
#if defined(KEY_UNDO)
      case KEY_UNDO:
#endif
	return CMD_UND;
	break;

      case CMD_REDRAW:
        erase();
        clear();
        break;

      case CMD_PRI:
#if defined(KEY_F)
      case KEY_F(4):
#endif
	return CMD_PRI;
	break;

      case CMD_JMP:
#if defined(KEY_F)
      case KEY_F(5):
#endif
	return CMD_JMP;
	break;

      case CMD_DON:
#if defined(KEY_F)
      case KEY_F(6):
#endif
	return CMD_DON;
	break;

      default: 
	if (!writeOk)
	  break;
	if (vmode == asciiMode  &&  !IS_VALID(key))
	{	beep();
		break;
	}
	if (vmode == hexMode    &&  !isxdigit(key))
	{	beep();
		break;
	}

	/* user typed a key.   If in the ascii side, allow the overstrike of 
           the ascii value If in the hex side, ensure user types valid hex 
           digits.
			      
	   reflect changes on both sides of the display.
	*/
	reflected = FALSE;
	if (vmode == asciiMode) {
            STANDOUT;
	    waddch(stdscr, (char)key);
	    maybe_dirty = TRUE;
	    wbuf[byte_pos] = (char)key;
	    reflect_change(byte_pos, hexMode, wbuf);
	    reflected = TRUE;
	  }
	else {
          hexa = hexb = 0;
          STANDOUT;			/* 1st nibble	*/
	  waddch(stdscr, (char)key);
          EXTRA_REFRESH;
	  hexa = key;
	  maybe_dirty = TRUE;
	  key = wgetch(stdscr);	/* 2nd nibble	*/
	  if (isxdigit(key)) {
            waddch(stdscr, (char)key);
	    hexb = key;
	    }
	  reflected = TRUE;
	  if (hexa != 0) {
            wbuf[byte_pos] &= 0x0f;
	    wbuf[byte_pos] |= hexval(hexa) << 4;
	    }
	  if (hexb != 0) {
            wbuf[byte_pos] &= 0x0f0;
	    wbuf[byte_pos] |= hexval(hexb);
	    }
	  if (hexa != 0 || hexb != 0)
	    reflect_change(byte_pos, asciiMode, wbuf);
	  }

	if ((key < 0  || key > 256) && key != CMD_LEFT
#if defined(KEY_LEFT)
	    && key != KEY_LEFT
#endif
	    ) /* function key ?? */
	  have_key = TRUE;
	else
	if (byte_pos < BSIZE - 1  &&  (reflected || key == ' ')
            && key != CMD_LEFT
#if defined(KEY_LEFT)
	    && key != KEY_LEFT
#endif
	    )
	  byte_pos++;

        STANDEND;
	if (key == CMD_UND)
	  return CMD_UND;
#if defined(KEY_F)
        if (key == KEY_F(5))
	  return CMD_UND;
#endif
	break;
      } /* switch */
    } /* while */
}

/* =======================================================================
 * Name - reflect_change
 *
 * Purpose - ensure the byte is placed on the screen in the appropriate panel.
 *
 * Arguments:  position in wbuf that has changed
 *	       vmode -- HEX versus ASCII side
 *	       wbuf -- buffer with change
 *
 * Returns     function return -- 
 *
 *========================================================================
 */
#if defined(FLG_PROTOTYPE)
static void reflect_change(int byte, ViewModeEType vmode, char *wbuf)
#else
static void reflect_change(byte, vmode, wbuf)
int	byte;
ViewModeEType vmode;
char	*wbuf;
#endif
{
  int		x,y;
  char	hbuf[2];

  whereis(byte, vmode, &x, &y);
  wmove(stdscr, y, x);
  if (vmode == asciiMode)
    waddch(stdscr, T_TO_A(wbuf[byte]));
  else {
    ttox(hbuf, wbuf[byte]);
    waddch(stdscr, hbuf[0]);
    waddch(stdscr, hbuf[1]);
    }
  EXTRA_REFRESH;
}

/* =======================================================================
 * Name - hexval
 *
 * Purpose - convert an ascii character into a hex nibble.
 *
 * Arguments:  c -- the char
 *
 * Returns     function return -- its equivalent as an int
 *
 *========================================================================
 */
#if defined(FLG_PROTOTYPE)
static int hexval(int c)
#else
static int hexval(c)
int	c; /*	0123456789abcdef expected */
#endif
{
  if (c >= '0'  &&  c <= '9')
    return c - '0';
  return (tolower(c) - 'a' + 10) & 0x0f;
}

/* =======================================================================
 * Name - whereis
 *
 * Purpose - given the relative byte position of a byte within the 256 byte
 *	     buffer, compute the screen coordinates to be used.
 *
 * Arguments:  pos -- relative byte position in wbuf
 *             which -- HEX versus ASCII side of display
 *
 * Returns     function return -- 
 *	       *x -- column of byte on display
 *	       *y -- row of byte on display
 *
 *========================================================================
 */
#if defined(FLG_PROTOTYPE)
static void whereis(int pos, ViewModeEType which, int *x, int *y)
#else
static void whereis(pos, which, x, y)
int	pos;	/*	relative byte position in wbuf	*/
ViewModeEType	which;	/*	HEX or ASC side of display	*/
int	*x, *y;	/*	returned x,y coordinate on display	*/
#endif
{
  if (which == asciiMode) {
    *x = pos % 16 + FIRSTA;
    *y = pos / 16 + FIRSTL;
    }
  else {
    *x = (pos % 16) * 3 + FIRSTH;
    *y = pos / 16 + FIRSTL;
    }
}

/* =======================================================================
 * Name - ttox
 *
 * Purpose - for a given character, return the 2 character hex equivalent.
 *
 * Arguments:  c -- character to convert
 *             x -- location of result
 *
 * Returns     function return -- 
 *	       x[0], x[1] -- the two nibbles
 *
 *========================================================================
 */
#if defined(FLG_PROTOTYPE)
static void ttox(char x[], int c)
#else
static void ttox(x, c)
char x[];
int c;
#endif
{
  x[1] = binToHex[c & 0x0f];
  x[0] = binToHex[(c >> 4) & 0x0f];
}

/* =======================================================================
 * Name - search
 *
 * Purpose - search the file for a string of bytes.  return the byte pointer
 *	     if found.  If not found, return -1
 *
 * Arguments:  fp -- file descriptor of file to search
 *	       s -- string to search for
 *             n -- length of s
 *             here -- current byte position
 *
 * Returns     function return -- position of file that matches string, -1L
 *                                otherwise
 *
 *========================================================================
 */
#if defined(FLG_PROTOTYPE)
static long search(int fp, char s[], int n, long here)
#else
static long search(fp, s, n, here)
int	fp;
char	s[];
int	n;
long	here;
#endif
{
  char	buf[MAX_RETURNED_FROM_DISK];
  int		nn, i, j;
  long	ptr;

  ptr = here;

  while (TRUE) {
    nn = doRead(fp, ptr, buf, sizeof buf);
    if (nn < n)
      break;
    for (i = 0;  i < nn - n + 1;  i++) {
      for (j = 0;  j < n;  j++)	/*	test string match 	*/
	if (s[j] != buf[i + j])
	  break;
	if (j == n) {
	  return ptr + i; /*	valid match found */
	  }
      }
    ptr += (sizeof buf) + (long)(1 - n); /*	with n-1 overlap */
    }
  return -1L; /* no match found	*/
}

/* =======================================================================
 * Name - cvthex
 *
 * Purpose - convert a string of hex 'characters' into bytes.
 *	     attempt to allow blanks or not as user sees fit.
 *
 * Arguments:  s -- null-terminated input string, e.g. c524fe
 *             sx -- address in which to place converted string
 *
 * Returns     function return -- size of resulting string
 *	       *sx -- contains converted bytes
 *
 *========================================================================
 */
#if defined(FLG_PROTOTYPE)
static int cvthex(char s[], char *sx)
#else
static int cvthex(s, sx)
char *s; /*	input string eg:  c524fe */
char	*sx; /*	returned packed bytes. */
#endif
{
  int		n;
  int		 c1, c2;

  n = 0;
  while (*s != '\0') {
    if (*s == ' ') {
      s++;
      continue;
      }
    c1 = hexval(*s++);
    if (*s == ' ')
      sx[n++] = (char)c1;
    else {
      c2 = hexval(*s++);
      sx[n++] = (char)((c1 << 4) | c2);
      }
    }
  return n;
}

/* =======================================================================
 * Name - usage
 *
 * Purpose - Print short blurb
 *
 * Arguments:  None
 *
 * Returns     function return -- 
 *
 *========================================================================
 */
#if defined(FLG_PROTOTYPE)
static void usage(char name[])
#else
static void usage(name)
char name[];
#endif
{
  printf("File Modifier 2.0 (binary file editor), patch level %d\n",
      PATCHLEVEL);
  printf("Usage: %s filename\n", name);
}

/* =======================================================================
 * Name - main
 *
 * Purpose - where it all starts
 *
 * Arguments:  argv, argc noise
 *
 * Returns     function return -- status code to operating system (not reached)
 *
 *========================================================================
 */
#if defined(FLG_PROTOTYPE)
int main(int argc, char *argv[])
#else
int main(argc, argv)
int		argc;
char	*argv[];
#endif
{
  int		fp;
  char	fileBuf[BSIZE];		/*	file i/o buffer	*/
  char	scratchBuf[BSIZE];	/*	work buffer copy of fileBuf*/
  long	fpos, fsearch;		/*	current file position	*/
  int		nbytes;		/*	number of bytes read */
  int		key, yesno;
  char	resp[100];		/*	user response */
  char	stext[100];		/*	last ascii search string */
  char	shex[100];		/*	last hex search string */
  char	hexstring[50];
  int		nn;
  struct stat statbuf;
		  
  if (argc == 1) {
    usage(argv[0]);
    EXIT_SUCCESS;
    }

  signal(SIGTERM, freakout);
  initscr();
#if defined(MSDOS)
{ /* make cursor easier to see */
union REGS inRegs, outRegs;

inRegs.h.ah = 1; /* Set Cursor Type */
inRegs.h.ch = 0;
inRegs.h.cl = 13;
int86(0x10 /* VIDEO_IO*/, &inRegs, &outRegs);
}
#endif
  /*	determine size - for directories or regular files.  */
  strcpy(srcName, argv[1]);
  if (stat(srcName, &statbuf) == -1) {
    cleanup();
    perror("stat() failure");
    EXIT_ERR;
    }
  if ((statbuf.st_mode & S_IFMT) == S_IFDIR
      ||  (statbuf.st_mode & S_IFMT) == S_IFREG)
    max_fpos = (long) statbuf.st_size;
  else
    max_fpos = 0x7fffffffL;	/*	infinity??? */

  if (max_fpos <= 0) { /* what's the point? */
    cleanup();
    fprintf(stderr, "file %s is zero-length!\n", srcName);
    EXIT_ERR;
    }

  fp = doOpen(srcName, FALSE);
  if (fp == -1) {
    fp = doOpen(srcName, TRUE);
    if (fp == -1) {
      cleanup();
      perror("file is unavailable");
      EXIT_ERR;
      }
    writeOk = FALSE;	/*	read only */
    }
  else
    writeOk = TRUE;	/*	read/write	*/

  clear();
  NONCANONICAL;

#if (defined(SYSV_CURSES) || defined(M_XENIX)) && !defined(VMS)
  keypad(stdscr, TRUE);
#endif
  key = CMD_REDRAW;
  fpos = 0;
  byte_pos = 0;
  stext[0] = '\0';
  shex[0] = '\0';

  while (key != CMD_DON) {
    switch (key) {
      case CMD_FWD:	/*	next screen load */
	if (fpos + BSIZE < max_fpos)
	  fpos += BSIZE;
        else
          beep();
	break;

      case CMD_BAK:	/*	previous screen load	*/
	fpos -= BSIZE;
	if (fpos < 0L) {
	  fpos = 0L;
	  byte_pos = 0;
          beep();
	  }
	break;

      case CMD_JMP:	/*	goto byte address */
        CANONICAL;
	mvwaddstr(stdscr, 1, PROMPTCOL, "jump address = ");
	wclrtoeol(stdscr);
        EXTRA_REFRESH;
	wgetstr(stdscr, resp);
	sscanf(resp, "%lx", &fpos);
	if (fpos < 0)
	  fpos = 0;
	else
	if (fpos > max_fpos)
	  fpos = max_fpos;
	byte_pos = (int)(fpos & 0x0ff);
	fpos &= ~((long) BSIZE - 1);
	NONCANONICAL;
	break;

      case CMD_SEARCH:	/*	search ascii text */
	CANONICAL;
        if (vmode == asciiMode) {
	  if (*stext != '\0') {
	    wmove(stdscr, 2, PROMPTCOL);
	    wprintw(stdscr, "( / = use '%s')", stext);
            }
	  mvwaddstr(stdscr, 1, PROMPTCOL, "ascii search = ");
          }
        else {
	  if (*shex != '\0') {
	    wmove(stdscr, 2, PROMPTCOL);
	    wprintw(stdscr, "( / = use '%s')", shex);
	    }
	  mvwaddstr(stdscr, 1, PROMPTCOL, "hex search = ");
          }
	wclrtoeol(stdscr);
        EXTRA_REFRESH;
	wgetstr(stdscr, resp);
	NONCANONICAL;
        if (vmode == asciiMode) {
	  if (resp[0] == '\0')
	    strcpy(resp, stext);
          else
	    strcpy(stext, resp);
	  fsearch = search(fp, resp, strlen(resp), fpos + byte_pos + 1);
          }
        else {
	  if (resp[0] == '\0')
	    strcpy(resp, shex);
          else
	    strcpy(shex, resp);
	  nn = cvthex(resp, hexstring);
	  fsearch = search(fp, hexstring, nn, fpos + byte_pos + 1);
          }
	if (fsearch != -1) {
	  fpos = fsearch;
	  byte_pos = (int)(fpos & 0x0ff);
	  fpos &= ~((long) BSIZE - 1);
	  }
	break;

      case CMD_REDRAW:
        erase();
        clear();
        break;

      case CMD_PRI:
	CANONICAL;
	if (printFile == NULL) {
	  mvwaddstr(stdscr, 1, PROMPTCOL, "print file = ");
	  wclrtoeol(stdscr);
          EXTRA_REFRESH;
	  wgetstr(stdscr, resp);
	  if (resp[0] == '\0') {
	    NONCANONICAL;
	    break;
	    }
	  printFile = fopen(resp, "a+");
	  if (printFile == NULL) { /* display err */
            char message[200];

            sprintf(message, "fopen() error, errno = %d", errno);
	    mvwaddstr(stdscr, 1, PROMPTCOL, message);
	    wclrtoeol(stdscr);
	    NONCANONICAL;
	    break;
	    }
	  }

	mvwaddstr(stdscr, 1, PROMPTCOL, "print comment = ");
	wclrtoeol(stdscr);
        EXTRA_REFRESH;
	wgetstr(stdscr, resp);
	NONCANONICAL;
	printit(resp, fpos + byte_pos, scratchBuf, nbytes);
	mvwaddstr(stdscr, 1, PROMPTCOL, "printed");
	wclrtoeol(stdscr);
	break;		

      case CMD_UND:	/*	undo current changes */
	break;
		    
      default: ;
      }

    /*	get a 256 byte block of file.  copy to a working buffer (scratchBuf) */
    
    nbytes = doRead(fp, fpos, fileBuf, BSIZE);
    memcpy(scratchBuf, fileBuf, nbytes);
    maybe_dirty = FALSE;
    display(fpos, scratchBuf, nbytes);

    /*	allow user to modify file */
    
    key = edit(scratchBuf, nbytes, fpos);
    if (key == CMD_UND)
      continue;
    if (!writeOk || !maybe_dirty || memcmp(fileBuf, scratchBuf, BSIZE) == 0)
      continue;

    echo();
    mvwaddstr(stdscr, 1, PROMPTCOL, "apply changes (y/n): ");
    EXTRA_REFRESH;
    yesno = wgetch(stdscr);
    noecho();
    if (yesno == 'Y' || yesno == 'y')
      doWrite(fp, fpos, scratchBuf, nbytes);
    else
      memcpy(scratchBuf, fileBuf, nbytes);		/* in case PRInt */
    } /* while */
  EXTRA_REFRESH;
  doClose(fp);
  cleanup();
  EXIT_SUCCESS;

return 0; /* for lint */
}
