/********************************************
 D by Richard Conn
  
	D displays a sorted listing of files.  This listing takes
two basic forms (commands are on lines beginning with a $):
  
(1) short form (less than or equal to SHORTLIMIT files in listing) -
  
$ d d*
 d                        32768 rwx
 d.c                       9411 rw-
  
2 Files, 42179 Bytes
  
(2) long form (more than SHORTLIMIT files in listing), with the sort
down the columns -
  
$ d
 d                        32768 rwx |  prform.l                  7715 rw-
 d.c                       9411 rw- |  xdir_orig.c               1802 rw-
 prform.c                 13680 rw- |  xdir_very_old.c          12452 rw-
  
6 Files, 77828 Bytes
  
	If the listing includes directory names, these
names are prefixed with a / and sorted after the files:
  
$ d -a ~
 .cshrc                    1362 rwx |  .rootmenu                  404 rw-
 .defaults                  561 rw- |  .suntools                  464 rw-
 .history                   264 rw- | /benchmark_results          512 rwx
 .indent.pro                 20 rw- | /bin                        512 rwx
 .kermrc                     50 rw- | /components                2048 rwx
 .kermrc_ttyb               279 rw- | /diagrams                   512 rwx
 .login                     835 rw- | /upload                      24 rwx
 .logout                     83 rw- | /upload1                    512 rwx
 .mailrc                    499 rwx | /work                       512 rwx
 .prrc                       38 rw- | /xmc                        512 rwx
  
20 Files, 10003 Bytes
  
	One major distinguishing feature of D is that D presents
the rwx permissions from the point of view of the user running the
program.  If a file is owned by the user, the user rwx permissions
are shown; if the file is not owned by the user but owned by someone
in the user's group, the group rwx permissions are shown; and if the
file is owned by someone outside the user's group, the world rwx
permissions are shown.
  
	Command-line syntax and options are:
  
 Parameters: [-option ...] [filename ...]
 Options:
  -1  Output in one column
  -a  Display all files (including hidden ones)
  -d  Omit directory names from display
  -H  Print this help message
  -s  Suppress display of total file count and size
  
	-1 tells D to ignore the SHORTLIMIT option and always
display in a single column.
	-a displays the hidden files (beginning with a dot) as
well as the other files.
	-d omits directory names (beginning with a / in the listing)
from the display.
	-s suppresses the total file count and size at the bottom
of the listing.
  
     Several options may be placed after a single dash (eg, -1ad).
  
Installation:
  
 One of the following symbols must be
 defined in the compiler command line:
	BSD    - any flavor of BSD UNIX
	POSIX  - standard POSIX (IEEE Std 1003.1, April 1986), untested
	SYSVR2 - Release 2 of UNIX SYSTEM V
	SYSVR3 - Release 3 of UNIX SYSTEM V
 Example:
	cc -o d -DBSD d.c
  
Acknowledgments:
	Special thanks to Frank Wancho for providing the SYSTEM V
code specifics and checking the program out under several SYSTEM V
machines.  Thanks also to Frank for his review effort and constructive
comments.
 ********************************************/

#ifdef SYSVR2
#define SYSV
#endif

#ifdef SYSVR3
#define SYSV
#endif

/* Global Parameters */

#define VERSION "D 2.0"

#define CURDIR "."

#define OMIT1 "."
#define OMIT2 ".."

#define DIRLEADLEN 1		/* chars in dir indicator field */
#define NAMELEN 22		/* chars in file name field */
#define SIZELEN 8		/* chars in file size field */
#define RWXLEN 4		/* leading space plus rwx */
#define FORMATLEN (DIRLEADLEN + NAMELEN + SIZELEN + RWXLEN)

#define SHORTLIMIT 5		/* if fewer than or same as
				 * SHORTLIMIT files, don't split */

typedef struct entry {
    char            string[FORMATLEN + 1];
    struct entry   *next;
}               ENT, *ENTPTR;

ENT            *efirst;

char            format[20];

int             printall = 0;
int             summary = 1;
int             onecol = 0;
int             omitdir = 0;

int             filecount = 0;
long            totalsize = 0;

/********************************************
 Begin BSD 4.x UNIX Implementation Specifics
 ********************************************/
#ifdef BSD

#include <stdio.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <sys/stat.h>

dirsize(dirp)
    struct direct  *dirp;
{
    return DIRSIZ(dirp);
}

/********************************************
 End   BSD 4.x UNIX Implementation Specifics
 ********************************************/
#endif				/* BSD */

/********************************************
 Begin POSIX Implementation Specifics
 ********************************************/
#ifdef POSIX

#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <stat.h>

dirsize(dirp)
    struct dirent  *dirp;
{
    return strlen(dirp->d_name) + 1;
}

/********************************************
 End   POSIX Implementation Specifics
 ********************************************/
#endif				/* POSIX */

/********************************************
 Begin UNIX SYSTEM V Implementation Specifics
 ********************************************/
#ifdef SYSV

#include <stdio.h>

#ifdef SYSVR3
#include <sys/types.h>
#endif
#include <sys/file.h>
#include <sys/dir.h>
#include <sys/stat.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>

typedef struct {
    int             size;
    struct direct  *files;
    struct direct  *current;
}               DIR;

DIR            *
opendir(name)
{
    DIR            *dir;
    int             fd, open();
    struct stat     sbuf;
    int             x;

    if (stat(name, &sbuf) != 0)
	return (NULL);
    if (!(sbuf.st_mode & S_IFDIR)) {
	errno = ENOTDIR;
	return (NULL);
    }
    fd = open(name, O_RDONLY, 0);
    if (fd == -1)
	perror("open");
    if (fd == -1)
	return (NULL);
    if ((dir = (DIR *) malloc(sizeof(DIR))) == NULL) {
	close(fd);
	return (NULL);
    }
    dir->size = sbuf.st_size;
    if ((dir->files = (struct direct *) malloc(dir->size)) == 0) {
	close(fd);
	free(dir);
	return (NULL);
    }
    x = read(fd, dir->files, dir->size);
    if (x == -1)
	perror("read");
    dir->current = dir->files;
    close(fd);
    return (dir);
}

struct direct  *
readdir(dirp)
    DIR            *dirp;
{
    int             s = dirp->size / sizeof(struct direct);
    struct direct  *temp;
    if (dirp->current >= dirp->files + s)
	return (NULL);
    while (dirp->current->d_ino == 0)
	if (dirp->current >= dirp->files + s)
	    return (NULL);
	else
	    dirp->current++;
    if (dirp->current >= dirp->files + s)
	return (NULL);
    temp = dirp->current++;
    return (temp);
}

rewinddir(dirp)
    DIR            *dirp;
{
    dirp->current = dirp->files;
}

closedir(dirp)
    DIR            *dirp;
{
    free(dirp->files);
    free(dirp);
}

dirsize(dirp)
    struct direct  *dirp;
{
    return DIRSIZ;
}

/********************************************
 End   UNIX SYSTEM V Implementation Specifics
 ********************************************/
#endif				/* SYSV */

main(argc, argv)
    int             argc;
    char           *argv[];
{
    int             i;
    int             dirflag = 0;

    /* Set up first entry pointer and format string */
    efirst = NULL;
    sprintf(format, "%%c%%-%ds%%%dld %%c%%c%%c", NAMELEN, SIZELEN);

    /* Process command line arguments */
    for (i = 1; i < argc; i++) {
	if (*argv[i] == '-')
	    option(&argv[i][1]);
	else {
	    dirflag = 1;
	    dir(argv[i]);
	}
    }

    /* If no directories processed, process current directory */
    if (!dirflag)
	dir(CURDIR);

    /* Do dump of list */
    listdump();

    /* Print summary if requested */
    if (summary)
	printf("\n%d Files, %ld Bytes\n", filecount, totalsize);

    exit(0);
}

/* Process options */
option(text)
    char           *text;
{
    while (*text) {
	switch (*text) {
	case '1':
	    onecol = 1;
	    break;
	case 'a':
	    printall = 1;
	    break;
	case 'd':
	    omitdir = 1;
	    break;
	case 'H':
	    help();
	    break;
	case 's':
	    summary = 0;
	    break;
	default:
	    fprintf(stderr, " Invalid option at %s\n", text);
	    help();
	    break;
	}
	text++;
    }
}

/* Print help message */
help()
{
    fprintf(stderr, "%s\n", VERSION);
    fprintf(stderr, " Parameters: [-option ...] [filename ...]\n");
    fprintf(stderr, " Options:\n");
    fprintf(stderr, "  -1  Output in one column\n");
    fprintf(stderr, "  -a  Display all files (including hidden ones)\n");
    fprintf(stderr, "  -d  Omit directory names from display\n");
    fprintf(stderr, "  -H  Print this help message\n");
    fprintf(stderr, "  -s  Suppress display of total file count and size\n");
    exit(0);
}

/* Issue directory display */
dir(dirname)
    char           *dirname;
{
    DIR            *opendir(), *dirp;
#ifdef POSIX
    struct dirent  *current_entry, *readdir();
#else
    struct direct  *current_entry, *readdir();
#endif				/* POSIX */
    char            current_name[256];
    char            full_name[256];
    struct stat     statbuf;
    int             stat();
    char           *dummy, *strcpy(), *strcat(), *strncpy();

    /* Try to open dirname as a directory */
    if ((dirp = opendir(dirname)) == NULL) {
	/* dirname was not a directory, so must be a file */
	if (stat(dirname, &statbuf) == -1) {
	    fprintf(stderr, " File %s Not found\n", dirname);
	} else {
	    store_entry(dirname, &statbuf);
	}
    } else {
	/*
	 * dirname is a directory, so read and store all entries in
	 * it 
	 */
	for (current_entry = readdir(dirp); current_entry != NULL;
	     current_entry = readdir(dirp)) {
	    dummy = strncpy(current_name, current_entry->d_name,
			    dirsize(current_entry));
	    if (!(strcmp(current_name, OMIT1) == 0 ||
		  strcmp(current_name, OMIT2) == 0)) {
		/*
		 * Generate full file name (with directory prefix)
		 * and get status information of it and store 
		 */
		strcpy(full_name, dirname);
		strcat(full_name, "/");
		strcat(full_name, current_name);
		stat(full_name, &statbuf);
		store_entry(current_name, &statbuf);
	    }
	}
	closedir(dirp);
    }
}

/* Store a line of the directory display in the linked list */
store_entry(filename, statbuf)
    char           *filename;
    struct stat    *statbuf;
{
    char           *fname;
    char           *rover;
    ENTPTR          output, erover, elast;
#ifndef SYSV
    char           *malloc();
#endif				/* SYSV */
    int             shift;
    int             str, stw, stx;

    /*
     * Determine if file name is prefixed by a directory and set
     * fname to point to the first character of the file name 
     */
    fname = filename;
    rover = filename;
    while (*rover) {
	if (*rover == '/')
	    fname = ++rover;
	else
	    rover++;
    }

    /* Check for hidden file and printall option */
    if (!printall && (*fname == '.'))
	return;

    /* Check for directory name and omitdir option */
#ifdef POSIX
    if (omitdir && S_ISDIR(statbuf->st_mode))
#else
    if (omitdir && (statbuf->st_mode & S_IFDIR))
#endif				/* POSIX */
	return;

    /*
     * Increment file count and total size and allocate memory for
     * new entry 
     */
    filecount++;
    totalsize += statbuf->st_size;
    if ((output = (ENTPTR) (malloc(sizeof(ENT)))) == NULL) {
	fprintf(stderr, " Dynamic Memory Overflow\n");
	exit(1);
    }
    /* Truncate file name string if necessary */
    if (strlen(fname) > NAMELEN - 1) {
	fname[NAMELEN - 1] = '*';
	fname[NAMELEN] = '\0';
    }
    /*
     * Determine if we are same user, group, or world as target file
     * and set shift count accordingly 
     */
    if (statbuf->st_uid == getuid()) {
#ifdef POSIX
	str = statbuf->st_mode & S_IRUSR;
	stw = statbuf->st_mode & S_IWUSR;
	stx = statbuf->st_mode & S_IXUSR;
#else
	shift = 0;
#endif				/* POSIX */
    } else {
	if (statbuf->st_gid == getgid()) {
#ifdef POSIX
	    str = statbuf->st_mode & S_IRGRP;
	    stw = statbuf->st_mode & S_IWGRP;
	    stx = statbuf->st_mode & S_IXGRP;
#else
	    shift = 3;
#endif				/* POSIX */
	} else {
#ifdef POSIX
	    str = statbuf->st_mode & S_IROTH;
	    stw = statbuf->st_mode & S_IWOTH;
	    stx = statbuf->st_mode & S_IXOTH;
#else
	    shift = 6;
#endif				/* POSIX */
	}
    }
#ifndef POSIX
    str = statbuf->st_mode & (S_IREAD >> shift);
    stw = statbuf->st_mode & (S_IWRITE >> shift);
    stx = statbuf->st_mode & (S_IEXEC >> shift);
#endif				/* POSIX */

    /* Generate output string */
    sprintf(output->string, format,
#ifdef POSIX
	    ((S_ISDIR(statbuf->st_mode)) ? '/' : ' '),
#else
	    ((statbuf->st_mode & S_IFDIR) ? '/' : ' '),
#endif				/* POSIX */
	    fname,
	    statbuf->st_size,
	    (str ? 'r' : '-'),
	    (stw ? 'w' : '-'),
	    (stx ? 'x' : '-'));


    /* Place output string in linked list */
    if (efirst == NULL) {
	efirst = output;
	output->next = NULL;
    } else {
	erover = efirst;
	elast = NULL;
	while ((erover != NULL) && (strcmp(erover->string,
					   output->string) < 0)) {
	    elast = erover;
	    erover = erover->next;
	}
	output->next = erover;
	if (elast == NULL)
	    efirst = output;
	else
	    elast->next = output;
    }
}

/* Dump linked list to console */
listdump()
{
    ENTPTR          erover, erover2;
    int             i;

    if ((filecount <= SHORTLIMIT) || onecol) {
	erover = efirst;
	while (erover != NULL) {
	    printf("%s\n", erover->string);
	    erover = erover->next;
	}
    } else {
	erover = efirst;
	erover2 = efirst;
	for (i = 0; i < filecount / 2; i++)
	    erover2 = erover2->next;
	if (filecount % 2)
	    erover2 = erover2->next;
	for (i = 0; i < filecount / 2; i++) {
	    printf("%s | %s\n", erover->string, erover2->string);
	    erover = erover->next;
	    erover2 = erover2->next;
	}
	if (filecount % 2)
	    printf("%s\n", erover->string);
    }
}
