/*
 * capp - cut and paste processor
 *
 * capp is a batch-mode cut and paste processor for
 * dramatic reorganization of files. unlike m4 and
 * similar filter programs, it makes two passes:
 *	pass1: process all cut requests
 *	pass2: process all paste requests 
 * which enables the cut fragments to be pasted
 * somewhere *in front* of the actual cut location.
 * capp is especially useful in situations where
 * the reorganization of the file is such that most
 * screen editors will keep you on your terminal for
 * hours. It is even possible to enter the text with
 * future organization in mind, and simply run 
 * capp from within the editor. 
 *
 * by default, all cut sections that are not pasted
 * elsewhere get pasted to the end of the file in
 * sequence.
 *
 * Author: Ozan S. Yigit
 *	   Metasystems
 *	   Jan 3 1986
 *
 * Jan 07 1986 Oz	Getopt interface and misc. options.
 * Jan 03 1986 Oz	Initial cut.
 */
#include <stdio.h>
#include <signal.h>

char *divfile = "/tmp/cap*XXXXXX";	/* diversion file names */
#define UNIQUE	8

#define MAXOUT	10
#define BUFMAX	512
#define WR	1
#define RD	0

#define	CREAT	1
#define	USED	2

struct dfile {
	FILE *fp;
	char  fl;
	char *fn;
};

FILE *active;				/* active output file   */
FILE *temp;				/* temporary output     */
struct dfile outfile[MAXOUT];		/* diversion array      */
char *captemp;				/* filename for temp.   */
char buf[BUFMAX];			/* input buffer..       */

char stac = '%';			/* default starter char */
char cutc = 'C';			/* default "cut"   char */
char pasc = 'P';			/* default "paste" char */
char endc = 'E';			/* default "end"   char */

int onintr();
char *save();
char *malloc();
char *mktemp();

main(argc, argv)
char *argv[];
{
	extern char *optarg;
	extern int optind;

	int c;
	register char *p;

	if (signal(SIGINT, SIG_IGN) != SIG_IGN)
		(void) signal(SIGINT, onintr);

	while ((c = getopt(argc, argv, "c:p:e:s:")) != EOF)
		switch(c) {

		case 's':
			stac = *optarg;
			break;
		case 'c':
			cutc = *optarg;
			break;
		case 'p':
			pasc = *optarg;
			break;
		case 'e':
			endc = *optarg;
			break;
		case '?':
			usage();
		}


	captemp = mktemp(divfile);
	if ((temp = fopen(captemp, "w")) == NULL)
		cant(captemp, WR);
	active = temp;
	/*
	 * first pass: 	collect all CUT sections.
	 *		resuting file goes into temp. CUT
	 *		indicates that the section dissapears
	 *		from the input file, and stored in
	 *		one of the diversions, or in the
	 *		bitbucket.
	 */
	while ((p = fgets(buf, BUFMAX, stdin)) != NULL)
		if (*p++ == stac) {
			if ((c = *p++) == cutc) {	
				while (*p == ' ' || *p == '\t')
					p++;
				divert(atoi(p));
			}
			else if (c == endc)
				divert(0);
			else
				out(buf);
		}
		else
			out(buf);
	(void) fclose(temp);
	closeall();
	captemp[UNIQUE] = '*';
	if ((temp = fopen(captemp, "r")) == NULL)
		cant(captemp, RD);
	active = stdout;
	/*
	 * second pass: now read in all diversions. (PASTE)
	 *		since we keep diversions around,
	 *		it is possible to read them into
	 *		different places more than once.
	 */

	while ((p = fgets(buf, BUFMAX, temp)) != NULL)
		if (*p++ == stac) {
			if (*p++ = pasc) {
				while (*p == ' ' || *p == '\t')
					p++;
				undivert(atoi(p));
			}
			else
				out(buf);
		}
		else
			out(buf);
	/*
	 * at the end: read in all created
	 *	       but unused diversions.
	 */
	unused();
	rmall();
	exit(0);
}

divert(n) {
	if (n != 0) {
		if (n < 0 || n >= MAXOUT)
			n = 0;
		if (outfile[n].fp == NULL) {
			captemp[UNIQUE] = n + '0';
			if ((outfile[n].fp = fopen(captemp, "w")) == NULL)
				cant(captemp, WR);
			outfile[n].fn = save(captemp);
			outfile[n].fl |= CREAT;
		}
		active = outfile[n].fp;
	}
	else
		active = temp;
}

undivert(n) {
	if (n > 0 && n < MAXOUT)
		getdiv(n);
}

closeall() {
	register int n;

	for (n = 0; n < MAXOUT; n++)
		if (outfile[n].fp != NULL) {
			(void) fclose (outfile[n].fp);
			outfile[n].fp = NULL;
		}
}

unused() {
	register int n;

	for (n = 1; n < MAXOUT; n++)
		if (outfile[n].fl == CREAT)	/* created but not used */
			getdiv(n);
}

rmall() {
	register int n;

	for (n = 0; n < MAXOUT; n++)
		if (outfile[n].fl & CREAT) {
			if (outfile[n].fp != NULL)
				(void)	fclose (outfile[n].fp);
			(void) unlink(outfile[n].fn);
		}
	captemp[UNIQUE] = '*';
	(void) unlink(captemp);
}

getdiv(n) {
	register int c;

	if (!(outfile[n].fl & CREAT))
		fprintf(stderr,"diversion #%d is inactive.\n",n);
	else {
		if (outfile[n].fp == NULL) {
			if ((outfile[n].fp = fopen(outfile[n].fn, "r")) == NULL)
				cant(outfile[n].fn, RD);
		
		}
		else
			rewind(outfile[n].fp);
		outfile[n].fl |= USED;
		while ((c = getc(outfile[n].fp)) != EOF)
			putc(c, active);
	}
}

out(s)
char *s;
{
	while (*s)
		putc(*s++, active);
}

onintr() {
	rmall();
	fprintf(stderr, "\ninterrupted.\n");
	exit(1);
}

cant(s, n)
char *s;
{
	fprintf(stderr, "%s: cannot open for %s.\n", s, n ? "write" : "read");
	rmall();
	exit(1);
}

char *save(s)
char *s;
{
	char *p;
	if ((p = malloc(strlen(s) + 1)) != NULL)
		(void) strcpy(p, s);
	return(p);
}

usage() {
fprintf(stderr, "usage: capp [-s char] [-c char] [-e char] [-p char]\n");
	exit(1);
}
