/* agef
  
   SCCS ID	@(#)agef.c	1.6	7/9/87
  
   David S. Hayes, Site Manager
   Army Artificial Intelligence Center
   Pentagon HQDA (DACS-DMA)
   Washington, DC  20310-0200
  
   Phone:  202-694-6900
   Email:  merlin@hqda-ai.UUCP   merlin%hqda-ai@seismo.CSS.GOV
  
   +=======================================+
   | This program is in the public domain. |
   +=======================================+
  
   This program scans determines the amount of disk space taken up
   by files in a named directory.  The space is broken down
   according to the age of the files.  The typical use for this
   program is to determine what the aging breakdown of news
   articles is, so that appropriate expiration times can be
   established.
  
   Call via
  
	agef fn1 fn2 fn3 ...
  
   If any of the given filenames is a directory (the usual case),
   agef will recursively descend into it, and the output line will
   reflect the accumulated totals for all files in the directory.
*/

#include "patchlevel.h"

#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <stdio.h>

#include "customize.h"
#include "hash.h"

#define SECS_DAY	(24L * 60L * 60L)	/* seconds in one day */
#define TOTAL		(n_ages-1)	/* column number of total
					 * column */
#define SAME		0	/* for strcmp */
#define MAXUNS		((unsigned) -1L)
#define MAXAGES		40	/* max number of age columns */
#define BLOCKSIZE	512	/* size of a disk block */

#define K(x)		((x+1) >> 1)	/* convert stat(2) blocks into
					 * k's.  On my machine, a block
					 * is 512 bytes. */


extern char    *optarg;		/* from getopt(3) */
extern int      optind,
                opterr;


char           *Program;	/* our name */

short           sw_follow_links = 1;	/* follow symbolic links */

/* Types of inode times for sw_time. */
#define	MODIFIED	1
#define CHANGED		2
#define ACCESSED	3
short           sw_time = MODIFIED;
short           sw_summary;	/* print Grand Total line */


short           n_ages = 0;	/* how many age categories */
unsigned        ages[MAXAGES];	/* age categories */
int             inodes[MAXAGES];/* inode count */
long            sizes[MAXAGES];	/* block count */

char            topdir[NAMELEN];/* our starting directory */
long            today,
                time();		/* today's date */



main(argc, argv)
    int             argc;
    char           *argv[];
{
    int             i,
                    j;
    int             option;
    int             total_inodes[MAXAGES];	/* for grand totals */
    long            total_sizes[MAXAGES];

    Program = *argv;		/* save our name for error messages */

    /* Pick up options from command line */
    while ((option = getopt(argc, argv, "smacd:")) != EOF) {
	switch (option) {
	  case 's':
	    sw_follow_links = 0;
	    break;

	  case 'm':
	    sw_time = MODIFIED;
	    break;

	  case 'a':
	    sw_time = ACCESSED;
	    break;

	  case 'c':
	    sw_time = CHANGED;
	    break;

	  case 'd':
	    n_ages = 0;
	    while (*optarg) {
		ages[n_ages] = atoi(optarg);	/* get day number */
		if (ages[n_ages] == 0)	/* check, was it a number */
		    break;	/* no, exit the while loop */
		n_ages++;
		while (isdigit(*optarg))	/* advance over the
						 * digits */
		    optarg++;
		if (*optarg == ',')	/* skip a comma separator */
		    optarg++;

		if (n_ages > (MAXAGES - 2)) {
		    fprintf(stderr, "too many ages, max is %d\n", MAXAGES - 2);
		    exit(-1);
		};
	    };
	    ages[n_ages++] = MAXUNS;	/* "Over" column */
	    ages[n_ages++] = MAXUNS;	/* "Total" column */
	    break;
	};
    };

    /* If user didn't specify ages, make up some that sound good. */
    if (!n_ages) {
	n_ages = 6;
	ages[0] = 7;
	ages[1] = 14;
	ages[2] = 30;
	ages[3] = 45;
	ages[4] = MAXUNS;
	ages[5] = MAXUNS;
    };

    /* If user didn't specify targets, inspect current directory. */
    if (optind >= argc) {
	argc = 2;
	argv[1] = ".";
    };

    sw_summary = argc > 2;	/* should we do a grant total? */

    getwd(topdir);		/* find out where we are */
    today = time(0) / SECS_DAY;

    make_headers();		/* print column heades */

    /* Zero out grand totals */
    for (i = 0; i < n_ages; i++)
	total_inodes[i] = total_sizes[i] = 0;

    /* Inspect each argument */
    for (i = optind; i < argc; i++) {
	for (j = 0; j < n_ages; j++)
	    inodes[j] = sizes[j] = 0;

	chdir(topdir);		/* be sure to start from the same place */
	get_data(argv[i]);	/* this may change our cwd */

	display(argv[i], inodes, sizes);
	for (j = 0; j < n_ages; j++) {
	    total_inodes[j] += inodes[j];
	    total_sizes[j] += sizes[j];
	};
    };

    if (sw_summary) {
	putchar('\n');		/* blank line */
	display("Grand Total", total_inodes, total_sizes);
    };

#ifdef HSTATS
    fflush(stdout);
    h_stats();
#endif
    exit(0);
};



 /*
  * Get the aged data on a file whose name is given.  If the file is a
  * directory, go down into it, and get the data from all files inside. 
  */
get_data(path)
    char           *path;
{
    struct stat     stb;
    int             i;
    long            age;	/* file age in days */

#ifdef LSTAT
    if (sw_follow_links)
	stat(path, &stb);	/* follows symbolic links */
    else
	lstat(path, &stb);	/* doesn't follow symbolic links */
#else
    stat(path, &stb);
#endif

    /* Don't do it again if we've already done it once. */
    if (h_enter(stb.st_dev, stb.st_ino) == OLD)
	return;

    if ((stb.st_mode & S_IFMT) == S_IFDIR)
	down(path);
    if ((stb.st_mode & S_IFMT) == S_IFREG) {
	switch (sw_time) {
	  case MODIFIED:
	    age = today - stb.st_mtime / SECS_DAY;
	    break;
	  case CHANGED:
	    age = today - stb.st_ctime / SECS_DAY;
	    break;
	  case ACCESSED:
	    age = today - stb.st_atime / SECS_DAY;
	    break;
	};

	for (i = 0; i < TOTAL; i++) {
	    if (age <= ages[i]) {
		inodes[i]++;
		sizes[i] += K(stb.st_blocks);
		break;
	    };
	};
	inodes[TOTAL]++;
	sizes[TOTAL] += K(stb.st_blocks);
    };
}



 /*
  * We ran into a subdirectory.  Go down into it, and read everything
  * in there. 
  */

down(subdir)
    char           *subdir;
{
    OPEN           *dp;		/* stream from a directory */
    char            cwd[NAMELEN];
    READ           *file;	/* directory entry */

    if ((dp = opendir(subdir)) == NULL) {
	fprintf(stderr, "%s: can't read %s/%s\n", Program, topdir, subdir);
	return;
    };

    getwd(cwd);			/* remember where we are */
    chdir(subdir);		/* go into subdir */
    for (file = readdir(dp); file != NULL; file = readdir(dp))
	if (strcmp(NAME(*file), "..") != SAME)
	    get_data(NAME(*file));

    chdir(cwd);			/* go back where we were */
    closedir(dp);		/* shut down the directory */
}



 /*
  * Print one line of the table. 
  */
display(name, inodes, sizes)
    char           *name;
    int             inodes[];
long            sizes[];
{
    char            tmpstr[30];
    int             i;

    for (i = 0; i < n_ages; i++) {
	tmpstr[0] = '\0';
	if (inodes[i] || i == TOTAL) {
	    if (sizes[i] < 10000)
		sprintf(tmpstr, "%d %4ldk", inodes[i], sizes[i]);
	    else
		sprintf(tmpstr, FLOAT_FORMAT, inodes[i], sizes[i] / 1000.0);
	};
	printf("%10s ", tmpstr);
    };
    printf(" %s\n", name);
}


 /*
  * Print column headers, given the ages. 
  */
make_headers()
{
    char            header[15];
    int             i;

    for (i = 0; i < TOTAL; i++) {
	if (ages[i] == MAXUNS)
	    sprintf(header, "Over %d", ages[i - 1]);
	else
	    sprintf(header, "%d %s", ages[i], ages[i] > 1 ? "days" : "day");
	printf("%10s ", header);
    };
    printf("     Total  Name\n");
    for (i = 0; i < n_ages; i++)
	printf("---------- ");
    printf(" ----\n");
}
