/*
 * $Header: /src3/ecn/access/RCS/access.c,v 1.5 85/03/14 09:28:22 davy Exp $
 *
 * access.c - file access routines for access program.
 *
 * David A. Curry, April 1984
 *
 * Modified: 7/18/84 (also defs.h, main.c)
 *	The method of reading the user/group data was taken out of this
 *	program and placed in a separate program, mkaccessdb.  This
 *	program now reads in the files created by mkaccessdb, a huge
 *	time improvement.
 * 	--Dave Curry
 *
 * $Log:	access.c,v $
 * Revision 1.5  85/03/14  09:28:22  davy
 * Installed bug fix from Jeff Glass (security!jjg) in which we
 * weren't interpreting the file permissions right.
 * 
 * Revision 1.4  85/02/01  13:37:37  root
 * Fixed stupid bugs....
 * 
 * Revision 1.3  85/02/01  13:08:35  root
 * Fixed bug in permission printing routines.
 * 
 * Revision 1.3  85/02/01  13:03:23  root
 * 
 * 
 * Revision 1.2  84/07/19  09:07:29  root
 * fast version .. pre-structured
 * 
 * Revision 1.1  84/06/25  10:58:42  root
 * Initial revision
 * 
 */
#include "defs.h"

extern struct grp *ghead, *gtail;
extern struct user *uhead, *utail;

char *malloc();
extern int nusers;
extern char *pname;
extern char cur_dir[];

loaduids()
{
	struct stat sbuf;
	register FILE *fp;
	register struct user *u;

	if ((fp = fopen(USERFILE, "r")) == NULL) {
		fprintf(stderr, "%s: cannot open %s.\n", pname, USERFILE);
		exit(1);
	}
	
	fstat(fileno(fp), &sbuf);
	nusers = sbuf.st_size / sizeof(struct user);
	
	if ((uhead = (struct user *) malloc(nusers * sizeof(struct user))) == USNULL) {
		fprintf(stderr, "%s: not enough memory.\n", pname);
		exit(1);
	}
	
	u = uhead;
	
	while (fread((char *) u, USIZE, 1, fp) != NULL)
		u++;
	
	utail = u;
	fclose(fp);
}

loadgids()
{
	int ngids;
	struct stat sbuf;
	register FILE *fp;
	register struct grp *g;

	if ((fp = fopen(GROUPFILE, "r")) == NULL) {
		fprintf(stderr, "%s: cannot open %s.\n", pname, GROUPFILE);
		exit(1);
	}
	
	fstat(fileno(fp), &sbuf);
	ngids = sbuf.st_size / sizeof(struct grp);
	
	if ((ghead = (struct grp *) malloc(ngids * sizeof(struct grp))) == GRNULL) {
		fprintf(stderr, "%s: not enough memory.\n", pname);
		exit(1);
	}
	
	g = ghead;
	
	while (fread((char *) g, GSIZE, 1, fp) != NULL)
		g++;
	
	gtail = g;
	fclose(fp);
}

/*
 * process - gets full path to file, processes it and prints theresults.
 */
process(file)
char *file;
{
	short endpath;
	register char *s;
	char path[MAXPATHLEN];
	struct stat sbuf, sbuf2;
	
#ifdef DEBUG /*------------------------------------------------------------*/
	printf("process(%s)\n", file);
#endif /*------------------------------------------------------------------*/
	
	/*
	 * If beginning of pathname is '/', we don't need the
	 * current directory.  Otherwise, we have to make a path
	 * like <current directory>/file.
	 */
	if (*file == '/') {
		strcpy(path, file);
	}
	else {
		/*
		 * Put the pieces together, resolve "../"
		 * and things.
		 */
		copy(path, cur_dir, file);
	}

	/*
	 * Start with the root directory.
	 */
	if (stat(ROOTDIR, &sbuf) < 0) {
		fprintf(stderr, "%s: cannot stat %s\n", pname, ROOTDIR);
		return;
	}

	sbuf2 = sbuf;

	/*
	 * Since this is the first time through, we
	 * want to set the permissions to what they
	 * are here.
	 */
	doaccess(sbuf.st_uid, sbuf.st_gid, sbuf.st_mode, SET);
	
	s = path + 1;
	
	/*
	 * Now follow the path component by component.
	 */
	do {
		while ((*s != '/') && (*s != NULL))
			s++;
		
		endpath = (*s == '/' ? 0 : 1);
		*s = NULL;
		
		/*
		 * Save old stat structure.
		 */
		sbuf2 = sbuf;
		
		if (stat(path, &sbuf) < 0) {
			fprintf(stderr, "%s: cannot stat %s\n", pname, path);
			return;
		}
		
		/*
		 * Now we want to add the present permissions
		 * to those already there.
		 */
		doaccess(sbuf.st_uid, sbuf.st_gid, sbuf.st_mode, ADD);
		
		*s++ = '/';
	} while (!endpath);

	*--s = NULL;
	
	/*
	 * Print out the information.
	 */
	printout(path, sbuf, sbuf2);
}

/*
 * copy - copies s1 and s2 into dest, resolving things like "./" and "../".
 */
copy(dest, s1, s2)
char *dest, *s1, *s2;
{
	char *rindex();
	char buf[MAXPATHLEN];
	register char *s, *t;
	
	s = s2;
	strcpy(buf, s1);
	s1 = buf;
	
	/*
	 * Until we run out of things like "./" and "../"...
	 */
	while (!strncmp(s, "./", 2) || !strncmp(s, "../", 3)) {
		/*
		 * If it was "./", just skip over it.
		 */
		if (*(s+1) == '/') {
			s += 2;
			continue;
		}
		
		/* 
		 * If it was "../", we back up to the previous
		 * '/' in the s1 string.
		 */
		t = rindex(s1, '/');
		
		/*
		 * Should never be NULL, since s1[0] == '/'.
		 * If it's the beginning slash, then leave the
		 * slash there.
		 * If it's a different slash, just set it
		 * to NULL.
		 */
		if (t == NULL)
			break;
		else if (t == s1)
			*(t+1) = NULL;
		else
			*t = NULL;
		
		s += 3;
	}
	
	/*
	 * Now put together the pieces.
	 */
	strcpy(dest, s1);

	/*
	 * If all that remains of s is "..", skip
	 * backwards in dest.
	 */
	if (!strcmp(s, "..")) {
		t = rindex(dest, '/');
		
		if (t == dest)
			*(t+1) = NULL;
		else
			*t = NULL;
		
		*s = NULL;
	}

	/*
	 * Only add s if it is non-NULL and not equal to "."
	 */
	if ((*s != NULL) && (strcmp(s, ".") != 0)) {
		strcat(dest, "/");
		strcat(dest, s);
	}

#ifdef DEBUG /*------------------------------------------------------------*/
	printf("copy - final path is '%s'\n", dest);
#endif /*------------------------------------------------------------------*/

}

/*
 * doaccess - for each user, determines the permissions he gets given
 *	      that the file is owned by user u, group g, with mode m.
 */
doaccess(u, g, m, how)
int u, g, m, how;
{
	register int shmode;
	register struct user *p;
	register int uid, gid, mode, shift;
	
	uid = u;
	gid = g;
	mode = m;
	
	/*
	 * If how is SET, then we simply set the 
	 * bits in each user's structure to be "his"
	 * bits in mode.
	 */
	if (how == SET) {
		p = uhead;
		while (p != utail) {
			if (p->uid == uid)
				p->perm = (mode >> 6) & 07;
			else if (member(gid, p->groups))
				p->perm = (mode >> 3) & 07;
			else
				p->perm = mode & 07;

			/*
			 * Old permission bits get same as current.
			 */
			p->operm = p->perm;

			p++;
		}
		
		return;
	}
	
	/*
	 * Otherwise, we have to selectively turn on bits.
	 */
	p = uhead;
	while (p != utail) {
		if (p->uid == uid)
			shift = 6;
		else if (member(gid, p->groups))
			shift = 3;
		else
			shift = 0;
		
		/*
		 * Save the old permission bits.
		 */
		p->operm = p->perm;

		shmode = mode >> shift;

		/*
		 * If a given permission is on for this user, he is
		 * only allowed to gain that permission if he was
		 * able to search the previous directory.  We don't
		 * ever turn on execute permission, because once you
		 * lose it, you can never regain it.
		 *
		 * BUG FIX from security!jjg (Jeff Glass).
		 * This stuff should only happen if we had execute (search)
		 * permission on the previous directory, otherwise,
		 * we have to lose all permissions on the file (since
		 * we can't even get to it).  Thanks, Jeff!
		 */
		if ((p->operm & EXECP) == EXECP) {
			if ((shmode & READP) == READP)
				p->perm |= READP;
			else
				p->perm &= ~READP;
			
			if ((shmode & WRITEP) == WRITEP)
				p->perm |= WRITEP;
			else
				p->perm &= ~WRITEP;
		
			if ((shmode & EXECP) != EXECP)
				p->perm &= ~EXECP;
		}
		else {
			p->perm = 0;
		}

		p++;
	}
}

/*
 * member - determine if group g is in the group list gl.
 */
member(g, gl)
int g, *gl;
{
	register int i, gid;
	
	gid = g;
	
	for (i=0; (i < NGROUPS) && (gl[i] != NOGROUP); i++) {
		if (gl[i] == gid)
			return(1);
	}
	
	return(0);
}

/*
 * printout - print out the information
 */
printout(file, sbuf, sbuf2)
char *file;
struct stat sbuf, sbuf2;
{
	char *username(), *groupname(), *whatis();
	int ingroup, ningrp, nread, nexec, nwrite;

	printf("%s (%s):\n", file, whatis(file, sbuf.st_mode));

	/*
	 * Count number of users with each permission,
	 * count members of group, see if uid is in
	 * group.
	 */
	count(&nread, &nwrite, &nexec);
	ningrp = numingroup(sbuf.st_gid);
	ingroup = ingrp(sbuf.st_uid, sbuf.st_gid);
	
	/*
	 * Print the modes out.
	 */
	printf("\tReadable by: ");
	printmode(nread, READP, sbuf.st_uid, sbuf.st_gid, ningrp, ingroup);

	printf("\tWritable by: ");
	printmode(nwrite, WRITEP, sbuf.st_uid, sbuf.st_gid, ningrp, ingroup);

	printf("\t%sable by: ", (isdirectory(sbuf.st_mode) ? "Search" : "Execut"));
	printmode(nexec, EXECP, sbuf.st_uid, sbuf.st_gid, ningrp, ingroup);
	
	/*
	 * If the file is not a directory, we simply swap the old
	 * permissions (parent directory permissions) for each user.
	 * The same goes for an empty directory, since you needn't 
	 * be able to write the directory itself to remove it.  But,
	 * if the directory has files in it, you must be able to
	 * remove them too, so we AND the old permissions and the
	 * current permissions.
	 */
	if (!isdirectory(sbuf.st_mode))
		swapperm(SET);
	else if (emptydir(file))
		swapperm(SET);
	else
		swapperm(ADD);
		
	/*
	 * Now count everything again using the new permissions
	 * and print it out.
	 */
	printf("\tRemovable by: ");
	ningrp = numingroup(sbuf2.st_gid);
	count(&nread, &nwrite, &nexec);

	if (sbuf.st_gid != sbuf2.st_gid)
		ingroup = -1;
	else
		ingroup = ingrp(sbuf2.st_uid, sbuf2.st_gid);

	printmode(nwrite, WRITEP, sbuf2.st_uid, sbuf2.st_gid, ningrp, ingroup);

	printf("\n");
	fflush(stdout);
}

/*
 * printmode - prints out the number of people (nperm) who have permission
 *	       perm.  The file is owned by uid and gid, there are ningrp
 *	       people in group gid, and ingroup is 1 if uid is in group
 *	       gid.
 */
printmode(nperm, perm, uid, gid, ningrp, ingroup)
int nperm, perm, uid, gid, ningrp, ingroup;
{
	int n;
	
#ifdef DEBUG
	printf("PRINTMODE: nusers = %d\n", nusers);
	printf("PRINTMODE: nperm %d perm %d uid %d gid %d ningrp %d ingroup %d\n", nperm, perm, uid, gid, ningrp, ingroup);
#endif

	/*
	 * Each of these conditions handles one or two
	 * modes.  It's pretty bizarre.
	 */
	if (nperm == 0)
		printf("nobody\n");
	else if (nperm == nusers)
		printf("everybody\n");
	else if (nperm == 1)
		prsingle(perm, 1);
	else if (nperm == (nusers - 1)) {
		printf("everybody except ");
		prsingle(perm, 0);
	}
	else if (nperm == (ningrp + 1)) {
		printf("%s and ", username(uid));
		printf("members of group %s\n", groupname(gid));
	}
	else if ((n = commongroup(gid, perm)) != 0) {
		if (n < (ningrp / 2)) {
			printall(perm, 1);
		}
		else {
			printf("members of group %s", groupname(gid));

			if (n != ningrp) {
				printf(" execpt ");
				prgroup(gid, perm);
			}
		
			printf("\n");
		}
	}
	else if (ingroup == 1) {
		printf("everybody except members of group %s", groupname(gid));

		if (nperm == (nusers - ningrp + 1))
			printf(" excluding %s", username(uid));
			
		printf("\n");
	}
	else if (ingroup == 0) {
		printf("everybody except ");
		
		if (nperm == (nusers - ningrp))
			printf("members of group %s\n", groupname(gid));
		if (nperm == (nusers - ningrp - 1))
			printf("members of group %s and %s\n", groupname(gid), username(uid));
		if (nperm != (nusers - ningrp))
			printall(perm, 0);
	}
	else if (ingroup == -1) {
		printall(perm, 1);
	}
	else {
		/*
		 * Should never get here.  If we do, something
		 * is trashed somewhere.  This will give you
		 * a list of everyone in the password file.
		 */
		printall(perm, 1);
	}
}

/*
 * swapperm - if how is SET, sticks operm into perm.  Otherwise, ands them
 *	      together into perm.
 */
swapperm(how)
int how;
{
	register struct user *p;
	
	if (how == SET) {
		p = uhead;
		while (p != utail) {
			p->perm = p->operm;
			p++;
		}
	}
	else {
		p = uhead;
		while (p != utail) {
			p->perm &= p->operm;
			p++;
		}
	}
}

/*
 * count - count the number of users who have read, write, execute
 *	   permissions.
 */
count(r, w, x)
int *r, *w, *x;
{
	register struct user *p;
	register int nr, nw, nx;
	
	nr = 0;
	nw = 0;
	nx = 0;
	
	p = uhead;
	while (p != utail) {
		if ((p->perm & READP) == READP)
			nr++;
		if ((p->perm & WRITEP) == WRITEP)
			nw++;
		if ((p->perm & EXECP) == EXECP)
			nx++;

		p++;
	}
	
	*r = nr;
	*w = nw;
	*x = nx;
}

/*
 * numingroup - returns the number of RECORDED users in group g.  There
 *		may be other members in the group for whom we don't have
 *		records (because they don't have an account on this
 *		machine).
 */
numingroup(g)
int g;
{
	register int gid;
	register struct grp *p;
	
	gid = g;
	
	p = ghead;
	while (p != gtail) {
		if (p->gid == gid)
			return(p->nmembers);
					
		p++;
	}

	/*
	 * Should never hit this.
	 */
	return(0);
}

/*
 * commongroup - returns number of users who have permission m and are 
 *		 members of group g.  If a user with permission m who
 *		 is NOT a member of group g is found, 0 is returned.
 */
commongroup(g, m)
int g, m;
{
	register int n;
	register int gid, mode;
	register struct user *p;

	n = 0;
	gid = g;
	mode = m;
	
	p = uhead;
	while (p != utail) {
		if ((p->perm & mode) != mode) {
			p++;
			continue;
		}
		
		if (!member(gid, p->groups))
			return(0);
		n++;
		p++;
	}
	
	return(n);
}

/*
 * prsingle - print the name of the single user with permission m.  The o
 *	      argument reverses the sense of the check (0 = off, 1 = on).
 */
prsingle(m, o)
int m, o;
{
	register int off, mode;
	register struct user *p;
	
	off = o;
	mode = m;
	
	p = uhead;
	while (p != utail) {
		if (off == 0) {
			if ((p->perm & mode) == 0) {
				printf("%s\n", p->uname);
				return;
			}
		}
		else {
			if ((p->perm & mode) == mode) {
				printf("%s\n", p->uname);
				return;
			}
		}
		
		p++;
	}
}

/*
 * prgroup - print the names of any members of group g who do NOT have
 *	     permission m.  
 */
prgroup(g, m)
int g, m;
{
	short didone;
	register int gid, mode;
	register struct user *p;
	
	gid = g;
	mode = m;
	didone = 0;
	
	p = uhead;
	while (p != utail) {
		if ((p->perm & mode) == mode) {
			p++;
			continue;
		}
			
		if (!member(gid, p->groups)) {
			p++;
			continue;
		}

		printf("%s%s", (didone++ ? ", " : ""), p->uname);

		p++;
	}
	
	if (!didone)
		printf("???");
}

/*
 * printall - print the name of every user who has permission m if have is
 *	      1, or who doesn't have permission m if have is 0.
 */
printall(m, have)
int m, have;
{
	char ans[8];
	register struct user *p;
	register int mode, didone;
	int nread, nwrite, nexec, nperm;
	
	count(&nread, &nwrite, &nexec);
	
	if (have) {
		if (m == READP)
			nperm = nread;
		else if (m == WRITEP)
			nperm = nwrite;
		else
			nperm = nexec;
	}
	else {
		if (m == READP)
			nperm = nusers - nread;
		else if (m == WRITEP)
			nperm = nusers - nwrite;
		else
			nperm = nusers - nexec;
	}

	/*
	 * If there's greater than THRESHOLD names,
	 * they may not want to see them all, so we
	 * ask them.
	 */
	if (nperm > THRESHOLD) {
		printf("There are %d names in this list.\n", nperm);
		printf("\t\t      Do you really want to see them (y/[n])? ");
		fflush(stdout);

		nread = read(0, ans, sizeof(ans));
		ans[nread-1] = NULL;
		
		if ((*ans != 'y') && (*ans != 'Y'))
			return;
	}
	
	mode = m;
	didone = 0;

	p = uhead;
	while (p != utail) {
		if (have) {
			if ((p->perm & mode) == mode)
				printf("%s%s", (didone++ ? ", " : ""), p->uname);
		}
		else {
			if ((p->perm & mode) == 0)
				printf("%s%s", (didone++ ? ", " : ""), p->uname);
		}
		
		p++;
	}
	
	printf("\n");
}

/*
 * ingrp - different interface to member().
 */
ingrp(u, g)
int u, g;
{
	register int uid;
	register struct user *p;
	
	uid = u;
	
	/*
	 * Find the uid, do a member().
	 */
	p = uhead;
	while (p != utail) {
		if (p->uid == uid)
			return(member(g, p->groups));
		
		p++;
	}
	
	/*
	 * Should never get here.
	 */
	return(0);
}

/*
 * emptydir - returns 1 if directory is empty.
 */
emptydir(dir)
char *dir;
{
	register int n;
	DIR *dirp, *opendir();
	struct direct *d, *readdir();
	
	/*
	 * If we can't read the directory, assume
	 * it's non-empty.
	 */
	if ((dirp = opendir(dir)) == NULL)
		return(0);
	
	n = 0;
	while ((d = readdir(dirp)) != NULL) {
		if (d->d_ino == 0)
			continue;
		
		/*
		 * If we find 3 entries, it cannot
		 * be empty ("." + ".." + any file).
		 */
		if (n++ >= 3) {
			closedir(dirp);
			return(0);
		}
	}
	
	closedir(dirp);
	return(1);
}

/*
 * username - returns the name of the user associated with u.
 */
char *username(u)
int u;
{
	register int uid;
	register struct user *p;
	
	uid = u;
	
	p = uhead;
	while (p != utail) {
		if (p->uid == uid)
			return(p->uname);
		
		p++;
	}
	
	/*
	 * Should never get here.
	 */
	return("???");
}

/*
 * groupname - returns the name of the group associated with g.
 */
char *groupname(g)
int g;
{
	register int gid;
	register struct grp *gr;
	
	gid = g;

	gr = ghead;
	while (gr != gtail) {
		if (gr->gid == gid)
			return(gr->gname);
		
		gr++;
	}
	
	/*
	 * Should never get here.
	 */
	return("???");
}

/*
 * whatis - describe what this "thing" is.
 */
char *whatis(file, mode)
char *file;
int mode;
{
	int i;
	char name[128];
	static char buf[256];

	switch (mode & S_IFMT) {
	case S_IFREG:
		return("file");
		break;
	case S_IFDIR:
		return("directory");
		break;
	case S_IFCHR:
		return("character special file");
		break;
	case S_IFBLK:
		return("block special file");
		break;
	case S_IFSOCK:
		return("socket");
		break;
	case S_IFLNK:
		if ((i = readlink(file, name, sizeof(name))) < 0)
			sprintf(name, "???");
		else
			name[i] = NULL;
		sprintf(buf, "symbolic link to %s", name);
		return(buf);
		break;
	}

	return("???");
}

#ifdef LISTDEBUG /*--------------------------------------------------------*/
/*
 * printlist - prints out the linked list.
 */
printlist()
{
	register int i, didone;
	register struct user *p;
	
	p = uhead;
	while (p != utail) {
		didone = 0;
		printf("%6d%9s: ", p->uid, p->uname);

		printf("perm: %02o ", p->perm & 07);
		printf("groups: ");
		for (i=0; (i < NGROUPS) && (p->groups[i] != NOGROUP); i++)
			printf("%s%d", (didone++ ? ", " : ""), p->groups[i]);
		
		printf("\n");
		p++;
	}
	
	fflush(stdout);
}
#endif /*------------------------------------------------------------------*/
