static char *RCSid = "$Header: /src3/ecn/access/RCS/mkaccessdb.c,v 1.3 85/02/01 13:09:54 root Exp $";

/*
 * mkaccessdb - makes databases for access program
 *
 * This program runs through the password and group files and slurps
 * out the information we need.  It then writes this information out
 * to the files USERFILE and GROUPFILE, where it may be read in by
 * the access program.
 *
 * The purpose for all this is to speed up access' initialization time
 * at the expense of a little bit of disk space.
 *
 * David A. Curry, July 1984
 *
 * $Log:	mkaccessdb.c,v $
 * Revision 1.3  85/02/01  13:09:54  root
 * Rearranged some defines, etc. for easier compilation.
 * 
 * Revision 1.3  85/02/01  13:03:59  root
 * 
 * 
 * Revision 1.2  85/01/12  18:44:59  root
 * Modified so the group "other" hack is selectable with an ifdef.
 * 
 * Revision 1.1  84/07/19  09:08:12  root
 * Initial revision
 * 
 */
#include "defs.h"

/*
 * Redefine these, since we have slightly different
 * structures to use for building the linked list.
 */
#undef USIZE
#undef GSIZE
#undef USNULL
#undef GRNULL
#define GSIZE	sizeof(struct Grp)
#define USIZE	sizeof(struct User)
#define GRNULL	(struct Grp *) NULL
#define USNULL	(struct User *) NULL

long nusers;
char *malloc();
struct Grp *ghead, *gtail;
struct User *uhead, *utail;

main()
{
	struct rlimit rlim;
	
	/*
	 * Push up our limits.
	 */
	rlim.rlim_cur = rlim.rlim_max = RLIM_INFINITY;
	
	setrlimit(RLIMIT_CPU, &rlim);
	setrlimit(RLIMIT_RSS, &rlim);
	setrlimit(RLIMIT_DATA, &rlim);
	setrlimit(RLIMIT_STACK, &rlim);
	
	uhead = utail = USNULL;
	ghead = gtail = GRNULL;
	
	/*
	 * Load the files.
	 */
	loaduids();
	loadgids();

	/*
	 * Write the files.
	 */
	writeuids();
	writegids();
}

/*
 * loaduids - load in the password file
 */
loaduids()
{
	register int nu;
	struct User *uinsert();
	register struct User *q;
	struct passwd *ngetpwent();
	register struct passwd *p;
	
	nu = 0;
	
	/*
	 * Rewind the password file.
	 */
	nsetpwent();
	
	/*
	 * Grab entries until we hit EOF.
	 */
	while ((p = ngetpwent()) != PNULL) {
		/*
		 * Insert the uid into the linked
		 * list.
		 */
		if ((q = uinsert(p->pw_uid)) == USNULL) {
			fprintf(stderr, "mkaccessdb: out of memory\n");
			exit(1);
		}
		
		/*
		 * Permissions and groups are nothing
		 * for now.
		 */
		q->u.perm = 0;
		q->u.operm = 0;
		q->u.groups[0] = NOGROUP;
		strcpy(q->u.uname, p->pw_name);

		nu++;
	}

	/*
	 * Don't need the password file anymore.
	 */
	nendpwent();

	/*
	 * Record number of users.
	 */
	nusers = nu;
}

/*
 * loadgids - reads in thr group file and records who's a member of
 *	      what group.
 */
loadgids()
{
	register int i;
	register int members;
	struct Grp *ginsert();
	register struct Grp *gr;
	struct group *getgrent();
	register struct group *g;

	/*
	 * Rewind the group file.
	 */
	setgrent();
	
	while ((g = getgrent()) != GNULL) {
		members = 0;
		
#ifdef ALLINGROUP
		/*
		 * If there are no members of the group
		 * listed, then everyone is a member of
		 * the group.  This is a special case,
		 * primarily for group "other".
		 */
		if ((g->gr_mem[0] == NULL) && (!strcmp(g->gr_name, "other"))) {
			mark(NULL, g->gr_gid);
			members = nusers;
		}
		else {
#endif ALLINGROUP
			/*
			 * Mark each member listed as being in
			 * the group.
			 */
			for (i=0; g->gr_mem[i] != NULL; i++)
				members += mark(g->gr_mem[i], g->gr_gid);
#ifdef ALLINGROUP
		}
#endif ALLINGROUP
		
		/*
		 * Insert the group into the group list.
		 */
		if ((gr = ginsert(g->gr_gid)) == GRNULL) {
			fprintf(stderr, "mkaccessdb: out of memory\n");
			exit(1);
		}
		
		gr->g.nmembers = members;
		strcpy(gr->g.gname, g->gr_name);
	}
}

/*
 * uinsert - insert a new record for user id u into the linked list.  We
 *	     avoid running through the whole list at all costs.
 */
struct User *uinsert(u)
int u;
{
	register int uid;
	register struct User *p, *q;
	
	uid = u;
	
	/*
	 * If the head is NULL, we're just starting.
	 */
	if (uhead == USNULL) {
		if ((p = (struct User *) malloc(USIZE)) == USNULL)
			return(USNULL);
		
		uhead = p;
		utail = p;
		p->u.uid = uid;
		p->next = USNULL;

		return(p);
	}
	
	/*
	 * If uid < the uid in the head of the list, we
	 * need to insert the new record in front of the
	 * head.
	 */
	if (uid < uhead->u.uid) {
		if ((p = (struct User *) malloc(USIZE)) == USNULL)
			return(USNULL);
		
		p->u.uid = uid;
		p->next = uhead;
		uhead = p;
		
		return(p);
	}
	
	/*
	 * If uid > the uid at the end of the list,
	 * we simply tack uid onto the end.
	 */
	if (uid > utail->u.uid) {
		if ((p = (struct User *) malloc(USIZE)) ==  USNULL)
			return(USNULL);
		
		p->u.uid = uid;
		p->next = USNULL;
		utail->next = p;
		utail = p;
		
		return(p);
	}
	
	/*
	 * Otherwise, we have to run through the list
	 * until we find the appropriate place to stick
	 * the new record.
	 */
	p = q = uhead;
	while (q->u.uid < uid) {
		p = q;
		q = q->next;
	}
	
	if ((q = (struct User *) malloc(USIZE)) == USNULL)
		return(USNULL);
	
	q->u.uid = uid;
	q->next = p->next;
	p->next = q;
	
	return(q);
}

/*
 * ginsert - insert a new record for group id g into the linked list.  We
 *	     avoid running through the whole list at all costs.
 */
struct Grp *ginsert(g)
int g;
{
	register int gid;
	register struct Grp *p, *q;
	
	gid = g;
	
	/*
	 * If the head is NULL, we're just starting.
	 */
	if (ghead == GRNULL) {
		if ((p = (struct Grp *) malloc(GSIZE)) == GRNULL)
			return(GRNULL);
		
		ghead = p;
		gtail = p;
		p->g.gid = gid;
		p->next = GRNULL;

		return(p);
	}
	
	/*
	 * If gid < the gid in the head of the list, we
	 * need to insert the new record in front of the
	 * head.
	 */
	if (gid < ghead->g.gid) {
		if ((p = (struct Grp *) malloc(GSIZE)) == GRNULL)
			return(GRNULL);
		
		p->g.gid = gid;
		p->next = ghead;
		ghead = p;
		
		return(p);
	}
	
	/*
	 * If gid > the gid at the end of the list,
	 * we simply tack gid onto the end.
	 */
	if (gid > gtail->g.gid) {
		if ((p = (struct Grp *) malloc(GSIZE)) ==  GRNULL)
			return(GRNULL);
		
		p->g.gid = gid;
		p->next = GRNULL;
		gtail->next = p;
		gtail = p;
		
		return(p);
	}
	
	/*
	 * Otherwise, we have to run through the list
	 * until we find the appropriate place to stick
	 * the new record.
	 */
	p = q = ghead;
	while (q->g.gid < gid) {
		p = q;
		q = q->next;
	}
	
	if ((q = (struct Grp *) malloc(GSIZE)) == GRNULL)
		return(GRNULL);
	
	q->g.gid = gid;
	q->next = p->next;
	p->next = q;
	
	return(q);
}

/*
 * mark - marks user u as being in group g.  If u is NULL, marks
 *	  all users.  Returns 1 or 0 if a given user was found or not.
 */
mark(u, g)
char *u;
int g;
{
	register int i, gid;
	register char *uname;
	register struct User *p;
	
	uname = u;
	gid = g;
	
	/*
	 * If uname is NULL, mark every user.
	 */
	if (uname == NULL) {
		p = uhead;
		while (p != USNULL) {
			/*
			 * Skip to first unused entry.
			 */
			for (i=0; (i < NGROUPS) && (p->u.groups[i] != NOGROUP); i++);
			
			/*
			 * Mark the group.
			 */
			if (i < NGROUPS) {
				p->u.groups[i] = gid;
				
				if (i < (NGROUPS - 1))
					p->u.groups[i+1] = NOGROUP;
			}

			p = p->next;
		}
		return;
	}
	
	/*
	 * Find the specific uid we want.
	 */
	p = uhead;
	while ((p != USNULL) && strcmp(p->u.uname, uname))
		p = p->next;
	
	/*
	 * If we didn't find it, return 0.
	 */
	if (p == USNULL)
		return(0);
	
	/*
	 * Find unused entry.
	 */
	for (i=0; (i < NGROUPS) && (p->u.groups[i] != NOGROUP); i++);
	
	/*
	 * Mark it.
	 */
	if (i < NGROUPS) {
		p->u.groups[i] = gid;
		
		if (i < (NGROUPS - 1))
			p->u.groups[i+1] = NOGROUP;
	}
	
	return(1);
}

/*
 * writeuids - writes out the user structures to USERFILE.
 */
writeuids()
{
	register FILE *fp;
	register struct User *p;
	
	if ((fp = fopen(USERFILE, "w")) == NULL)
		exit(1);
	
	p = uhead;
	
	while (p != USNULL) {
		fwrite((char *) &(p->u), sizeof(struct user), 1, fp);
		p = p->next;
	}
	
	fclose(fp);
}

/*
 * writegids - writes out the gids to GROUPFILE.
 */
writegids()
{
	register FILE *fp;
	register struct Grp *p;
	
	if ((fp = fopen(GROUPFILE, "w")) == NULL)
		exit(1);
	
	p = ghead;
	
	while (p != GRNULL) {
		fwrite((char *) &(p->g), sizeof(struct grp), 1, fp);
		p = p->next;
	}
	
	fclose(fp);
}

#ifdef PASSWDBYUID /*------------------------------------------------------*/
/*
 * Modified getpwent() which tries to use a password file sorted by user
 * id.  If this fails, it uses the regular password file.  The password
 * file sorted by user id should be used if at all possible, it speeds
 * up the initialization process greatly.
 */
static char PASSWDUID[]		= PASSWDBYUID;
static char PASSWDNAME[]	= "/etc/passwd";

static FILE *pwf = NULL;
static char line[BUFSIZ+1];
static struct passwd passwd;

nsetpwent()
{
	if (pwf == NULL) {
		if ((pwf = fopen(PASSWDUID, "r")) == NULL)
			pwf = fopen(PASSWDNAME, "r");
	}
	else {
		rewind(pwf);
	}
}

nendpwent()
{
	if (pwf != NULL) {
		fclose(pwf);
		pwf = NULL;
	}
}

static char *pwskip(p)
register char *p;
{
	while (*p && *p != ':')
		++p;
	if (*p)
		*p++ = 0;
	return(p);
}

struct passwd *ngetpwent()
{
	register n;
	register char *p;

	if (pwf == NULL) {
		if ((pwf = fopen(PASSWDUID, "r")) == NULL)
			if ((pwf = fopen(PASSWDNAME, "r")) == NULL)
				return(NULL);
	}
	
	p = fgets(line, BUFSIZ, pwf);

	if (p == NULL || *p == '\0')
		return(NULL);

	passwd.pw_name = p;
	p = pwskip(p);
	passwd.pw_passwd = p;
	p = pwskip(p);
	passwd.pw_uid = atoi(p);
	p = pwskip(p);
	passwd.pw_gid = atoi(p);
	p = pwskip(p);
	passwd.pw_gecos = p;
	passwd.pw_comment = p;
	passwd.pw_quota = atoi(p);
	p = pwskip(p);
	passwd.pw_dir = p;
	p = pwskip(p);
	passwd.pw_shell = p;
	while (*p && *p != '\n')
		p++;
	*p = '\0';

	return(&passwd);
}
#endif PASSWDBYUID /*-------------------------------------------------------*/
