/*
 * Symbol tables, and keymap setup.
 * The terminal specific parts of building the
 * keymap has been moved to a better place.
 */
#include	"def.h"

/*
 * Heavily modified routine for the TINY version of MicroGnuEmacs.
 */

/*
 * Defined here so to collect the #ifdef MEYN config stuff in one file
 * If you set either MINDENT or MFILL, then you need to change the bindings
 * in this file to match: KCTRL|'M' -> newline-and-indent and KCTRL|'J' ->
 * insert-newline for MINDENT, and ' ' -> insert-with-wrap for MFILL.
 * MEYN is used for compile-time customization of the system for micros.
 */

int	mode = MFILL | MOVRSTK | MFLOW;

/*
 * Defined by "main.c".
 */
extern	int	ctrlg();		/* Abort out of things		*/
extern	int	quit();			/* Quit				*/

/*
 * Defined by "search.c".
 */
extern	int	forwsearch();		/* Search forward		*/
extern	int	backsearch();		/* Search backwards		*/
extern  int	searchagain();		/* Repeat last search command	*/

/*
 * Defined by "basic.c".
 */
extern	int	gotobol();		/* Move to start of line	*/
extern	int	backchar();		/* Move backward by characters	*/
extern	int	gotoeol();		/* Move to end of line		*/
extern	int	forwchar();		/* Move forward by characters	*/
extern	int	gotobob();		/* Move to start of buffer	*/
extern	int	gotoeob();		/* Move to end of buffer	*/
extern	int	forwline();		/* Move forward by lines	*/
extern	int	backline();		/* Move backward by lines	*/
extern	int	setmark();		/* Set mark			*/
/*
 * Defined by "buffer.c".
 */
extern	int	savebuffers();		/* Save unmodified buffers	*/
extern	int	notmodified();		/* Reset modification flag	*/

#ifdef	DIRLIST
/*
 * Defined by "dirlist.c".
 */
extern	int	dirlist();		/* Directory list.		*/
#endif

/*
 * Defined by "display.c"
 */
extern	int	rotatmode();		/* rotate mode-line help	*/

/*
 * Defined by "file.c".
 */
extern	int	filevisit();		/* Get a file, read write	*/
extern	int	filewrite();		/* Write a file			*/
extern	int	filesave();		/* Save current file		*/
extern	int	fileinsert();		/* Insert file into buffer	*/
#ifdef	BACKUP
extern	int	makebkfile();		/* Control backups on saves	*/
#endif

/*
 * Defined by "random.c".
 */
extern	int	selfinsert();		/* Insert character		*/
extern	int	showcpos();		/* Show the cursor position	*/
extern	int	openline();		/* Open up a blank line		*/
extern	int	newline();		/* Insert CR-LF			*/
extern	int	forwdel();		/* Forward delete		*/
extern	int	backdel();		/* Backward delete in		*/
extern	int	killline();		/* Kill forward			*/
extern	int	bsmapmode();		/* set bsmap mode		*/
extern	int	flowmode();		/* set flow mode		*/
extern	int	fillmode();		/* set word-wrap mode		*/
extern	int	insovrmode();		/* toggle insert overstrike mode */
extern	int	yank();			/* Yank back from killbuffer.	*/

/*
 * Defined by "region.c".
 */
extern	int	killregion();		/* Kill region.			*/
#ifdef	PREFIXREGION
extern	int	prefixregion();		/* Prefix all lines in region	*/
extern	int	setprefix();		/* Set line prefix string	*/
#endif

/*
 * Defined by "window.c".
 */
extern	int	reposition();		/* Reposition window		*/
extern	int	refresh();		/* Refresh the screen		*/

/*
 * Defined by "word.c".
 */
extern	int	delfword();		/* Delete forward word.		*/
extern	int	delbword();		/* Delete backward word.	*/

/*
 * Defined by "extend.c".
 */
extern	int	extend();		/* Extended commands.		*/
extern	int	startover();		/* reread file, restart edit	*/
extern	int	reallyquit();		/* kill emacs, don't save files	*/
extern	int	deleteline();		/* delete line from anywhere	*/
#ifdef	STARTUP
extern	int	evalexpr();		/* Extended commands (again)	*/
extern	int	evalbuffer();		/* Evaluate current buffer	*/
extern	int	evalfile();		/* Evaluate a file		*/
#endif

/*
 * defined by "paragraph.c" - the paragraph justification code.
 */
extern	int	gotobop();		/* Move to start of paragraph.	*/
extern	int	gotoeop();		/* Move to end of paragraph.	*/
extern	int	fillpara();		/* Justify a paragraph.		*/
extern	int	killpara();		/* Delete a paragraph.		*/
extern	int	setfillcol();		/* Set fill column for justify.	*/
extern	int	fillword();		/* Insert char with word wrap.	*/

#ifdef MISLOG
/*
 * defined by "newlog.c" - the mislog code.
 */
extern	int	newlog();		/* newlog function for MISLOG	*/
#endif

typedef	struct	{
	KEY	k_key;			/* Key to bind.			*/
	int	(*k_funcp)();		/* Function.			*/
	char	*k_name;		/* Function name string.	*/
}	KEYTAB;

/*
 * Default key binding table. This contains
 * the function names, the symbol table name, and (possibly)
 * a key binding for the builtin functions. There are no
 * bindings for C-U or C-X. These are done with special
 * code, but should be done normally.
 */
KEYTAB	key[] = {
	KCTRL|'C',	ctrlg,		"keyboard-quit",
	KCTRL|'D',	forwdel,	"delete-char",
	KCTRL|'E',	gotoeol,	"end-of-line",
	KCTRL|'I',	selfinsert,	"self-insert-command",
	KCTRL|'G',	ctrlg,		"keyboard-quit",
	KCTRL|'M',	newline,	"insert-newline",
	KCTRL|'R',	refresh,	"redraw-display",
	KCTRL|'S',	forwsearch,	"search-forward",
	KCTLX|KCTRL|'Q',quit,		"save-buffers-kill-emacs",
	KCTLX|KCTRL|'F',filevisit,	"find-file",
	KMETA|'X',	extend,		"execute-extended-command",
	KCTRL|'B',	backchar,	"backward-char",
	KCTRL|'F',	forwchar,	"forward-char",
	KCTRL|'N',	forwline,	"next-line",
	KCTRL|'P',	backline,	"previous-line",
	KCTLX|'=',	showcpos,	"what-cursor-position",
	KCTLX|'I',	fileinsert,	"insert-file",
	KMETA|'M',	setmark,	"set-mark-command",
	KMETA|'C',	killregion,	"kill-region",
	KMETA|'P',	yank,		"yank",
	' ',		fillword,	"insert-with-wrap",
	-1,		backsearch,	"search-backward",
	-1,		gotobol,	"beginning-of-line",
	-1,		reposition,	"recenter",
	-1,		killline,	"kill-line",
	-1,		openline,	"open-line",
#ifdef	DIRLIST
	-1,		dirlist,	"display-directory",
#endif
	-1,		filesave,	"save-buffer",
	-1,		filewrite,	"write-file",
	-1,		savebuffers,	"save-some-buffers",
	-1,		setfillcol,	"set-fill-column",
	-1,		gotoeob,	"end-of-buffer",
	-1,		gotobob,	"beginning-of-buffer",
	-1,		gotobop,	"backward-paragraph",
	-1,		gotoeop,	"forward-paragraph",
	-1,		fillpara,	"fill-paragraph",
	-1,		notmodified,	"not-modified",
#ifdef	STARTUP
	-1,		evalexpr,	"eval-expression",
	-1,		evalbuffer,	"eval-current-buffer",
	-1,		evalfile,	"load",
#endif
	-1,		bsmapmode,	"bsmap-mode",
	-1,		flowmode,	"flow-mode",
	-1,		fillmode,	"auto-fill-mode",
	-1,		searchagain,	"search-again",
	-1,		killpara,	"kill-paragraph",
#ifdef	PREFIXREGION
	-1,		prefixregion,	"prefix-region",
	-1,		setprefix,	"set-prefix-string",
#endif
#ifdef	BACKUP
	-1,		makebkfile,	"make-backup-files",
#endif
	/* new functions for TinyGnuEmacs */
	-1,		insovrmode,	"toggle-insert-overstrike",
	-1,		startover,	"restart-edit",
	-1,		reallyquit,	"kill-emacs",
	-1,		deleteline,	"delete-line",
	-1,		rotatmode,	"rotate-help",
#ifdef MISLOG
	-1,		newlog,		"newlog",
#endif
};

#define	NKEY	(sizeof(key) / sizeof(key[0]))

/*
 * Just some definitions, to keep ANSI compilers happy.
 */
VOID	keymapinit();
VOID	keyadd();
VOID	keydup();

/*
 * Symbol table lookup.
 * Return a pointer to the SYMBOL node, or NULL if
 * the symbol is not found.
 */
SYMBOL	*
symlookup(cp) register char *cp;
{
	register SYMBOL	*sp;

#ifdef	HASH
	sp = symbol[symhash(cp)];
#else
	sp = symbol[0];
#endif
	while (sp != NULL)
	{
		if (strcmp(cp, sp->s_name) == 0)
			return (sp);
#ifdef	HASH
		if ((sp->s_flags&SFEND) != 0) break;
#endif
		sp = sp->s_symp;
	}
	return (NULL);
}

#ifdef	HASH
/*
 * Take a string, and compute the symbol table
 * bucket number. This is done by adding all of the characters
 * together, and taking the sum mod NSHASH. The string probably
 * should not contain any GR characters; if it does the "*cp"
 * may get a nagative number on some machines, and the "%"
 * will return a negative number!
 */
symhash(cp) register char *cp;
{
	register int	c;
	register int	n;

	n = 0;
	while ((c = *cp++) != 0)
		n += c;
	return (n % NSHASH);
}
#endif

/*
 * Build initial keymap. The funny keys
 * (commands, odd control characters) are mapped using
 * a big table and calls to "keyadd". The printing characters
 * are done with some do-it-yourself handwaving. The terminal
 * specific keymap initialization code is called at the
 * very end to finish up. All errors are fatal.
 */
VOID
keymapinit()
{
	register SYMBOL	*sp;
	register KEYTAB	*kp;
	register int	i;

	for (i=0; i<NKEYS; ++i)
		binding[i] = NULL;
	for (kp = &key[0]; kp < &key[NKEY]; ++kp)
		keyadd(kp->k_key, kp->k_funcp, kp->k_name);
	keydup((KEY) (KCTLX|KCTRL|'C'),	"keyboard-quit");
	keydup((KEY) (KMETA|KCTRL|'C'),	"keyboard-quit");
	keyadd((KEY) (KMETA|0x7F), delbword,
				"backward-kill-word");
	keyadd((KEY) 0x7F, backdel,	"backward-delete-char");
	/*
	 * Should be bound by "tab" already.
	 */
	if ((sp=symlookup("self-insert-command")) == NULL)
		panic("no self-insert-command in keymapinit");
	if (binding[0x20] == NULL)	/* 0x20 == ' ' may already be	*/
		binding[0x20] = sp;	/* bound to insert-with-wrap	*/
	for (i=0x21; i<0x7F; ++i)
	{
		if (binding[i] != NULL)
			panic("nonull binding in keymapinit");
		binding[i] = sp;
	}
	ttykeymapinit();
#ifdef	HASH
	/* Link up the symbol table entries	*/
	for (sp = symbol[i = 0]; i < NSHASH-1; sp = sp->s_symp)
		if (sp->s_symp == NULL) sp->s_symp = symbol[++i];
#endif			
}

/*
 * Create a new builtin function "name"
 * with function "funcp". If the "new" is a real
 * key, bind it as a side effect. All errors
 * are fatal.
 */
VOID
keyadd(new, funcp, name) register KEY new; int (*funcp)(); char *name;
{
	register SYMBOL	*sp;
#ifdef	HASH
	register int	hash;
#endif

	if ((sp=(SYMBOL *)malloc(sizeof(SYMBOL))) == NULL)
		panic("No memory");
#ifdef	HASH
	hash = symhash(name);
	if (symbol[hash] == NULL) sp->s_flags |= SFEND;
	sp->s_symp = symbol[hash];
	symbol[hash] = sp;
#else
	sp->s_symp = symbol[0];
        symbol[0] = sp;
#endif
	sp->s_name = name;
	sp->s_funcp = funcp;
	if (new >= 0)
	{				/* Bind this key.	*/
		if (binding[new] != NULL)
		{
			char buf[80];
			sprintf(buf,"rebinding old symbol: %s", name);
			panic(buf);
		}
		binding[new] = sp;
	}
}

/*
 * Bind key "new" to the existing
 * routine "name". If the name cannot be found,
 * or the key is already bound, abort.
 */
VOID
keydup(new, name) register KEY new; char *name;
{
	register SYMBOL	*sp;

	if (binding[new]!=NULL || (sp=symlookup(name))==NULL)
	{
#ifdef	KEYDUP_ERROR
		fprintf (stderr, "keydup: binding[%d] = %x",
				new, binding[new]);
		fprintf (stderr, " and symlookup(%s) == %x\n", name, sp);
#endif
		panic("keydup");
	}
	binding[new] = sp;
}
