/*****************************************************************************
*
* %W%   %G%   %U%
*
*   NAME:           show_txt
*   DESCRIPTION:    Accell/C-hook for displaying a text field
*
*   ACCELL USAGE:   show_txt(target_variable)
*                     where:
*                        target_variable is the long name of
*                                        a UNIFY database field
*
*   AUTHOR:         Mark Hargrove (based on code in edit_txt)
*   DATE:           8/6/87
*
*   RETURNS:
*			0 - successful termination
*	     -1 - unsuccessful termintation
*
*      the buffer memory for the transfer of data from the database
*      into a temporary file and vice versa is allocated as size 
*      TRANS_BUF_SIZE or less.
*
*   Modification History:
*
******************************************************************************/
#include "fdesc.h"
#include "chookincl.h"
#include "dbtypes.h"
#include "domains.h"
#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#define TRANS_BUF_SIZE 8192L
#define YES   1
#define NO   0

extern int get_fld_num();

int   
show_txt(numargs, acclarg)
   int numargs;
   AVAL acclarg[];
{
   AVAL retval;
   char *field_name;  /* long name of database field */
   int fld;            /* field number */
   long text_size;   /* Actual size of text in DB */
   off_t   file_size;    /* Actual size of edited temp. file */
   long  t_size;      /* Number of bytes to transfer at a whack */
   int   rc;         /* Return code */
   char tname[20];   /* Name of temporary file */
   int fd;            /* File descriptor */
   char *b_ptr;      /* Pointer to transfer buffer */
   long b_size;      /* Size of transfer buffer */
   FIELDLIST flist;   /* Data transfer descriptor block for gfield and pfield */
   FLDESC fdsc;      /* Data base field descriptor */
   struct stat stat_buf; /* File status block */
   time_t   mod_time; /* Last modification time of file */
   char errbuf[256];  /* For output messages */
   char cmdbuf[256];  /* Command line buffer for edit command */
   char *editor;      /* Points to EDIT environment variable */
   int pid;             /* process id of child */


   /* set up for an error return condition */
   retval.aknd = A_INT; 
   retval.dfflg = YES;

   if (numargs != 1)
   {
      retval.aval.inval = -1;
      chookrt(&retval);
      return;
   }
   if (acclarg[0].dfflg == NO)
   {
      retval.aval.inval = -2;
      chookrt(&retval);
      return;
   }

   field_name = acclarg[0].aval.stval;

   if ((fld = get_fld_num(field_name)) < 0)
   {
      retval.aval.inval = -3;
      chookrt(&retval);
      return;
   }

   /* Initialization */
   b_ptr = NULL;
   tname[0] = '\0'; 
   fd = -1;

   /* Check for valid field type */
   if ((rc = fldesc( fld, &fdsc )) != 1)
   {
      sprintf(errbuf,"edit_txt: got bad field=%d",fld);
      prtmsg(0,22,errbuf);
      retval.aval.inval = terminate(fd, b_ptr, tname, -1);
      chookrt(&retval);
      return;
   }

   if ((rc = fdsc.f_typ) != TEXT )
   {
      sprintf(errbuf,"edit_txt: got non-TEXT field=%d",fld);
      prtmsg(0,22,errbuf);
      retval.aval.inval = terminate(fd, b_ptr, tname, -1);
      chookrt(&retval);
      return;
   }

   /* Get size of actual text in database */
   if ((rc = gsize(fld,&text_size)) != 0)
   {
      sprintf(errbuf,"edit_txt: can't retrieve gsize for field=%d",fld);
      prtmsg(0,22,errbuf);
      retval.aval.inval = terminate(fd, b_ptr, tname, -1);
      chookrt(&retval);
      return;
   }
   
   /* Open a temporary file */
   strcpy(tname,"edtdbXXXXXX");
   mktemp(tname);
   if ((fd=open(tname, O_RDWR | O_CREAT | O_TRUNC, 0666 )) < 0)
   {
      sprintf(errbuf,"edit_txt: can't open temp. file=%s",tname);
      prtmsg(0,22,errbuf);
      retval.aval.inval = terminate(-1,b_ptr,"",-1);
      chookrt(&retval);
      return;
   }

   /* Open buffer for transfer of data from database to temp file */
   b_size = (text_size > TRANS_BUF_SIZE) ? TRANS_BUF_SIZE : text_size;
   if ((b_ptr = (char *)malloc((unsigned)b_size)) == NULL)
   {
      sprintf(errbuf,"edit_txt: can't retrieve gsize for field=%d",fld);
      prtmsg(0,22,errbuf);
      retval.aval.inval = terminate(fd, b_ptr, tname, -1);
      chookrt(&retval);
      return;
   }

   /* Transfer data from database to temporary file */
   flist.fnum = fld;
   flist.fdata = b_ptr;
   flist.mode = FL_TRUNC;
   flist.start = (long)0;
   while(text_size > 0)
   {
      t_size = (text_size > b_size) ? b_size : text_size;
      flist.size = t_size;
      if ((rc = gfield(fld,(char *)0,&flist)) != 0)
      {
         sprintf(errbuf,"edit_txt: gfield error=%d",rc); 
         prtmsg(0,22,errbuf);
         retval.aval.inval = terminate(fd, b_ptr, tname, -1);
         chookrt(&retval);
         return;
      }

      if ((rc = write(fd, b_ptr, (unsigned)t_size)) != t_size)
      {
         sprintf(errbuf,"edit_txt: write error on file=%s",tname);  
         prtmsg(0,22,errbuf);
         retval.aval.inval = terminate(fd, b_ptr, tname, -1);
         chookrt(&retval);
         return;
      }
      flist.start += t_size;
      text_size -= t_size;
   }

   /* Free previously allocated buffer */
   free(b_ptr);
   b_ptr = NULL;

   /* Note the file modification time */
   if ((rc = fstat(fd, &stat_buf)) != 0)
   {
      sprintf(errbuf,"edit_txt: can't fstat file=%s",tname);
      prtmsg(0,22,errbuf);
      retval.aval.inval = terminate(fd, b_ptr, tname, -1);
      chookrt(&retval);
      return;
   }
   mod_time = stat_buf.st_mtime;

   /* Check for EDIT environment variable */
   if ((editor = (char *)getenv("EDIT")) == NULL)
   {
      editor = "/usr/bin/vedit";
      sprintf(errbuf,"EDIT environment variable is not set, using vedit editor");
      prtmsg(0,22,errbuf);
   }

   /* Invoke the editor */
   /* setcook();   */
	/*
   strcpy(cmdbuf, editor);
   strcat(cmdbuf, " ");
   strcat(cmdbuf, tname);
   system(cmdbuf);
	*/

   if ((pid = fork()) == 0)
   {
      if (execlp(editor, editor, tname, NULL) == -1)
      {
         sprintf(errbuf, "can't exec %s, %s", editor, tname);
         prtmsg(0,22,errbuf);
         exit(-1);
      }
   }
   else
   {
      wait(NULL);
   }
   setraw();
	retval.aval.inval = terminate(fd, b_ptr, tname, 0);
	chookrt(&retval);
	return;
}
