/*****************************************************************************
 *
 *
 *   NAME:           edit_txt
 *   DESCRIPTION:    Accell/C-hook for editing a text field
 *
 *   ACCELL USAGE:   edit_txt(target_variable)
 *                     where:
 *                        target_variable is the long name of
 *                                        a UNIFY database field
 *
 *   AUTHOR:         Howie Johnson
 *                   ^^^^^^^^^^^^^
 *                   Not the world's prettiest C programmer
 *
 *   DATE:           7/17/87
 * 
 *   NOTES:
 *       0  indicates successful termination - buffer not modified
 *       1  indicates successful termination - buffer modified
 *      -1  unsuccessful termination
 *
 *      the buffer memory for the transfer of data from the database
 *      into a temporary file and vice versa is allocated as size 
 *      TRANS_BUF_SIZE or less.
 *
 *   Modification History:
 *   $Log:	edit_txt.c,v $
 * Revision 1.3  88/01/25  12:54:58  mark
 * .dbv bug workaround code installed by scotth.
 * 
 * Revision 1.2  88/01/20  16:24:03  scotth
 * baseline before code changes to workaround dbv problem
 * 
 *
 *****************************************************************************/

#include "fdesc.h"
#include "chookincl.h"
#include "dbtypes.h"
#include "domains.h"
#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <termio.h>

/* do NOT change the following value unless you know *exactly* what
* you're doing
*/
#define TRANS_BUF_SIZE 32752L	/* 32K-16 buffer size */

#define YES    1
#define NO     0

#ifdef DEBUG
#  define DCLOSE      {if (errlog != NULL) {fclose( errlog); errlog=NULL;}}
#  define DPRINT0(s)      {if (errlog != NULL) fprintf( errlog, s);}
#  define DPRINT1(s,a1)      {if (errlog != NULL) fprintf( errlog, s, a1);}
#  define DPRINT2(s,a1,a2)   {if (errlog != NULL) fprintf( errlog, s, a1, a2);}
#  define DPRINT3(s,a1,a2,a3)   {if (errlog != NULL) fprintf( errlog, s, a1, a2, a3);}
#  define DFLUSH      {if (errlog != NULL) fflush( errlog);}
#else
#  define DCLOSE
#  define DPRINT0(s)
#  define DPRINT1(s,a1)
#  define DPRINT2(s,a1,a2)
#  define DPRINT3(s,a1,a2,a3)
#  define DFLUSH
#endif

static char RCS_ID[]="$Header: edit_txt.c,v 1.3 88/01/25 12:54:58 mark Exp $";
static struct termio term_buf, new_termio;
static long dbv_alloc();
static FILE *errlog;
extern int get_fld_num();
extern int errno;
extern int terminate();

int   
edit_txt(numargs, acclarg)
   int numargs;
   AVAL acclarg[];
{
   AVAL retval;
   char *field_name;		/* long name of database field */
   int fld;			/* field number */
   long text_size;		/* Actual size of text in DB */
   off_t   file_size;		/* Actual size of edited temp. file */
   long  t_size;		/* Number of bytes to transfer at a whack */
   int   rc;			/* Return code */
   char tname[20];		/* Name of temporary file */
   int fd;			/* File descriptor */
   char *b_ptr;			/* Pointer to transfer buffer */
   long b_size;			/* Size of transfer buffer */
   FIELDLIST flist;		/* Data transfer descriptor block for gfield and pfield */
   FLDESC fdsc;			/* Data base field descriptor */
   struct stat stat_buf;	/* File status block */
   time_t   mod_time;		/* Last modification time of file */
   char errbuf[256];		/* For output messages */
   char cmdbuf[256];		/* Command line buffer for edit command */
   char *editor;		/* Points to EDIT environment variable */
   int pid;			/* process id of child */
   int wait_sts;		/* Status returned by wait() */
   char *Debug;			/* MISBUG environment value */
   extern char *getenv();

/*********************************** work around code **********************/
   long orig_size, padded_size;
/***************************************************************************/

   /* set up for an error return condition */
   retval.aknd  = A_INT; 
   retval.dfflg = YES;
   Debug = NULL;
   errlog = (FILE *)NULL;

#ifdef DEBUG
   if ((Debug = getenv("MISBUG")) != NULL)
   {
      if (*Debug == '\0')
      {
         Debug = NULL;
         errlog = (FILE *)NULL;
      }
      else if ((errlog = fopen("ferrlog", "a+")) == NULL)
      {
         fprintf(stderr, "couldn't open the errlog for append\n");
         return -1;
      }
      fprintf(stderr, "starting debug, value='%s'\n", Debug);
   }
   else
   {
      fprintf(stderr, "no debug this time\n");
   }
#endif

   DPRINT0("=============== entering edit_txt ============================\n");
   if (numargs < 1)
   {
      DCLOSE;
      retval.aval.inval = -1;
      chookrt(&retval);
      return;
   }
   if (acclarg[0].dfflg == NO) 
   {
      DCLOSE;
      retval.aval.inval = -2;
      chookrt(&retval);
      return;
   }

   field_name = acclarg[0].aval.stval;

   if ((fld = get_fld_num(field_name)) < 0)
   {
      DCLOSE;
      retval.aval.inval = -3;
      chookrt(&retval);
      return;
   }

   /* Initialization */
   b_ptr = NULL;
   tname[0] = '\0'; 
   fd = -1;

   /* Check for valid field type */
   if ((rc = fldesc( fld, &fdsc )) != 1)
   {
      sprintf(errbuf,"edit_txt: got bad field=%d",fld);
      DPRINT1("edit_txt: got bad field=%d\n",fld);
      prtmsg(0,22,errbuf);
      retval.aval.inval = terminate(fd, b_ptr, tname, -1);
      DCLOSE;
      chookrt(&retval);
      return;
   }

   if ((rc = fdsc.f_typ) != TEXT )
   {
      sprintf(errbuf,"edit_txt: got non-TEXT field=%d",fld);
      DPRINT1("edit_txt: got non-TEXT field=%d",fld);
      prtmsg(0,22,errbuf);
      retval.aval.inval = terminate(fd, b_ptr, tname, -1);
      DCLOSE;
      chookrt(&retval);
      return;
   }

   /* Get size of actual text in database */
   if ((rc = gsize(fld,&text_size)) != 0)
   {
      sprintf(errbuf,"edit_txt: can't retrieve gsize for field=%d",fld);
      DPRINT1("edit_txt: can't retrieve gsize for field=%d",fld);
      prtmsg(0,22,errbuf);
      retval.aval.inval = terminate(fd, b_ptr, tname, -1);
      DCLOSE;
      chookrt(&retval);
      return;
   }

/*********************************** work around code **********************/
   /* remember the size of the original text! */
   orig_size = text_size;
   DPRINT1("edit_txt: original size=%d\n", orig_size);
/***************************************************************************/

   /* Open a temporary file */
   strcpy(tname,"edtdbXXXXXX");
   mktemp(tname);
   if ((fd=open(tname, O_RDWR | O_CREAT | O_TRUNC, 0666 )) < 0)
   {
      sprintf(errbuf,"edit_txt: can't open temp. file=%s",tname);
      DPRINT1("edit_txt: can't open temp. file=%s",tname);
      prtmsg(0,22,errbuf);
      retval.aval.inval = terminate(-1,b_ptr,"",-1);
      DCLOSE;
      chookrt(&retval);
      return;
   }

   /* Open buffer for transfer of data from database to temp file */
   b_size = (text_size > TRANS_BUF_SIZE) ? TRANS_BUF_SIZE : text_size;
   DPRINT1("edit_txt: malloc'ed size = %d\n", b_size);
   if ((b_ptr = (char *)malloc((unsigned)b_size)) == NULL)
   {
      sprintf(errbuf,"edit_txt: can't malloc buffer for data transfer");
      DPRINT0("edit_txt: can't malloc buffer for data transfer");
      prtmsg(0,22,errbuf);
      retval.aval.inval = terminate(fd, b_ptr, tname, -1);
      DCLOSE;
      chookrt(&retval);
      return;
   }

   /* Transfer data from database to temporary file */
   flist.fnum = fld;
   flist.fdata = b_ptr;
   flist.mode = FL_TRUNC;
   flist.start = 0L;
   while(text_size > 0L)
   {
      t_size = (text_size > b_size) ? b_size : text_size;
      flist.size = t_size;
      DPRINT3("edit_txt: flist.fnum=%d, flist.start=%d, flist.size=%d\n",
                         flist.fnum, flist.start, flist.size);
      if ((rc = gfield(fld, (char *)0, &flist)) != 0)
      {
         sprintf(errbuf,"edit_txt: gfield error=%d",rc); 
         DPRINT1("\tedit_txt: gfield error=%d",rc); 
         prtmsg(0,22,errbuf);
         retval.aval.inval = terminate(fd, b_ptr, tname, -1);
         DCLOSE;
         chookrt(&retval);
         return;
      }

      if ((rc = write(fd, b_ptr, (unsigned)t_size)) != t_size)
      {
         sprintf(errbuf,"edit_txt: write error on file=%s",tname);  
         DPRINT1("edit_txt: write error on file=%s",tname);  
         prtmsg(0,22,errbuf);
         retval.aval.inval = terminate(fd, b_ptr, tname, -1);
         DCLOSE;
         chookrt(&retval);
         return;
      }
      flist.start += t_size;
      text_size -= t_size;
   }

   /* Free previously allocated buffer */
   free(b_ptr);
   b_ptr = NULL;

   /* Note the file modification time */
   if ((rc = fstat(fd, &stat_buf)) != 0)
   {
      sprintf(errbuf,"edit_txt: can't fstat file=%s",tname);
      DPRINT1("edit_txt: can't fstat file=%s",tname);
      prtmsg(0,22,errbuf);
      retval.aval.inval = terminate(fd, b_ptr, tname, -1);
      DCLOSE;
      chookrt(&retval);
      return;
   }
   mod_time = stat_buf.st_mtime;

   /* Check for EDIT environment variable */
   if ((editor = (char *)getenv("EDIT")) == NULL)
   {
      editor = "/usr/bin/vedit";
      sprintf(errbuf,"EDIT environment variable is not set, using vedit");
      prtmsg(0,22,errbuf);
   }


   /* save stty settings */
   ioctl(0,TCGETA, &term_buf);

   DPRINT0( "TTY Settings:\n");
   DPRINT1( "c_cflag:   %o\n", term_buf.c_cflag);
   DPRINT1( "c_iflag:   %o\n", term_buf.c_iflag);
   DPRINT1( "c_oflag:   %o\n", term_buf.c_oflag);
   DPRINT1( "c_lflag:   %o\n", term_buf.c_lflag);
   DFLUSH;

   /* Establish new termio values for raw input mode */
   new_termio = term_buf;
   new_termio.c_iflag = IGNBRK;
   new_termio.c_lflag = 0;
   new_termio.c_cc[VMIN]  = 1;
   new_termio.c_cc[VTIME] = 0;

   /* set the new values */
   ioctl(0,TCSETA, &new_termio);

   /* Invoke the editor */
   if ((pid = fork()) == 0)
   {
      if (execlp(editor, editor, tname, NULL) == -1)
      {
         sprintf(errbuf, "can't exec %s, %s", editor, tname);
         DPRINT2( "can't exec %s, %s", editor, tname);
         prtmsg(0,22,errbuf);
         DCLOSE;
         exit(-1);
      }
   }
   else
   {

      DPRINT1( "waiting for pid %d\n", pid);
      DFLUSH;
      while((rc=wait(&wait_sts)) != pid )
      {

         DPRINT3( "! wait returned BAD pid %d, status %d, errno %d\n", rc, wait_sts,errno);
         DFLUSH;
         if( rc == -1 )
         {
            DPRINT1( "! killing process no. %d\n",pid);
            DFLUSH;
            rc=kill(pid,9);
            DPRINT2( "kill return code=%d, errno=%d\n",rc,errno);
            DFLUSH;
            break;
         }
      }

      DPRINT2( "  wait returned OK pid %d, status %d\n", rc, wait_sts);
      DFLUSH;

   }
   /* restore stty settings */
   ioctl( 0, TCSETA, &term_buf);

   DPRINT0("edit_txt: after edit\n");
   /* Write file to database if necessary */
   if ((rc = fstat(fd, &stat_buf)) != 0)
   {
      sprintf(errbuf,"edit_txt: can't fstat file=%s",tname);
      DPRINT1("edit_txt: can't fstat file=%s",tname);
      prtmsg(0,22,errbuf);
      retval.aval.inval = terminate(fd, b_ptr, tname, -1);
      DCLOSE;
      chookrt(&retval);
      return;
   }

   if (stat_buf.st_mtime != mod_time) /* then buffer was modified */
   {
      /* Open buffer for transfer of data from database to temp. file */
      file_size = stat_buf.st_size;

/*********************************** work around code **********************/
      DPRINT1("edit_txt: file_size=%d\n", file_size);
      padded_size = dbv_alloc((long)file_size);
      b_size = (padded_size > TRANS_BUF_SIZE) ? TRANS_BUF_SIZE : padded_size;
      DPRINT2("edit_txt: padded_size=%d, b_size=%d\n", padded_size, b_size);
/***************************************************************************/

      if ((b_ptr = (char *)malloc((unsigned)b_size)) == NULL)
      {
         sprintf(errbuf,"edit_txt: can't retrieve gsize for field=%d",fld);
         DPRINT1("edit_txt: can't retrieve gsize for field=%d",fld);
         prtmsg(0,22,errbuf);
         retval.aval.inval = terminate(fd, b_ptr, tname, -1);
         DCLOSE;
         chookrt(&retval);
         return;
      }
   
      /* Transfer data from database to temp. file */
      flist.fnum = fld;
      flist.fdata = b_ptr;
      flist.mode = FL_TRUNC;
      flist.start = 0L;
      (void) lseek(fd,0L,0);   /* Rewind the temp file */

/*********************** work around code modifications **********************/

      while(file_size > 0)
      {
         t_size = ((file_size > b_size) ? b_size : file_size);
         flist.size = ((padded_size > b_size) ? b_size : padded_size);
         DPRINT3("\tedit_txt: flist.start=%d, flist.size=%d, t_size=%d\n",
                  flist.start, flist.size, t_size);
         (void)memset( b_ptr, '\0', b_size);
         if ((rc=read(fd, b_ptr, (unsigned)t_size)) != t_size)
         {
            DPRINT1("edit_txt: read error, rc=%d\n", rc);
            sprintf(errbuf, "edit_txt: error reading file, rc=%d", rc);
            prtmsg(0,22,errbuf);
            retval.aval.inval = terminate(fd, b_ptr, tname, -1);
            chookrt(&retval);
            return;
         }

         if ((rc=pfield(fld,(char *)0,&flist)) != 0)
         {
            sprintf(errbuf,"edit_txt: pfield error=%d",rc); 
            prtmsg(0,22,errbuf);
            retval.aval.inval = terminate(fd, b_ptr, tname, -1);
            DCLOSE;
            chookrt(&retval);
            return;
         }
         flist.start += t_size;
         file_size   -= t_size;
         padded_size -= t_size;
      }

/************************************************************************/

      retval.aval.inval = terminate(fd, b_ptr, tname, 1);
      chookrt(&retval);
   } /* End if (stat_buf . . . */
   else /* buffer was NOT modified */
   {
      retval.aval.inval = terminate(fd, b_ptr, tname, 0);
      chookrt(&retval);
   }
   DCLOSE;
   return;
}

/* Clean up all open files, buffers, and remove temp. file from directory */
int
terminate(fd, b_ptr, file_name, val)
   int fd;      /* File descriptor */
   char *b_ptr;      /* Buffer area */
   char *file_name;   /* Pointer to file name */
   int  val;      /* Value to return */
{
   if (fd >= 0)
   {
      close(fd);
   }   

   if (b_ptr != NULL)
   {
      free(b_ptr);
   }

   if (*file_name != '\0')
   {
      unlink(file_name);
   }
   return(val);
}

/****************************************************************/

struct allocsize {
   long  boundary;	/* break point for allocation changes	*/
   long  increment;	/* allocation increment below this boundary	*/
};
static struct allocsize btab[] = {
      { 32768L-16L, 32L },
      { (1024L*1024L)-16L, 32768L },
      { (1024L*1024L*1024L)-16L, (1024L*1024L) }
   };
#define BTABSIZ (sizeof(btab) / sizeof(btab[0]))

static long
dbv_alloc(insize)
   long insize;
{
   register int i;
   register long retval;

   for (i=0; i < BTABSIZ; ++i)
   {
      if (insize <= btab[i].boundary)
      {
         break;
      }
   }
   retval =  ((insize / btab[i].increment) + 1) * btab[i].increment - 16L;
   if (retval < insize)
   {
      retval += btab[i].increment;
   }
   return retval;
}
