
/*
 * asm.c
 * Mainline.
 *
 */
#include <stdio.h>
#include "asm.h"
static char *ident="@(#)asmain.c	1.4";

/*
 * This is the mainline.
 * It scans the command line,
 * collects up a source file,
 * sets option flags and calls
 * the assembler proper.
 */
static int cflag = 0;		/* cross reference on flag	*/

main(argc, argv)
char *argv[];
{
	register int i, c;
	register char *p;
	char *file;

	file = NULL;
	for(i=1; i<argc; ++i) {
		p = argv[i];
		if(*p++ == '-') {
			while(c = *p++)
				switch(c) {

				case 'l':
				case 'L':
					++lflag;
					break;
				case 'c':
				case 'C':
					++lflag;
					++cflag;
					break;

				case 'n':
				case 'N':
					++nflag;
					break;

				case 's':
				case 'S':
					++lflag;
					++sflag;
					break;
				default:
					usage();
				}
		}
		else
		{
			file = argv[i];
			if(file == NULL)
				usage();
			assemble(file);
		}
	}
}

/*
 * Assemble a file.
 */

assemble(file)
char *file;
{
	char fn[40];
	int index;

	strncpy(title,file,20);
	for ( index = 0 ; index < HASH ; index ++)
		hashtbl[ index ] = NULL;
	name(fn, file, "asm", 0);
	if((src=fopen(fn, "r")) == NULL) {
		fprintf(stderr, "%s: cannot open\n", fn);
		exit(1);
	}
	if(lflag) {
		name(fn, file, "lst", 1);
		if((lst=fopen(fn, "w")) == NULL) {
			fprintf(stderr, "%s: cannot create\n", fn);
			exit(1);
		}
	}
	if(!nflag) {
		name(fn, file, "hex", 1);
		if((obj=fopen(fn, "w")) == NULL) {
			fprintf(stderr, "%s: cannot create\n", fn);
			exit(1);
		}
		fprintf(obj, "\n");
	}
	page = 1;
	for(pass=0; pass<2; ++pass) {
		skip = 0; 			/* not currently skipping*/
		rewind(src);
		errcnt = 0;
		lineno = 0;
		dot->s_type = S_ABS;
		dot->s_flag = SF_ASG;
		dot->s_value = 0;
		if(pass && lflag )
			top(0);

		while(fgets(sbuf, SRCMAX , src) != NULL) {
			++lineno;
			sptr = sbuf;
			cptr = cbuf;
			eptr = ebuf;
			asmline();
			if(pass) {
				outerrors();
				if(lflag)
					outlisting();
			}
		}
	}
	if(errcnt)
		printf(" Total errors = %5d\n",errcnt);
	if(lflag){
		if(errcnt){
			fprintf(lst,"\n Total number of errors = %5d \n",errcnt)
				;
		}
		table();
		fclose(lst);
	}
	if(!nflag){
		cflush(1);
		fclose(obj);
	}
	printf(" Total bytes assembled = %04X\n",dot->s_value);
	fclose(src);
}

/*
 * If the user screws up, put out
 * a usage message.
 * Then quit.
 * Not much sense staying around.
 */
usage()
{
	fprintf(stderr, "Usage:%4s  [-ln] file file ...\n",TASK);
	fprintf(stderr, "where: \n");
	fprintf(stderr, "      l = make a listing \n");
	fprintf(stderr, " and  n = don't make an object file\n");
	exit(1);
}

/*
 * Build RSX file names.
 * The mode argument is either 0
 * which means default, or 1 which
 * means replace with.
 */
name(fn, file, type, mode)
char *fn, *file, *type;
{
	register char *p1, *p2;
	register int c;

	p1 = fn;
	p2 = file;
	while((c = *p2++) && c!='.')
		*p1++ = c;
	if(mode == 0) {
		if(c == '.') {
			do {
				*p1++ = c;
			} 
			while(c = *p2++);
		} 
		else {
			*p1++ = '.';
			p2 = type;
			while(c = *p2++)
				*p1++ = c;
		}
	} 
	else {
		*p1++ = '.';
		p2 = type;
		while(c = *p2++)
			*p1++ = c;
	}
	*p1 = '\0';
}

/*
 * Output code byte.
 * Save it in the per line
 * buffer for outlisting.
 * Update dot.
 */
codeb(b)
{
	b &= 0377;
	if(cptr < &cbuf[CLMAX])
		*cptr++ = b;
	if(pass && !nflag) {
		if(crec>=CBMAX || cadr+crec!=dot->s_value) {
			cflush(0);
			cadr = dot->s_value;
		}
		crbf[crec++] = b;
	}
	++dot->s_value;
}

/*
 * Output a word.
 * Low then high.
 */
codew(w)
{
	if(BSWAP)codeb(w>>8);
	codeb(w);
	if(!BSWAP)codeb(w>>8);
}

/*
 * Signal error.
 * Add it to the error buffer
 * if not already there.
 */
err(c)
{
	register char *p;

	errcnt+=1;
	p = ebuf;
	if(eptr == &ebuf[ERRMAX-1]) c ='*';
	while(p < eptr)
		if(*p++ == c)
			return;
	*p++ = c;
	if ( p> &ebuf[ERRMAX])
		--p;
	eptr = p;
}

/*
 * Format a line for the
 * listing.
 * More work than you would
 * think.
 * Only called if -l.
 */
outlisting()
{
	register int nbytes;
	register char *cp;
	int w1, w2, w3, w4;

	if(listmode == NLIST)
		return;
	for(cp = eptr; cp < &ebuf[ERRMAX-1]; *cp++ = ' ');
	*cp='\0';
	if(!(--pline ))
		top(1);
	if(listmode == SLIST)
		fprintf(lst, "%.6s      ", ebuf);
	else
		fprintf(lst, "%.6s %04X ", ebuf, listaddr);
	if(listmode == ALIST)
		fprintf(lst, "%9s%4d %s", "", lineno, sbuf);
	else {
		nbytes = cptr-cbuf;
		w1 = cbuf[0]&0377;
		w2 = cbuf[1]&0377;
		w3 = cbuf[2]&0377;
		w4 = cbuf[3]&0377;
		switch(nbytes) {
		case 0:
			fprintf(lst, "%9s", ""); 
			break;
		case 1:
			fprintf(lst, "%02X%7s", w1, ""); 
			break;
		case 2:
			fprintf(lst, "%02X%02X%5s", w1, w2, ""); 
			break;
		case 3:
			fprintf(lst, "%02X%02X%02X%3s", w1, w2, w3,""); 
			break;
		default:
			fprintf(lst, "%02X%02X%02X%02X%1s", w1, w2, w3,w4,"");
		}
		fprintf(lst, "%4d\t%s", lineno, sbuf);
		cp = &cbuf[4];
		while((nbytes -= 4) > 0) {
			if( --pline < 0  )
				top(1);
			listaddr += 4;
			fprintf(lst, "%5s%04X ", "",listaddr );
			switch(nbytes) {
			case 0:	
				break;
			case 1:
				w1 = cp[0]&0377;
				fprintf(lst, "%02X\n", w1);
				break;
			case 2:
				w1 = cp[0]&0377;
				w2 = cp[1]&0377;
				fprintf(lst, "%02X%02X\n", w1, w2);
				break;
			case 3:
				w1 = cp[0]&0377;
				w2 = cp[1]&0377;
				w3 = cp[2]&0377;
				fprintf(lst, "%02X%02X%02X\n", w1, w2, w3);
				break;
			default:
				w1 = cp[0]&0377;
				w2 = cp[1]&0377;
				w3 = cp[2]&0377;
				w4 = cp[3]&0377;
				fprintf(lst, "%02X%02X%02X%02X\n", w1, w2, w3,w4);
			}
			cp += 4;
		}
	}
}

/*
 * Write errors to the tty.
 */
outerrors()
{
	register char *p;
	if( lflag )
		return;

	if( eptr > ebuf ){
		*eptr='\0';
		printf("%s\t%04d\t%s", ebuf, lineno, sbuf);
	}
}

/*
 * Flush the binary code
 * buffer.
 */

cflush(lf)
int lf;
{
	int chksum;		/* checksum for hex file format */
	register int i;

	if(crec == 0)
		return;
	fprintf( obj, ":%02X%04X00",crec,cadr );
	chksum = 0;
	chksum += crec;
	chksum += (( cadr >> 8)& 0xFF);
	chksum += ( cadr & 0xFF);
	for(i=0; i<crec; ++i){
		fprintf( obj,"%02X", crbf[ i ]&0xFF );
		chksum += crbf[ i ]&0xFF;
	}
	fprintf( obj,"%02X\n" , ( 00 -( chksum &0xFF))&0xFF );
	crec = 0;
	if(lf)
		fprintf(obj,":00000000\n");
}
/* 
 * Print out the user symbol table
 */
table()
{
	register struct sym *mine ;
	register struct hu *howu;
	int line,indx,count;

	count =line = 0;
	page = 1;
	fprintf(lst,"\f\n%-40s  Symbol table dump \t\t  Page %3d\n\n\n",
	title,page++) ;
	for( indx = 0 ; indx < HASH ; indx++){
		for(mine = hashtbl[ indx ]; mine != NULL; mine = mine->right ){
			if( line >= mxline ){
				fprintf(lst,
				"\f\n Symbol table dump page %3d\n\n\n"
				    ,page++);
				line = 0;
			}
			fprintf(lst,"%10s = %04X  ",mine->s_name,mine->s_value);
			if( cflag ){
				for( howu = mine->howused; howu != NULL ;
						 howu = howu->next ){
					    if( count >= 10 ){
						fprintf(lst,"\n                   ");
						line++;
						count = 0;
					}
					fprintf(lst,"%6d %c :", howu->line, howu->how );
					count++;
				}
				count = 0;
				line++;
				fprintf(lst,"\n");
			}
			else{ 
				if( count >= 4 ){
					line++;
					count = 0;
					fprintf(lst,"\n");
				}
				count++;
			}
		}
	}
	fprintf(lst,"\f\n");
	fflush(lst);
}

/*
 *
 * Top - Print the label at the top of the page
 *
 */

top(mode)
int mode;
{

	if(mode)putc('\f',lst);
	pline = mxline;
	fprintf(lst,"\n %-40s KSE cross assembler for the %-5s",title,MICRO);
	fprintf(lst,"      page %3d",page++);
	fprintf(lst," \n\n\n");
}

int
hash( id )
char *id;
{
	if( *id >= 'a' && *id <= 'z' )
		return( *id - 'a');
	else if( *id == '?' )
		return( 26 );
	else
		return( 27 );
}
struct sym *
lkup( id , howu )
char *id;
char howu;
{
	struct sym *np;
	struct hu  *usage;

	for( np = hashtbl[hash(id)]; np != NULL; np = np->right ){
		if( strcmp( id, np->s_name ) == 0){
			if( cflag ){
				for( usage = np->howused;
				usage->line != lineno && usage->next != NULL;
				usage = usage->next )
					    ;
				if( usage->line != lineno ){
					usage->next =(struct hu *) 
						malloc( sizeof( *usage));
					if( usage->next != NULL){
						usage->next->line = lineno;
						usage->next->how  =howu;
						usage->next->next = NULL;
					} 
				}
			}
			return( np );
		}
	}
	return(NULL );
}
struct sym *
install( id , howu )
char *id;
char howu;
{
	struct sym *np,*lp;
	struct sym *lkup();
	int hashval;

	if( (np = lkup( id , howu ) ) == NULL ) {
		np= ( struct sym *)malloc( sizeof( *np ));
		if(np == NULL ){
			fprintf(stderr,"in line %d ",lineno);
			fprintf(stderr,"Symbol table overflow\n");
			return( NULL );
		}
		np->right = NULL;
		np->left  = NULL;
		np->s_type= 0;
		np->s_flag= 0;
		np->s_value = 0;
		if( cflag ){
			np->howused = (struct hu *)malloc( sizeof( *np->howused));
			if( np->howused != NULL ){
				np->howused->how = howu;
				np->howused->line = lineno;
				np->howused->next = NULL;
			}
		}
		strcpy(np->s_name,id);
		hashval = hash( id );
		if( hashtbl[ hashval ] == NULL ){
			hashtbl[ hashval ] = np;
		}
		else if( strcmp( id , hashtbl[hashval]->s_name ) > 0 ){
			hashtbl[ hashval ]->left = np;
			np->right = hashtbl[ hashval ];
			hashtbl[ hashval] = np;
		} 
		else
			for( lp = hashtbl[hashval];lp != NULL;lp = lp->right) {
				if( lp->right == NULL ){
					lp->right = np;
					np->left  = lp;
					break;	
				}
				else if ( strcmp( id, lp->right->s_name ) < 0){
					np->right = lp->right;
					lp->right->left = np;
					lp->right = np;
					np->left  = lp;
					break;
				}
			}
	}
	return( np );
}
/*
 * Lookup id.
 * The table is either the pst or
 * the ust.
 */
struct sym *
lookup(id, stp, howu)
char *id;
register struct sym *stp;
char howu;
{
	if(stp == ust)
		return(install( id ,howu ));
	while(stp < pptr) 
		if(strcmp(id, stp->s_name) == 0)
			break;
		else
			++stp;
	if( stp >= pptr )
		return( NULL);
	else
		return( stp );
}
