/*************************************************************************/
/*                                                                       */
/*                               CICA2CSV.CPP                            */
/*                                                                       */
/*       A program to convert CICA index files into                      */
/*       comma seperated value files (CSV) suitable for import           */
/*       into common spreadsheets and relational data base managers.     */
/*                                                                       */
/*       This program and associated source & object files are released  */
/*       to the public domain.  You may do what you want with these      */
/*       files, so long as you do not sell them, or remove the original  */
/*       authors's credits.                                              */
/*                                                                       */
/*                               Allen W. Todd                           */
/*                          2711 N.W. Gibson Hill Road                   */
/*                           Albany, Oregon   97321                      */
/*                               (503) 926-2986                          */
/*                                                                       */
/*                      internet:  todda@jacobs.cs.orst.edu              */
/*                                                                       */
/*                                Version 2.2                            */
/*                               April 10, 1993                          */
/*                                                                       */
/*************************************************************************/

#include <owl.h>
#include <edit.h>
#include <listbox.h>
#include <combobox.h>
#include <bchkbox.h>
#include <checkbox.h>
#include <bradio.h>
#include <radiobut.h>
#include <scrollba.h>
#include <dialog.h>
#include <bwcc.h>
#include "RES_DEFS.H"

#include <stdio.h>
#include <filedial.h>

enum DateFormats {MDY, DMY, YMD};

extern long convert_csv (char *infile_name,
                         char *outfile_name,
                         char *seperator,
                         char *date_delim,
                         char *string_delim,
                         int  field_names,
                         DateFormats DateFormat);

// Define application class derived from  TApplication

class TCICA2CSV : public TApplication
{
public:
  TCICA2CSV(LPSTR AName, HINSTANCE hInstance, HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine, int nCmdShow)
    : TApplication(AName, hInstance, hPrevInstance, lpCmdLine, nCmdShow) {};
    virtual void InitMainWindow();
};

// Declare TMainWindow, a TDialog descendant
class TMainWindow : public TDialog 
{
public:
   TMainWindow(PTWindowsObject AParent, LPSTR ATitle);
   ~TMainWindow();
   void SetupWindow();

   // TMainWindow Control Objects;
   PTEdit EditInputFileTxt;
   PTEdit EditOutputFileTxt;
   PTEdit EditFieldSeperatorTxt;
   PTEdit EditDateDelimiterTxt;
   PTEdit EditTextDelimiterTxt;
   TBCheckBox *PTBCheckFirstLineNamesOpt;
   TRadioButton *PTRadioMDY_Button;
   TRadioButton *PTRadioDMY_Button;
   TRadioButton *PTRadioYMD_Button;

   // TMainWindow Message Response Functions
      virtual void TMainWindowMDY_Button(RTMessage Msg) = [ID_FIRST + MDY_Button];
   virtual void TMainWindowDMY_Button(RTMessage Msg) = [ID_FIRST + DMY_Button];
   virtual void TMainWindowYMD_Button(RTMessage Msg) = [ID_FIRST + YMD_Button];
   virtual void TMainWindowAboutButton(RTMessage Msg) = [ID_FIRST + AboutButton];
   virtual void TMainWindowSelectInputFileButton(RTMessage Msg) = [ID_FIRST + SelectInputFileButton];
   virtual void TMainWindowSelectOutputFileButton(RTMessage Msg) = [ID_FIRST + SelectOutputFileButton];
   virtual void TMainWindowFirstLineNamesOpt(RTMessage Msg) = [ID_FIRST + FirstLineNamesOpt];
   virtual void Ok(RTMessage Msg) = [ID_FIRST + IDOK];
   virtual void Cancel(RTMessage Msg) = [ID_FIRST + IDCANCEL];

   //TMainWindow Variables
   char InputFileName[MAXPATH];
   char OutputFileName[MAXPATH];
   char TextDelimiter[2];
   char DateDelimiter[2];
   char FieldSeperator[2];
   int  WriteFieldNames;
   DateFormats DateFormat;

protected:
   virtual void GetWindowClass(WNDCLASS _FAR & AWndClass);
   virtual LPSTR GetClassName();
   DateFormats GetDefDateFormat();
};


/****************************************************
 * TMainWindow implementations: 
 ****************************************************/

// Define TMainWindow, a TWindow constructor
TMainWindow::TMainWindow(PTWindowsObject AParent, LPSTR ATitle)
                         : TDialog(AParent, ATitle)
{
   EditInputFileTxt = new TEdit(this, InputFileTxt, 80);
   EditOutputFileTxt = new TEdit(this, OutputFileTxt, 80);
   EditFieldSeperatorTxt = new TEdit(this, FieldSeperatorTxt, 80);
   EditDateDelimiterTxt = new TEdit(this, DateDelimiterTxt, 80);
   EditTextDelimiterTxt = new TEdit(this, TextDelimiterTxt, 80);
   PTBCheckFirstLineNamesOpt = new TBCheckBox(this, FirstLineNamesOpt, 0);
   PTRadioMDY_Button = new TRadioButton(this, MDY_Button, 0);
   PTRadioDMY_Button = new TRadioButton(this, DMY_Button, 0);
   PTRadioYMD_Button = new TRadioButton(this, YMD_Button, 0);
   WriteFieldNames = FALSE;
}

// Define TMainWindow destructor
TMainWindow::~TMainWindow()
{
}

// Get Windows Default Date Format
DateFormats TMainWindow::GetDefDateFormat()
{
   char  buffer[80];

   GetProfileString ("Intl", "sShortDate", "M/D/Y", buffer, sizeof (buffer));
   strupr (buffer);
   if ( (buffer[0] == 'M') &&
        (buffer[2] == 'D') &&
        (buffer[4] == 'Y')    ) return (MDY);
   else if ( (buffer[0] == 'D') &&
             (buffer[2] == 'M') &&
             (buffer[4] == 'Y')    ) return (DMY);
   else if ( (buffer[0] == 'Y') &&
             (buffer[3] == 'M') &&
             (buffer[5] == 'D')    ) return (YMD);
   else return (MDY);
}


// TMainWindow setup code
void TMainWindow::SetupWindow()
{
   TDialog::SetupWindow();
   DateFormat = GetDefDateFormat();
   switch (DateFormat)
   {
      case MDY:  PTRadioMDY_Button->Check();
      break;

      case DMY:  PTRadioDMY_Button->Check();
      break;

      case YMD:  PTRadioYMD_Button->Check();
      break;
   }
}

LPSTR TMainWindow::GetClassName()
{
     return "BorDlg_01";
}

void TMainWindow::GetWindowClass(WNDCLASS _FAR & AWndClass)
{

   TDialog::GetWindowClass(AWndClass);
   AWndClass.lpfnWndProc = BWCCDefDlgProc;
   AWndClass.hIcon = LoadIcon( GetApplication()->hInstance,
                                   MAKEINTRESOURCE( cica2csv_icon ) );
}

void TMainWindow::TMainWindowSelectInputFileButton(RTMessage Msg)
{
   switch(Msg.LP.Hi)
   {
      case BN_CLICKED :
      strcpy(InputFileName, "*.*");
      if ( GetApplication()->ExecDialog(new TFileDialog(this,
                                                        SD_FILEOPEN,
                                                        InputFileName)) == IDOK )
         {
         EditInputFileTxt->SetText(InputFileName);
         }
      break;
   }
}

void TMainWindow::TMainWindowSelectOutputFileButton(RTMessage Msg)
{
   switch(Msg.LP.Hi)
   {
      case BN_CLICKED :
      OutputFileName[0] = '\0';
      if ( GetApplication()->ExecDialog(new TFileDialog(this,
                                                        SD_FILESAVE,
                                                        OutputFileName)) == IDOK )
         {
         EditOutputFileTxt->SetText(OutputFileName);
         }
      break;
   }
}

void TMainWindow::TMainWindowFirstLineNamesOpt(RTMessage Msg)
{
   switch(Msg.LP.Hi)
   {
      case BN_CLICKED :
      WriteFieldNames = !WriteFieldNames;
      break;
   }
}

void TMainWindow::TMainWindowMDY_Button(RTMessage Msg)
{
   switch(Msg.LP.Hi)
   {
      case BN_CLICKED : DateFormat = MDY;
      break;
   }
}

void TMainWindow::TMainWindowDMY_Button(RTMessage Msg)
{
   switch(Msg.LP.Hi)
   {
      case BN_CLICKED : DateFormat = DMY;
      break;
   }
}

void TMainWindow::TMainWindowYMD_Button(RTMessage Msg)
{
   switch(Msg.LP.Hi)
   {
      case BN_CLICKED : DateFormat = YMD;
      break;
   }
}


void TMainWindow::Ok(RTMessage Msg)
{
   long       result;
   char       s[80];
   HCURSOR    hcurSave;
   
   switch(Msg.LP.Hi)
   {
      case BN_CLICKED :
      EditInputFileTxt->GetText (InputFileName, sizeof(InputFileName));
      EditOutputFileTxt->GetText (OutputFileName, sizeof(OutputFileName));
      EditTextDelimiterTxt->GetText (TextDelimiter, sizeof(TextDelimiter));
      EditDateDelimiterTxt->GetText (DateDelimiter, sizeof(DateDelimiter));
      EditFieldSeperatorTxt->GetText (FieldSeperator, sizeof(FieldSeperator));
      // do it!
      hcurSave = SetCursor (LoadCursor (NULL, IDC_WAIT));   // put up hourglass
      result = convert_csv (InputFileName,
                            OutputFileName,
                            FieldSeperator,
                            DateDelimiter,
                            TextDelimiter,
                            WriteFieldNames,
                            DateFormat);
      SetCursor (hcurSave); // restore original cursor
      if (result == -1)
         MessageBox(HWindow, "Can NOT Open Input File", "System Error", MB_OK | MB_ICONHAND);
      else if (result == -2)
         MessageBox(HWindow, "Can NOT Create File", "System Error", MB_OK | MB_ICONHAND);
      else
         {
         sprintf (s, "Total Lines Processed = %ld", result);
         MessageBox(HWindow, s, "Conversion Complete", MB_OK | MB_ICONINFORMATION);
         }
//    TDialog::Ok(Msg);
      break;
   }
}

void TMainWindow::Cancel(RTMessage Msg)
{
   switch(Msg.LP.Hi)
   {
      case BN_CLICKED :
      TDialog::Cancel(Msg);
      break;
   }
}

_CLASSDEF(TAbout_DlgDlg)

// Declare TAbout_DlgDlg, a TDialog descendant
class TAbout_DlgDlg : public TDialog 
{
public:

   TAbout_DlgDlg(PTWindowsObject AParent, LPSTR AName);
   ~TAbout_DlgDlg();
   virtual void SetupWindow();
   virtual void Ok(RTMessage Msg) = [ID_FIRST + IDOK];

};

// Place all Window or field interaction here, not in the constructor.
void TAbout_DlgDlg::SetupWindow()
{
   TDialog::SetupWindow();
}

// Define TAbout_DlgDlg, a TDialog constructor
TAbout_DlgDlg::TAbout_DlgDlg(PTWindowsObject AParent, LPSTR AName)
               :TDialog(AParent, AName)
{
}

// Define TAbout_DlgDlg destructor
TAbout_DlgDlg::~TAbout_DlgDlg()
{
}

void TAbout_DlgDlg::Ok(RTMessage Msg)
{
   switch(Msg.LP.Hi)
   {
      case BN_CLICKED :
         TDialog::Ok(Msg);
      break;
   }
}

void TMainWindow::TMainWindowAboutButton(RTMessage Msg)
{
   switch(Msg.LP.Hi)
   {
      case BN_CLICKED :
      // Execute modal dialog
      if(GetModule()->ExecDialog(
                  new TAbout_DlgDlg(this, "About_Dlg")) == IDOK)
      {
      }
      break;
   }
}


/***************************************************
 * TCICA2CSVApp method implementations:
 ***************************************************/

// Construct the TCICA2CSV's MainWindow of type TMainWindow
void TCICA2CSV::InitMainWindow()
{
   MainWindow = new TMainWindow(NULL, "Main_Dlg");
}

// Main program
int PASCAL WinMain(HINSTANCE hInstance,
                   HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine,
                   int nCmdShow)
{
   HINSTANCE hBorLibrary;

   hBorLibrary = LoadLibrary("bwcc.dll");

   if((UINT)hBorLibrary <= 32)
      MessageBox(NULL, "Unable to load Borland Controls", "System Error", MB_OK | MB_ICONHAND);

   TCICA2CSV CICA2CSV ("cica2csv", hInstance, hPrevInstance,
      lpCmdLine, nCmdShow);

   CICA2CSV.Run();

   if((UINT)hBorLibrary > 32)
      FreeLibrary(hBorLibrary);

   return CICA2CSV.Status;
}
