#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if defined(MSDOS)
#define DO_FOPEN 1                                             /* use fopen */
#define ANSI_DECL           /* use ANSI C function declarations/definitions */
#define FOPEN_R_ARG "rb"
#define FOPEN_W_ARG "wb"
#define EOL_STR "\r\n"                               /* end of line marking */
#define SYSTEM_ID "MSDOS"

#elif defined(VMS)
#include <file.h>                                   /* O_... flags for open */
#include <unixio.h>                                       /* open prototype */
#define DO_FOPEN 0                                     /* use open & fdopen */
#define ANSI_DECL           /* use ANSI C function declarations/definitions */
#define OPEN_R_ARG  O_RDONLY, 0
#define OPEN_WN_ARG  O_WRONLY | O_TRUNC | O_CREAT, 0, "rat = cr","rfm = var"
#define OPEN_WD_ARG  O_WRONLY | O_TRUNC | O_CREAT, 0, "rfm = var"
#define FOPEN_R_ARG "r"
#define FOPEN_W_ARG "w"
#define EOL_STR "\n"                                 /* end of line marking */
#define SYSTEM_ID "VMS"

#else                 /* don't use ANSI C function declarations/definitions */
#include <fcntl.h>                                  /* O_... flags for open */
#include <sys/types.h>
#include <sys/stat.h>                               /* S_... flags for open */
#define DO_FOPEN 0                                     /* use open & fdopen */
#if !defined(O_RAW)
#define O_RAW  0
#endif
#define OPEN_R_ARG  O_RAW | O_RDONLY, 0
#define OPEN_WN_ARG  O_WRONLY | O_TRUNC | O_CREAT, S_IREAD | S_IWRITE
#define OPEN_WD_ARG  O_RAW | O_WRONLY | O_TRUNC | O_CREAT, S_IREAD | S_IWRITE
#define FOPEN_R_ARG "r"
#define FOPEN_W_ARG "w"
#define EOL_STR "\n"                                 /* end of line marking */
#define SYSTEM_ID "UNIX"

#endif

/*----------------------- revision history ---------------------------------
 * 26-10-92 (JAD): Pre/post encoding text used \n instead of \r\n for MSDOS.
 * 08-04-92 (JAD): VMS specific help text restricted to VMS.
 *          Documented NCDC.
 * 01-05-91 (AM): added smaller buffer for VMS; added /s option:
 *          Stream_LF decoding for VMS.
 * 10-04-91 (JAD): added osize = 0 in decode; added get_start()
 *          because continuation files received from SIMTEL don't
 *          start with begin. When decodeing was expecting multiple
 *          of four in last line, now computes # actually expected
 *          (when encoding still writing multiple of four.
 *  9-04-91 (JAD): changed VMS OPEN_W_ARG, added pre-ANSI C function
 *          headers for UNIX.
 *  8-04-91 (JAD): Introduced MSDOS/VMS/UNIX versions.
 *  4-04-91 (JAD): Bug fix: table was not kept when found at top of
 *          encoded file.
 *--------------------------------------------------------------------------*/

/*=========================== global help ==================================*/
char  sProgram[] = "NCDC";
char  sVersion[] = "version 1.51";
char  sAuthor[] = "J.A. Doornik.";

static char  *sUsage[]  = /* array of strings instead of one string for VAX */
{
#if defined(VMS)
    "\nUsage:  NCDC infile [outfile] [/u] [/s] [/##]\n",
#else
    "\nUsage:  NCDC infile [outfile] [/u] [/##]\n",
#endif
    "Decodes if infile has no extension or extension ENC, XXE or UUE;",
    "Encodes otherwise.\n",
    "Encoding: uses XXencoding or UUencoding if /u is specified",
    "\t  /## sets maximum outfile size to ## kilobytes",
    "\t  Eg NCDC test.zip /50 could lead to",
    "\t  test.enc, test.en2 and test.en3 being created.",
    "Decoding: decoding method is derived from first encoded line.\n",
#if defined(VMS)
    "Decoded file defaults in Variable length format.",
    "\t  /s  : decoded file in Stream_LF format.\n",
#endif
    "Example : ncdc test.         XXencode file test into test.enc",
    "          ncdc test          XXdecode file test.enc into test",
    NULL
};

#define  FALSE 0
#define  TRUE  !FALSE
#define  byte  unsigned char

/*---------------- files and  internal buffer for in/output ----------------*/
#define CONFIRM     0
#define NO_CONFIRM  1
#define IN_BUFLEN 20480
#if defined(VMS)
#define OUT_BUFLEN 512                                         /* 0.5K buffer */
#else
#define OUT_BUFLEN 20480                                       /* 20K buffer */
#endif
byte    Stream_LF = FALSE;        /* indicates Stream_LF decoded file on VAX */
char    sInbuf[IN_BUFLEN], sOutbuf[OUT_BUFLEN];
char    sOutfile[80] = "", sInfile[80] = "";
FILE   *FILEin, *FILEout;

/*-------------------- UUencode information --------------------------------*/
char    sUUtable[]                                       /* character table */
     = "`!\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_";
int     iUUlen = 'M';

/*-------------------- XXencode information --------------------------------*/
char    sXXtable[]                                       /* character table */
     = "+-0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
int     iXXlen = 'h';

/*---------------------- coding information --------------------------------*/
#define ERROR    0
#define WARNING  1
#define MESSAGE  2
#define NCODE    0
#define DCODE    1

#define BIT6MASK 0x3F

#define INLEN  45   /* # chars in line to be encoded, must be multiple of 3 */
#define OUTLEN 60           /* INLEN original bytes is OUTLEN encoded bytes */

#define UNDEF_CH -2               /* value if ch undefined in character map */

char    sMethod[] = "XXencode";
int     aiNNmap[257];                                   /* holds active map */
int    *piNNmap = aiNNmap + 1;             /* now piNNmap[-1] == aiNNmap[0] */
                                           /* assumed is that EOF equals -1 */
int     iNNlen;                    /* # bytes in a full length encoded line */
char    sDefault_ext[] = ".enc";                       /* default extension */
int     cLine;                          /* current lineNumber in input file */
int     chFirst = 0; /* first decoding char found in get_ouffname/get_start */
                                               /* this is to avoid ungetc() */
#define MAXLINELEN 80
byte abLine[MAXLINELEN + 1];      /* input line, must be unsigned for <<,>> */


/*--------------------------- message --------------------------------------*/
#if defined(MSDOS) | defined(VMS)
#include <stdarg.h>

void message(int level, int line, char *msg, ...)
{
    va_list args;

    printf("%s ", sProgram);
    if (line) printf("%s (%d) ", sInfile, line);

    if (level == ERROR)         printf("error: ");
    else if (level == WARNING)  printf("warning: ");

    va_start(args, msg);                   /* args points to first argument */
    vfprintf(stdout, msg, args);    va_end(args);
    printf(".\n");

    if (level == ERROR)  exit(0);
}
#else
#include <varargs.h>

message(level, line, va_alist)  int level, line; va_dcl
{
    va_list args;  char *msg;

    printf("%s ", sProgram);
    if (line) printf("%s (%d) ", sInfile, line);

    if (level == ERROR)         printf("error: ");
    else if (level == WARNING)  printf("warning: ");

    va_start(args);                         /* args points to format string */
    msg = va_arg(args, char *);            /* args points to first argument */
    vfprintf(stdout, msg, args);    va_end(args);
    printf(".\n");

    if (level == ERROR)  exit(0);
}
#endif
/*--------------------------- END message ----------------------------------*/


/*------------------------- fbase/fext -------------------------------------
 * fname = path + base + extension (incl. dot)
 * fbase returns pointer to start of base.
 * fext returns pointer to start of extension, or end of file name.
 *--------------------------------------------------------------------------*/
#if defined(ANSI_DECL)
  char *fbase(char *fname)
#else
  char *fbase(fname)  char *fname;
#endif
{
    char *s;

    if ( (s = strrchr(fname, '\\')) != NULL)
        s++;                      /* points to first character after last \ */
    else if ( (s = strrchr(fname, ':')) != NULL)
        s++;                      /* points to first character after last : */
    else
        s = fname;                                 /* whole name, no : or \ */
return(s);
}

#if defined(ANSI_DECL)
  char *fext(char *fname)
#else
  char *fext(fname)  char *fname;
#endif
{
    char *s;

    if ( (s = strrchr(fname, '.')) == NULL)
        s = fname + strlen(fname);            /* points to extension or end */
return(s);
}
/*------------------------- END fbase/fext ---------------------------------*/


/*------------------------------ fopen_r/fopen_w ---------------------------
 * does low level or high level opening, according to value of DO_FOPEN
 * MSC: fopen "wb" is at least 5 times faster than "w"
 *--------------------------------------------------------------------------*/
#if defined(ANSI_DECL)
  FILE *fopen_r(char *sfi, char *ibuf)
#else
  FILE *fopen_r(sfi, ibuf)  char *sfi, *ibuf;
#endif
{
    FILE *ifd = NULL;  int ifh;

#if DO_FOPEN
    ifd = fopen(sfi, FOPEN_R_ARG);
#else
    if ( (ifh = open(sfi, OPEN_R_ARG)) != -1)
        ifd = fdopen(ifh, FOPEN_R_ARG);
#endif

    if (ifd == NULL)   message(ERROR, 0, "cannot open input file %s", sfi);
    if (ibuf != NULL)  setvbuf(ifd, ibuf, _IOFBF, IN_BUFLEN);

return(ifd);
}

#if defined(ANSI_DECL)
  FILE *fopen_w(char *sfo, char *obuf, int confirm, int ncdc)
#else
  FILE *fopen_w(sfo, obuf, confirm, ncdc)  char *sfo, *obuf; int confirm, ncdc;
#endif
{
    FILE *ofd = NULL;  int  yn, ofh;

    if (confirm == CONFIRM && (ofd = fopen(sfo, "r")) != NULL)
    {
        fprintf(stderr, "%s already exists", sfo);

#if !defined(VMS)                        /* VMS just makes a higher version */
        fprintf(stderr, ", overwrite (Y/N)? ");
        do
        {   yn = getchar();  yn = toupper(yn);
        }  while (yn != 'Y' && yn != 'N');
        if (yn == 'N') message(ERROR, 0, "%s not overwritten", sfo);
#else
        fprintf(stderr, ".\n");
#endif

        fclose(ofd);
    }

#if DO_FOPEN
    ofd = fopen(sfo, FOPEN_W_ARG);
#else
    if ( ncdc == DCODE && (Stream_LF == TRUE ) )
        ofd = fopen(sfo, FOPEN_W_ARG);
    else if ( (ncdc == NCODE && (ofh = open(sfo, OPEN_WN_ARG)) != -1)
           || (ncdc == DCODE && (ofh = open(sfo, OPEN_WD_ARG)) != -1) )
             ofd = fdopen(ofh, FOPEN_W_ARG);
#endif

    if (ofd == NULL)   message(ERROR, 0, "cannot open output file %s", sfo);
    if (obuf != NULL)  setvbuf(ofd, obuf, _IOFBF, OUT_BUFLEN);

return(ofd);
}
/*-------------------------- END fopen_r/fopen_w ---------------------------*/


/*--------------------------- get_encmethod --------------------------------
 * Determines the encoding method, and sets up the decoding info.
 * Assumes that infile is at the start of the encoded text.
 *--------------------------------------------------------------------------*/
#if defined(ANSI_DECL)
  void get_encmethod(FILE *infile, int notable)
#else
  get_encmethod(infile, notable)  FILE *infile; int notable;
#endif
{
    int i, ch;

    chFirst = getc(infile);                /* peek ahead to get line length */

    if (notable)                               /* determine encoding method */
    {   if (chFirst == iUUlen)
        {   sMethod[0] = sMethod[1] = 'U';
            for (i = 0; i < 64; i++)
                sXXtable[i] = sUUtable[i];        /* replace XXmap by UUmap */
            iXXlen = iUUlen;
        }
        else if (chFirst != iXXlen)
            message(ERROR, 0, "unknown encoding method");
    }
    else
    {   sMethod[0] = sMethod[1] = 'U';       /* table -> call it UUencode */
        iXXlen = chFirst;
    }

    for (i = 0; i <= 255; i++)                       /* setup character map */
        piNNmap[i] = UNDEF_CH;
    for (i = 0; i < 64; i++)
    {   ch = sXXtable[i];
        if (piNNmap[ch] != UNDEF_CH)
            message(ERROR, 0, "character %c already defined in table", ch);
        else
           piNNmap[ch] = i;
    }
    if (EOF != -1)  message(ERROR, 0, "EOF is not -1");
    piNNmap[EOF] = EOF;

    iNNlen = piNNmap[iXXlen];           /* replace encoded value by # bytes */
}
/*--------------------------- END get_encmethod ----------------------------*/


/*----------------------- get_outfname/get_start ---------------------------
 * get_outfname gets the original name of the encoded file from the input
 *   file. If a table is found, this is stored in sXXtable.
 *   Determines encoding method.
 * get_start locates the start in the next file
 *--------------------------------------------------------------------------*/
#if defined(ANSI_DECL)
  void get_outfname(FILE *infile, char *infname, char *outfname)
#else
  get_outfname(infile,infname,outfname) FILE *infile;char *infname, *outfname;
#endif
{
    int  i, notable = TRUE;

    cLine = 1;                         /* current line number in input file */

    while (TRUE)
    {   if (fgets(abLine, MAXLINELEN, infile) == NULL)
            break;                                /* EOF: break out of loop */
        cLine++;

        if (strncmp(abLine, "begin", 5) == 0)           /* yes: begin found */
        {   sscanf(abLine + 6, " %*s %s ", outfname);
            get_encmethod(infile, notable);
            return;
        }
        else if (strncmp(abLine, "table", 5) == 0)
        {   notable = FALSE;
            fgets(abLine, MAXLINELEN, infile);           /* 1st half of map */
            fgets(abLine + 32, MAXLINELEN - 32, infile); /* 2nd half of map */
            for (i = 0; i < 64; i++)
                sXXtable[i] = abLine[i];        /* replace XXmap by new map */
        }
        else if (strncmp(abLine, "charmap", 7) == 0)
        {   notable = FALSE;
            for (i = 0; i < 64; i++)
                sXXtable[i] = abLine[i + 8];    /* replace XXmap by new map */
        }
    }
    message(ERROR, 0, "nothing to decode in file %s", infname);
}

#if defined(ANSI_DECL)
  FILE *get_start(char *infname)
#else
  FILE *get_start(infname)  char *infname;
#endif
{
    int  ch, nobegin = TRUE;  FILE *infile;

    cLine = 1;                         /* current line number in input file */
    chFirst = 0;
    infile = fopen_r(infname, sInbuf);

    if ( (ch = getc(infile)) == iXXlen)   nobegin = FALSE;
    while (nobegin)
    {   abLine[0] = ch;
        if (ch != '\n' && fgets(abLine + 1, MAXLINELEN, infile) == NULL)
            break;                                /* EOF: break out of loop */
        cLine++;
        if (strncmp(abLine, "begin", 5) == 0)
            nobegin = FALSE;
        else if ( (ch = getc(infile)) == iXXlen)
            nobegin = FALSE;
    }
    if (nobegin)
        message(ERROR, 0, "nothing to decode in file %s", infname);
    else if (ch == iXXlen)
    {   chFirst = ch;
        message(WARNING, cLine,
            "line starting with \'%c\' found before \"begin\"", chFirst);
    }
return(infile);
}
/*----------------------- END get_outfname/get_start -----------------------*/


/*------------------------- encode -----------------------------------------*/
#if defined(ANSI_DECL)
  int  encode_file(long *fsize, int mxlines)
#else
  int  encode_file(fsize, mxlines)  long *fsize; int mxlines;
#endif
{
    byte *buf = abLine + 1; /* 1st char of abLine is quant., rest is buffer */
    int  i, j, len, chbytes3;
    int  nbytes3;       /* # decoded bytes (so 4 * nbytes / 3 when encoded) */

    chbytes3 = sXXtable[INLEN];
    do
    {                                /* get INLEN bytes from the input file */
        nbytes3 = fread(buf, sizeof(char), INLEN, FILEin);
        *fsize += nbytes3;

        if (nbytes3 == INLEN)
        {   abLine[0] = chbytes3;                    /* store quantity char */
            len = OUTLEN;
        }
        else
        {   abLine[0] = sXXtable[nbytes3];           /* store quantity char */
            for (i = nbytes3; i < INLEN; i++)  buf[i] = 0;
            nbytes3 = (nbytes3 + 2) / 3;             /* take ceiling of div */
            len = 4 * nbytes3;                        /* # of encoded bytes */
            nbytes3 *= 3;         /* translate from 3-byte packets to bytes */
        }

        for (i = nbytes3 - 3, j = len - 4; i >= 0; i -= 3, j -= 4)/* encode */
        {
            buf[j + 3] =
                sXXtable[buf[i + 2] & BIT6MASK];
            buf[j + 2] =
                sXXtable[((buf[i+1] << 2) + (buf[i+2] >> 6)) & BIT6MASK];
            buf[j + 1] =
                sXXtable[((buf[i] << 4)   + (buf[i+1] >> 4)) & BIT6MASK];
            buf[j]     =
                sXXtable[(buf[i] >> 2) & BIT6MASK];
        }
#if defined(MSDOS)
        buf[len++] = '\r';         /* file opened binary --> \r\n for MSDOS */
#endif
        buf[len++] = '\n';                     /* add end of line character */

        if (fwrite(abLine, len + 1, 1, FILEout) != 1)
           message(ERROR, cLine, "write error on output file");
        cLine++;                                   /* increment linecounter */

    } while (nbytes3 > 0 && cLine != mxlines);

return(nbytes3);
}

#if defined(ANSI_DECL)
  void encode(int mxlines)
#else
  encode(mxlines)  int mxlines;
#endif
{
    int  i, fnum = 1;  char *outext;  long  fsize = 0;

    printf("\n%s --- %s ---> %s\n", sInfile, sMethod, sOutfile);

    fprintf(FILEout, "%s %s (%s)%s", sMethod, sVersion, SYSTEM_ID, EOL_STR);
    fprintf(FILEout, "begin 644 %s%s", fbase(sInfile), EOL_STR);
    cLine = 3;                         /* current line number in input file */
    outext = fext(sOutfile);

    while (encode_file(&fsize, mxlines) > 0)
    {
        fclose(FILEout);
        outext[3] = ++fnum % 10 + '0';  /* create new : increment extension */
        if (fnum / 10 > 0)  outext[2] = fnum / 10 + '0';

        FILEout = fopen_w(sOutfile, sOutbuf, NO_CONFIRM, NCODE);

        printf("%*s --- %s ---> %s\n", strlen(sInfile), " ", sMethod, sOutfile);
        fprintf(FILEout, "part %d%s", fnum, EOL_STR);
        fprintf(FILEout, "begin 644 %s%s", fbase(sInfile), EOL_STR);
        cLine = 3;
    }
    fprintf(FILEout,"end%s%ld bytes%s", EOL_STR, fsize, EOL_STR);
    if (fnum > 0)  fprintf(FILEout,"encoded into %d file(s)%s", fnum, EOL_STR);
}
/*--------------------------- END encode -----------------------------------*/


/*------------------------- decode -----------------------------------------
 * decode_file skips undefined characters until the first normal character
 *    is found in the input file. The line counter is incremented and the
 *    line to be decoded is read into abLine.
 *    Returns the # bytes (# after decoding, so 4 * nbytes / 3 when encoded)
 *    in the line or EOF for eof (nothing more to decode).
 *    A line could end in \r\n (MSDOS), \n (VMS/UNIX?), or _EOF (unexpected
 *    end-of-file).
 *    Emailing to Southampton led to ' '\r\n being the end of line,
 *    reason why undefined chars are skipped.
 * then it replaces the 4 byte packets (6-bits per byte) in abLine into 3 byte
 *    packets (8-bits per byte), and writes it to the output file.
 *--------------------------------------------------------------------------*/
#if defined(ANSI_DECL)
  int  recover(int i, int nbytes3, int len)
#else
  int  recover(i, nbytes3, len)  int i, nbytes3, len;
#endif
{
    int  j, ch;
                                                           /* line to short */
    do
    {   if (i >= len)  return(sXXtable[0]);/*no problem: at end of decoding */
        if (piNNmap[abLine[i]] == EOF)
        {   for ( ; i < len; i++)  abLine[i] = sXXtable[0];
            message(WARNING, cLine, "unexpected end of file");
            return(sXXtable[0]);                        /* return null char */
        }
        if (abLine[i] == '\n')
        {
            cLine++;                               /* increment linecounter */
            if (nbytes3 == iNNlen)                 /* no: glue broken lines */
                message(WARNING, cLine - 1, "unexpected line break - glued");
            else if (nbytes3 < iNNlen)
            {   message(WARNING, cLine - 1, "unexpected end of line");
                for ( ; i < len; i++)  abLine[i] = sXXtable[0];
                return(sXXtable[0]);
            }
            else
                message(WARNING, cLine - 1, "file probably corrupted");
        }
        else if (abLine[i] >= ' ')
            message(WARNING, cLine, "skipped illegal character \'%c\'", abLine[i]);
        else
            message(WARNING, cLine, "skipped illegal character %#x", abLine[i]);

                  /* now remove offending character, and add new one at end */
        for (j = i ; j < len - 1; j++)  abLine[j] = abLine[j + 1];
        if ( (ch = getc(FILEin)) == EOF)  ch = sXXtable[0];
        abLine[j] = ch;

    } while (piNNmap[abLine[i]] < 0);

return(piNNmap[abLine[i]]);
}

#if defined(ANSI_DECL)
  int  decode_file(long *fsize)
#else
  int  decode_file(fsize)  long *fsize;
#endif
{
    int  i, j, len, ch;  int  k1, k2, k3, k4;
    int  nbytes3;       /* # decoded bytes (so 4 * nbytes / 3 when encoded) */

    if (chFirst == 0)  chFirst = getc(FILEin);
    nbytes3 = piNNmap[chFirst];            /* read first quantity character */
    if (nbytes3 == EOF)  return(EOF);

    do
    {   len = (nbytes3 * 4) / 3 + (nbytes3 % 3 > 0);  /* # of encoded bytes */

        if (fread(abLine, len, 1, FILEin) < 1)
            message(WARNING, cLine, "unexpected end of file");

        for (i = j = 0; i < nbytes3; i += 3, j += 4)     /* decode the line */
        {
            if ( (k1 = piNNmap[abLine[j]]) < 0)
                k1 = recover(j, nbytes3, len);
            if ( (k2 = piNNmap[abLine[j + 1]]) < 0)
                k2 = recover(j + 1, nbytes3, len);
            if ( (k3 = piNNmap[abLine[j + 2]]) < 0)
                k3 = recover(j + 2, nbytes3, len);
            if ( (k4 = piNNmap[abLine[j + 3]]) < 0)
                k4 = recover(j + 3, nbytes3, len);
            abLine[i]     = (k1 << 2) + (k2 >> 4);
            abLine[i + 1] = (k2 << 4) + (k3 >> 2);
            abLine[i + 2] = (k3 << 6) +  k4;
        }
                                                            /* and write it */
        if (fwrite(abLine, nbytes3, 1, FILEout) != 1)
            message(ERROR, cLine, "write error on output file");
        *fsize += nbytes3;
                               /* skip remaining characters on current line */
        for (i = 0, ch = getc(FILEin); piNNmap[ch] >= 0; i++)
            ch = getc(FILEin);
        if (i > 0 && nbytes3 == iNNlen)/*no message for ending shorter line */
            message(WARNING, cLine, "end of line expected");

        do                           /* locate first character on next line */
        {   if (ch == '\n')  cLine++;              /* increment linecounter */
            ch = getc(FILEin);
        }  while ( (nbytes3 = piNNmap[ch]) == UNDEF_CH);

    } while (nbytes3 > 0);

return(nbytes3);     /* returns either 0 (last line) or EOF (more expected) */
}


#if defined(ANSI_DECL)
  void  decode(void)
#else
  decode()
#endif
{
    int  fnum = 1;  char *inext;  long  fsize = 0, osize = 0;
    char dummy[80];

    printf("\n%s --- %s ---> %s ", sInfile, sMethod, sOutfile);

#if defined(VMS)
    if (Stream_LF == TRUE)
      printf(" [Stream_LF]\n");
    else
      printf(" [Variable length]\n");
#else
    printf("\n");
#endif
    inext = fext(sInfile);

    while (decode_file(&fsize) == EOF)
    {
        fclose(FILEin);
        inext[3] = ++fnum % 10 + '0';     /* find next: increment extension */
        if (fnum / 10 > 0)  inext[2] = fnum / 10 + '0';
        
        printf("%s --- %s --->\n", sInfile, sMethod);
        FILEin = get_start(sInfile);          /* find start of encoded text */
    }
                                                     /* check trailing line */
    fscanf(FILEin, " %s %ld", abLine, &osize);
    cLine++;
    if (strncmp(abLine, "end", 3) != 0)
        message(WARNING, cLine, "end line not found");
    if (osize > 0 && osize != fsize)
    {   fscanf(FILEin, " %s", abLine);
        if (abLine[0] != 'l')  message(WARNING, 0, "file size different");
    }
}
/*--------------------------- END decode -----------------------------------*/


/*----------------------------- main ---------------------------------------*/
#if defined(ANSI_DECL)
  main(int argc, char *argv[])
#else
  main(argc, argv)  int argc; char *argv[];
#endif
{
    int  i, mxlines = -1;  char *ext, temp[80];

    printf("\n%s %s (%s), %s\n", sProgram, sVersion, SYSTEM_ID, sAuthor);

    if (--argc == 0 || strcmp(*++argv,"?") == 0)  /* check the command line */
    {   for (i = 0; sUsage[i] != NULL; i++)  printf("%s\n",sUsage[i]);
        return;                           /* return(0) gives message on VAX */
    }
    while (argc > 0)
    {
	if (**argv == '/')
        {   if (*++*argv == 'u' || **argv == 'U')
            {   sMethod[0] = sMethod[1] = 'U';
                for (i = 0; i < 64; i++)
                    sXXtable[i] = sUUtable[i];    /* replace XXmap by UUmap */
                iXXlen = iUUlen;
            }
            else if (**argv == 's' || **argv == 'S') /* VAX-VMS Stream_LF */
            {   Stream_LF = TRUE;
            }
            else if (isdigit(**argv))
            {   mxlines = atoi(*argv);     /* max no of kilobytes in a file */
                mxlines = mxlines * 16;                        /* --> lines */
            }
        }
        else if (*sInfile == '\0')
            strcpy(sInfile, *argv);              /* copy name of input file */
        else if (*sOutfile == '\0')
            strcpy(sOutfile, *argv);          /* user specified output file */

        --argc;  ++argv;
    }                        /* end of exploration of command line switches */

    for (i = 0; sOutfile[i] != '\0'; i++)  sOutfile[i] = tolower(sOutfile[i]);
    for (i = 0;  sInfile[i] != '\0'; i++)   sInfile[i] = tolower( sInfile[i]);
    ext = fext(sInfile);

    if (*ext == '\0' || strncmp(ext, sDefault_ext, 4) == 0
        || strncmp(ext, ".xxe", 4) == 0 || strncmp(ext, ".uue", 4) == 0)
    {
        sMethod[2] = 'd';  sMethod[3] = 'e';
        if (*ext == '\0')  strcat(sInfile, sDefault_ext);/* add default ext */
        FILEin = fopen_r(sInfile, sInbuf);

        get_outfname(FILEin, sInfile, temp);/* temp holds recorded sOutfile */
        if (*sOutfile != '\0')       /* yes: user override of recorded name */
            message(WARNING, 0, "%s specified outfile as %s", sInfile, temp);
        else
            strcpy(sOutfile, temp);      /* store recorded name in sOutfile */
        FILEout = fopen_w(sOutfile, sOutbuf, CONFIRM, DCODE);

        decode();
    }
    else
    {
        FILEin = fopen_r(sInfile, sInbuf);

        if (*sOutfile == '\0')       /* no output file name on command line */
            strcpy(sOutfile, sInfile);
        strcpy(fext(sOutfile), sDefault_ext);   /* replace or add extension */
        if (strcmp(sOutfile, sInfile) == 0)
            message(ERROR, 0, "input and output filename must differ");
        FILEout = fopen_w(sOutfile, sOutbuf, CONFIRM, NCODE);

        encode(mxlines);
    }
}
