
/*-
 * Copyright (c) 1995 The Apache Group. All rights reserved.
 * 
 *
 * Apache httpd license
 * ====================
 * 
 *
 * This is the license for the Apache Server. It covers all the
 * files which come in this distribution, and should never be removed.
 * 
 * The "Apache Group" has based this server, called "Apache", on
 * public domain code distributed under the name "NCSA httpd 1.3".
 * 
 * NCSA httpd 1.3 was placed in the public domain by the National Center 
 * for Supercomputing Applications at the University of Illinois 
 * at Urbana-Champaign.
 * 
 * As requested by NCSA we acknowledge,
 * 
 *  "Portions developed at the National Center for Supercomputing
 *   Applications at the University of Illinois at Urbana-Champaign."
 *
 * Copyright on the sections of code added by the "Apache Group" belong
 * to the "Apache Group" and/or the original authors. The "Apache Group" and
 * authors hereby grant permission for their code, along with the
 * public domain NCSA code, to be distributed under the "Apache" name.
 * 
 * Reuse of "Apache Group" code outside of the Apache distribution should
 * be acknowledged with the following quoted text, to be included with any new
 * work;
 * 
 * "Portions developed by the "Apache Group", taken with permission 
 *  from the Apache Server   http://www.apache.org/apache/   "
 *
 *
 * Permission is hereby granted to anyone to redistribute Apache under
 * the "Apache" name. We do not grant permission for the resale of Apache, but
 * we do grant permission for vendors to bundle Apache free with other software,
 * or to charge a reasonable price for redistribution, provided it is made
 * clear that Apache is free. Permission is also granted for vendors to 
 * sell support for for Apache. We explicitly forbid the redistribution of 
 * Apache under any other name.
 * 
 * The "Apache Group" makes no promises to support "Apache". Users and
 * sellers of Apache support, and users of "Apache Group" code, should be 
 * aware that they use "Apache" or portions of the "Apache Group" code at 
 * their own risk. While every effort has been made to ensure that "Apache"
 * is robust and safe to use, we will not accept responsibility for any
 * damage caused, or loss of data or income which results from its use.
 * 
 */



/*
 * str.c: string utility things
 * 
 * 3/21/93 Rob McCool
 * 
 */


#include "httpd.h"

#include "stream.h"

char *get_time() {
    time_t t;
    char *time_string;

    t=time(NULL);
    time_string = ctime(&t);
    time_string[strlen(time_string) - 1] = '\0';
    return (time_string);
}

char *gm_timestr_822(time_t sec) {
    return ht_time(sec,HTTP_TIME_FORMAT, 1);
}

char *ht_time(time_t t, char *fmt, int gmt) {
    static char ts[MAX_STRING_LEN];
    struct tm *tms;

    tms = (gmt ? gmtime(&t) : localtime(&t));

    /* check return code? */
    strftime(ts,MAX_STRING_LEN,fmt,tms);
    return ts;
}

/* What a pain in the ass. */
struct tm *get_gmtoff(long *tz) {
    time_t tt;
    struct tm *t;

    tt = time(NULL);
    t = localtime(&tt);
#if defined(BSD) && !defined(AUX) && !defined(APOLLO)
    *tz = t->tm_gmtoff;
#else
    *tz = - timezone;
    if(t->tm_isdst)
        *tz += 3600;
#endif
    return t;
}


static char *months[] = {
    "Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"
};


int find_month(char *mon) {
    register int x;

    for(x=0;x<12;x++)
        if(!strcmp(months[x],mon))
            return x;
    return -1;
}

/* Roy owes Rob beer. */
/* This would be considerably easier if strptime or timegm were portable */

int later_than(struct tm *lms, char *ims) {
    char *ip;
    char mname[MAX_STRING_LEN];
    int year = 0, month = 0, day = 0, hour = 0, min = 0, sec = 0, x;

    /* Whatever format we're looking at, it will start with weekday. */
    /* Skip to first space. */
    if(!(ip = strchr(ims,' ')))
        return 0;
    else
        while(isspace(*ip))
            ++ip;

    if(isalpha(*ip)) {
        /* ctime */
        sscanf(ip,"%s %d %d:%d:%d %*s %d",mname,&day,&hour,&min,&sec,&year);
    }
    else if(ip[2] == '-') {
        /* RFC 850 (normal HTTP) */
        char t[MAX_STRING_LEN];
        sscanf(ip,"%s %d:%d:%d",t,&hour,&min,&sec);
        t[2] = '\0';
        day = atoi(t);
        t[6] = '\0';
        strcpy(mname,&t[3]);
        x = atoi(&t[7]);
        /* Prevent wraparound from ambiguity */
        if(x < 70)
            x += 100;
        year = 1900 + x;
    }
    else {
        /* RFC 822 */
        sscanf(ip,"%d %s %d %d:%d:%d",&day,mname,&year,&hour,&min,&sec);
    }
    month = find_month(mname);

    if((x = (1900+lms->tm_year) - year))
        return x < 0;
    if((x = lms->tm_mon - month))
        return x < 0;
    if((x = lms->tm_mday - day))
        return x < 0;
    if((x = lms->tm_hour - hour))
        return x < 0;
    if((x = lms->tm_min - min))
        return x < 0;
    if((x = lms->tm_sec - sec))
        return x < 0;

    return 1;
}


/* Match = 0, NoMatch = 1, Abort = -1 */
/* Based loosely on sections of wildmat.c by Rich Salz */
int strcmp_match(char *str, char *exp) {
    int x,y;

    for(x=0,y=0;exp[y];++y,++x) {
        if((!str[x]) && (exp[y] != '*'))
            return -1;
        if(exp[y] == '*') {
            while(exp[++y] == '*');
            if(!exp[y])
                return 0;
            while(str[x]) {
                int ret;
                if((ret = strcmp_match(&str[x++],&exp[y])) != 1)
                    return ret;
            }
            return -1;
        } else 
            if((exp[y] != '?') && (str[x] != exp[y]))
                return 1;
    }
    return (str[x] != '\0');
}

int is_matchexp(char *str) {
    register int x;

    for(x=0;str[x];x++)
        if((str[x] == '*') || (str[x] == '?'))
            return 1;
    return 0;
}

void strsubfirst(int start,char *dest, char *src)
{
  int src_len, dest_len, i;

  if ((src_len=strlen(src))<start){  /** src "fits" in dest **/
    for (i=0;dest[i]=src[i];i++);
    for (i=src_len;dest[i]=dest[i-src_len+start];i++);
  }
  else {                             /** src doesn't fit in dest **/
    for (dest_len=strlen(dest),i=dest_len+src_len-start;i>=src_len;i--)
      dest[i] = dest[i-src_len+start];
    for (i=0;i<src_len;i++) dest[i]=src[i];
  }
}

/*
 * Parse .. so we don't compromise security
 */
void getparents(char *name)
{
    int l=0,w=0;
    const char *lookfor="..";

    while(name[l]!='\0') {
        if(name[l]!=lookfor[w]) (w>0 ? (l-=(w-1),w=0) : l++);
        else {
            if(lookfor[++w]=='\0') {
                if((name[l+1]=='\0') || (name[l+1]=='/') &&
                   (((l > 3) && (name[l-2] == '/')) || (l<=3))) {
                    register int m=l+1,n;

                    l=l-3;
                    if(l>=0) {
                        while((l!=0) && (name[l]!='/')) --l;
                    }
                    else l=0;
                    n=l;
                    while(name[n]=name[m]) (++n,++m);
                    w=0;
                }
                else w=0;
            }
            else ++l;
        }
    }
}

void no2slash(char *name) {
    register int x,y;

    for(x=0; name[x]; x++)
        if(x && (name[x-1] == '/') && (name[x] == '/'))
            for(y=x+1;name[y-1];y++)
                name[y-1] = name[y];
}

void make_dirstr(char *s, int n, char *d) {
    register int x,f;

    for(x=0,f=0;s[x];x++) {
        if((d[x] = s[x]) == '/')
            if((++f) == n) {
                d[x] = '\0';
                return;
            }
    }
    d[x] = '\0';
}

int count_dirs(char *path) {
    register int x,n;

    for(x=0,n=0;path[x];x++)
        if(path[x] == '/') n++;
    return n;
}


void strcpy_dir(char *d, char *s) {
    register int x;

    for(x=0;s[x];x++)
        d[x] = s[x];

    if(s[x-1] != '/') d[x++] = '/';
    d[x] = '\0';
}

void chdir_file(char *file) {
    int i;

    if((i = rind(file,'/')) == -1)
        return;
    file[i] = '\0';
    chdir(file);
    file[i] = '/';
}

void http2cgi(char *w) {
    register int x;

    for(x=strlen(w);x != -1; --x)
        w[x+5]= (w[x] == '-' ? '_' : toupper(w[x]));
    strncpy(w,"HTTP_",5);
}

void getline_timed_out() {
    char errstr[MAX_STRING_LEN];

    sprintf(errstr,"timed out waiting for %s",remote_name);
    log_error(errstr);
    fclose(stdin);
    fclose(stdout);
    exit(0);
}

/*
 * This is a bit of a hack. We are relying on httpd not trying to
 * read two streams simultaneously with getline(), so we use a single
 * static buffer
 */
static STREAM getlinestr;
static unsigned char getlinebuff[8192];

/* this must be called at the beginning of any group of getlines() */
void getlineinit(int fd)
{
    getlinestr.base = getlinebuff;
    getlinestr.size = 8192;
    sdopen(fd, &getlinestr);
}

/*
 * try to be as compatible as possible with the old getline behaviour,
 * no matter how bad it was.
 */
int getline(char *s, int n, int f, unsigned int timeout) {
    int rv;

/* call the buffered routine */
    rv = sgets((unsigned char *)s, n, timeout, &getlinestr);
    if (rv == -1)  /* error */
    {
	if (errno == ETIMEDOUT) getline_timed_out();
	s[0] = '\0';
	return 1;  /* what about other errors? */
    }
    if (rv == 0) return 1;
    if (s[rv-1] == '\n') s[rv-1] = '\0';  /* but what about overlong lines? */
    return 0;
}

void getword(char *word, char *line, char stop) {
    int x = 0,y;

    for(x=0;((line[x]) && (line[x] != stop));x++)
        word[x] = line[x];

    word[x] = '\0';
    if(line[x]) ++x;
    y=0;

    while(line[y++] = line[x++]);
}

void cfg_getword(char *word, char *line) {
    int x=0,y;
    
    for(x=0;line[x] && isspace(line[x]);x++);
    y=0;
    while(1) {
        if(!(word[y] = line[x]))
            break;
        if(isspace(line[x]))
            if((!x) || (line[x-1] != '\\'))
                break;
        if(line[x] != '\\') ++y;
        ++x;
    }
    word[y] = '\0';
    while(line[x] && isspace(line[x])) ++x;
    for(y=0;line[y] = line[x];++x,++y);
}

int
cfg_getline(char *s, int n, FILE *f) {
    register int i=0, c;

    s[0] = '\0';
    /* skip leading whitespace */
    do {
        c = getc(f);
    } while (c == '\t' || c == ' ');

    while(1) {
        if((c == '\t') || (c == ' ')) {
            s[i++] = ' ';
            while((c == '\t') || (c == ' ')) 
                c = getc(f);
        }
        if(c == CR) {
            c = getc(f);
        }
        if(c == EOF || c == 0x4 || c == LF || i == (n-1)) {
            /* blast trailing whitespace */
            while(i && (s[i-1] == ' ')) --i;
            s[i] = '\0';
            return (feof(f) ? 1 : 0);
        }
        s[i] = c;
        ++i;
        c = getc(f);
    }
}

void escape_shell_cmd(char *cmd) {
    register int x,y,l;

    l=strlen(cmd);
    for(x=0;cmd[x];x++) {
        if(ind("&;`'\"|*?~<>^()[]{}$\\",cmd[x]) != -1){
            for(y=l+1;y>x;y--)
                cmd[y] = cmd[y-1];
            l++; /* length has been increased */
            cmd[x] = '\\';
            x++; /* skip the character */
        }
    }
}

void plustospace(char *str) {
    register int x;

    for(x=0;str[x];x++) if(str[x] == '+') str[x] = ' ';
}

void spacetoplus(char *str) {
    register int x;

    for(x=0;str[x];x++) if(str[x] == ' ') str[x] = '+';
}

char x2c(char *what) {
    register char digit;

    digit = ((what[0] >= 'A') ? ((what[0] & 0xdf) - 'A')+10 : (what[0] - '0'));
    digit *= 16;
    digit += (what[1] >= 'A' ? ((what[1] & 0xdf) - 'A')+10 : (what[1] - '0'));
    return(digit);
}

void unescape_url(char *url) {
    register int x,y;

    for(x=0,y=0;url[y];++x,++y) {
        if((url[x] = url[y]) == '%') {
            url[x] = x2c(&url[y+1]);
            y+=2;
        }
    }
    url[x] = '\0';
}

#define c2x(what,where) sprintf(where,"%%%2x",what)

void escape_url(char *url) {
    register int x,y;
    register char digit;
    char *copy;

    copy = strdup(url);
            
    for(x=0,y=0;copy[x];x++,y++) {
        if(ind("% ?+&",url[y] = copy[x]) != -1) {
            c2x(copy[x],&url[y]);
            y+=2;
        }
    }
    url[y] = '\0';
    free(copy);
}

void escape_uri(char *url) {
    register int x,y;
    register char digit;
    char *copy;

    copy = strdup(url);
            
    for(x=0,y=0;copy[x];x++,y++) {
        if(ind(":% ?+&",url[y] = copy[x]) != -1) {
            c2x(copy[x],&url[y]);
            y+=2;
        }
    }
    url[y] = '\0';
    free(copy);
}

void make_full_path(char *src1,char *src2,char *dst) {
    register int x,y;

    for(x=0;dst[x] = src1[x];x++);

    if(!x) dst[x++] = '/';
    else if((dst[x-1] != '/'))
        dst[x++] = '/';

    for(y=0;dst[x] = src2[y];x++,y++);
}

int is_directory(char *path) {
    struct stat finfo;

    if(stat(path,&finfo) == -1)
        return 0; /* in error condition, just return no */

    return(S_ISDIR(finfo.st_mode));
}

int is_url(char *u) {
    register int x;

    for(x=0;u[x] != ':';x++)
        if((!u[x]) || (!isalpha(u[x])))
            return 0;

    if((u[x+1] == '/') && (u[x+2] == '/'))
        return 1;
    else return 0;
}

char *make_env_str(char *name, char *value, FILE *out) {
    char *t,*tp;

    if(!(t = (char *)malloc(strlen(name)+strlen(value)+2)))
        die(NO_MEMORY,"make_env_str",out);

    for(tp=t;*tp = *name;tp++,name++);
    for(*tp++ = '=';*tp = *value;tp++,value++);
    return t;
}

char *replace_env_str(char **env, char *name, char *value, FILE *out) {
    char *p;
    register int i, len;

    for (i = 0, len = strlen(name); env[i]; i++)
       if (strncmp(env[i], name, len) == 0) {
        free(env[i]);
        env[i] = make_env_str(name, value, out);
        break;
       }
}


char **new_env(char **env, int to_add, int *pos) {
    char **newenv;
    
    if(!env) {
        *pos = 0;
        /*  Apache. Following line replaced with version which updates the
              global var pointer to the read environment vars.
            If we don't remember where the new version is, we can never
              hope to reuse the variables after a custom error response redirect
        return (char **)malloc((to_add+1)*sizeof(char *));    
        */
        newenv = (char **)malloc((to_add+1)*sizeof(char *));
        newenv[to_add] = NULL;
    }
    else {
        int x;

        for(x=0;env[x];x++);
        if(!(newenv = (char **)malloc((to_add+x+1)*(sizeof(char *)))))
            return NULL;
        for(x=0;env[x];x++)
            newenv[x] = env[x];
        newenv[to_add + x] = NULL;
        *pos = x;
        free(env);
        
          /* Apache. again we need to rember where the vars are at all times */
    }
    return (in_headers_env = newenv);
}

void free_env(char **env) {
    int x;

    for(x=0;env[x];x++)
        free(env[x]);
    free(env);
}

int can_exec(struct stat *finfo) {
    if(user_id == finfo->st_uid)
        if(finfo->st_mode & S_IXUSR)
            return 1;
    if(group_id == finfo->st_gid)
        if(finfo->st_mode & S_IXGRP)
            return 1;
    return (finfo->st_mode & S_IXOTH);
}

#ifdef NEED_STRDUP
char *strdup (char *str)
{
  char *dup;

  if(!(dup = (char *)malloc (strlen (str) + 1)))
      return NULL;
  dup = strcpy (dup, str);

  return dup;
}
#endif

/* The following two routines were donated for SVR4 by Andreas Vogel */
#ifdef NEED_STRCASECMP
int strcasecmp (const char *a, const char *b)
{
    const char *p = a;
    const char *q = b;
    for (p = a, q = b; *p && *q; p++, q++)
    {
      int diff = tolower(*p) - tolower(*q);
      if (diff) return diff;
    }
    if (*p) return 1;       /* p was longer than q */
    if (*q) return -1;      /* p was shorter than q */
    return 0;               /* Exact match */
}

#endif

#ifdef NEED_STRNCASECMP
int strncasecmp (const char *a, const char *b, int n)
{
    const char *p = a;
    const char *q = b;

    for (p = a, q = b; /*NOTHING*/; p++, q++)
    {
      int diff;
      if (p == a + n) return 0;     /*   Match up to n characters */
      if (!(*p && *q)) return *p - *q;
      diff = tolower(*p) - tolower(*q);
      if (diff) return diff;
    }
    /*NOTREACHED*/
}
#endif



#ifdef NEED_INITGROUPS
int initgroups(const char *name, gid_t basegid)
{
  gid_t groups[NGROUPS_MAX];
  struct group *g;
  int index = 0;

  groups[index++] = basegid;

  while (index < NGROUPS_MAX && ((g = getgrent()) != NULL))
    if (g->gr_gid != basegid)
    {
      char **names;

      for (names = g->gr_mem; *names != NULL; ++names)
        if (!strcmp(*names, name))
          groups[index++] = g->gr_gid;
    }

  return setgroups(index, groups);
}
#endif

#ifdef NEED_WAITPID
/* From ikluft@amdahl.com */
/* this is not ideal but it works for SVR3 variants */
/* httpd does not use the options so this doesn't implement them */
int waitpid(pid_t pid, int *statusp, int options)
{
    int tmp_pid;
    if ( kill ( pid,0 ) == -1) {
        errno=ECHILD;
        return -1;
    }
    while ((( tmp_pid = wait(statusp)) != pid) && ( tmp_pid != -1 ));
    return tmp_pid;
}
#endif

int ind(const char *s, char c) {
    register int x;

    for(x=0;s[x];x++)
        if(s[x] == c) return x;

    return -1;
}

int rind(char *s, char c) {
    register int x;

    for(x=strlen(s)-1;x != -1;x--)
        if(s[x] == c) return x;

    return -1;
}

void str_tolower(char *str) {
    while(*str) {
        *str = tolower(*str);
        ++str;
    }
}
        
uid_t uname2id(char *name) {
    struct passwd *ent;

    if(name[0] == '#') 
        return(atoi(&name[1]));

    if(!(ent = getpwnam(name))) {
        fprintf(stderr,"httpd: bad user name %s\n",name);
        exit(1);
    }
    else return(ent->pw_uid);
}

gid_t gname2id(char *name) {
    struct group *ent;

    if(name[0] == '#') 
        return(atoi(&name[1]));

    if(!(ent = getgrnam(name))) {
        fprintf(stderr,"httpd: bad group name %s\n",name);
        exit(1);
    }
    else return(ent->gr_gid);
}

int get_portnum(int sd,FILE *out) {
    struct sockaddr addr;
    int len;

    len = sizeof(struct sockaddr);
    if(getsockname(sd,&addr,&len) < 0)
        die(SERVER_ERROR,"could not get port number",out);
    return ntohs(((struct sockaddr_in *)&addr)->sin_port);
}

#ifdef VIRTUAL_HOST
struct in_addr get_local_addr(int sd,FILE *out) {
    struct sockaddr addr;
    int len;

    len = sizeof(struct sockaddr);
    if(getsockname(sd,&addr,&len) < 0)
        die(SERVER_ERROR,"could not get local address",out);
    return ((struct sockaddr_in *)&addr)->sin_addr;
}
#endif /* VIRTUAL_HOST */

char *find_fqdn(struct hostent *p) {
    int x;

    if(ind(p->h_name,'.') == -1) {
        for(x=0;p->h_aliases[x];++x) {
            if((ind(p->h_aliases[x],'.') != -1) && 
               (!strncmp(p->h_aliases[x],p->h_name,strlen(p->h_name))))
                return strdup(p->h_aliases[x]);
        }
        return NULL;
    } else return strdup(p->h_name);
}

/*
 * Yet another speed feature --- nameserver cache.
 *
 * This is maintained by the parent process, to which we send new entries.
 */

/*
 * These two parameters have been chosen with care.
 * Do not change them unless you have good statistical evidence
 * that it would be beneficial. You would only ever want to increase
 * NSCACHE_SIZE, and it probably won't do any good.
 */
#define NSCACHE_NAMELEN 44
#define NSCACHE_SIZE 100

/*
 * This structure must be less than PIPE_BUF bytes in size, which
 * POSIX mandates as being >= 512.
 */
struct nscache_entry
{
  struct in_addr addr;
  unsigned int access_time;
  char name[NSCACHE_NAMELEN];
};

static struct nscache_entry *nscache = NULL;  /* the cache */
static unsigned int nscache_timer = 0;        /* one clock tick per request */
static int nscache_fdwr=-1;                   /* resolved data out here */

/*
 * Reset the nameserver cache, allocating memory when necessary.
 * Parameters:
 *   FILE *errors   stream for fatal error reporting
 *   int fdwr       if not -1, then the fd on which to write host data.
 */
void
init_nameserver_cache(FILE *errors, int fdwr)
{
#ifndef MINIMAL_DNS
    if (nscache == NULL)
    {
	nscache = malloc(NSCACHE_SIZE * sizeof(struct nscache_entry));
        if (nscache == NULL) die(NO_MEMORY,"init_nameserver_cache", errors);
    }
    memset(nscache, '\0', NSCACHE_SIZE * sizeof(struct nscache_entry));
    nscache_timer = 0;
    if (fdwr != -1) nscache_fdwr = fdwr;
#endif
}
 
void
get_remote_host(int fd) {
    struct sockaddr addr;
    int len, i;
    struct in_addr *iaddr;
    struct hostent *hptr;
    extern int h_errno;  /* some machines don't have this in their headers */

    len = sizeof(struct sockaddr);

    if ((getpeername(fd, &addr, &len)) < 0) {
        remote_host=NULL;
        remote_ip=NULL;
        remote_name="UNKNOWN_HOST";
        return;
    }
    iaddr = &(((struct sockaddr_in *)&addr)->sin_addr);
    remote_ip = inet_ntoa(*iaddr);
    remote_host= NULL;         /* default values */
    remote_name = remote_ip;

#ifndef MINIMAL_DNS
/* firstly, search the cache */
    if (nscache != NULL)
    {
	for (i=0; i < NSCACHE_SIZE; i++)
	    if (memcmp(&nscache[i].addr, iaddr, sizeof(struct in_addr)) == 0)
		break;
	if (i < NSCACHE_SIZE) /* found it */
	{
/* a zero length name indicates a cached failed or spoof lookup */
	    if (nscache[i].name[0] != '\0')
	    {
		remote_host = nscache[i].name;
		remote_name = remote_host;
	    }
	    return;
	}
    }

    hptr = gethostbyaddr((char *)iaddr, sizeof(struct in_addr), AF_INET);
    if(hptr != NULL)
    {
        char **haddr;

	remote_host = strdup(hptr->h_name);
	str_tolower(remote_host);

#ifdef MAXIMUM_DNS
    /* Grrr. Check THAT name to make sure it's really the name of the addr. */
    /* Code from Harald Hanche-Olsen <hanche@imf.unit.no> */
        hptr = gethostbyname(remote_host);
        if (hptr) {
            for(haddr=hptr->h_addr_list;*haddr;haddr++) {
                if(((struct in_addr *)(*haddr))->s_addr == iaddr->s_addr)
                    break;
            }
        }
        if((!hptr) || (!(*haddr)))
            remote_host = NULL;
#endif
    }

/* cache successful responses, or negative responses which weren't temporary */
    if ((remote_host != NULL && strlen(remote_host) < NSCACHE_NAMELEN) ||
      (remote_host == NULL && h_errno  != TRY_AGAIN && h_errno != NO_RECOVERY))
    {
	struct nscache_entry ent;

	ent.addr = *iaddr;
	ent.access_time = nscache_timer;
	if (remote_host == NULL) ent.name[0] = '\0';
	else strcpy(ent.name, remote_host);
/* don't care if this works */
	(void)write(nscache_fdwr, &ent, sizeof(struct nscache_entry));
    }

    if (remote_host != NULL) remote_name = remote_host;
#endif /* MINIMAL_DNS */
}

/*
 * Update the name server cache. N.B. this is run by the parent process,
 * _as root_, so be very careful.
 */
void
update_nameserver_cache(int fdrd)
{
    struct nscache_entry ent;
    int nrd, i, oldest, oldt;

#ifndef MINIMAL_DNS
    if (nscache != NULL)
    {
	nscache_timer++;  /* another tick */
/* try and read it */
	nrd = read(fdrd, &ent, sizeof(struct nscache_entry));
/*
 * there are several different semantics for what happens when there is no
 * data for a non-blocking read, including:
 *  POSIX  O_NONBLOCK      return -1,  errno = EAGAIN,
 *  BSD    O_NDELAY        return -1,  errno = EWOULDBLOCK,
 *  SYSV   O_NDELAY        return 0. (yuk)
 */
	if (nrd != sizeof(struct nscache_entry)) return;

/* find the oldest cache entry */
	oldest = 0;
	oldt = nscache_timer;
	
	for (i=0; i < NSCACHE_SIZE; i++)
	{
	    if (nscache[i].access_time < oldt)
	    {
		oldest = i;
		oldt = nscache[i].access_time;
	    }
	    if (memcmp(&nscache[i].addr,&ent.addr,sizeof(struct in_addr) == 0))
	    {
		nscache[i].access_time = nscache_timer;
		return;
	    }
	}
/* not hit */
	nscache[oldest].addr = ent.addr;
	nscache[oldest].access_time = nscache_timer;
	strcpy(nscache[oldest].name, ent.name);
    }
#endif /* MINIMAL_DNS */
}



char *get_remote_logname(FILE *fd) {
    int len;
    char *result;
#if defined(NEXT) || defined(BSD4_4) || defined(SOLARIS2) || defined(LINUX)
    struct sockaddr sa_server, sa_client;
#else
    struct sockaddr_in sa_server,sa_client;
#endif

    len = sizeof(sa_client);
    if(getpeername(fileno(stdout),&sa_client,&len) != -1) {
        len = sizeof(sa_server);
        if(getsockname(fileno(stdout),&sa_server,&len) == -1)
            result = "unknown";
        else
            result = rfc931((struct sockaddr_in *) & sa_client,
                                    (struct sockaddr_in *) & sa_server);
    }
    else result = "unknown";

    return result; /* robm=pinhead */
}

void get_local_host()
{
    char str[128];
    int len = 128;

    if(!server_hostname) {
        struct hostent *p;
        gethostname(str, len);
        if((!(p=gethostbyname(str))) || (!(server_hostname = find_fqdn(p)))) {
            fprintf(stderr,"httpd: cannot determine local host name.\n");
            fprintf(stderr,"Use ServerName to set it manually.\n");
            exit(1);
        }
    }
}

void construct_url(char *d, char *s) {
  if (port == 80) {
    sprintf(d,"http://%s%s",server_hostname,s);
  } else {
    sprintf(d,"http://%s:%d%s",server_hostname,port,s);
  }
/*    escape_url(d); */
}

/* aaaack but it's fast and const should make it shared text page. */
const int pr2six[256]={
    64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,
    64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,62,64,64,64,63,
    52,53,54,55,56,57,58,59,60,61,64,64,64,64,64,64,64,0,1,2,3,4,5,6,7,8,9,
    10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,64,64,64,64,64,64,26,27,
    28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,
    64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,
    64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,
    64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,
    64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,
    64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,64,
    64,64,64,64,64,64,64,64,64,64,64,64,64
};

void uudecode(char *bufcoded, unsigned char *bufplain, int outbufsize) {
    int nbytesdecoded, j;
    register unsigned char *bufin;
    register unsigned char *bufout = bufplain;
    register int nprbytes;
    
    /* Strip leading whitespace. */
    
    while(*bufcoded==' ' || *bufcoded == '\t') bufcoded++;
    
    /* Figure out how many characters are in the input buffer.
     * If this would decode into more bytes than would fit into
     * the output buffer, adjust the number of input bytes downwards.
     */
    bufin = (unsigned char *)bufcoded;
    while(pr2six[*(bufin++)] <= 63);
    nprbytes = (char *)bufin - bufcoded - 1;
    nbytesdecoded = ((nprbytes+3)/4) * 3;
    if(nbytesdecoded > outbufsize) {
        nprbytes = (outbufsize*4)/3;
    }
    
    bufin = (unsigned char *)bufcoded;
    
    while (nprbytes > 0) {
        *(bufout++) = 
            (unsigned char) (pr2six[*bufin] << 2 | pr2six[bufin[1]] >> 4);
        *(bufout++) = 
            (unsigned char) (pr2six[bufin[1]] << 4 | pr2six[bufin[2]] >> 2);
        *(bufout++) = 
            (unsigned char) (pr2six[bufin[2]] << 6 | pr2six[bufin[3]]);
        bufin += 4;
        nprbytes -= 4;
    }
    
    if(nprbytes & 03) {
        if(pr2six[bufin[-2]] > 63)
            nbytesdecoded -= 2;
        else
            nbytesdecoded -= 1;
    }
    bufplain[nbytesdecoded] = '\0';
}

   /* New Apache routine to map error responses into array indicies */
   /*  e.g.  400 -> 0,  500 -> 1,  502 -> 2 ...                     */
   /* the indicies have no significance                             */
int index_of_response(int err_no) {
   char *cptr, err_string[4];
   static char *response_codes = RESPONSE_CODE_LIST;
   int index_number;
   
   sprintf(err_string,"%3d",err_no);
   
   cptr = response_codes;
   cptr++;
   index_number = 0;
   while (*cptr && strncmp(cptr, err_string, 3)) { 
      cptr += 4;
      index_number++;
   }
   return index_number;
}

   /* New Apache routine to rename existing environoment variables so  */
   /* that they have a X_FIXED prefix                                  */
   
void rename_original_env(char **env, FILE *fd) {
   int x = 0;
   char *cptr;
   
   if (env == NULL) return; 
   
   while(env[x] != NULL) {
                                               /* v length of REDIRECT_ + 1*/
     if(!(cptr = (char *) malloc((strlen(env[x])+1+9)*sizeof(char))))
       die(NO_MEMORY,"make_env_str",fd);

     sprintf(cptr,"REDIRECT_%s",env[x]);
     free(env[x]);    /* Discard old environment variable */
     env[x] = cptr;
     x++;
   }
   
   env[x] = NULL;
}

/* This routine records the original status of an error/problem as
    an environment variable and as a global variable for use in redirected
    error/problems.
*/
char **add_redirection_status_env(char **env,FILE *out, int status) {
    char t[HUGE_STRING_LEN],*env_path;
    int x;

    if(!(env = new_env(env,1,&x)))
        die(NO_MEMORY,"add_redirection_status_env",out);
    
    sprintf(t,"%d",status);
    env[x++] = make_env_str("REDIRECT_STATUS",t,out);
    original_status = status;

    env[x] = NULL;
    in_headers_env = env;
    return env;     /* Missing from earlier patch */
}

/*
  Instead of resetting everything prior to a redirect, we will do something
  far more useful; we'll keep as much info as we can for the benefit of the
  URL being called.
*/
void prepare_for_redirect(FILE *fd, int orig_status, char *new_url) {

     /* Reset some of the important things that now change */
     content_type[0] = '\0';
     content_length = -1;
     auth_line[0] = '\0';
     content_encoding[0] = '\0';       
     status_line = NULL;
     status = 0;
     out_headers = NULL;
     location[0] = '\0';
     sprintf(the_request, "GET %s", new_url);
 
       /* rename existing headers, e.g. USER_AGENT -> REDIRECT_USER_AGENT */
     rename_original_env(in_headers_env, fd);
                   
       /* Add another env var called REDIRECT_STATUS */ 
     add_redirection_status_env(in_headers_env, fd, orig_status);
}
