
/*-
 * Copyright (c) 1995 The Apache Group. All rights reserved.
 * 
 *
 * Apache httpd license
 * ====================
 * 
 *
 * This is the license for the Apache Server. It covers all the
 * files which come in this distribution, and should never be removed.
 * 
 * The "Apache Group" has based this server, called "Apache", on
 * public domain code distributed under the name "NCSA httpd 1.3".
 * 
 * NCSA httpd 1.3 was placed in the public domain by the National Center 
 * for Supercomputing Applications at the University of Illinois 
 * at Urbana-Champaign.
 * 
 * As requested by NCSA we acknowledge,
 * 
 *  "Portions developed at the National Center for Supercomputing
 *   Applications at the University of Illinois at Urbana-Champaign."
 *
 * Copyright on the sections of code added by the "Apache Group" belong
 * to the "Apache Group" and/or the original authors. The "Apache Group" and
 * authors hereby grant permission for their code, along with the
 * public domain NCSA code, to be distributed under the "Apache" name.
 * 
 * Reuse of "Apache Group" code outside of the Apache distribution should
 * be acknowledged with the following quoted text, to be included with any new
 * work;
 * 
 * "Portions developed by the "Apache Group", taken with permission 
 *  from the Apache Server   http://www.apache.org/apache/   "
 *
 *
 * Permission is hereby granted to anyone to redistribute Apache under
 * the "Apache" name. We do not grant permission for the resale of Apache, but
 * we do grant permission for vendors to bundle Apache free with other software,
 * or to charge a reasonable price for redistribution, provided it is made
 * clear that Apache is free. Permission is also granted for vendors to 
 * sell support for for Apache. We explicitly forbid the redistribution of 
 * Apache under any other name.
 * 
 * The "Apache Group" makes no promises to support "Apache". Users and
 * sellers of Apache support, and users of "Apache Group" code, should be 
 * aware that they use "Apache" or portions of the "Apache Group" code at 
 * their own risk. While every effort has been made to ensure that "Apache"
 * is robust and safe to use, we will not accept responsibility for any
 * damage caused, or loss of data or income which results from its use.
 * 
 */



/*
 * http_include.c: Handles the server-parsed HTML documents
 * 
 * Rob McCool
 * 
 */

#include "httpd.h"

#define STARTING_SEQUENCE "<!--#"
#define ENDING_SEQUENCE "-->"
#define DEFAULT_ERROR_MSG "[an error occurred while processing this directive]"
#define DEFAULT_TIME_FORMAT "%A, %d-%b-%y %T %Z"
#define SIZEFMT_BYTES 0
#define SIZEFMT_KMG 1

/* These are stored statically so that they can be reformatted quickly */
static time_t date,lm;

static void decodehtml(char *s);
static char *get_tag(FILE *in, char *tag, int dodecode);
static int get_directive(FILE *in, char *d);

/* ------------------------ Environment function -------------------------- */

#define NUM_INCLUDE_VARS 5

char **add_include_vars(char **env,char *file, char *path_args, char *args, 
                        char *timefmt,FILE *out)
{
    int x;
    struct stat finfo;
    char ufile[HUGE_STRING_LEN];
    char *t;

    if(!(env = new_env(env,NUM_INCLUDE_VARS,&x)))
        die(NO_MEMORY,"add_include_vars",out);
    date = time(NULL);
    env[x++] = make_env_str("DATE_LOCAL",ht_time(date,timefmt,0),out);
    env[x++] = make_env_str("DATE_GMT",ht_time(date,timefmt,1),out);

    if(stat(file,&finfo) != -1) {
        lm = finfo.st_mtime;
        env[x++] = make_env_str("LAST_MODIFIED",ht_time(lm,timefmt,0),out);
    }
    strcpy(ufile,file);
    unmunge_name(ufile);
    env[x++] = make_env_str("DOCUMENT_URI",ufile,out);
    if(t = strrchr(ufile,'/'))
        ++t;
    else
        t = ufile;
    env[x++] = make_env_str("DOCUMENT_NAME",t,out);
    env[x] = NULL;
    return env;
}

#define GET_CHAR(f,c,r) \
 { \
   int i = getc(f); \
   if(feof(f) || ferror(f) || (i == -1)) { \
        fclose(f); \
        return r; \
   } \
   c = (char)i; \
 }

/* --------------------------- Parser functions --------------------------- */

int find_string(FILE *in,char *str, FILE *out) {
    int x,l=strlen(str),p;
    char c;

    p=0;
    while(1) {
        GET_CHAR(in,c,1);
        if(c == str[p]) {
            if((++p) == l)
                return 0;
        }
        else {
            if(out) {
                if(p) {
                    for(x=0;x<p;x++) {
                        putc(str[x],out);
                        ++bytes_sent;
                    }
                }
                putc(c,out);
                ++bytes_sent;
            }
            p=0;
        }
    }
}

/*
 * decodes a string containing html entities or numeric character references.
 * 's' is overwritten with the decoded string.
 * If 's' is syntatically incorrect, then the followed fixups will be made:
 *   unknown entities will be left undecoded;
 *   references to unused numeric characters will be deleted.
 *   In particular, &#00; will not be decoded, but will be deleted.
 *
 * drtr
 */

/* maximum length of any ISO-LATIN-1 HTML entity name. */
#define MAXENTLEN (6)
typedef const char *cchar;

static void
decodehtml(char *s)
{
    int val, i, j;
    char *p=s;
    cchar ents;
    static const cchar entlist[MAXENTLEN+1]={
	NULL,  /* 0 */
	NULL,  /* 1 */
	"lt\074gt\076", /* 2 */
	"amp\046ETH\320eth\360", /* 3 */
	"quot\042Auml\304Euml\313Iuml\317Ouml\326Uuml\334auml\344euml\353\
iuml\357ouml\366uuml\374yuml\377", /* 4 */
	"Acirc\302Aring\305AElig\306Ecirc\312Icirc\316Ocirc\324Ucirc\333\
THORN\336szlig\337acirc\342aring\345aelig\346ecirc\352icirc\356ocirc\364\
ucirc\373thorn\376", /* 5 */
	"Agrave\300Aacute\301Atilde\303Ccedil\307Egrave\310Eacute\311\
Igrave\314Iacute\315Ntilde\321Ograve\322Oacute\323Otilde\325Oslash\330\
Ugrave\331Uacute\332Yacute\335agrave\340aacute\341atilde\343ccedil\347\
egrave\350eacute\351igrave\354iacute\355ntilde\361ograve\362oacute\363\
otilde\365oslash\370ugrave\371uacute\372yacute\375" /* 6 */
    };

    for (; *s != '\0'; s++, p++)
    {
	if (*s != '&')
	{
	    *p = *s;
	    continue;
	}
/* find end of entity */
	for (i=1; s[i] != ';' && s[i] != '\0'; i++) ;

	if (s[i] == '\0')  /* treat as normal data */
	{
	    *p = *s;
	    continue;
	}

/* is it numeric ? */
	if (s[1] == '#')
	{
	    for (j=2, val=0; j < i && isdigit(s[j]); j++)
		val = val * 10 + s[j] - '0';
	    s += i;
	    if (j < i || val <= 8 || (val >= 11 && val <= 31) ||
		(val >= 127 && val <= 160) || val >= 256)
		p--;  /* no data to output */
	    else
		*p = val;
	} else
	{
	    j = i-1;
	    if (i-1 > MAXENTLEN || entlist[i-1] == NULL) /* wrong length */
	    {
		*p = '&';
		continue;  /* skip it */
	    }
	    for (ents=entlist[i-1]; *ents != '\0'; ents += i)
		if (strncmp(s+1, ents, i-1) == 0) break;

	    if (*ents == '\0')
		*p = '&';  /* unknown */
	    else
	    {
		*p = ((const unsigned char *)ents)[i-1];
		s += i;
	    }
	}
    }

    *p = '\0';
}

/*
 * extract the next tag name and value.
 * if there are no more tags, set the tag name to 'done'
 * the tag value is html decoded if dodecode is non-zero
 */
static char *
get_tag(FILE *in, char *tag, int dodecode) {
    char *t = tag, *tag_val, c, term;
    int n;

    n = 0;

    do { /* skip whitespace */
	GET_CHAR(in,c,NULL);
    } while (isspace(c));

/* tags can't start with - */
    if(c == '-') {
        GET_CHAR(in,c,NULL);
        if(c == '-') {
            do {
		GET_CHAR(in,c,NULL);
	    } while (isspace(c));
            if(c == '>') {
                strcpy(tag,"done");
                return tag;
            }
        }
	return NULL; /* failed */
    }

/* find end of tag name */
    while(1) {
        if(++n == MAX_STRING_LEN) {
            t[MAX_STRING_LEN - 1] = '\0';
            return NULL;
        }
	if(c == '=' || isspace(c)) break;
	*(t++) = tolower(c);
        GET_CHAR(in,c,NULL);
    }

    *t++ = '\0';
    tag_val = t;

    while (isspace(c)) GET_CHAR(in, c, NULL); /* space before = */
    if (c != '=') return NULL;

    do {
	GET_CHAR(in,c,NULL);  /* space after = */
    } while (isspace(c));

/* we should allow a 'name' as a value */
    if (c != '"' && c != '\'') return NULL;
    term = c;
    while(1) {
	GET_CHAR(in,c,NULL);
	if(++n == MAX_STRING_LEN) {
	    t[MAX_STRING_LEN - 1] = '\0';
	    return NULL;
	}
	if (c == term) break;
	*(t++) = c;
    }
    *t = '\0';
    if (dodecode) decodehtml(tag_val);
    return tag_val;
}

static int
get_directive(FILE *in, char *d) {
    char c;

    /* skip initial whitespace */
    while(1) {
        GET_CHAR(in,c,1);
        if(!isspace(c))
            break;
    }
    /* now get directive */
    while(1) {
        *d++ = tolower(c);
        GET_CHAR(in,c,1);
        if(isspace(c))
            break;
    }
    *d = '\0';
    return 0;
}

/* --------------------------- Action handlers ---------------------------- */


void send_parsed_content(char *file, FILE *f, FILE *fd, 
                         char *path_args, char *args,
                         char **env,int noexec);

int send_included_file(char *file, FILE *out, char **env, char *fn) 
{
    FILE *f;
    struct stat finfo;
    int allow;char op,i;

    if(stat(file,&finfo) == -1)
        return -1;
    evaluate_access(file,&finfo,M_GET,&allow,&op,out);
    if(!allow)
        return -1;
    set_content_type(file);
#ifdef XBITHACK
    if((op & OPT_INCLUDES) &&
        (!strcmp(content_type,INCLUDES_MAGIC_TYPE) ||
         !strcmp(content_type,INCLUDES_MAGIC_TYPE3) ||
         (finfo.st_mode & S_IXUSR) )) {
#else
    if((op & OPT_INCLUDES) && (!strcmp(content_type,INCLUDES_MAGIC_TYPE)
			       ||!strcmp(content_type,INCLUDES_MAGIC_TYPE3))) {
#endif
        if(!(f = fopen(file,"r")))
            return -1;
        send_parsed_content(file,f,out,"","",env,op & OPT_INCNOEXEC);
        chdir_file(fn); /* grumble */
    }
    else if(!strcmp(content_type,CGI_MAGIC_TYPE))
        return -1;
    else {
        if(!(f=fopen(file,"r")))
            return -1;
        send_fd(f,out,NULL);
        fclose(f);
    }
    return 0;
}

int handle_include(FILE *in, FILE *out, char *fn, char **env, char *error) {
    char tag[MAX_STRING_LEN],errstr[MAX_STRING_LEN];
    char *tag_val;

    while(1) {
        if(!(tag_val = get_tag(in,tag,1)))
            return 1;
        if(!strcmp(tag,"file")) {
            char dir[MAX_STRING_LEN],to_send[MAX_STRING_LEN];

            getparents(tag_val); /* get rid of any nasties */
            getwd(dir);
            make_full_path(dir,tag_val,to_send);
            if(send_included_file(to_send,out,env,fn)) {
                sprintf(errstr,"unable to include %s in parsed file %s",
                        tag_val, fn);
                log_error_noclose(errstr);
                bytes_sent += fprintf(out,"%s",error);
            }            
        } 
        else if(!strcmp(tag,"virtual")) {
            if(translate_name(tag_val,out) != STD_DOCUMENT) {
                bytes_sent += fprintf(out,"%s",error);
                log_error_noclose(errstr);
            }  
            else if(send_included_file(tag_val,out,env,fn)) {
                sprintf(errstr,"unable to include %s in parsed file %s",
                        tag_val, fn);
                log_error_noclose(errstr);
                bytes_sent += fprintf(out,"%s",error);
            }
        } 
        else if(!strcmp(tag,"done"))
            return 0;
        else {
            sprintf(errstr,"unknown parameter %s to tag echo in %s",tag,fn);
            log_error_noclose(errstr);
            bytes_sent += fprintf(out,"%s",error);
        }
    }
}

int handle_echo(FILE *in, FILE *out, char *file, char *error, char **env) {
    char tag[MAX_STRING_LEN];
    char *tag_val;

    while(1) {
        if(!(tag_val = get_tag(in,tag,1)))
            return 1;
        if(!strcmp(tag,"var")) {
            int x, i, len;

	    len = strlen(tag_val);
            for(x=0;env[x] != NULL; x++) {
                i = ind(env[x],'=');
                if(i == len && strncmp(env[x], tag_val, i) == 0) {
                    bytes_sent += fprintf(out,"%s",&env[x][i+1]);
                    break;
                }
            }
            if(!env[x]) bytes_sent += fprintf(out,"(none)");
        } else if(!strcmp(tag,"done"))
            return 0;
        else {
            char errstr[MAX_STRING_LEN];
            sprintf(errstr,"unknown parameter %s to tag echo in %s",tag,file);
            log_error_noclose(errstr);
            bytes_sent += fprintf(out,"%s",error);
        }
    }
}

int include_cgi(char *s, char *pargs, char *args, char **env, FILE *out) 
{
    char *argp,op,d[HUGE_STRING_LEN];
    int allow,check_cgiopt;
    struct stat finfo;

    getparents(s);
    if(s[0] == '/') {
        strcpy(d,s);
        if(translate_name(d,out) != SCRIPT_CGI)
            return -1;
        check_cgiopt=0;
    } else {
        char dir[MAX_STRING_LEN];
        getwd(dir);
        make_full_path(dir,s,d);
        check_cgiopt=1;
    }
    /* No hardwired path info or query allowed */
    if(stat(d,&finfo) == -1)
        return -1;

    evaluate_access(d,&finfo,M_GET,&allow,&op,out);
    if((!allow) || (check_cgiopt && (!(op & OPT_EXECCGI))))
        return -1;

    if(cgi_stub("GET",d,pargs,args,env,&finfo,-1,out) == REDIRECT_URL)
        bytes_sent += fprintf(out,"<A HREF=\"%s\">%s</A>",location,location);
    return 0;
}

static int ipid;
void kill_include_child() {
    char errstr[MAX_STRING_LEN];
    sprintf(errstr,"killing command process %d",ipid);
    log_error_noclose(errstr);
    kill(ipid,SIGKILL);
    waitpid(ipid,NULL,0);
}

int include_cmd(char *s, char *pargs, char *args, char **env, FILE *out) {
    int p[2],x;
    FILE *f;

    if(pipe(p) == -1)
        die(SERVER_ERROR,"httpd: could not create IPC pipe",out);
    if((ipid = fork()) == -1)
        die(SERVER_ERROR,"httpd: could not fork new process",out);
    if(!ipid) {
        char *argv0;

        if(pargs[0] || args[0]) {
            if(!(env = new_env(env,4,&x)))
                return -1;
            if(pargs[0]) {
                char p2[HUGE_STRING_LEN];
                
                escape_shell_cmd(pargs);
                env[x++] = make_env_str("PATH_INFO",pargs,out);
                strcpy(p2,pargs);
                if (translate_name(p2,out) != BAD_URL)
		    env[x++] = make_env_str("PATH_TRANSLATED",p2,out);
            }
            if(args[0]) {
                env[x++] = make_env_str("QUERY_STRING",args,out);
                unescape_url(args);
                escape_shell_cmd(args);
                env[x++] = make_env_str("QUERY_STRING_UNESCAPED",args,out);
            }
            env[x] = NULL;
        }

        close(p[0]);
        if(p[1] != STDOUT_FILENO) {
            dup2(p[1],STDOUT_FILENO);
            close(p[1]);
        }
        error_log2stderr();
        if(!(argv0 = strrchr(SHELL_PATH,'/')))
            argv0=SHELL_PATH;
        if(execle(SHELL_PATH,argv0,"-c",s,(char *)0,env) == -1) {
            fprintf(stderr,"httpd: exec of %s failed, errno is %d\n",
                    SHELL_PATH,errno);
            exit(1);
        }
    }
    close(p[1]);
    if(!(f=fdopen(p[0],"r"))) {
        waitpid(ipid,NULL,0);
        return -1;
    }
    send_fd(f,out,kill_include_child);
    fclose(f);
    waitpid(ipid,NULL,0);
    return 0;
}


int handle_exec(FILE *in, FILE *out, char *file, char *path_args, char *args,
                char *error, char **env)
{
    char tag[MAX_STRING_LEN],errstr[MAX_STRING_LEN];
    char *tag_val;

    while(1) {
        if(!(tag_val = get_tag(in,tag,1)))
            return 1;
        if(!strcmp(tag,"cmd")) {
            if(include_cmd(tag_val,path_args,args,env,out) == -1) {
                sprintf(errstr,"invalid command exec %s in %s",tag_val,file);
                log_error_noclose(errstr);
                bytes_sent += fprintf(out,"%s",error);
            }
            /* just in case some stooge changed directories */
            chdir_file(file);
        } 
        else if(!strcmp(tag,"cgi")) {
            if(include_cgi(tag_val,path_args,args,env,out) == -1) {
                sprintf(errstr,"invalid CGI ref %s in %s",tag_val,file);
                log_error_noclose(errstr);
                bytes_sent += fprintf(out,"%s",error);
            }
            /* grumble groan */
            chdir_file(file);
        }
        else if(!strcmp(tag,"done"))
            return 0;
        else {
            char errstr[MAX_STRING_LEN];
            sprintf(errstr,"unknown parameter %s to tag echo in %s",tag,file);
            log_error_noclose(errstr);
            bytes_sent += fprintf(out,"%s",error);
        }
    }

}

int handle_config(FILE *in, FILE *out, char *file, char *error, char *tf,
                  int *sizefmt, char **env) {
    char tag[MAX_STRING_LEN];
    char *tag_val;

    while(1) {
        if(!(tag_val = get_tag(in,tag,0)))
            return 1;
        if(!strcmp(tag,"errmsg"))
            strcpy(error,tag_val);
        else if(!strcmp(tag,"timefmt")) {
            strcpy(tf,tag_val);
            replace_env_str(env, "DATE_LOCAL", ht_time(date,tf,0), out);
            replace_env_str(env, "DATE_GMT", ht_time(date,tf,1), out);
            replace_env_str(env, "LAST_MODIFIED", ht_time(lm,tf,0), out);
        }
        else if(!strcmp(tag,"sizefmt")) {
	    decodehtml(tag_val);
            if(!strcmp(tag_val,"bytes"))
                *sizefmt = SIZEFMT_BYTES;
            else if(!strcmp(tag_val,"abbrev"))
                *sizefmt = SIZEFMT_KMG;
        } 
        else if(!strcmp(tag,"done"))
            return 0;
        else {
            char errstr[MAX_STRING_LEN];
            sprintf(errstr,"unknown parameter %s to tag config in %s",
                    tag,file);
            log_error_noclose(errstr);
            bytes_sent += fprintf(out,"%s",error);
        }
    }
}



int find_file(FILE *out, char *file, char *directive, char *tag, 
              char *tag_val, struct stat *finfo, char *error)
{
    char errstr[MAX_STRING_LEN], dir[MAX_STRING_LEN], to_send[MAX_STRING_LEN];

    if(!strcmp(tag,"file")) {
        getparents(tag_val); /* get rid of any nasties */
        getwd(dir);
        make_full_path(dir,tag_val,to_send);
        if(stat(to_send,finfo) == -1) {
            sprintf(errstr,
                    "unable to get information about %s in parsed file %s",
                    to_send,file);
            log_error_noclose(errstr);
            bytes_sent += fprintf(out,"%s",error);
            return -1;
        }
        return 0;
    }
    else if(!strcmp(tag,"virtual")) {
        if(translate_name(tag_val,out) != STD_DOCUMENT) {
            bytes_sent += fprintf(out,"%s",error);
            log_error_noclose(errstr);
        }  
        else if(stat(tag_val,finfo) == -1) {
            sprintf(errstr,
                    "unable to get information about %s in parsed file %s",
                    to_send,file);
            log_error_noclose(errstr);
            bytes_sent += fprintf(out,"%s",error);
            return -1;
        }
        return 0;
    }
    else {
        sprintf(errstr,"unknown parameter %s to tag %s in %s",
                tag,directive,file);
        log_error_noclose(errstr);
        bytes_sent += fprintf(out,"%s",error);
        return -1;
    }
}


int handle_fsize(FILE *in, FILE *out, char *file, char *error, int sizefmt,
                 char **env) 
{
    char tag[MAX_STRING_LEN];
    char *tag_val;
    struct stat finfo;

    while(1) {
        if(!(tag_val = get_tag(in,tag,1)))
            return 1;
        else if(!strcmp(tag,"done"))
            return 0;
        else if(!find_file(out,file,"fsize",tag,tag_val,&finfo,error)) {
            if(sizefmt == SIZEFMT_KMG) {
                send_size(finfo.st_size,out);
                bytes_sent += 5;
            }
            else {
                int l,x;
                sprintf(tag,"%ld",finfo.st_size);
                l = strlen(tag); /* grrr */
                for(x=0;x<l;x++) {
                    if(x && (!((l-x) % 3))) {
                        fputc(',',out);
                        ++bytes_sent;
                    }
                    fputc(tag[x],out);
                    ++bytes_sent;
                }
            }
        }
    }
}

int handle_flastmod(FILE *in, FILE *out, char *file, char *error, char *tf,
                    char **env) 
{
    char tag[MAX_STRING_LEN];
    char *tag_val;
    struct stat finfo;

    while(1) {
        if(!(tag_val = get_tag(in,tag,1)))
            return 1;
        else if(!strcmp(tag,"done"))
            return 0;
        else if(!find_file(out,file,"flastmod",tag,tag_val,&finfo,error))
            bytes_sent += fprintf(out,"%s",ht_time(finfo.st_mtime,tf,0));
    }
}    



/* -------------------------- The main function --------------------------- */

/* This is a stub which parses a file descriptor. */

void send_parsed_content(char *file, FILE *f, FILE *fd, 
                         char *path_args, char *args,
                         char **env,int noexec)
{
    char directive[MAX_STRING_LEN], error[MAX_STRING_LEN], c;
    char timefmt[MAX_STRING_LEN], errstr[MAX_STRING_LEN];
    int ret, sizefmt;

    strcpy(error,DEFAULT_ERROR_MSG);
    strcpy(timefmt,DEFAULT_TIME_FORMAT);
    sizefmt = SIZEFMT_KMG;

    chdir_file(file);

    while(1) {
        if(!find_string(f,STARTING_SEQUENCE,fd)) {
            if(get_directive(f,directive))
                return;
            if(!strcmp(directive,"exec")) {
                if(noexec) {
                    sprintf(errstr,"httpd: exec used but not allowed in %s",
                            file);
                    log_error_noclose(errstr);
                    bytes_sent += fprintf(fd,"%s",error);
                    ret = find_string(f,ENDING_SEQUENCE,NULL);
                } else 
                    ret=handle_exec(f,fd,file,path_args,args,error,env);
            } 
            else if(!strcmp(directive,"config"))
                ret=handle_config(f,fd,file,error,timefmt,&sizefmt,env);
            else if(!strcmp(directive,"include"))
                ret=handle_include(f,fd,file,env,error);
            else if(!strcmp(directive,"echo"))
                ret=handle_echo(f,fd,file,error,env);
            else if(!strcmp(directive,"fsize"))
                ret=handle_fsize(f,fd,file,error,sizefmt,env);
            else if(!strcmp(directive,"flastmod"))
                ret=handle_flastmod(f,fd,file,error,timefmt,env);
            else {
                sprintf(errstr,"httpd: unknown directive %s in parsed doc %s",
                        directive,file);
                log_error_noclose(errstr);
                bytes_sent += fprintf(fd,"%s",error);
                ret=find_string(f,ENDING_SEQUENCE,NULL);
            }
            if(ret) {
                sprintf(errstr,"httpd: premature EOF in parsed file %s",file);
                log_error_noclose(errstr);
                return;
            }
        } else 
            return;
    }
}

/* Called by send_file */

void send_parsed_file(char *file, FILE *fd, char *path_args, char *args,
                      int noexec) 
{
    FILE *f;
    char **env;

    if(!(f=fopen(file,"r"))) {
        log_reason("file permissions deny server access",file);
        unmunge_name(file);
        die(FORBIDDEN,file,fd);
    }
    strcpy(content_type,"text/html");
    if(!assbackwards)
        send_http_header(fd);
    if(header_only)
        return;

    /* Make sure no children inherit our buffers */
    fflush(fd);
    assbackwards = 1; /* make sure no headers get inserted anymore */
    alarm(timeout);

    env = add_include_vars(in_headers_env,file,path_args,args,
                           DEFAULT_TIME_FORMAT,fd);
    env = add_common_vars(env,fd);

    send_parsed_content(file,f,fd,path_args,args,env,noexec);
    /*free_env(env);*/
}
