
/*-
 * Copyright (c) 1995 The Apache Group. All rights reserved.
 * 
 *
 * Apache httpd license
 * ====================
 * 
 *
 * This is the license for the Apache Server. It covers all the
 * files which come in this distribution, and should never be removed.
 * 
 * The "Apache Group" has based this server, called "Apache", on
 * public domain code distributed under the name "NCSA httpd 1.3".
 * 
 * NCSA httpd 1.3 was placed in the public domain by the National Center 
 * for Supercomputing Applications at the University of Illinois 
 * at Urbana-Champaign.
 * 
 * As requested by NCSA we acknowledge,
 * 
 *  "Portions developed at the National Center for Supercomputing
 *   Applications at the University of Illinois at Urbana-Champaign."
 *
 * Copyright on the sections of code added by the "Apache Group" belong
 * to the "Apache Group" and/or the original authors. The "Apache Group" and
 * authors hereby grant permission for their code, along with the
 * public domain NCSA code, to be distributed under the "Apache" name.
 * 
 * Reuse of "Apache Group" code outside of the Apache distribution should
 * be acknowledged with the following quoted text, to be included with any new
 * work;
 * 
 * "Portions developed by the "Apache Group", taken with permission 
 *  from the Apache Server   http://www.apache.org/apache/   "
 *
 *
 * Permission is hereby granted to anyone to redistribute Apache under
 * the "Apache" name. We do not grant permission for the resale of Apache, but
 * we do grant permission for vendors to bundle Apache free with other software,
 * or to charge a reasonable price for redistribution, provided it is made
 * clear that Apache is free. Permission is also granted for vendors to 
 * sell support for for Apache. We explicitly forbid the redistribution of 
 * Apache under any other name.
 * 
 * The "Apache Group" makes no promises to support "Apache". Users and
 * sellers of Apache support, and users of "Apache Group" code, should be 
 * aware that they use "Apache" or portions of the "Apache Group" code at 
 * their own risk. While every effort has been made to ensure that "Apache"
 * is robust and safe to use, we will not accept responsibility for any
 * damage caused, or loss of data or income which results from its use.
 * 
 */



/*
 * httpd.c: simple http daemon for answering WWW file requests
 *
 * 
 * Rob McCool 3/21/93
 * 
 */


#include "httpd.h"


void usage(char *bin) {
    fprintf(stderr,"Usage: %s [-d directory] [-f file] [-v]\n",bin);
    fprintf(stderr,"-d directory : specify an alternate initial ServerRoot\n");
    fprintf(stderr,"-f file : specify an alternate ServerConfigFile\n");
    exit(1);
}

void htexit(int status, FILE *out) {
    exit(status);
}

int sd;
pid_t pgrp;

void detach() {
    int x;

    chdir("/");
    if((x = fork()) > 0)
        exit(0);
    else if(x == -1) {
        fprintf(stderr,"httpd: unable to fork new process\n");
        perror("fork");
        exit(1);
    }
#ifndef NO_SETSID
    if((pgrp=setsid()) == -1) {
        fprintf(stderr,"httpd: setsid failed\n");
        perror("setsid");
        exit(1);
    }
#else

/* OS/2 dosen't support groups */
#ifndef __EMX__
    if((pgrp=setpgrp(getpid(),0)) == -1) {
        fprintf(stderr,"httpd: setpgrp failed\n");
        perror("setpgrp");
        exit(1);
    }
#endif    

#endif    
}

void sig_term() {
    log_error("httpd: caught SIGTERM, shutting down");
#ifndef NO_KILLPG
    killpg(pgrp,SIGKILL);
#else
    kill(-pgrp,SIGKILL);
#endif
    shutdown(sd,2);
    close(sd);
}

#if defined(BSD) || defined(__EMX__)
void ign() {
#ifndef NEXT
    int status;
#else
    union wait status;
#endif
    pid_t pid;
#ifdef __EMX__
    int rc;
    while( (rc = waitpid(-1, &status, WNOHANG)) > 0);
    /* Tried wait(), but cause process to block on call. */
    /* while( (pid = wait(&status)) > 0); */
#else               
    while( (pid = wait3(&status, WNOHANG, NULL)) > 0);
#endif    
}
#endif

void bus_error() {
    log_error("httpd: caught SIGBUS, dumping core");
    chdir(server_root);
    abort();         
}

void seg_fault() {
    log_error("httpd: caught SIGSEGV, dumping core");
    chdir(server_root);
    abort();
}

void set_signals();

/* Reset group privileges, after rereading the config files
 * (our uid may have changed, and if so, we want the new perms).
 *
 * Don't reset the uid yet --- we do that only in the child process,
 * so as not to lose any root privs.  But we can set the group stuff
 * now, once, and avoid the overhead of doing all this on each
 * connection.
 *
 * rst 23/Jan/95
 * Fixed to use the username as set in the config files, rather than
 * the lookup of to uid.
 * drtr 17/Feb/95
 */
  
static void set_group_privs()
{
  if(!getuid()) {
      char *name;
  
/* Get username if passed as a uid */
      if (user_name[0] == '#') {
      struct passwd* ent;
      uid_t uid=atoi(&user_name[1]);

      ent = getpwuid(uid);
      if (ent == NULL)
          die(SERVER_ERROR,"couldn't determine user name from uid",
          stdout);
      name = ent->pw_name;
      } else name = user_name;

/* OS/2 dosen't support groups */
#ifndef __EMX__
    /* Reset `groups' attributes. */
    
    if (initgroups(name, group_id) == -1)
      die(SERVER_ERROR,"unable to setgroups",stdout);

    if (setgid(group_id) == -1)
      die(SERVER_ERROR,"unable to change gid",stdout);
#endif      
  }
}

void restart() {
    log_error_noclose("httpd: caught SIGHUP, restarting");
    kill_mime();
    kill_security();
    kill_indexing();
#ifdef VIRTUAL_HOST
    close_virtual_logs();
    kill_virtual_host_config();
#endif /* VIRTUAL_HOST */
    if(server_hostname) {
        free(server_hostname);
        server_hostname = NULL;
    }
    init_nameserver_cache(error_log, -1);  /* flush cache */
    read_config(error_log);
    set_group_privs();
    close_logs();
    open_logs();
    log_error_noclose("httpd: successful restart");
    get_local_host();
    set_signals();
}

void set_signals() {
    signal(SIGSEGV,(void (*)())seg_fault);
    signal(SIGBUS,(void (*)())bus_error);
    signal(SIGTERM,(void (*)())sig_term);
    signal(SIGHUP,(void (*)())restart);

#if defined(BSD) || defined(__EMX__)
    signal(SIGCHLD,(void (*)())ign);
#else
    signal(SIGCHLD,SIG_IGN);
#endif
}

/* More idiot speed-hacking --- the first time conversion makes the C
 * library open the files containing the locale definition and time
 * zone.  If this hasn't happened in the parent process, it happens in
 * the children, once per connection --- and it does add up.
 *
 * rst 1/23/95
 */

void speed_hack_libs()
{
  time_t dummy_time_t = time(NULL);
  struct tm *dummy_time = localtime (&dummy_time_t);
  struct tm *other_dummy_time = gmtime (&dummy_time_t);
  char buf[MAX_STRING_LEN];

  strftime (buf, MAX_STRING_LEN, "%d/%b/%Y:%H:%M:%S", dummy_time);
}

void standalone_main() {
    int csd, clen,pid, one=1, cachefds[2];
#if defined(NEXT) || defined(BSD4_4) || defined(SOLARIS2) || defined(LINUX) || defined(__EMX__)
    struct sockaddr_in sa_server;
    struct sockaddr sa_client;
#else
    struct sockaddr_in sa_server,sa_client;
#endif

#ifdef __EMX__
    int emxign=1;
#ifdef __EMX__FORKFIX
    int emxsleep=1;
#endif    
#endif    

#ifndef MINIMAL_DNS
/* open pipe for returning ns cache data */
    if (pipe(cachefds) == -1)
    {
        perror("pipe");
        fprintf(stderr,"httpd: could not create pipe\n");
        exit(1);
    }
/* prefer posix-style non-blocking for the read and write ends */
#ifdef O_NONBLOCK
    if (fcntl(cachefds[0], F_SETFL, (int)O_NONBLOCK) == -1 ||
    fcntl(cachefds[1], F_SETFL, (int)O_NONBLOCK) == -1 )
#else
    if (fcntl(cachefds[0], F_SETFL, (int)O_NDELAY) == -1 ||
    fcntl(cachefds[1], F_SETFL, (int)O_NDELAY) == -1)
#endif
    {
    perror("fcntl");
    fprintf(stderr, "http: could not set non-blocking I/O on pipe\n");
    exit(1);
    }
    /* try setting close on exec, for safety */
    if (fcntl(cachefds[0], F_SETFD, (int)FD_CLOEXEC) == -1 ||
    fcntl(cachefds[1], F_SETFD, (int)FD_CLOEXEC) == -1)
    {
    perror("fcntl");
    fprintf(stderr, "http: could not set close-on-exec on pipe\n");
    exit(1);
    }

    init_nameserver_cache(stderr, cachefds[1]);
#endif /* MINIMAL_DNS */

    detach();

    if ((sd = socket(AF_INET,SOCK_STREAM,IPPROTO_TCP)) == -1) {
        fprintf(stderr,"httpd: could not get socket\n");
        perror("socket");
        exit(1);
    }

    if((setsockopt(sd,SOL_SOCKET,SO_REUSEADDR,(const char *)&one,sizeof(one)))
       == -1) {
        fprintf(stderr,"httpd: could not set socket option\n");
        perror("setsockopt");
        exit(1);
    }
    bzero((char *) &sa_server, sizeof(sa_server));
    sa_server.sin_family=AF_INET;
#ifdef VIRTUAL_HOST
    sa_server.sin_addr=bind_address;
#else /* VIRTUAL_HOST */
    sa_server.sin_addr.s_addr=htonl(INADDR_ANY);
#endif /* VIRTUAL_HOST */
    sa_server.sin_port=htons(port);
    if(bind(sd,(struct sockaddr *) &sa_server,sizeof(sa_server)) == -1) {
#ifdef VIRTUAL_HOST
    if (bind_address.s_addr != htonl(INADDR_ANY))
        fprintf(stderr,"httpd: could not bind to address %s port %d\n",
            inet_ntoa(bind_address), port);
    else
#endif /* VIRTUAL_HOST */
        fprintf(stderr,"httpd: could not bind to port %d\n",port);
        perror("bind");
        exit(1);
    }
    listen(sd,128);

    speed_hack_libs();
    set_signals();
    log_pid();

    while(1) {
      retry:
        clen=sizeof(sa_client);
        if((csd=accept(sd,&sa_client,&clen)) == -1) {
            if(errno == EINTR)  {
#if defined(BSD) || defined(__EMX__)
                ign();
#endif
                goto retry;
#ifdef __EMX__            
            } else {
                /* Added for debbugging of exact cause accept() failure. */
                /* fprintf(stderr,"Socket Accept Error: %i -> %s\n",errno,strerror(errno)); */
#endif                
            }
            log_error("socket error: accept failed");
            goto retry;
        }
        /* we do this here so that logs can be opened as root */
        if((pid = fork()) == -1)
            log_error("unable to fork new process");
        else if(!pid) {
            struct passwd* pwent;
#ifdef NOTDEF        
            struct linger sl;

            sl.l_onoff = 1;
            sl.l_linger = 600; /* currently ignored anyway */
            /* this should check error status, but it's not crucial */
            setsockopt(csd,SOL_SOCKET,SO_LINGER,&sl,sizeof(sl));
#endif        

            close(0);
            close(1);
            dup2(csd,0);
            dup2(csd,1);
#ifndef __EMX__
            /* Since EMX allows inheriting socket handles through a shared 
            pool of handles if we close it here the parent process will trap, 
            because of an invalid handle. */
            close(sd);

            /* EMX won't allow child processes to inherit handles with FD_CLOEXEC flag. */
            (void)fcntl(csd, F_SETFD, (int)FD_CLOEXEC);
#endif            

            remote_logname = (!do_rfc931 ? NULL :
                              rfc931((struct sockaddr_in *)&sa_client,
                                     &sa_server));

            /* Only try to switch if we're running as root */
            if(!getuid()) {
                /* Now, make absolutely certain we don't have any privileges
                 * except those mentioned in the configuration file. */
                /* Reset `groups' attribute. */

                /* Already set group privs after reading config file;
                 * now need just to reset uid.
                 */

                if (setuid(user_id) == -1)
                    die(SERVER_ERROR,"unable to change uid",stdout);
            }
            process_request(0,stdout);
#ifdef __EMX__FORKFIX
            /* EMX seems to have a problem balancing fork()ed processes.
            The first or second fork() will sometimes block until all other
            fork()s complete. This causes problems with clients that
            dynamically display the page. They must get at least the start of
            the image before they can calculate offsets for spacing other
            images and text. Putting a sleep() here seems to fix this problem,
            but slows server response. */
            if (emxsleep == __EMX__FORKFIX) {
                sleep(1); 
                fprintf(stderr,"Sleep\n");
            }
#endif
            fclose(stdin);
            fclose(stdout);
            shutdown(csd,2);
            close(csd); 
            exit(0);
        }
#ifdef __EMX__
        /* EMX requires that we read the status() of the forked process.
           Otherwise they will stackup and Apache will stop fork()ing. 
           But we can limit how often we call it. Refer to __EMX__IGNFIX
           in httpd.h */
        if (emxign == __EMX__IGNFIX) {
            ign();
            fprintf(stderr,"Ign\n");
        }
        /* Make sure we only call ign() every so many requests.
           As defined in emxfix.h */
        emxign++;
        if (emxign > __EMX__IGNFIX)
            emxign = 1;
                
#ifdef __EMX__FORKFIX
        emxsleep++;
        if (emxsleep > __EMX__FORKFIX)
            emxsleep = 1;
#endif            
#endif

        close(csd); 
#ifndef MINIMAL_DNS
        update_nameserver_cache(cachefds[0]); /* this does not block */
#endif                        
    }
}

extern char *optarg;
extern int optind;

main(int argc, char *argv[])
{
    int c, init;

    strcpy(server_root,HTTPD_ROOT);
    make_full_path(server_root,SERVER_CONFIG_FILE,server_confname);

    while((c = getopt(argc,argv,"d:f:v")) != -1) {
        switch(c) {
          case 'd':
            strcpy(server_root,optarg);
      make_full_path(server_root,SERVER_CONFIG_FILE,server_confname);
            break;
          case 'f':
            strcpy(server_confname,optarg);
            break;
          case 'v':
            printf("NCSA httpd version %s.\n",SERVER_VERSION);
            exit(1);
          case '?':
            usage(argv[0]);
        }
    }
    for (init=0; init <= RESPONSE_CODES; init++) 
        response_code_strings[init] = NULL;

#ifdef __EMX__
    printf("%s \n",SERVER_VERSION);
    printf("OS/2 port by Garey Smiley <garey@slink.com> \n");
#endif
        
    read_config(stderr);
    set_group_privs();
    open_logs();
    get_local_host();

    if(standalone)
        standalone_main();
    else {
        user_id = getuid();
        group_id = getgid();

        port = get_portnum(fileno(stdout),stdout);
        if(do_rfc931)
            remote_logname = get_remote_logname(stdout);
        process_request(0,stdout);
    }
    fclose(stdin);
    fclose(stdout);
    exit(0);
}
