/*
 * Demo file using the GraphIO library to import and display a graphics
 * file. It can also save the graphic in the metafile format courtesy of
 * the GraphIO library. Most of the code is standard for Windows programs
 * (Petzold's book has more details on that). The only interesting part
 * is the code to import/display/export the graphic.
 *
 * Written by Hippocrates Sendoukas, Los Angeles, September 1993
 */
#include <windows.h>
#include <commdlg.h>
#include <string.h>
#include "graphio.h"
#include "grview.h"

/*
 * The variable "scalemethod" controls the scaling of the metafile. If it
 * is equal to USEINCHES, the program will use the bounding box info as
 * well as the "inch" field in the matafile to determine its extents. If
 * scalemethod is equal to USEPIXELS, the program uses only the bounding
 * box and disregards the "inch" field; this is useful when the metafile
 * contains a bitmap; in that case, the bounding box dimensions are simply
 * the bitmap dimensions (in pixels), and the inch field is irrelevant.
 * If scalemethod is equal to TOWINDOW, the program ignores the bounding
 * box and the inch field, and scales the metafile to fit in the window.
 * This is the only possible method for standard (plain) metafiles.
 */
#define	USEINCHES	0
#define	USEPIXELS	1
#define	TOWINDOW	2
static	int		scalemethod = USEINCHES;

/*
 * If the metafile is large enough, we need to use the scrollbars to view
 * different portions of the graphic. The most efficient method is to scroll
 * the appropriate part of the window, and then paint only the newly uncovered
 * part. This works well with most metafiles, but it can produce incorrect
 * results under certain circumstances. If for example the metafile contains
 * a shading instruction and the graphics adapter does not support enough
 * colors, Windows tries to approximate the requested color by dithering, but
 * it cannot do the right thing since it paints only a small portion of the
 * metafile. For this reason, the variable "full_updates" controls the
 * screen update method when scrolling the window; if it is equal to zero,
 * then incremental updating is used; if it is non-zero, the entire window
 * is repainted for every movement; this is noticeably slower, but the
 * output will be correct. The program uses incremental updating by default;
 * if the display is incorrect, you can simply change the method from the
 * "Screen" submenu.
 */
static	int		full_updates = 0;

static	int		windowx,windowy;	/* dimensions of client area */
static	int		metax,metay;		/* desired dimensions of the */
						/* metafile (in pixels) */
static	int		maxx,maxy,curx,cury;
static	PICTINFO	curmf;
static	char		*progname = "Graphics Viewer";
/*  <- Msg */

static void
Msg(HWND hwnd,char *s)
{
  MessageBox(hwnd,s,progname,MB_ICONSTOP|MB_OK);
}

/*  -> Msg */
/*  <- do_size */

static void
do_size(HWND hwnd,int nx,int ny)
{
  HDC hdc;
  long temp;

  windowx  = nx;
  windowy  = ny;
  if (curmf.inch && scalemethod!=TOWINDOW)
    {
      hdc   = GetDC(hwnd);
      temp  = curmf.bbox.right - curmf.bbox.left;
      if (scalemethod==USEINCHES)
        {
          temp *= GetDeviceCaps(hdc,LOGPIXELSX);
          temp /= curmf.inch;
        }
      metax = (int) temp;
      temp  = curmf.bbox.bottom - curmf.bbox.top;
      if (scalemethod==USEINCHES)
        {
          temp *= GetDeviceCaps(hdc,LOGPIXELSY);
          temp /= curmf.inch;
        }
      metay = (int) temp;
      ReleaseDC(hwnd,hdc);
    }
  else
    {
      metax = windowx;
      metay = windowy;
    }
  maxx	= max(0,metax-windowx);
  maxy	= max(0,metay-windowy);
  curx	= min(curx,maxx);
  SetScrollRange(hwnd,SB_HORZ,0,maxx,FALSE);
  SetScrollPos(hwnd,SB_HORZ,curx,TRUE);
  cury	= min(cury,maxy);
  SetScrollRange(hwnd,SB_VERT,0,maxy,FALSE);
  SetScrollPos(hwnd,SB_VERT,cury,TRUE);
}

/*  -> do_size */
/*  <- do_save */

static void
do_save(HWND hwnd)
{
  int		doplace;
  OPENFILENAME	ofn;
  char		mask[60],buf[256];

  if (curmf.hmf==0)
    {
      MessageBeep(0);
      return;
    }
  doplace = (curmf.inch != 0);
  memcpy(mask,"Placeable Metafile\0*.wmf\0Standard Metafile\0*.wmf\0",50);
  buf[0] = '\0';
  memset(&ofn,0,sizeof(ofn));
  ofn.lStructSize	= sizeof(ofn);
  ofn.hwndOwner		= hwnd;
  ofn.lpstrFile		= buf;
  ofn.nMaxFile		= sizeof(buf);
  ofn.lpstrFilter	= mask;
  ofn.nFilterIndex	= (doplace) ? 1 : 2;
  ofn.lpstrDefExt	= "wmf";
  ofn.Flags		= OFN_HIDEREADONLY | OFN_PATHMUSTEXIST |
                          OFN_OVERWRITEPROMPT;
  if (!GetSaveFileName(&ofn)) return;
  if (doplace)  doplace = (ofn.nFilterIndex == 1);
  switch (save_meta(&curmf,0,buf,doplace))
    {
      case GREXP_NOMEM:
        Msg(hwnd,"Out of memory");
        break;
      case GREXP_NODISC:
        Msg(hwnd,"Out of disc space");
        break;
      case GREXP_BADFNAME:
        Msg(hwnd,"Cannot open the requested file");
        break;
      case GREXP_BADPARMS:
        Msg(hwnd,"Internal error: bad parameters");
        break;
    }
}

/*  -> do_save */
/*  <- do_open */

static void
do_open(HWND hwnd)
{
  int		n;
  HDC		hdc;
  OPENFILENAME	ofn;
  char		buf[256],mask[2048];
  static DWORD	filtind = 1;
  static char	*inisection = "MS Graphic Import Filters";

  get_openmask(mask,sizeof(mask),inisection);
  buf[0] = '\0';
  memset(&ofn,0,sizeof(ofn));
  ofn.lStructSize	= sizeof(ofn);
  ofn.hwndOwner		= hwnd;
  ofn.lpstrFile		= buf;
  ofn.nMaxFile		= sizeof(buf);
  ofn.lpstrFilter	= mask;
  ofn.lpstrDefExt	= "";
  ofn.nFilterIndex	= filtind;
  ofn.Flags		= OFN_HIDEREADONLY | OFN_FILEMUSTEXIST |
			  OFN_PATHMUSTEXIST;
  if (!GetOpenFileName(&ofn)) return;
  filtind = ofn.nFilterIndex;
  if (curmf.hmf) DeleteMetaFile(curmf.hmf);
  memset(&curmf,0,sizeof(curmf));
  SetWindowText(hwnd,progname);
  hdc = GetDC(hwnd);
  n = import_graphic(hdc,hwnd,buf,&curmf,0,inisection);
  ReleaseDC(hwnd,hdc);
  switch (n)
    {
      case GRIMP_OK:
        do_size(hwnd,windowx,windowy);
        SetWindowText(hwnd,buf);
        break;
      case GRIMP_NOMEM:
        Msg(hwnd,"Out of memory");
        break;
      case GRIMP_NOFILE:
        Msg(hwnd,"Cannot open the requested file");
        break;
      case GRIMP_NOTYPE:
        Msg(hwnd,"Cannot find any filter for this type of file");
        break;
      case GRIMP_NOFILTER:
        Msg(hwnd,"Cannot find the filter specified in win.ini");
        break;
      case GRIMP_NOCONV:
        Msg(hwnd,"The filter cannot import the file");
        break;
      case GRIMP_NOINIFILT:
        Msg(hwnd,"Cannot initialize the graphics filter");
        break;
      default:
        Msg(hwnd,"Internal error");
    }
  InvalidateRect(hwnd,NULL,TRUE);
}

/*  -> do_open */
/*  <- do_paint */

static void
do_paint(HWND hwnd)
{
  HDC		hdc;
  HPEN		hp;
  PAINTSTRUCT	ps;

  hdc = BeginPaint(hwnd,&ps);
  if (curmf.hmf)
    {
      /*
       * Put a border around the metafile so we can see
       * the dimensions easily.
       */
      hp = SelectObject(hdc,CreatePen(PS_SOLID,1,
                                      GetSysColor(COLOR_WINDOWTEXT)));
      MoveTo(hdc,0,metay-cury);
      LineTo(hdc,metax-curx,metay-cury);
      LineTo(hdc,metax-curx,-cury);
      DeleteObject(SelectObject(hdc,hp));

      /*
       * Display the metafile
       */
      SetMapMode(hdc,MM_ANISOTROPIC);
      if (curmf.inch>0)
        {
          /*
           * This is required for some stupid metafiles that do not
           * set the window origin and extents. If the metafile is
           * ok, it will override them
           */
          SetWindowOrg(hdc,curmf.bbox.left,curmf.bbox.top);
          SetWindowExt(hdc,curmf.bbox.right-curmf.bbox.left,
                           curmf.bbox.bottom-curmf.bbox.top);
        }
      SetViewportExt(hdc,metax,metay);
      SetViewportOrg(hdc,-curx,-cury);
      PlayMetaFile(hdc,curmf.hmf);
    }
  EndPaint(hwnd,&ps);
}

/*  -> do_paint */
/*  <- do_scroll */

static int
do_scroll(WORD wparam,LONG lparam,int curv,int maxv,int window)
{
  int delta;

  delta = 0;
  switch (wparam)
    {
      case SB_LINEUP:	delta = -1;		break;
      case SB_LINEDOWN:	delta = 1;		break;
      case SB_PAGEUP:	delta = -window/2;	break;
      case SB_PAGEDOWN:	delta = window/2;	break;
      case SB_THUMBPOSITION: delta = LOWORD(lparam)-curv; break;
    }
  return max(-curv,min(delta,maxv-curv));
}

/*  -> do_scroll */
/*  <- do_info */

static void
do_info(HWND hwnd)
{
  char *title,temp[256];

  if (curmf.hmf==0)
    {
      MessageBeep(0);
      return;
    }
  if (curmf.inch)
    {
      title = "Placeable Metafile";
      wsprintf(temp,"Bounding Box:\t%d %d %d %d\n"
                    "Inch:\t\t%d\n"
                    "Pixel dimen.:\t%d x %d",
                    curmf.bbox.left,curmf.bbox.top,
                    curmf.bbox.right,curmf.bbox.bottom,
                    (int)curmf.inch,metax,metay);
    }
  else
    {
      title = "Standard metafile";
      wsprintf(temp,"Window size: %d x %d",metax,metay);
    }
  MessageBox(hwnd,temp,title,MB_OK);
}

/*  -> do_info */
/*  <- put_checkmarks */

static void
put_checkmarks(HWND hwnd)
{
  HMENU h;

  h = GetMenu(hwnd);
  CheckMenuItem(h,IDM_INCH,MF_UNCHECKED|MF_BYCOMMAND);
  CheckMenuItem(h,IDM_PIXELS,MF_UNCHECKED|MF_BYCOMMAND);
  CheckMenuItem(h,IDM_WINDOW,MF_UNCHECKED|MF_BYCOMMAND);
  CheckMenuItem(h,IDM_INCH+scalemethod,MF_CHECKED|MF_BYCOMMAND);
  CheckMenuItem(h,IDM_INCR,MF_UNCHECKED|MF_BYCOMMAND);
  CheckMenuItem(h,IDM_FULL,MF_UNCHECKED|MF_BYCOMMAND);
  CheckMenuItem(h,IDM_INCR+full_updates,MF_CHECKED|MF_BYCOMMAND);
}

/*  -> put_checkmarks */
/*  <- WndProc */

LONG FAR PASCAL
WndProc(HWND hwnd,WORD msg,WORD wparam,LONG lparam)
{
  int delta;

  switch (msg)
    {
/*  <-       WM_CREATE */

      case WM_CREATE:
        metax = metay = curx = cury = 0;
        memset(&curmf,0,sizeof(curmf));
        put_checkmarks(hwnd);
        break;
/*  ->       WM_CREATE */
/*  <-       WM_SIZE */

      case WM_SIZE:
        do_size(hwnd,LOWORD(lparam),HIWORD(lparam));
        return 0;
/*  ->       WM_SIZE */
/*  <-       WM_HSCROLL */

      case WM_HSCROLL:
        if ( (delta=do_scroll(wparam,lparam,curx,maxx,windowx)) != 0 )
          {
            curx += delta;
            SetScrollPos(hwnd,SB_HORZ,curx,TRUE);
            if (full_updates)	InvalidateRect(hwnd,NULL,TRUE);
            else		ScrollWindow(hwnd,-delta,0,NULL,NULL);
            UpdateWindow(hwnd);
          }
        return 0;
/*  ->       WM_HSCROLL */
/*  <-       WM_VSCROLL */

      case WM_VSCROLL:
        if ( (delta=do_scroll(wparam,lparam,cury,maxy,windowy)) != 0 )
          {
            cury += delta;
            SetScrollPos(hwnd,SB_VERT,cury,TRUE);
            if (full_updates)	InvalidateRect(hwnd,NULL,TRUE);
            else		ScrollWindow(hwnd,0,-delta,NULL,NULL);
            UpdateWindow(hwnd);
          }
        return 0;
/*  ->       WM_VSCROLL */
/*  <-       WM_COMMAND */

      case WM_COMMAND:
        switch (wparam)
          {
            case IDM_OPEN:
              do_open(hwnd);
              return 0;
            case IDM_SAVE:
              do_save(hwnd);
              return 0;
            case IDM_EXIT:
              PostMessage(hwnd,WM_CLOSE,0,0);
              return 0;
            case IDM_INCH:
            case IDM_PIXELS:
            case IDM_WINDOW:
              delta = wparam-IDM_INCH;
              if (delta != scalemethod)
                {
                  scalemethod = delta;
                  put_checkmarks(hwnd);
                  do_size(hwnd,windowx,windowy);
                  InvalidateRect(hwnd,NULL,TRUE);
                }
              return 0;
            case IDM_INCR:
            case IDM_FULL:
              delta = wparam-IDM_INCR;
              if (delta != full_updates)
                {
                  full_updates = delta;
                  put_checkmarks(hwnd);
                  InvalidateRect(hwnd,NULL,TRUE);
                }
              return 0;
            case IDM_INFO:
              do_info(hwnd);
              return 0;
          }
        break;
/*  ->       WM_COMMAND */
/*  <-       WM_PAINT */

      case WM_PAINT:
        do_paint(hwnd);
        return 0;
/*  ->       WM_PAINT */
/*  <-       WM_CLOSE */

      case WM_CLOSE:
        DestroyWindow(hwnd);
        return 0;
/*  ->       WM_CLOSE */
/*  <-       WM_DESTROY */

      case WM_DESTROY:
        if (curmf.hmf) DeleteMetaFile(curmf.hmf);
        PostQuitMessage(0);
        return 0;
/*  ->       WM_DESTROY */
    }
  return DefWindowProc(hwnd,msg,wparam,lparam);
}

/*  -> WndProc */
/*  <- WinMain */

#pragma argsused
int PASCAL
WinMain(HANDLE hinstance,HANDLE hprevinst,LPSTR cmdline,int cmdshow)
{
  MSG		msg;
  WNDCLASS	wc;
  HWND		hwnd;
  HANDLE	haccel;

  wc.lpszClassName	= progname;
  wc.hInstance		= hinstance;
  wc.lpfnWndProc	= WndProc;
  wc.hCursor		= LoadCursor(0,IDC_ARROW);
  wc.hIcon		= LoadIcon(hinstance,"grview");
  wc.lpszMenuName	= "grview";
  wc.hbrBackground	= COLOR_WINDOW+1;
  wc.style		= CS_HREDRAW | CS_VREDRAW;
  wc.cbClsExtra		= 0;
  wc.cbWndExtra		= 0;
  RegisterClass(&wc);

  hwnd = CreateWindow(progname,progname,
               WS_OVERLAPPEDWINDOW | WS_HSCROLL | WS_VSCROLL,
               0,0,CW_USEDEFAULT,CW_USEDEFAULT,0,0,hinstance,NULL);
  ShowWindow(hwnd,cmdshow);
  UpdateWindow(hwnd);
  haccel = LoadAccelerators(hinstance,"grview");
  while (GetMessage(&msg,0,0,0))
    if (!TranslateAccelerator(hwnd,haccel,&msg))
      {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
      }
  return 0;
}
/*  -> WinMain */
