/*  <- declarations */
/*
 * All filters can use the routines below. This file provides reasonable
 * substitutes for the file and memory allocation functions of the C
 * library. There are two advantages to these functions: they use the
 * Windows API, so the code size is small, and they are fully reentrant.
 * If you use standard C functions, make sure that they don't use static
 * data; the results could be disastrous under the shared data model of
 * Windows 3.x
 *
 * Written by Hippocrates Sendoukas, Los Angeles, September 1993
 */
#include <windows.h>
#include <string.h>
#include "graphio.h"
#include "filter.h"
#ifndef _WIN32_
#  define ___16_BIT___	1	/* Don't define this on a 32-bit compiler */
#endif
typedef long		Int32;
typedef short		Int16;
typedef signed char	Int8;
typedef unsigned long	Uns32;
typedef unsigned short	Uns16;
typedef unsigned char	Uns8;

#define BUFFERSIZE	8192

typedef struct
{
  int		fh,bufptr,buflen,eof;
  Uns8		buf[BUFFERSIZE];
} MYFILE, FAR *LPMYFILE;

#define	EOF			(-1)
#define	SEEK_SET		0
#define	SEEK_CUR		1
#define	SEEK_END		2
#define malloc(n)		mymalloc(n)
#define free(p)			myfree(p)
#define fopen(name,mode)	myfopen(name)
#define fclose(f)		myfclose(f)
#define getc(f)			mygetc(f)
#define fseek(f,ofs,whence)	myfseek(f,ofs,whence)
#define fread(buf,sz,n,f)	myread(buf,(sz)*(n),f)
#define feof(f)			( (f)->eof != 0 )
#define FILE			MYFILE
#define readUns8(f)		( (Uns8)getc(f) )
/*  -> declarations */
/*  <- mymalloc */

static void FAR * PASCAL NEAR
mymalloc(unsigned long size)
{
  HANDLE	h;
  LPSTR		p;

  if ( (h=GlobalAlloc(GMEM_MOVEABLE,size)) == 0 ) return NULL;
  if ( (p=GlobalLock(h)) == NULL ) GlobalFree(h);
  return p;
}

/*  -> mymalloc */
/*  <- myfree */

static void PASCAL NEAR
myfree(void FAR *p)
{
  HANDLE h;

  h = LOWORD(GlobalHandle(HIWORD(p)));
  GlobalUnlock(h);
  GlobalFree(h);
}

/*  -> myfree */
/*  <- mygetc */

static int PASCAL NEAR
mygetc(LPMYFILE f)
{
  if (f->bufptr>=f->buflen)
    {
      f->bufptr = 0;
      if ( (f->buflen = _lread(f->fh,(LPSTR)f->buf,BUFFERSIZE)) <= 0 )
        {
          f->eof = 1;
          return EOF;
        }
    }
  return f->buf[(f->bufptr)++];
}

/*  -> mygetc */
/*  <- myread */

static int PASCAL NEAR
myread(void *buf,unsigned requested,LPMYFILE f)
{
  unsigned available;

  available = f->buflen - f->bufptr;
  if (available>=requested)
    {
      memcpy(buf,f->buf + f->bufptr,requested);
      f->bufptr += requested;
      return requested;
    }
  else
    {
      if (available)
        {
          memcpy(buf,f->buf + f->bufptr, available);
          f->bufptr += available;
          requested -= available;
          ((LPSTR)buf) += available;
        }
      if (requested>=BUFFERSIZE)
        {
          if ( _lread(f->fh,buf,requested) != requested )
            {
              f->eof = 1;
              return EOF;
            }
        }
      else
        {
          if (  (f->buflen = _lread(f->fh,(LPSTR)f->buf,BUFFERSIZE))  <
                (int)requested  )
            {
              f->bufptr = f->buflen;
              f->eof = 1;
              return EOF;
            }
          memcpy(buf,f->buf,requested);
          f->bufptr = requested;
        }
      return requested + available;
    }
}

/*  -> myread */
/*  <- myfseek */

static long PASCAL NEAR
myfseek(LPMYFILE f,long offset,int fromwhere)
{
  long l;

  f->bufptr = f->buflen = 0;
  f->eof =  ( (l=_llseek(f->fh,offset,fromwhere)) == -1L );
  return l;
}

/*  -> myfseek */
/*  <- myfopen */

static LPMYFILE PASCAL NEAR
myfopen(LPSTR fname)
{
  LPMYFILE	f;
  OFSTRUCT	of;

  if ( (f=malloc(sizeof(MYFILE))) == NULL )  return NULL;
  if ( (f->fh=OpenFile(fname,&of,OF_READ)) < 0 )
    {
      free(f);
      return NULL;
    }
  f->bufptr = f->buflen = f->eof = 0;
  return f;
}

/*  -> myfopen */
/*  <- myfclose */

static void PASCAL NEAR
myfclose(LPMYFILE f)
{
  _lclose(f->fh);
  free(f);
}

/*  -> myfclose */
/*  <- readUns16 */

static Uns16 PASCAL NEAR
readUns16(FILE *fp)
{
  register Uns16 l;

  l = getc(fp);
  return( ((Uns16)getc(fp) << 8 ) | l );
}

/*  -> readUns16 */
/*  <- readUns32 */

static Uns32 PASCAL NEAR
readUns32(FILE *fp)
{
  unsigned	h,l;

  l	 = getc(fp);
  l	|= (unsigned)getc(fp) << 8;
  h	 = getc(fp);
  h	|= (unsigned)getc(fp) << 8;
  return( l | ((Uns32)h << 16) );
}

/*  -> readUns32 */
/*  <- make_palette */

static HPALETTE PASCAL NEAR
make_palette(unsigned colors,RGBQUAD FAR *rgb)
{
  unsigned		i;
  HPALETTE		hpal;
  LPPALETTEENTRY	pe;
  LPLOGPALETTE		pal;

  if ( (pal=malloc(sizeof(LOGPALETTE)+colors*sizeof(PALETTEENTRY)))==NULL)
    return 0;
  pal->palVersion = 0x0300;
  pal->palNumEntries = colors;
  pe = &(pal->palPalEntry[0]);
  for (i=0; i<colors; i++,pe++,rgb++)
    {
      pe->peRed		= rgb->rgbRed;
      pe->peGreen	= rgb->rgbGreen;
      pe->peBlue	= rgb->rgbBlue;
      pe->peFlags	= PC_NOCOLLAPSE;
    }
  hpal = CreatePalette(pal);
  free(pal);
  return hpal;
}

/*  -> make_palette */
