{$A+,B-,D-,E-,F-,I-,L-,N-,O+,R-,S-,V-}
(******************************************************************************)
(*                      Hydra Bi-directional Protocol                         *)
(*                                                   *)
(*                                                                            *)
(*                      CRC 16/32 Calculation Routines                        *)
(*                                                                            *)
(* BY: Adam Blake                                      Wandoo Valley Software *)
(*     Arjen Lentz                                         and Lentz Software *)
(* VER: 1.00                                                      Development *)
(* DATE: 5th August 1993                                   (c) Copyright 1993 *)
(* LANGUAGE: Turbo Pascal v6.0                  All Rights Reserved Worldwide *)
(******************************************************************************)
Unit Crc;
Interface

type
  Crc16Table     = ^Crc16TableType;
  Crc16TableType = array[0..255] of word;

  Crc32Table     = ^Crc32TableType;
  Crc32TableType = array[0..255] of longint;


const
(* CRC-16 used by ARC and LH, using Crc16___ routines                         *)
  CRC16A_POLY      = $A001;              (* Generator polynomial number       *)
  CRC16A_INIT      = $0;                 (* Initial CRC value for calculation *)
  CRC16A_TEST      = $0;                 (* Result to test for at receiver    *)

(* CRC-16 CCITT proper                                                        *)
  CRC16_POLY       = $8408;              (* Generator polynomial number       *)
  CRC16_INIT       = $FFFF;              (* Initial CRC value for calculation *)
  CRC16_TEST       = $F0B8;              (* Result to test for at receiver    *)

(* CRC-16 CCITT upside-down                                                   *)
  CRC16R_POLY      = $1021;              (* Generator polynomial number       *)
  CRC16R_INIT      = $0;                 (* Initial CRC value for calculation *)
  CRC16R_TEST      = $0;                 (* Result to test for at receiver    *)

(* CRC-32 CCITT                                                               *)
  CRC32_POLY      = $EDB88320;          (* Generator polynomial number        *)
  CRC32_INIT      = $FFFFFFFF;          (* Initial CRC value for calculation  *)
  CRC32_TEST      = $DEBB20E3;          (* Result to test for at receiver     *)

(* Number of items in CRC table                                               *)
  CRC_TABSIZE = 256;                    (* Normal 256-entry table             *)


(*** CRC-16 proper, used for both CCITT and the one used by ARC             ***)
Procedure Crc16Init( CrcTab : CRC16Table; Poly : word );
Function  Crc16Block( CrcTab : CRC16Table; Crc : word; Buf : pointer; Len : word ) : word;
Function  Crc16Upd( CrcTab : CRC16Table; Crc : word; Buf : pointer; Ch : Byte ) : word;
Function  Crc16Post( Crc : word ) : word;

(*** CRC-16 upside-down, transmitted high-byte first                        ***)
Procedure Crc16RInit( CrcTab : CRC16Table; Poly : word );
Function  Crc16RBlock( CrcTab : CRC16Table; Crc : word; Buf : pointer; Len : word ) : word;
Function  Crc16RUpd( CrcTab : CRC16Table; Crc : word; Buf : pointer; Ch : Byte ) : word;
Function  Crc16RPost( Crc : word ) : word;

(*** CRC-32                                                                 ***)
Procedure Crc32Init( CrcTab : CRC32Table; Poly : longint );
Function  Crc32Block( CrcTab : CRC32Table; Crc : longint; Buf : pointer; Len : word ) : longint;
Function  Crc32Upd( CrcTab : CRC32Table; Crc : longint; Ch : Byte ) : longint;
Function  Crc32Post( Crc : longint ) : longint;

(******************************************************************************)
Implementation

{$L CRC.OBJ}
Procedure  Crc16Init( CrcTab : CRC16Table; Poly : word ); external;
Function   Crc16Block( CrcTab : CRC16Table; Crc : word; Buf : pointer; Len : word ) : word; external;
Procedure  Crc16RInit( CrcTab : CRC16Table; Poly : word ); external;
Function   Crc16RBlock( CrcTab : CRC16Table; Crc : word; Buf : pointer; Len : word ) : word; external;
Procedure  Crc32Init( CrcTab : CRC32Table; Poly : longint ); external;
Function   Crc32Block( CrcTab : CRC32Table; Crc : longint; Buf : pointer; Len : word ) : longint; external;


Function Crc16APost( Crc : word ) : word;
begin
  Crc16APost := (not Crc);              (* CRC Postconditioning before xmit  *)
end;


Function Crc16Upd( CrcTab : CRC16Table; Crc : word; Buf : pointer; Ch : Byte ) : word;
begin
  Crc16Upd := CrcTab^[(Crc XOR Ch) AND $FF] XOR (Crc SHR 8);
end;


Function Crc16Post( Crc : word ) : word;
begin
  Crc16Post := (not Crc);               (* CRC Postconditioning before xmit  *)
end;


Function Crc16RUpd( CrcTab : CRC16Table; Crc : word; Buf : pointer; Ch : Byte ) : word;
begin
  Crc16RUpd := CrcTab^[((Crc SHR 8) XOR Ch) AND $FF] XOR (Crc SHL 8);
end;


Function Crc16RPost( Crc : word ) : word;
begin
  Crc16RPost := Crc;                    (* CRC Postconditioning before xmit  *)
end;


Function Crc32Upd( CrcTab : CRC32Table; Crc : longint; Ch : Byte ) : longint;
begin
  Crc32Upd := CrcTab^[(Crc XOR Ch) AND $FF] XOR (Crc SHR 8);
end;


Function Crc32Post( Crc : longint ) : longint;
begin
  Crc32Post := (not Crc);               (* CRC Postconditioning before xmit  *)
end;

(**********************************MAINLINE************************************)

end.

