/*
** Exception Library -- General exception handling for ANSI C programs
** 
** Copyright (C) 1992 Computational Vision and Active Perception Lab. (CVAP),
**                    Royal Institute of Technology, Stockholm.
**
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public
** License as published by the Free Software Foundation; either
** version 2 of the License, or (at your option) any later version.
** 
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
** 
** You should have received a copy of the GNU Library General Public
** License along with this library (see COPYING-LIB); if not, write to 
** the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, 
** USA.
** 
**                            Written by
**                                 
**                  Harald Winroth, Matti Rendahl
**      Computational Vision and Active Perception Laboratory
**                  Royal Institute of Technology
**                        S-100 44 Stockholm
**                              Sweden
**                                 
** Report bugs to candela-bug@bion.kth.se, and direct all inquiries to 
** candela@bion.kth.se.
**
*/

#include <stdio.h>

#include <exception/exc-sig-macros.h>

int exc_sig_type;

/*
 * Let _EXC_SIG expand to a declaration AND a definition
 */

#undef _EXC_SIG
#define _EXC_SIG(LIST) 							      \
    _EXC_SIG_DECLARE(LIST);						      \
    _EXC_SIG_DEFINE(LIST);						      \
    _EXC_SIG_HANDLER(LIST)						      \

#include <exception/exc-sig.h> /* exc-sig-macros.h will not be read again */

char *exc_sig_name (excSig sig)
{
    return sig.name;
}

int exc_sig_number (excSig sig)
{
    return sig.number;
}

int exc_sig_exception_handler (void *e, void *e_type, void *h_data)
{
    excSig es;

    if (!EXC_IN_DOMAIN (e_type, exc_sig_type))
	exc_rethrow ();

    es = *(excSig *)e;

    fprintf (stderr, "\nexc_sig: Received %s (signal %d)\n", 
	     exc_sig_name (es), exc_sig_number (es));

    return 1;
}

excSigHandler exc_signal (int sig)
{
    static int exception_handler_installed = 0;

    if (!exception_handler_installed)
    {
	EXC_INSTALL_HANDLER (exc_any, exc_sig_exception_handler, NULL);
	exception_handler_installed = 1;
    }

#if defined(SVR4) || defined(SYSV) || defined(_SYSV_)
   {
       
       sigset_t mask;
       struct sigaction act, old_act;

       if (sigemptyset (&mask))
	   exc_fatal ("exc_signal: sigemptyset() failed.");

       act.sa_handler = exc_sig_handler;
       act.sa_mask = mask;
       act.sa_flags = SA_NODEFER | SA_SIGINFO;

       if (sigaction (sig, &act, &old_act))
	   exc_fatal ("exc_signal: sigaction() failed.");

       return old_act.sa_handler;
   }
#else
    return signal (sig, exc_sig_handler);
#endif
}

#if defined(SVR4) || defined(SYSV) || defined(_SYSV_)
void exc_signals (sigset_t *set)
#define SIG_IS_MEMBER(SET, SIG) sigismember (SET, SIG)
#else
void exc_signals (int set)
#define SIG_IS_MEMBER(SET, SIG) ((SET) & sigmask (SIG))
#endif
{
  /*
   * This is not completely portable, but neither is the rest of the 
   * exception/signal stuff here...
   */

  int nsignals = sizeof(exc_sig)/sizeof(excSig);
  excSig *arr = (excSig *) &exc_sig;
  int i;

  for (i=0; i<nsignals; i++)
    if (SIG_IS_MEMBER (set, arr[i].number))
	(void) exc_signal (arr[i].number);
}

#undef SIG_IS_MEMBER

