/*
** Exception Library -- General exception handling for ANSI C programs
** 
** Copyright (C) 1992 Computational Vision and Active Perception Lab. (CVAP),
**                    Royal Institute of Technology, Stockholm.
**
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public
** License as published by the Free Software Foundation; either
** version 2 of the License, or (at your option) any later version.
** 
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
** 
** You should have received a copy of the GNU Library General Public
** License along with this library (see COPYING-LIB); if not, write to 
** the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, 
** USA.
** 
**                            Written by
**                                 
**                  Harald Winroth, Matti Rendahl
**      Computational Vision and Active Perception Laboratory
**                  Royal Institute of Technology
**                        S-100 44 Stockholm
**                              Sweden
**                                 
** Report bugs to candela-bug@bion.kth.se, and direct all inquiries to 
** candela@bion.kth.se.
**
*/

#ifndef _exception_exc_sig_h
#define _exception_exc_sig_h

#include <signal.h>

#if defined(SVR4) || defined(SYSV) || defined(_SYSV_)
#include <siginfo.h>
#include <ucontext.h>
#endif

#include <exception/exception.h>
#include <exception/exc-sig-macros.h>

/*
 * The _EXC_SIG_HANDLER_ARGS definition should match the system's declaration
 * of signal handler arguments. The SIG_ARG parameter is the name of the 
 * formal parameter that should contain the signal number (it will be used in 
 * the body of the handler).
 */

#if defined(__sun__) /* SunOS 4 and 5 */
#define _EXC_SIG_HANDLER_ARGS(SIG_ARG_NAME)				      \
    int SIG_ARG_NAME
#elif defined(sgi) /* Silicon Graphics IRIX */
#define _EXC_SIG_HANDLER_ARGS(SIG_ARG_NAME)				      \
    int SIG_ARG_NAME
#elif defined(SVR4) || defined(SVR4) || defined(SYSV) || defined(_SYSV_)
#define _EXC_SIG_HANDLER_ARGS(SIG_ARG_NAME)				      \
    int SIG_ARG_NAME, siginfo_t *infop, ucontext_t *ucp
#else
#define _EXC_SIG_HANDLER_ARGS(SIG_ARG_NAME)				      \
    int SIG_ARG_NAME, ...
#endif

typedef void (*excSigHandler) (_EXC_SIG_HANDLER_ARGS (sig));

typedef struct
{
    char *name;
    int number;

} excSig;

/*
 * Declare excSigDomain
 */

#include <exception/exc-sig-list.h>

/*
 * exc_sig is the domain of all signal exceptions, its type is excSigDomain. 
 * The type of the individual exceptions is excSig. Note: The location (index)
 * of the signals in the exc_sig struct will NOT in general be the same as the
 * signal number.
 */

extern excSigDomain exc_sig;
extern int exc_sig_type;

/*
 * These functions returns information about an excSig exception.
 */

extern char *exc_sig_name (excSig sig);
extern int exc_sig_number (excSig sig);

/*
 * exc_sig_exception_handler() is an exception handler (NOT a signal
 * handler) which is installed automaically by exc_signal(). It prints 
 * an error messages for excSig exceptions that are not caught.
 */

extern int exc_sig_exception_handler (void *e, void *e_type, void *h_data);

/*
 * exc_sig_handler is a signal handler that translates signals to exceptions.
 * It is installed by exc_signal().
 */

extern void exc_sig_handler (_EXC_SIG_HANDLER_ARGS (sig));

/*
 * exc_signal() installs exc_sig_handler() for a specific signal, just as
 * signal(2) does.
 */

extern excSigHandler exc_signal (int sig);

/*
 * exc_signals() works like exc_signal() but installs exc_sig_handler() for
 * multiple signals. This list is specified by a mask (int) in BSD and by
 * an sigset_t* in SYSV. The previous handles are not returned.
 */

#if defined(SVR4) || defined(SYSV) || defined(_SYSV_)
extern void exc_signals (sigset_t *set);
#else
extern void exc_signals (int mask);
#endif

#endif /* !_exception_exc_sig_h */
