/***
*
*  Getsys.prg
*
*  Standard Clipper 5.2 GET/READ Subsystem
*
*  Copyright (c) 1991-1993, Computer Associates International, Inc.
*  All rights reserved.
*
*  This version adds the following public functions:
*
*     ReadKill( [<lKill>] )       --> lKill
*     ReadUpdated( [<lUpdated>] ) --> lUpdated
*     ReadFormat( [<bFormat>] )   --> bFormat | NIL
*
*  NOTE: compile with /m /n /w
*
* Mouse modification history:
*
*   Version   Date     Who            Reason
*    V01.00   5/14/91  L J Letendre   Initial Version forClipper 5.01
*    V01.01   5/18/91  L J Letendre   Fixed exit out of WHEN clause with reads
*                                     after use of mouse to get to field. Added
*                                     RDCoolSpot and RDWarmSpot.
*    V01.02   6/30/91  L J Letendre   Fixed parameters in RDCoolSpot and
*                                     RDWarmSpot function statements
*    V01.03  12/12/91  L J Letendre   Added Hot Spot ID number to call to
*                                     code block
*    V1.04   12/30/91  L J Letendre   Added TRANSPARENT definition which
*                                     allows one to compile and load with
*                                     other modules without modifying the
*                                     definition of READ and READ SAVE.
*    V1.05   01/28/92  L J Letendre   Fixed procname and procline passed
*                                     to a SET KEY function. Fixed trashing
*                                     of updated get when hotspot function
*                                     called
*    V1.06   01/10/93  L J Letendre   Added change CA made to GETSYS with
*                                     version 5.01a
*    V2.00   03/07/93  L J Letendre   Converted Clipper 5.2 GETSYS.PRG
*                                     Changed variable names to reflect
*                                     Hungarian notation
*/
* Mouse Change begins
* If you want to link this routine with your programs without modifying the
* definition of READ and READ SAVE remove the comment * in front of the
* following #define. This removes the flexibility of calling readmodal directly
* without getting mouse support however.
*#define TRANSPARENT
#ifdef TRANSPARENT
	#stdout
	#stdout Getsys compiled in TRANSPARENT Mode
#else
	#stdout
	#stdout GetSys compiled in Seperate Mode
#endif

* End Mouse Change

#include "Inkey.ch"
#include "Getexit.ch"

#define K_UNDO          K_CTRL_U


//
// State variables for active READ
//
STATIC sbFormat
STATIC slUpdated := .F.
STATIC slKillRead
STATIC slBumpTop
STATIC slBumpBot
STATIC snLastExitState
STATIC snLastPos
STATIC soActiveGet
STATIC scReadProcName
STATIC snReadProcLine


//
// Format of array used to preserve state variables
//
#define GSV_KILLREAD       1
#define GSV_BUMPTOP        2
#define GSV_BUMPBOT        3
#define GSV_LASTEXIT       4
#define GSV_LASTPOS        5
#define GSV_ACTIVEGET      6
#define GSV_READVAR        7
#define GSV_READPROCNAME   8
#define GSV_READPROCLINE   9

#define GSV_COUNT          9

* Start Mouse Change - Add
* additions for mouse support

static aGetCoords:={}   // save the coordinates of all of the gets
static aExcldMouse:={}   // regions of the screen to exclude from selection 
static nFreeExclude:=0  // next free exclude location 0=none
static aMouseSpot:={}    // region of screen where mouse click will cause action
static nFreeSpot:=0     // next free aMouseSpot location 0=none
static nCurrentLevel:=0 // the current nesting level of reads
static nMouseLevel:=1   // mouse can select gets from all levels >= nMouseLevel
static lMouseOn:=.F.    // signals if the mouse should be turned on
static lMouseInited:=.F. // mouse has been initialized
static nDescendLevel:=0 // the level we must decend to when mouse is hit
static nSavePos:=0      // save value of new for decending reads when mouse hit
static lIgnoreMouse:=.F. // ignore mouse if present flag
#define Mouserow	1
#define Mouseleft	2
#define Mouseright	3

* End Mouse Change


/***
*
*  ReadModal()
*
*  Standard modal READ on an array of GETs
*
*/
* Begin Mouse Change
#ifdef TRANSPARENT
FUNCTION ReadModal2( GetList, nPos, nProcLevel)
#else
FUNCTION ReadModal( GetList, nPos, nProcLevel )
#endif
* End Mouse Change

   LOCAL oGet
   LOCAL aSavGetSysVars

* Begin Mouse Change
* Fix for adding another level in determining the calling proc name
   IF nProcLevel=NIL
      nProcLevel=0
   ENDIF
* End Mouse Change

   IF ( VALTYPE( sbFormat ) == "B" )
      EVAL( sbFormat )
   ENDIF

   IF ( EMPTY( GetList ) )
      
      // S'87 compatibility
      SETPOS( MAXROW() - 1, 0 )
      RETURN (.F.)                  // NOTE

   ENDIF

* Start Mouse Change - Add
* Keep track of how deep we are in READs

     nCurrentLevel++
* End Mouse Change

   // Preserve state variables
   aSavGetSysVars := ClearGetSysVars()

   // Set these for use in SET KEYs
   scReadProcName := PROCNAME( 1+nProcLevel )
   snReadProcLine := PROCLINE( 1+nProcLevel )

   // Set initial GET to be read
   IF !( VALTYPE( nPos ) == "N" .AND. nPos > 0 )
      nPos := Settle( Getlist, 0 )
   ENDIF

   WHILE !( nPos == 0 )

      // Get next GET from list and post it as the active GET
      PostActiveGet( oGet := GetList[ nPos ] )

      // Read the GET
      IF ( VALTYPE( oGet:reader ) == "B" )
* Start Mouse Change - Added pos to calls to get:reader and GetReader
* necessary to know which get we are currently in.
         EVAL( oGet:reader, oGet, nPos )    // Use custom reader block
      ELSE
         GetReader( oGet, nPos )            // Use standard reader
      ENDIF
* End Mouse Change

* Start Mouse Change - Add

		if nDescendLevel=nCurrentLevel
* We have run down to where we want to be
			nPos:=nSavePos
			nDescendLevel=0

		elseif nDescendLevel!=0
* keep running down to the next level
			nPos:=0

		else

* End Mouse Change

      // Move to next GET based on exit condition
               nPos := Settle( GetList, nPos )
* Start Mouse Change
		endif
* End Mouse Change

   ENDDO


   // Restore state variables
   RestoreGetSysVars( aSavGetSysVars )

   // S'87 compatibility
   SETPOS( MAXROW() - 1, 0 )

* Start Mouse Change - Add
* Exiting so decrement level counter

   nCurrentLevel--

* End Mouse Change

   RETURN ( slUpdated )



/***
*
*  GetReader()
*
*  Standard modal read of a single GET
*
*/
* Start Mouse Change - Added nPos
PROCEDURE GetReader( oGet, nPos )
* End of mouse change

* Start Mouse Change - Add
local mouse_key:=0
local mouse_row:=0
local mouse_col:=0
local nKey, nTime
* End Mouse Change

   // Read the GET if the WHEN condition is satisfied
* Start Mouse Change - don't enter if we are running down

	IF ( GetPreValidate(oGet) .and. nDescendLevel=0)

* end mouse change

      // Activate the GET for reading
      oGet:setFocus()

      WHILE ( oGet:exitState == GE_NOEXIT )

         // Check for initial typeout (no editable positions)
         IF ( oGet:typeOut )
            oGet:exitState := GE_ENTER
         ENDIF

         // Apply keystrokes until exit
         WHILE ( oGet:exitState == GE_NOEXIT )

* Start Mouse Change
* Under normal conditions use the way it was done before

               IF !lMouseOn
                    GetApplyKey( oGet, Inkey(0) )

               ELSE
* Set up a loop and poll the mouse and keyboard
                    nKey=0
                    mouse_key=0
                    nDescendLevel=0  // handle VALID=.F. return

* Show and hide the cursor so that others won't have to worry
* about writing over it.
                    FT_MSHOWCRS()

                    DO WHILE (nKey=0).AND.(mouse_key=0)

                         mouse_key=FT_MGETPOS(@mouse_row,@mouse_col)
                         nKey=INKEY()

* if we have input from the mouse then convert the mouse coordinates
* Clear button press counts so call can determine double click easily

                         IF mouse_key>0
                              nTime=SECOND()
                              IF mouse_key%2=1 // left button 
                                   FT_MBUTPRS(0)
                              ENDIF
                              IF (INT(mouse_key/2)%2)=1 // right button
                                   FT_MBUTPRS(1)
                              ENDIF
                              IF (mouse_key>=4) // middle button
                                   FT_MBUTPRS(2)
                              ENDIF
                              mouse_row=INT(mouse_row/8)
                              mouse_col=INT(mouse_col/8)
                         ENDIF
                    ENDDO
                    FT_MHIDECRS()

                    IF nKey=0
* Input from the mouse so act on it

                         GetMouseFunc(mouse_key,mouse_row,mouse_col,;
                                        nPos,nTime)
                    ELSE        // we have normal input from the keyboard

                         GetApplyKey( oGet, nKey )
                    ENDIF
               ENDIF

* End Mouse change

         ENDDO

         // Disallow exit if the VALID condition is not satisfied
         IF ( !GetPostValidate( oGet ) )
            oGet:exitState := GE_NOEXIT
         ENDIF
      ENDDO

      // De-activate the GET
      oGet:killFocus()

   ENDIF

   RETURN



/***
*
*  GetApplyKey()
*
*  Apply a single INKEY() keystroke to a GET
*
*  NOTE: GET must have focus.
*
*/
PROCEDURE GetApplyKey( oGet, nKey )

   LOCAL cKey
   LOCAL bKeyBlock

   // Check for SET KEY first
   IF !( ( bKeyBlock := setkey( nKey ) ) == NIL )
      GetDoSetKey( bKeyBlock, oGet )
      RETURN                           // NOTE
   ENDIF

   DO CASE
   CASE ( nKey == K_UP )
      oGet:exitState := GE_UP

   CASE ( nKey == K_SH_TAB )
      oGet:exitState := GE_UP

   CASE ( nKey == K_DOWN )
      oGet:exitState := GE_DOWN

   CASE ( nKey == K_TAB )
      oGet:exitState := GE_DOWN

   CASE ( nKey == K_ENTER )
      oGet:exitState := GE_ENTER

   CASE ( nKey == K_ESC )
      IF ( SET( _SET_ESCAPE ) )
         
         oGet:undo()
         oGet:exitState := GE_ESCAPE

      ENDIF

   CASE ( nKey == K_PGUP )
      oGet:exitState := GE_WRITE

   CASE ( nKey == K_PGDN )
      oGet:exitState := GE_WRITE

   CASE ( nKey == K_CTRL_HOME )
      oGet:exitState := GE_TOP


#ifdef CTRL_END_SPECIAL

   // Both ^W and ^End go to the last GET
   CASE ( nKey == K_CTRL_END )
      oGet:exitState := GE_BOTTOM

#else

   // Both ^W and ^End terminate the READ (the default)
   CASE ( nKey == K_CTRL_W )
      oGet:exitState := GE_WRITE

#endif


   CASE ( nKey == K_INS )
      SET( _SET_INSERT, !SET( _SET_INSERT ) )
      ShowScoreboard()

   CASE ( nKey == K_UNDO )
      oGet:undo()

   CASE ( nKey == K_HOME )
      oGet:home()

   CASE ( nKey == K_END )
      oGet:end()

   CASE ( nKey == K_RIGHT )
      oGet:right()

   CASE ( nKey == K_LEFT )
      oGet:left()

   CASE ( nKey == K_CTRL_RIGHT )
      oGet:wordRight()

   CASE ( nKey == K_CTRL_LEFT )
      oGet:wordLeft()

   CASE ( nKey == K_BS )
      oGet:backSpace()

   CASE ( nKey == K_DEL )
      oGet:delete()

   CASE ( nKey == K_CTRL_T )
      oGet:delWordRight()

   CASE ( nKey == K_CTRL_Y )
      oGet:delEnd()

   CASE ( nKey == K_CTRL_BS )
      oGet:delWordLeft()

   OTHERWISE

      IF ( nKey >= 32 .AND. nKey <= 255 )

         cKey := CHR( nKey )

         IF ( oGet:type == "N" .AND. ( cKey == "." .OR. cKey == "," ) )
            oGet:toDecPos()
         ELSE
            
            IF ( SET( _SET_INSERT ) )
               oGet:insert( cKey )
            ELSE
               oGet:overstrike( cKey )
            ENDIF

            IF ( oGet:typeOut )
               IF ( SET( _SET_BELL ) )
                  ?? CHR(7)
               ENDIF

               IF ( !SET( _SET_CONFIRM ) )
                  oGet:exitState := GE_ENTER
               ENDIF
            ENDIF

         ENDIF

      ENDIF

   ENDCASE

   RETURN



/***
*
*  GetPreValidate()
*
*  Test entry condition (WHEN clause) for a GET
*
*/
FUNCTION GetPreValidate( oGet )

   LOCAL lSavUpdated
   LOCAL lWhen := .T.

   IF !( oGet:preBlock == NIL )

      lSavUpdated := slUpdated

      lWhen := EVAL( oGet:preBlock, oGet )

      oGet:display()

      ShowScoreBoard()
      slUpdated := lSavUpdated

   ENDIF

   IF ( slKillRead )
      
      lWhen := .F.
      oGet:exitState := GE_ESCAPE       // Provokes ReadModal() exit

   ELSEIF ( !lWhen )
      
      oGet:exitState := GE_WHEN         // Indicates failure

   ELSE
      
      oGet:exitState := GE_NOEXIT       // Prepares for editing

   END

   RETURN ( lWhen )



/***
*
*  GetPostValidate()
*
*  Test exit condition (VALID clause) for a GET
*
*  NOTE: Bad dates are rejected in such a way as to preserve edit buffer
*
*/
FUNCTION GetPostValidate( oGet )

   LOCAL lSavUpdated
   LOCAL lValid := .T.


   IF ( oGet:exitState == GE_ESCAPE )
      RETURN ( .T. )                   // NOTE
   ENDIF

   IF ( oGet:badDate() )
      oGet:home()
      DateMsg()
      ShowScoreboard()
      RETURN ( .F. )                   // NOTE
   ENDIF

   // If editing occurred, assign the new value to the variable
   IF ( oGet:changed )
      oGet:assign()
      slUpdated := .T.
   ENDIF

   // Reform edit buffer, set cursor to home position, redisplay
   oGet:reset()

   // Check VALID condition if specified
   IF !( oGet:postBlock == NIL )

      lSavUpdated := slUpdated

      // S'87 compatibility
      SETPOS( oGet:row, oGet:col + LEN( oGet:buffer ) )

      lValid := EVAL( oGet:postBlock, oGet )

      // Reset S'87 compatibility cursor position
      SETPOS( oGet:row, oGet:col )

      ShowScoreBoard()
      oGet:updateBuffer()

      slUpdated := lSavUpdated

      IF ( slKillRead )
         oGet:exitState := GE_ESCAPE      // Provokes ReadModal() exit
         lValid := .T.

      ENDIF
   ENDIF

   RETURN ( lValid )



/***
*
*  GetDoSetKey()
*
*  Process SET KEY during editing
*
*/
PROCEDURE GetDoSetKey( keyBlock, oGet )

   LOCAL lSavUpdated

   // If editing has occurred, assign variable
   IF ( oGet:changed )
      oGet:assign()
      slUpdated := .T.
   ENDIF

   lSavUpdated := slUpdated

   EVAL( keyBlock, scReadProcName, snReadProcLine, ReadVar() )

   ShowScoreboard()
   oGet:updateBuffer()

   slUpdated := lSavUpdated

   IF ( slKillRead )
      oGet:exitState := GE_ESCAPE      // provokes ReadModal() exit
   ENDIF

   RETURN





/***
*              READ services
*/



/***
*
*  Settle()
*
*  Returns new position in array of Get objects, based on:
*     - current position
*     - exitState of Get object at current position
*
*  NOTES: return value of 0 indicates termination of READ
*         exitState of old Get is transferred to new Get
*
*/
STATIC FUNCTION Settle( GetList, nPos )

   LOCAL nExitState

   IF ( nPos == 0 )
      nExitState := GE_DOWN
   ELSE
      nExitState := GetList[ nPos ]:exitState
   ENDIF

   IF ( nExitState == GE_ESCAPE .or. nExitState == GE_WRITE )
      RETURN ( 0 )               // NOTE
   ENDIF

   IF !( nExitState == GE_WHEN )
      // Reset state info
      snLastPos := nPos
      slBumpTop := .F.
      slBumpBot := .F.
   ELSE
      // Re-use last exitState, do not disturb state info
      nExitState := snLastExitState
   ENDIF

   //
   // Move
   //
   DO CASE
   CASE ( nExitState == GE_UP )
      nPos--

   CASE ( nExitState == GE_DOWN )
      nPos++

   CASE ( nExitState == GE_TOP )
      nPos       := 1
      slBumpTop  := .T.
      nExitState := GE_DOWN

   CASE ( nExitState == GE_BOTTOM )
      nPos       := LEN( GetList )
      slBumpBot  := .T.
      nExitState := GE_UP

   CASE ( nExitState == GE_ENTER )
      nPos++
* Start Mouse Change - Add

   CASE ( nExitState == GE_MOUSE )
* we got here by hitting the mouse within the same READ so change the GET

      nExitState:=IIF(nPos<nSavePos,GE_DOWN,GE_UP)
      nPos:=nSavePos

* End Mouse Change

   ENDCASE

   //
   // Bounce
   //
   IF ( nPos == 0 )                       // Bumped top
      IF ( !ReadExit() .and. !slBumpBot )
         slBumpTop  := .T.
         nPos       := snLastPos
         nExitState := GE_DOWN
      ENDIF

   ELSEIF ( nPos == len( GetList ) + 1 )  // Bumped bottom
      IF ( !ReadExit() .and. !( nExitState == GE_ENTER ) .and. !slBumpTop )
         slBumpBot  := .T.
         nPos       := snLastPos
         nExitState := GE_UP
      ELSE
         nPos := 0
      ENDIF
   ENDIF

   // Record exit state
   snLastExitState := nExitState

   IF !( nPos == 0 )
      GetList[ nPos ]:exitState := nExitState
   ENDIF
   
   RETURN ( nPos )



/***
*
*  PostActiveGet()
*
*  Post active GET for ReadVar(), GetActive()
*
*/
STATIC PROCEDURE PostActiveGet( oGet )

   GetActive( oGet )
   ReadVar( GetReadVar( oGet ) )

   ShowScoreBoard()

   RETURN



/***
*
*  ClearGetSysVars()
*
*  Save and clear READ state variables. Return array of saved values
*
*  NOTE: 'Updated' status is cleared but not saved (S'87 compatibility)
*/
STATIC FUNCTION ClearGetSysVars()

   LOCAL aSavSysVars[ GSV_COUNT ]

   // Save current sys vars
   aSavSysVars[ GSV_KILLREAD ]     := slKillRead
   aSavSysVars[ GSV_BUMPTOP ]      := slBumpTop
   aSavSysVars[ GSV_BUMPBOT ]      := slBumpBot
   aSavSysVars[ GSV_LASTEXIT ]     := snLastExitState
   aSavSysVars[ GSV_LASTPOS ]      := snLastPos
   aSavSysVars[ GSV_ACTIVEGET ]    := GetActive( NIL )
   aSavSysVars[ GSV_READVAR ]      := ReadVar( "" )
   aSavSysVars[ GSV_READPROCNAME ] := scReadProcName
   aSavSysVars[ GSV_READPROCLINE ] := snReadProcLine

   // Re-init old ones
   slKillRead      := .F.
   slBumpTop       := .F.
   slBumpBot       := .F.
   snLastExitState := 0
   snLastPos       := 0
   scReadProcName  := ""
   snReadProcLine  := 0
   slUpdated       := .F.

   RETURN ( aSavSysVars )



/***
*
*  RestoreGetSysVars()
*
*  Restore READ state variables from array of saved values
*
*  NOTE: 'Updated' status is not restored (S'87 compatibility)
*
*/
STATIC PROCEDURE RestoreGetSysVars( aSavSysVars )

   slKillRead      := aSavSysVars[ GSV_KILLREAD ]
   slBumpTop       := aSavSysVars[ GSV_BUMPTOP ]
   slBumpBot       := aSavSysVars[ GSV_BUMPBOT ]
   snLastExitState := aSavSysVars[ GSV_LASTEXIT ]
   snLastPos       := aSavSysVars[ GSV_LASTPOS ]

   GetActive( aSavSysVars[ GSV_ACTIVEGET ] )

   ReadVar( aSavSysVars[ GSV_READVAR ] )

   scReadProcName  := aSavSysVars[ GSV_READPROCNAME ]
   snReadProcLine  := aSavSysVars[ GSV_READPROCLINE ]

   RETURN



/***
*
*  GetReadVar()
*
*  Set READVAR() value from a GET
*
*/
STATIC FUNCTION GetReadVar( oGet )

   LOCAL cName := UPPER( oGet:name )
   LOCAL i

   // The following code includes subscripts in the name returned by
   // this FUNCTIONtion, if the get variable is an array element
   //
   // Subscripts are retrieved from the oGet:subscript instance variable
   //
   // NOTE: Incompatible with Summer 87
   //
   IF !( oGet:subscript == NIL )
      FOR i := 1 TO LEN( oGet:subscript )
         cName += "[" + LTRIM( STR( oGet:subscript[i] ) ) + "]"
      NEXT
   END

   RETURN ( cName )





/***
*              System Services
*/



/***
*
*  __SetFormat()
*  
*  SET FORMAT service
*
*/
PROCEDURE __SetFormat( b )
   sbFormat := IF( VALTYPE( b ) == "B", b, NIL )
   RETURN



/***
*
*  __KillRead()
*
*  CLEAR GETS service
*
*/
PROCEDURE __KillRead()
   slKillRead := .T.
   RETURN



/***
*
*  GetActive()
*
*  Retrieves currently active GET object
*/
FUNCTION GetActive( g )

   LOCAL oldActive := soActiveGet

   IF ( PCOUNT() > 0 )
      soActiveGet := g
   ENDIF

   RETURN ( oldActive )



/***
*
*  Updated()
*
*/
FUNCTION Updated()
   RETURN slUpdated



/***
*
*  ReadExit()
*
*/
FUNCTION ReadExit( lNew )
   RETURN ( SET( _SET_EXIT, lNew ) )



/***
*
*  ReadInsert()
*
*/
FUNCTION ReadInsert( lNew )
   RETURN ( SET( _SET_INSERT, lNew ) )



/***
*              Wacky Compatibility Services
*/


// Display coordinates for SCOREBOARD
#define SCORE_ROW      0
#define SCORE_COL      60


/***
*
*  ShowScoreboard()
*
*/
STATIC PROCEDURE ShowScoreboard()

   LOCAL nRow
   LOCAL nCol

   IF ( SET( _SET_SCOREBOARD ) )
      nRow := ROW()
      nCol := COL()

      SETPOS( SCORE_ROW, SCORE_COL )
      DISPOUT( IF( SET( _SET_INSERT ), "Ins", "   " ) )
      SETPOS( nRow, nCol )
   ENDIF

   RETURN



/***
*
*  DateMsg()
*
*/
STATIC PROCEDURE DateMsg()

   LOCAL nRow
   LOCAL nCol

   IF ( SET( _SET_SCOREBOARD ) )
      
      nRow := ROW()
      nCol := COL()

      SETPOS( SCORE_ROW, SCORE_COL )
      DISPOUT( "Invalid Date" )
      SETPOS( nRow, nCol )

      WHILE ( NEXTKEY() == 0 )
      END

      SETPOS( SCORE_ROW, SCORE_COL )
      DISPOUT( SPACE( 12 ) )
      SETPOS( nRow, nCol )

   ENDIF

   RETURN



/***
*
*  RangeCheck()
*
*  NOTE: Unused second param for 5.00 compatibility.
*
*/
FUNCTION RangeCheck( oGet, junk, lo, hi )

   LOCAL cMsg, nRow, nCol
   LOCAL xValue

   IF ( !oGet:changed )
      RETURN ( .T. )          // NOTE
   ENDIF

   xValue := oGet:varGet()

   IF ( xValue >= lo .and. xValue <= hi )
      RETURN ( .T. )          // NOTE
   ENDIF

   IF ( SET(_SET_SCOREBOARD) )
      
      cMsg := "Range: " + LTRIM( TRANSFORM( lo, "" ) ) + ;
            " - " + LTRIM( TRANSFORM( hi, "" ) )

      IF ( LEN( cMsg ) > MAXCOL() )
         cMsg := SUBSTR( cMsg, 1, MAXCOL() )
      ENDIF

      nRow := ROW()
      nCol := COL()

      SETPOS( SCORE_ROW, MIN( 60, MAXCOL() - LEN( cMsg ) ) )
      DISPOUT( cMsg )
      SETPOS( nRow, nCol )

      WHILE ( NEXTKEY() == 0 )
      END

      SETPOS( SCORE_ROW, MIN( 60, MAXCOL() - LEN( cMsg ) ) )
      DISPOUT( SPACE( LEN( cMsg ) ) )
      SETPOS( nRow, nCol )

   ENDIF

   RETURN ( .F. )



/***
*
*  ReadKill()
*
*/
FUNCTION ReadKill( lKill )

   LOCAL lSavKill := slKillRead

   IF ( PCOUNT() > 0 )
      slKillRead := lKill
   ENDIF

   RETURN ( lSavKill )



/***
*
*  ReadUpdated()
*
*/
FUNCTION ReadUpdated( lUpdated )
   
   LOCAL lSavUpdated := slUpdated
   
   IF ( PCOUNT() > 0 )
      slUpdated := lUpdated
   ENDIF

   RETURN ( lSavUpdated )
      


/***
*
*  ReadFormat()
*
*/
FUNCTION ReadFormat( b )
   
   LOCAL bSavFormat := sbFormat

   IF ( PCOUNT() > 0 )
      sbFormat := b
   ENDIF

   RETURN ( bSavFormat )
      
* Start Mouse Change - Add
*
* The remainder of the routines are Copyright (c) 1991-1993 by Leo J. Letendre.
* Permission is automatically granted to those who wish to use these
* routines in any application. Permission is not granted to anyone wishing
* to include these in any third party shareware or commercial library.
*
* For further information please contact: Leo Letendre, CIS: 73607,233
*
*******
** RdMouseLvl()
*
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     4/19/91   LJ Letendre   Initial Version
*
 * Author....:Leo Letendre CIS: 73607,233
/*  $DOC$
 *  $FUNCNAME$
 *     RDMOUSELVL()
 *  $CATEGORY$
 *     GET/READ
 *  $ONELINER$
 *     Sets/Gets Level of nested READS currently mouse active
 *  $SYNTAX$
 *      RDMouseLvl(<nLevel>) -> nCurLevel
 *  $ARGUMENTS$
 *      <nLevel> - integer value of the level of READs below which
 *               the mouse is inactive. READs are numbered in 
 *               order of execution starting at 1.
 *
 *  $RETURNS$
 *      Current status of active levels of reads
 *  $DESCRIPTION$
 *      This procedure sets the flag which controls which levels of READS
 *      are accesable to the mouse. If the calling  argument is missing then 
 *      it just returns the current setting.
 *
 *      Is usually called prior to READ but could be called from within
 *      a WHEN clause.
 *
 *      Note: To get the current level of read use RDCurLevel().
 *
 *  $EXAMPLES$
 *      RDMouseLvl(RDCurLvl())  // turn off mouse use for all but the next 
 *                              // read
 *      READ
 *  $SEEALSO$
 *      RDMOUSELVL()
 *  $INCLUDE$
 *
 *  $END$
 */

FUNCTION RDMouseLvl(nLevel)

*
* Local variables
*
LOCAL oldsetting    // setting prior to call

* Save old setting now in case we change it

oldsetting=nMouseLevel

* Set according to input

IF VALTYPE(nLevel)=="N"
	nMouseLevel=MAX(1,nLevel)
ENDIF

RETURN oldsetting

* End of RDMouseLvl

*******
** RDMouseOn()
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     4/20/91   LJ Letendre   Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     RDMOUSEON()
 *  $CATEGORY$
 *     GET/READ
 *  $ONELINER$
 *     Set/Return status of using mouse in READ
 *  $SYNTAX$
 *     RDMOUSEON(<lOn>) -> lCurrent
 *  $ARGUMENTS$
 *      <lOn> - Logical = .T. for mouse being active
 *                      = .F. for mouse being ignored
 *  $RETURNS$
 *      Current status as a logical
 *  $DESCRIPTION$
 *      This procedure sets the flag which controls whether the mouse
 *      is active for the current read. If the calling argument is 
 *      missing then it just returns the current setting.
 *
 *      This routine does not signal an error if an incorrect calling
 *      parameter is passed. In this case it acks as if the caller passed
 *      none.
 *
 *      This routine should not normally need to be called but is included
 *      on the off chance that it will be.
 *
 *  $EXAMPLES$
 *      lCurMouse=RDMouseOn(.T.)
 *  $SEEALSO$
 *
 *  $INCLUDE$
 *
 *  $END$
 */

FUNCTION RDMouseOn(lOn)

*
* Local variables
*
LOCAL oldsetting    // setting prior to call

* Save old setting now in case we change it

oldsetting=lMouseOn

* Set according to input

IF VALTYPE(lOn)=="L"
	lMouseOn=lOn
ENDIF

RETURN oldsetting

* End of RDMouseOn

*****
* MouseCoord()
*
* This function looks at the current getlist and sets up an array of 
* coordinates which define the begining and end of each get.
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     4/20/91   LJ Letendre   Initial Version
*         V1.01     5/10/91   LJ Letendre   Added GetList to call
*
* Calling parameters:  aGetList - the array of gets to be processed
*
* Returns: NIL
*
* Notes: This routine sets the length of the get equal to the size of the
*        string returned by the TRANSFORM function which currently follows
*        all of the picture functions of the GET which is counter to the
*        inference given in the documentation. Since this simplifies the job
*        we will use it here and hope that it doesn't change.
*
*        This routine should be called prior to issuing the READ command
*        or calling readmodal().
*
STATIC FUNCTION MouseCoord(aGetList)

*
* Local variables
*
LOCAL nGetlen   // length of getlist
LOCAL i         // index counter
LOCAL g         // local copy of each object in the getlist
LOCAL nLevel	 // next READ level

* expand the size of the corrdinate array as needed

nLevel=nCurrentLevel+1
ASIZE(aGetCoords,nLevel)
nGetlen=LEN(aGetList)
aGetCoords[nlevel]={}
ASIZE(aGetCoords[nLevel],nGetlen)

* now go through each element of the getlist and determine the size

FOR i=1 TO nGetlen

	g=aGetList[i]

* Expand the array to three levels

	aGetCoords[nlevel,i]=;
		{g:row, g:col, g:col+LEN(TRANSFORM(g:VARGET(),g:picture))-1 }

NEXT

* Return to caller work complete

RETURN NIL	

* End of mousecoord

*****
*
* GetMouseFunc()
*
* This function determines if the mouse is in a Get and if it is then at
* which level it needs to drop to and which get it needs to move to. 
* Additionally this routine processes any screen "hot spots" which are selected
* by the user. 
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     4/20/91   LJ Letendre   Initial Version
*         V1.01     5/10/91   LJL           Clipper 5.01 update
*                                           Added row, col and time to code
*                                           block eval call.
*         V1.03    12/14/91   LJL           Added hot spot id number to call
*                                           to service code block
*         V1.02    12/26/91   LJL           Changed hot spot release function
*                                           to FT_MGETPOS since this gives 
*                                           current button status.
*
* Calling Parameters:
*                    nButton - Button which was hit - reserved for future
*                    nRow - Row coordinate of mouse pointer when button hit
*                    nCol - Col coordinate of mouse pointer
*                    nPos - On input contains the current get number
*                    nTime - The time that the button was clicked
* Returns: NIL
*
* Notes: The processing of "Hot Spots" occurs prior to checking for the
* cursor being in a GET field. Therefore, the code block will be executed
* without the check for moving the cursor to another get.
*
*

STATIC FUNCTION GetMouseFunc(nButton,nRow,nCol,nPos,nTime)

*
* Local variables:
*
LOCAL i, j, working, ending, k
LOCAL working2, des_pos, old_pos
LOCAL g

* Check the general hot spots first

working=GeneralSpot(nButton, nRow, nCol, nTime)


* First check to see if any action hot spots were clicked on

j=1
DO WHILE (j<=LEN(aMouseSpot).AND.working)

* Check coordinates
	working=.NOT.(aMouseSpot[j,9].AND.;
				nRow>=aMouseSpot[j,1].AND.nRow<=aMouseSpot[j,3].AND.;
				nCol>=aMouseSpot[j,2].AND.nCol<=aMouseSpot[j,4].AND.;
				(aMouseSpot[j,8]=0.OR.aMouseSpot[j,8]=nButton))

* If we have a match then execute the code block

	IF .NOT.working

* if editing has occurred, assign variable

		if ( soActiveGet:changed )
			soActiveGet:Assign()
			slUpdated := .t.
		end


* do the request

		EVAL(aMouseSpot[j,5],nButton,nRow,nCol,nTime,j)

* in case var was reassigned in the code block
		soActiveGet:updateBuffer()

* in case insert status was diddled in SET KEY code
          ShowScoreboard()
* Wait for release if requested
		IF aMouseSpot[j,7]
			DO WHILE FT_MGETPOS()!=0
			ENDDO
		ENDIF

* Pause for the minimum amount of time

		sleep(aMouseSpot[j,6],nTime)
	ENDIF
* increment counter
	j++

ENDDO

* if we did not find a hit then continue on
IF working

* Search for a match in coordinates

	i = nCurrentLevel
	ending=IIF(nMouseLevel<1,1,nMouseLevel)
	working2=.T.

	DO WHILE (i>=ending.AND.working.AND.working2)
		j=1
		DO WHILE (j<=LEN(aGetCoords[i]).AND.working.AND.working2)
* First check the row since that has no spanning value
			IF aGetCoords[i,j,Mouserow]=nRow;
				.AND.nCol>=aGetCoords[i,j,Mouseleft];
				.AND.nCol<=aGetCoords[i,j,Mouseright]

				working=.F.
* Now check to see if it is excluded at this level and all higher

				k=1
				DO WHILE (k<=LEN(aExcldMouse).AND.working2)
					IF i<=aExcldMouse[k,5];
						.AND.nRow>=aExcldMouse[k,1];
						.AND.nRow<=aExcldMouse[k,3];
						.AND.nCol>=aExcldMouse[k,2];
						.AND.nCol<=aExcldMouse[k,4]

* If we are in an excluded area then kick us out of the whole routine

						working2=.F.
						working=.T.   // so no work gets done below
					ENDIF
					k++
				ENDDO
			ENDIF
			j++
		ENDDO
		i--
	ENDDO

* If working is false then we found a match so Do the work

	IF .NOT.working
		i++
	     nSavePos=j-1
* Now based upon which key was hit either move the active get- left key
* or move the cursor - right key

* Left button

		IF nButton=1

* If we are at the correct level then do not adjust for it
			IF i==nCurrentLevel
* Change get if necessary
				IF (nSavePos!=nPos)
					soActiveGet:exitState:=GE_MOUSE
				ENDIF
* at a different level then kill read and Get

			ELSE
				nDescendLevel=i
				soActiveGet:exitState:=GE_ESCAPE
				slKillRead=.T.
			ENDIF

* Right Button

		ELSEIF nButton=2

* See if we are in the correct get and if so move the cursor
			IF i==nCurrentLevel.AND.nSavePos==nPos
* Get desired position - a little funky due to scrollable fields

				soActiveGet:display()   // make sure cursor is showing
				old_pos=soActiveGet:pos
				des_pos=old_pos+ncol-COL()
				IF des_pos=soActiveGet:DecPos
					des_pos++
				ENDIF
* Move
				IF des_pos<old_pos
					old_pos--
					DO WHILE soActiveGet:pos>des_pos.AND.;
						.NOT.(old_pos==soActiveGet:pos)
						old_pos=soActiveGet:pos

						soActiveGet:left()
* Check for non-movement
					ENDDO
				ELSEIF des_pos>old_pos
					old_pos--
					DO WHILE soActiveGet:pos<des_pos.AND.;
						.NOT.(old_pos==soActiveGet:pos)
						old_pos=soActiveGet:pos

						soActiveGet:right()
* Check for non-movement
					ENDDO
				ENDIF

			ENDIF
		ENDIF
* Wait for release so that scrolling or hot spots are not
* messed up by not releasing quickly enough

		DO WHILE FT_MBUTREL(0)!=0
		ENDDO

	ENDIF


ENDIF

* Now return

RETURN NIL

* End of  GetMouseFunc


******
*
* RDExclMouse()
*
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     4/20/91   LJ Letendre   Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     RDEXCLMOUSE()
 *  $CATEGORY$
 *     GET/READ
 *  $ONELINER$
 *     Select region where mouse has no action in Gets
 *  $SYNTAX$
 *     RDExclMouse(<nTopRow>, <nLeftCol>, <nBotRow>, <nRightCol>) -> nId
 *  $ARGUMENTS$
 *      <nTopRow> - the top row of the area to exclude
 *      <nLeftCol> - the left column of the area
 *      <nBotRow> - the bottom row of the area to exclude
 *      <nRightCol> - the right column of the area to exclude
 *     
 *  $RETURNS$
 *     <nID> which is an ID number identifying the region so that the
 *              region may be restored to active status with a call to
 *              RDREMEXCL(ID)
 *  $DESCRIPTION$
 *      This routine allows the caller to exclude gets in an area of 
 *      the screen from being selected at the current and all lower levels.
 *     
 *  $EXAMPLES$
 *     nId1=RDExclMouse(10,10,15,30)   // Exclude area under a new GET/READ
 *     xSaveScr=SAVESCREEN(10,10,15,30)
 *     @ 10,10 CLEAR TO 15,30
 *     @ 10,10,15,30 BOX
 *     @ 11,12 SAY "Enter Special Code:
 *     cCode=SPACE(20)
 *     @ 12,12 GET cCode
 *     READ
 *     RDRemExcl(nId1)
 *  $SEEALSO$
 *     RDREMEXCL() RDCURLEVEL()
 *  $INCLUDE$
 *     
 *  $END$
 */
*
FUNCTION RDExclMouse(nTopRow, nLeftCol, nBotRow, nRightCol)
*
* Local parameters
LOCAL next_exclude, i, working

* Check size of exclude array and enlarge if necessary

IF (nFreeExclude=0)
	AADD(aExcldMouse,{})
	next_exclude=LEN(aExcldMouse)
ELSE
	next_exclude=nFreeExclude
	i=next_exclude+1
	nFreeExclude=0
	working=.T.
* find next free location
	DO WHILE i<LEN(aExcldMouse).AND.working
		IF aExcldMouse[i,1]=-1
			working=.F.
			nFreeExclude=i
		ENDIF
		i++
	ENDDO

ENDIF

* Now add the coordinates

aExcldMouse[next_exclude]={nTopRow, nLeftCol, nBotRow, nRightCol,nCurrentLevel}

RETURN next_exclude

* End of RDExclMouse

******
*
* RDRemExcl()
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     4/20/91   LJ Letendre   Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     RDREMEXCL()
 *  $CATEGORY$
 *     GET/READ
 *  $ONELINER$
 *     Clears an mouse excluded area set up by RDExclMouse()
 *  $SYNTAX$
 *     RDRemExcl( <nID> ) -> NIL
 *  $ARGUMENTS$
 *     <nID> - which is the ID number of the region to place
 *           back to active duty. It is given by RDExclMouse.
 *  $RETURNS$
 *     NIL
 *  $DESCRIPTION$
 *     This subroutine clears the excluded region set by ExcludeMouse 
 *     for the currently active READ
 *     
 *  $EXAMPLES$
 *     nId=RDExclMouse(10,10,15,30)
 *     <code>
 *     RDRemExcl(nId)
 *  $SEEALSO$
 *     RDEXCLMOUSE()
 *  $INCLUDE$
 *     
 *  $END$
 */
* Calling Parameters : *
* Returns: NIL
*
FUNCTION RDRemExcl(nID)
*
* Local variables
LOCAL i, working

aExcldMouse[nID,1]:=aExcldMouse[nID,2]:=-1
aExcldMouse[nID,3]:=aExcldMouse[nID,4]:=-1
aExcldMouse[nID,5]=0
IF nID<nFreeExclude.OR.nFreeExclude=0
	nFreeExclude=nID
ENDIF

* shrink the array if possible.

i=LEN(aExcldMouse)
working=.T.
DO WHILE i>=nFreeExclude.AND.working
	working=(aExcldMouse[i,1]<0)
	i--
ENDDO
ASIZE(aExcldMouse,i+1)

RETURN NIL

* End of RemExclude

******
*
* MOUSEREAD()
*
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     4/20/91   LJ Letendre   Initial Version
*         V1.01     5/10/91   LJ Letendre   changed call to MouseCoord
*         V1.02    12/30/91   LJ Letendre   Added transparent mode
/*  $DOC$
 *  $FUNCNAME$
 *     MOUSEREAD()
 *  $CATEGORY$
 *     GET/READ
 *  $ONELINER$
 *     Sets up GET/READ with mouse capabilities
 *  $SYNTAX$
 *     MOUSEREAD(<GetList>, [<nPos>]) -> lUpdated
 *  $ARGUMENTS$
 *     <GetList> - The list of gets normally passed to readmodal
 *     <nPos> - Undocumented starting get position. Undocumented in
 *              Clipper 5.2 docs.
 *  $RETURNS$
 *     <lUpdated> - Which is normally returned by readmodal
 *  $DESCRIPTION$
 *      This routine calls all of the appropriate routines to do a
 *      read using the mouse for field selection.
 *
 *      NOTE: When this routine is compiled with TRANSPARENT defined,
 *      this routine is called readmodal. Thus in transparent mode, no
 *      modifications to the calling clipper code must be made. If
 *      TRANSPARENT is not defined then either the definition of READ must
 *      be changed or MOUSEREAD(GetList) must be called explicitly.
 *     
 *  $EXAMPLES$
 *     With TRANSPARENT defined at compile time:
 *     @ 1,2 GET ...
 *     READ
 *
 *     Without TRANSPARENT defined:
 *
 *     #include "mouseget.ch"  // which includes:
 *                       // #command READ => MouseRead(GetList); GetList:={}
 *     @ 1,2 GET ...
 *     READ
 *
 *  $SEEALSO$
 *     
 *  $INCLUDE$
 *     
 *  $END$
 */

*
#ifdef TRANSPARENT
FUNCTION ReadModal( aGetList, nPos )
#else
FUNCTION MOUSEREAD(aGetList, nPos)
#endif
*
* Local variables:
LOCAL savemouse, nX, nY, result

* See if we have a mouse and reset it. This does no harm on multiple reads.

IF !lIgnoreMouse

* Initialize mouse if necessary

	lMouseInited:=FT_MINIT()

	IF lMouseInited
* Set the mouse flag on saving the old
		lMouseOn=.T.
* Save the coordinates of the fields
		mousecoord(aGetList)
	ENDIF

ENDIF

* Now do the read

#ifdef TRANSPARENT
result=ReadModal2(aGetList,nPos,1)
#else
result=ReadModal(aGetList,nPos,1)
#endif
IF !lIgnoreMouse
* Reset the world

	ASIZE(aGetCoords,MAX(nCurrentLevel,1))

ENDIF

RETURN result

* End of MOUSEREAD

******
*
* RDHotSpot()
*
* This function allows the caller to define a location on the screen which
* if clicked on with the mouse will cause an action to take place.
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     4/20/91   LJ Letendre   Initial Version
*         V1.01     5/10/91   LJL           Added row, col and time to code
*                                           block eval call documentation.
* 
/*  $DOC$
 *  $FUNCNAME$
 *     RDHOTSPOT()
 *  $CATEGORY$
 *     GET/READ
 *  $ONELINER$
 *     Defines Mouse Hot spots for GET/READ
 *  $SYNTAX$
 *     RDHotSpot( <nTopRow>, <nLeftCol>, <nBotRow>, <nRightCol>, <bAction>, ;
 *                <nButton>, <nSleep>, <lRelease>) -> nId
 *
 *  $ARGUMENTS$
 *     <nTopRow> - the top row of the area 
 *     <nLeftCol> - the left column of the area
 *     <nBotRow> - the bottom row of the area
 *     <nRightCol> - the right column of the area 
 *     <bAction> - Code block which will be executed when
 *              mouse is clicked in the area
 *     <nButton> - Optional button number for action to occur. IF
 *              equal to 0 or NIL, the action occurs on 
 *              clicking anybutton (the code block can decide 
 *              what to do with based upon the button). If equal
 *              to 1, code block executes only on left click,
 *              if equal to 2 only on right click and if equal
 *              to 4(?) then the middle button.
 *     <nSleep> - Optional value of a minimum time (in seconds) to
 *              wait between servicing multiple button presses. 
 *              Prevents routine from operating too quickly and 
 *              reading the press of a button multiple times 
 *              when not intended. If =NIL then the default value
 *              is used (see MDefSleep()).
 *     <lRelease> - Optional Logical Value. If set to .T. the
 *              servicing routine will pause after the completion
 *              of bAction for the release of the mouse button(s)
 *              Useful for guaranteeing no multiple hits on
 *              an area. If =NIL then the default is used (see
 *              MDefRelease())
 *     
 *  $RETURNS$
 *     nId which is an ID to be used to remove the area with a call
 *              to RDRemHotSpot(nId)
 *  $DESCRIPTION$
 *     This routine defines a hot spot for READ, which will be activated 
 *     it the user clicks the mouse in the defined area. The action which is
 *     executed is defined by the code block bAction which is called with
 *     four arguments:
 *
 *                 nButNum: the number of the button pressed with
 *                          1=left, 2=right, 4=middle(?).
 *                 nRow: The row that the mouse cursor was in when it
 *                       was clicked
 *                 nCol: The column that the mouse cursor was in when it
 *                       was clicked
 *                 nTime: The time returned by SECOND() shortly after the
 *                       button was clicked.
 *                 nId:   The hot spot Id number.
 *
 *        Thus the code block should have a form similar to the following
 *        if one wishes to use the button/cursor information:
 *
 *      {|nButNum, nRow, nCol, nTime, nId| MyFunc(NButNum,nRow,nCol,nTime,nId)}
 *
 *  $EXAMPLES$
 *      RDHotSpot(1,10,1,20,{|| ShowHelp()},1,,.T.) // hot spot shows help
 *  $SEEALSO$
 *      RDCOOLSPOT() RDWARMSPOT() RDREMHOTSPOT()
 *  $INCLUDE$
 *
 *  $END$
 */
*
FUNCTION RDHotSpot( nTopRow, nLeftCol, nBotRow, nRightCol, bAction, nButton,;
				 nSleep, lRelease)
*
* Local variables
*

* Entry point

* Now add the coordinates

RETURN AddHotSpot(aMouseSpot,@nFreeSpot,;
                 {nTopRow, nLeftCol, nBotRow, nRightCol, bAction,;
                  IIF(nSleep=NIL,MDefSleep(),nSleep),;
                  IIF(lRelease=NIL,MDefRelease(),lRelease),;
                  IIF(nButton=NIL,0,nButton),.T.})

* End of RDAddHotSpot

******
*
* RDRemHotSpot()
*
* This subroutine clears the specified Hotspot 
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     4/20/91   LJ Letendre   Initial Version
*         V1.01     10/25/91  LJ Letendre   Call generic routine to share code
*
/*  $DOC$
 *  $FUNCNAME$
 *     RDREMHOTSPOT()
 *  $CATEGORY$
 *     GET/READ
 *  $ONELINER$
 *     This subroutine clears the specified READ Hot Spot 
 *  $SYNTAX$
 *     RDRemHotSpot( <nId> ) -> NIL
 *  $ARGUMENTS$
 *     <nID> - the ID number of the region to remove from active duty. 
 *             It is given by RDHotSpot.
 *  $RETURNS$
 *      NIL
 *  $DESCRIPTION$
 *      This routine removes a mouse hot spot from the GET/READ list of active 
 *      hot spots.
 *  $EXAMPLES$
 *      nHelpId=RDHotSpot(1,1,1,10,{|| ShowHelp()})
 *
 *     <GET/READ code>
 *
 *      RDRemHotSpot(nHelpId)
 *  $SEEALSO$
 *      RDHOTSPOT() RDCOOLSPOT() RDWARMSPOT()
 *  $INCLUDE$
 *
 *  $END$
 */
*
FUNCTION RDRemHotSpot(nID)
*
* Local variables
*

* Call service routine

nFreeSpot=RemHotSpot(nId, aMouseSpot, nFreeSpot)

RETURN NIL

* End of RDRemHotSpot

******
*
* RDCoolSpot()
*
* This subroutine deactivates the specified HotSpot without deleting it
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     5/17/91   LJ Letendre   Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     RDCOOLSPOT()
 *  $CATEGORY$
 *     GET/READ
 *  $ONELINER$
 *     This subroutine deactivates the specified GET/READ Hot Spot
 *  $SYNTAX$
 *     RDCOOLSPOT(<nId>) -> NIL
 *  $ARGUMENTS$
 *      <nID> - the ID number of the GET/READ Hot Spot to remove from active 
 *           duty. It is given by RDHotSpot.
 *  $RETURNS$
 *      NIL
 *  $DESCRIPTION$
 *      This routine deactivates the specified hot spot without removing it
 *      from the list of hot spots. It can later be reactivated with
 *      RDWarmSpot()
 *
 *  $EXAMPLES$
 *      nHelpId=RDHotSpot(1,1,1,10,{|| ShowHelp()}
 *
 *      FUNCTION ShowHelp()
 *      RDCoolSpot(nHelpId)   // Cool off the help hot spot
 *      <Get/READ code>       // get type of help needed
 *      RDWarmSpot(nHelpId)   // reactivate before returning
 *      RETURN NIL
 *
 *  $SEEALSO$
 *      RDWARMSPOT() RDREMHOTSPOT() RDHOTSPOT()
 *  $INCLUDE$
 *
 *  $END$
 */
*
FUNCTION RDCoolSpot(nID)
*
* Local variables
*

aMouseSpot[nid,9]=.F.

RETURN NIL

* End of RDCoolSpot

******
*
* RDWarmSpot()
*
* This subroutine reactivates the specified HotSpot which was deactivated
* by RDCoolSpot
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     5/17/91   LJ Letendre   Initial Version
/*  $DOC$
 *  $FUNCNAME$
 *     RDWARMSPOT()
 *  $CATEGORY$
 *     GET/READ
 *  $ONELINER$
 *     This subroutine reactivates the specified GET/READ Hot Spot
 *  $SYNTAX$
 *     RDWARMSPOT(<nId>) -> NIL
 *  $ARGUMENTS$
 *      <nID> - the ID number of the READ Hot Spot to return to active duty. 
 *           It is given by RDHotSpot and should have been deactivated by
 *           RDCoolSpot()
 *
 *  $RETURNS$
 *      NIL
 *  $DESCRIPTION$
 *      This routine reactivates the specified READ hot spot after having 
 *      been deactivated by RDCoolSpot(). 
 *
 *  $EXAMPLES$
 *      RDWarmSpot(nHelpId)   // Turn the help hot spot back on
 *
 *  $SEEALSO$
 *      RDCOOLSPOT() RDREMHOTSPOT() RDHOTSPOT()
 *  $INCLUDE$
 *
 *  $END$
 */
*
FUNCTION RDWarmSpot(nID)
*
* Local variables
*

aMouseSpot[nId,9]=.T.

RETURN NIL

* End of RDWarmSpot

******
*
* RDCurLevel()
*
* This routine will return the current level of READs
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     4/25/91   LJL       Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     RDCURLEVEL()
 *  $CATEGORY$
 *     GET/READ
 *  $ONELINER$
 *     This routine will return the current level of READs
 *  $SYNTAX$
 *     RDCURLEVEL() -> <nCurLevel>
 *  $ARGUMENTS$
 *     None
 *  $RETURNS$
 *     <nCurLevel> The current depth of the READs. The level number
 *          is the current count of nested READs starting at 1.
 *  $DESCRIPTION$
 *     This number is useful when calling RDExclMouse() for excluding the 
 *     mouse from lower level READs.
 *  $EXAMPLES$
 *     nCurLevel=RDCurLevel()
 *     RDExclMouse(nCurLevel)
 *  $SEEALSO$
 *     RDEXCLMOUSE() RDREMEXCL()
 *  $INCLUDE$
 *     
 *  $END$
 */
*
FUNCTION RDCurLevel

RETURN nCurrentLevel

* End of RDCurLevel


******
*
*   RDDescend()
*
* Purpose: This routine returns a logical indicating if the GET/READ system is
* descending to a lower level read based upon a request by the mouse system
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     5/10/91   LJL       Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     RDDESCEND()
 *  $CATEGORY$
 *     GET/READ
 *  $ONELINER$
 *     Returns logical indicating READ systems descending status
 *  $SYNTAX$
 *     RDDESCEND() -> <lDescend>
 *  $ARGUMENTS$
 *     None
 *  $RETURNS$
 *     <lDescend> which is .T. if the GET/READ system will descend 
 *           to a lower level than the current one and is .F. if it 
 *           will stay at the same level.
 *  $DESCRIPTION$
 *     This routine can be useful when a READ is performed in a VALID clause.
 *     It is the mouse equivalent of the keyboard test LASTKEY()=K_ESC.
 *  $EXAMPLES$
 *     IF RDDescend()
 *        <Check other gets>
 *     ENDIF
 *  $SEEALSO$
 *     
 *  $INCLUDE$
 *     
 *  $END$
 */
*
FUNCTION RDDescend

RETURN (nDescendLevel!=0)

* End of RDDescend

*****
*
* function RDIgnoreMouse()
*
* Purpose: force the routines to ignore the mouse and perform the overhead
*          necessary for mouse support
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     5/10/91   LJL       Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     RDIGNOREMOUSE()
 *  $CATEGORY$
 *     GET/READ
 *  $ONELINER$
 *     Force READ to ignore the presence of the mouse
 *  $SYNTAX$
 *     RDIGNOREMOUSE(<lIgnore>) -> <lCurMouse>
 *  $ARGUMENTS$
 *     <lIgnore> - logical for ignoring mouse .T. = act as if mouse 
 *             is not present. If absent just returns current setting
 *  $RETURNS$
 *     <lCurMouse> - The current setting of mouse ignore flag.
 *  $DESCRIPTION$
 *     This routine allow one to force the GET/READ system to ignore the
 *     presence of the mouse and to not incur the overhead associated with
 *     it.
 *  $EXAMPLES$
 *     lCurIgnore=RDIgnoreMouse(.T.)
 *  $SEEALSO$
 *     
 *  $INCLUDE$
 *     
 *  $END$
 */
*
FUNCTION RDIgnoreMouse(lIgnore)

* Local Parameters
LOCAL oldsetting

*Save old value
oldsetting=lIgnoreMouse

IF lIgnore!=NIL
	lIgnoreMouse=lIgnore
ENDIF

RETURN oldsetting

* End of RDIgnoreMouse

******
*
* function RDCancelDescend()
*
* Purpose: to cancel the effect of a mouse click on a lower level GET
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     5/11/91   LJL       Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     RDCANCELDESCEND()
 *  $CATEGORY$
 *     GET/READ
 *  $ONELINER$
 *     Cancel the run down of a READ to a lower level READ
 *  $SYNTAX$
 *     RDCancelDescend() -> NIL
 *  $ARGUMENTS$
 *     None
 *  $RETURNS$
 *     NIL
 *  $DESCRIPTION$
 *     At times it may be useful to cancel the effect of the mouse causing
 *     a run down to a lower level read. This function will provide that
 *     effect.
 *  $EXAMPLES$
 *    FUNCTION ValidateSomething()
 *    IF xSomeValue<>xCorrectValue()
 *
 *        RDCancelDescend() 
 *        GetCorrectValue()
 *    ENDIF
 *  $SEEALSO$
 *     
 *  $INCLUDE$
 *     
 *  $END$
 */

FUNCTION RDCancelDescend

nDescendLevel=0

RETURN NIL

* End of RDCancelDescend

* End Mouse Change
