/***
*  Scrolbar.prg
*  Implements a scroll bar that can be updated as the cursor moves down
*  in a TBrowse object, ACHOICE(), DBEDIT(), or MEMOEDIT().
*
*  Copyright (c) 1990, Nantucket Corp.  All rights reserved.
*  David R. Allison
*
*  Modified to correct algorithm errors and to add support for horizontal 
*  scroll bars.  Leo J. Letendre 8/31/91
*
*  Note: Compile with /N/W
*
*/

#include "scrolbr2.ch"

/***
*  ScrollBarNew( <nTopRow>, <nTopColumn>, <nEndRowCol>, 
*     <cColorString>, <nInitPosition>, <nDirection> ) --> aScrollBar
*  
*  Create a new scroll bar array with the specified coordinates
*
*/
FUNCTION ScrollBarNew( nTopRow, nTopColumn, nEndRowCol, ;
                        cColorString, nInitPosition, nDirection )

   LOCAL aScrollBar := ARRAY( SB_ELEMENTS )

   // Set the direction - horizontal or vertical

   IF nDirection == NIL
      nDirection := SB_VERTICAL
   ENDIF
   aScrollBar[ SB_DIRECTION ] := nDirection


   aScrollBar[ SB_ROWTOP ]    := nTopRow
   aScrollBar[ SB_COLTOP ]    := nTopColumn
   aScrollBar[ SB_ROWBOTTOM ] := IIF(nDirection==SB_VERTICAL,nEndRowCol,nTopRow)
   aScrollBar[ SB_COLBOTTOM ] := IIF(nDirection==SB_VERTICAL,nTopColumn,nEndRowCol)

   // Set the default color to Red on Black if none specified
   IF cColorString == NIL
      cColorString := "R/W"
   ENDIF
   aScrollBar[ SB_COLOR ]     := cColorString

   // Set the starting position
   IF nInitPosition == NIL
      nInitPosition := 1
   ENDIF
   aScrollBar[ SB_POSITION ]  := nInitPosition


   RETURN aScrollBar

/***
*  ScrollBarDisplay( <aScrollBar> ) --> aScrollBar
*  Display a scoll bar array to the screen
*
*/
FUNCTION ScrollBarDisplay( aScrollBar )
   LOCAL cOldColor, nRow, nCol

   cOldColor := SETCOLOR( aScrollBar[ SB_COLOR ] )

   // Draw the arrows
   IF aScrollBar[ SB_DIRECTION ] = SB_VERTICAL
      @ aScrollBar[ SB_ROWTOP ], aScrollBar[ SB_COLTOP ] SAY SB_UPARROW
      @ aScrollBar[ SB_ROWBOTTOM ], aScrollBar[ SB_COLBOTTOM ] SAY SB_DNARROW
   // Draw the background
      FOR nRow := (aScrollBar[ SB_ROWTOP ] + 1) ;
         TO (aScrollBar[ SB_ROWBOTTOM ] - 1)
             @ nRow, aScrollBar[ SB_COLTOP ] SAY SB_BACKGROUND
      NEXT
   ELSE
      @ aScrollBar[ SB_ROWTOP ], aScrollBar[ SB_COLTOP ] SAY SB_LEFTARROW
      @ aScrollBar[ SB_ROWBOTTOM ], aScrollBar[ SB_COLBOTTOM ] SAY SB_RIGHTARROW
   // Draw the background
      FOR nCol := (aScrollBar[ SB_COLTOP ] + 1) ;
         TO (aScrollBar[ SB_COLBOTTOM ] - 1)
             @ aScrollBar[ SB_ROWTOP ], nCol SAY SB_BACKGROUND
      NEXT
   ENDIF

   SETCOLOR( cOldColor )

   RETURN aScrollBar

/***
*  ScrollBarUpdate( <aScrollBar>, <nCurrent>, <nTotal>,
*     <lForceUpdate> ) --> aScrollBar
*
*  Update scroll bar array with new tab position and redisplay tab
*
*/
FUNCTION ScrollBarUpdate( aScrollBar, nCurrent, nTotal, lForceUpdate )

   LOCAL cOldColor, nNewPosition
   LOCAL nScrollHeight

   nScrollHeight := IIF(aScrollBar[SB_DIRECTION]==SB_VERTICAL,;
                      aScrollBar[SB_ROWBOTTOM] - aScrollBar[SB_ROWTOP] - 2,;
                      aScrollBar[SB_COLBOTTOM] - aScrollBar[SB_COLTOP] - 2)
   IF nTotal < 1
      nTotal := 1
   ENDIF

   IF nCurrent < 1
      nCurrent := 1
   ENDIF

   IF nCurrent > nTotal
      nCurrent := nTotal
   ENDIF

   IF lForceUpdate == NIL
      lForceUpdate := .F.
   ENDIF

   cOldColor := SETCOLOR( aScrollBar[ SB_COLOR ] )

   // Determine the new position - corrected algorithm problems

   nNewPosition = 1 + ROUND( ((nCurrent-1)/MAX(1,nTotal-1)) * nScrollHeight, 0)

   // Overwrite the old position (if different), then draw in the new one
   IF nNewPosition <> aScrollBar[ SB_POSITION ] .OR. lForceUpdate

   // vertical scroll bar

     IF aScrollBar[ SB_DIRECTION ] == SB_VERTICAL
         @ (aScrollBar[ SB_POSITION ] + aScrollBar[ SB_ROWTOP ]), ;
            aScrollBar[ SB_COLTOP ] SAY SB_BACKGROUND
         @ (nNewPosition + aScrollBar[ SB_ROWTOP ]),;
            aScrollBar[ SB_COLTOP ] SAY SB_HIGHLIGHT

   // horizontal scroll bar
     ELSE

         @ aScrollBar[ SB_ROWTOP ], (aScrollBar[ SB_POSITION ];
           + aScrollBar[ SB_COLTOP ]) SAY SB_BACKGROUND
         @ aScrollBar[ SB_ROWTOP ], (nNewPosition;
           + aScrollBar[ SB_COLTOP ]) SAY SB_HIGHLIGHT
     ENDIF

      aScrollBar[ SB_POSITION ] := nNewPosition
   ENDIF

   SETCOLOR( cOldColor )

   RETURN aScrollBar

* End of scrollbar.prg