/*
 * File......: MOUSBROW.PRG
 * Author....: Leo Letendre CIS: 73607,233
 * Date......: 10/2/93
 * Revision..: V2.0       // update to clipper 5.2c release
 * Log file..: 
 * 
 * Adapted from TBDEMO.PRG and TBWHILE.PRG from Nantucket and Jim Orlowski
 * respectively.
 *
 * Copyright 1991-1993 Leo J. Letendre
 * Permission is automatically granted to those who wish to use these
 * routines in any application. Permission is not granted to anyone wishing
 * to include these in any third party shareware or commercial library.
 *
 *
 * Modification history:
 * ---------------------
 * 10/1/91  Fixed single record problem in determining the number of records
 *          in the active portion of the database.
 * 10/5/91  Removed all references to shadow boxes. Too many individual
 *          preferences to keep everybody happy.
 * 10/6/91  Added compile time option for left or right side scroll bar.
 * 10/20/91 Made the movement by the scroll bar more efficient
 * 10/25/91 Made default sleeping and release values for mouse events generic
 * 12/14/91 Added Hot Spot ID number to call to associated code block
 *  2/29/92 Added optional alternate record counters <bRecordNum> and
 *          <bRecordTotal>
 *  4/17/92 Changed algorithm for getting the logical record number slightly
 *  4/21/92 Added BRSetScroll() for turning individual scroll bars on/off
 *  5/15/92 Separated out buildbrowse() so user would have flexibility in 
 *          setting up a browse while another subsystem was running
 *  5/25/92 Changed some of the algorithm for moving in database based upon
 *          clicking in the scrollbar
 *  6/3/92  Added check for edits and appends not being allowed so that
 *          instoggle doesn't get loaded which in turn forces READMODAL()
 *          to be loaded
 *  6/6/92  Added the default action of waiting for right button release on
 *          scroll bar arrows for page up and down. A define was added to
 *          override this behavior. 
 * 6/18/92  Added check for being on top line with database smaller than
 *          allotted area and on the last record.
 *  7/5/92  Added additional fields in aField to handle gets better and added
 *          explicit field widths
 *  7/9/92  Fixed bug which lost character or mouse input when not stabilized
 *          when on got to top of grand loop but keyed or moused something
 * 7/14/92  Added code to explicitly control scroll bar colors
 *11/10/92  Added function call to user supplied function whenever browse is
 *          moved. Added code to handle empty databases with scrollbars. Fixed
 *          code to better handle unindexed browses. Added exit codes to 
 *          routines which return REFRESH_* values for explicit exit of browse.
 *11/11/92  Added support for NTXPOS() from NTX2.ZIP written by 
 *          Graham D. McKechnie modified by Brian Marasca.
 *12/28/92  Separated functions so that CreateBrowse, RunBrowse and 
 *          DestroyBrowse could be created and called separately. Separated 
 *          demo into separate file.
 * 2/15/93  Added support for additional actions when user double clicks. These
 *          are exiting and calling a code block.
 * 2/16/93  Added SetWhileBlock and SetWhileKey routines to allow changing of
 *          the search criteria within a reusable browse object
 * 3/25/93  Increment the index order number by 2 for call to NTXPOS()
 * 5/31/93  Checked for empty field array which hung computer
 * 6/13/93  Used NTXREC() to speed moving via the scroll bar
 * 10/02/93 Added explicit key handler routine so routine can be used as a
 *          picklist type routine. TbWhileBot() and TbWhileTop() are no longer
 *          static routines.
 * 10/30/93 Removed increment of index order number by 2 since different
 *          versions of NTXPOS and NTXREC are included for Clipper 5.01 and 5.2.
 * $Log$
 *
 */

#command DEFAULT <param> TO <val> [, <paramn> TO <valn> ];
=> ;
         <param> := IIF(<param> = NIL, <val>, <param> ) ;
         [; <paramn> := IIF(<paramn> = NIL, <valn>, <paramn> ) ]


#include "inkey.ch"
#include "setcurs.ch"
#include "error.ch"
#include "mousbrow.ch"
#include "mbrowse.ch"
#include "scrolbr2.ch"

* Remove the comments on any of the following to perform the desired actions
* Note the arguments passed to the routine are not changed when any of these
* defines are active.

#define LEFT_SCROLL        // switches scroll area from right side to left side
* #define NO_EDITS         // eliminates the code which allows editing of fields
* #define NO_APPENDS       // eliminates the code which allows adding records
* #define NO_SIDE_SCROLL   // eliminates the code which puts the scroll bar on
                           // the side
* #define NO_BOTTOM_SCROLL // eliminates the code which puts the scroll bar on
                           // the bottom
* #define NO_MOUSE         // eliminates the code for mouse support
* #define FASTPAGE         // allows multiple page up/down without releasing
                           // right mouse button.

* #define USE_NTXPOS       // use NTXPOS for determining position in database
                           // and NTXREC to speed moving within the database
* Test to see if both NO_EDITS and NO_APPENDS are set
#ifdef NO_APPENDS
	#ifdef NO_EDITS
		#define NEITHER_CHANGER
	#endif
#endif
#ifdef NO_EDITS
	#stdout 
	#stdout No edits allowed
#endif
#ifdef NO_APPENDS
	#stdout 
	#stdout No appending allowed
#endif
#ifdef USE_NTXPOS
	#stdout 
	#stdout NTXPOS in use
#endif

STATIC aMouseSpot:={}    // Holder of mouse hot spots
STATIC nFreeSpot:=0      // next free spot in aMouseSpot 0 = none
STATIC lIgnoreMouse:=.F. // Ignore mouse even if it is there
STATIC lMouseUsed:=.F.   // Indicates if the mouse is in use
STATIC nLastRec          // closest record to go to if while is .F. on an
                         // edited record
STATIC oBrowseCurrent:=NIL // The current browse object

STATIC lVertScroll:=.T.  // Use vertical scroll bars
STATIC lHorizScroll:=.T. // Use horizontal scroll bars
STATIC cScrollBarColor   // Scroll bar color string
STATIC cDefScrollColor:=NIL // Default scroll bar color

STATIC nBrowseMoved:=0     // Browse moved in some direction given by values
                           // defined in mbrowse.ch
STATIC lMouseExit:=.F.     // Mouse used to exit

#define nBotDepth        1    // the depth of the bottom separator
#define COUNT_BOTH       1    // Count both record number and number of records
#define RECNUM           2    // Count only the number of records

#define DOUBLE_NONE      0    // do nothing on a double click
#define DOUBLE_BLOCK     4096 // Execute a code block on double click


/*  $DOC$
 *  $FUNCNAME$
 *     MBROWSE()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     Browse database limited by a while condition with mouse support
 *  $SYNTAX$
 *     MBROWSE( <aFields>, <bWhileCond>, <cKey>,
 *                 [,<nFreeze>] [,<lSaveScrn>] [,<cColorList>]
 *                 [,<nTop>] [,<nLeft>]
 *                 [,<nBottom>] [,<nRight>], [,<bScreenInit>] [,<nReadInit>]
 *                 [,<bAltReader>] [,<nExitKey>] [,<lAppendMode>]
 *                 [,<AppendReader>] [,<bRecordNum>], [,bRecordTotal>]
 *                 [,<bMovedFunction>] [,<xDoubleAction>] [,<bKeyHandler>] ) ;
 *           -> <nRecno>
 *  $ARGUMENTS$
 *     <aFields> is multidimensional array of fields you want to display. 
 *               Each row contains up to nine members containing:
 *
 *        Index 1: Column heading
 *        Index 2: Code block for displaying information. Note: if you are
 *                 going to do an a read, this must code block must be written
 *                 so that if a parameter is passed, it will set the GET 
 *                 variable. For example: {|x| IIF(x=NIL,Last,Last:=x) }
 *                 can be used by both browse and the GET system.
 *        Index 3: Optional logical indicating if field can be edited if you
 *                 are allowing edits. By default, if you are allowing edits
 *                 (see below) it is .T., and, obviously, if you are not .F.
 *        Index 4: Optional Picture function for get on this column.
 *        Index 5: Optional Code block to be executed as a WHEN clause.
 *        Index 6: Optional Code block to be executed as a VALID clause.
 *        Index 7: Optional TBColumn:ColorBlock code block for this column
 *        Index 8: Optional Get code block. This code block allows one to 
 *                 format a column with the code block in index 2 and still
 *                 do an edit on the column in spite of your fancy formating.
 *                 The most obvious example is a value that gets formatted
 *                 to show the meaning of a logical yet you wish to edit that
 *                 field. If this is NIL and you wish to do an edit, the block
 *                 from index 2 will be used.
 *        Index 9: Optional column width.
 *
 *        Example to set up last name and first name in array:
 *        aFields := {}
 *        AADD(aFields,{"Last Name",{||Names->Last},.T.,"!XXXXXXXXXXXXXX" })
 *        AADD(aFields,{"First Name",{||Names->First} } )
 *
 *     <bWhileCond> is the limiting WHILE condition as a block.
 *        If cKey (below) is NIL. This is automatically set to {|| .T.}.
 *        If you will be allowing appends you will want to allow arguments to
 *        be passed so that you can use this while block on data entered by the
 *        user prior to adding it to the database. The third example below
 *        shows such a block (as does the example program above).
 *
 *        Example 1: { ||Names->Last == "JONES" }
 *        Example 2: { ||Names->Last == "JONES" .AND. Names->First == "A"  }
 *        Example 3: { |x| IIF(x=NIL, Names->Last = cKey, x = cKey)}
 *
 *     <cKey> is the key to find top condition of WHILE.  
 *        If NIL the entire database is browsed.
 *        cLast  := "JONES     "
 *        cFirst := "A"
 *        Example 1: cKey := cLast
 *        Example 2: cKey := cLast + cFirst
 *
 *     <nFreeze> is number of fields to freeze in TBrowse.  Defaults
 *        to 0 if not passed.
 *
 *     <lSaveScrn> is a logical indicating whether or not you want to
 *        save the screen from the calling program.  Defaults to .T. if
 *        not passed.
 *
 *     <cColorList> is a list of colors for the TBrowse columns.
 *        The 1st color is used as SAY/TBrowse Background and the
 *        3rd and 4th colors are used as part of column:defColor := {3, 4}
 *        The 1st Color is used for the scroll bars if used.
 *
 *        Thus if you pass a cColorList, you MUST pass at least 4 colors.
 *        Defaults to "N/W, N/BG, B/W, B/BG, B/W, B/BG, R/W, B/R" if not 
 *        passed.
 *
 *     <nTop>, <nLeft>, <nBottom>, <nRight> are the coordinates of
 *        the area to display the TBrowse in.  Defaults to 2, 2,
 *        MaxRow(), MaxCol() i.e. full screen.
 *
 *     <bScreenInit> is a code block which calls a routine to initialize the
 *        screen after it has been cleared. You could place help or mouse hot
 *        spots on the screen with this routine. If NIL then nothing happens.
 *
 *     <nReadInit> indicates how a read is initiated. The constants 
 *        GET_ON_RETURN and DOUBLE_CLICK_GET are defined in MBROWSE.CH. 
 *        GET_ON_RETURN indicates that a return will initiate a GET on the 
 *        current field. DOUBLE_CLICK_GET indicates that a mouse double click
 *        of the left button will initiate a GET on the current field. Either 
 *        one or the sum of the two can be used to indicate how to initiate 
 *        a GET. Passing NO_GET or NIL will prevent any editing.
 *       Example:
 *            GET_ON_RETURN+DOUBLE_CLICK_GET   will allow both the mouse and
 *                                             return initiated gets
 *
 *        NOTE: <xDoubleAction> is a more general way to indicate desired
 *        action on mouse double click. This method is support for 
 *        compatability purposes.
 *
 *     <bAltReader> is a code block which is called in place of the internal
 *        reader when the user requests an edit. If NIL then the internal one 
 *        is called. This gives more flexibility in doing reads and actually
 *        can be used for functions other than reads (especially with the 
 *        mouse). The alternate reader is passed the browse object and is 
 *        expected to return either a logical indicating if the browse should
 *        continue or one of the REFRESH_* constants defined in MBROWSE.CH.
 *
 *     <nExitKey> is a InKey code of a key which will exit the browse other
 *        than the escape key which is always active. By default is is set 
 *        to K_RETURN if GETS are not allowed and nothing if they are 
 *        allowed. 
 *
 *     <lAppendMode> indicates if appends to the database are allowed.
 *
 *     <bAppendReader> is a code block which will perform a read and append
 *        when the user requests an append. The routine receives no arguments
 *        but must return the number of records meeting the current while
 *        condition that were appended to the database. NOTE: if you are
 *        appending records under a while condition, do not include records 
 *        which you may have added but which do not meet the criteria.
 *
 *     <bRecordNum> Is an optional code block which will return the
 *        sequence number of the current record within the current scope
 *        including the influence of any WHILE clause given to the browse.
 *        The value is used in presenting the scroll bar. It returns 
 *        the record number of the current record. Unless you are using 
 *        the natural order of the database, this value is NOT equal to 
 *        RECNO(). If the argument is not given, then the default routine 
 *        is used. One might use this routine if one knows, for example, 
 *        that only 1% of a very large database will be used for the 
 *        browse and you have a faster way of counting than the default 
 *        method.
 *
 *     <bRecordTotal> is an optional code block which returns the total number
 *        of records which fall under the scope of the browse. Note: if the
 *        database contains deleted records the value of LASTREC() is 
 *        incorrect if set deleted is on. If no value is given then the
 *        the default method is used when scroll bars are in use.
 *
 *     <bMovedFunction> is an optional code block which will be called whenever
 *        the browse is moved. It is passed two arguments, the browse object
 *        and either CHANGED_RECORD, PANNED_RECORD or their sum (defined in 
 *        MBROWSE.CH). These values indicate that the record has changed or the 
 *        column has changed, respectively. No test to see if the actual 
 *        change has occured. That is, if the user continues to attempt to 
 *        scroll off of the top, bMovedFunction will continue to be called
 *        with the appropriate value of CHANGED_RECORD. It is based upon 
 *        the user's request through either a key stroke or mouse click.
 *
 *     <xDoubleAction> is an optional value which indicates what action is
 *        to be taken when a double click occurs within a field. (As usual,
 *        the double click must occur at the same location.)  If the value of
 *        <xDoubleAction> is DOUBLE_CLICK_EXIT, the browse will exit. If the
 *        value is DOUBLE_CLICK_GET, a get/read is performed. (This is 
 *        redundant with <nReadInit> which is maintained for compatability with
 *        previous versions.) If the value of <xDoubleAction> is a code block,
 *        then the code block is executed when the double click is detected.
 *        The code block should return one of the REFRESH_* constants found in
 *        MBROWSE.CH. The values passed to the code block are <nButton>,
 *        <nRow>, <nCol>, <nTime>, <oBrowse> which are the button used, the
 *        screen coordinates of the mouse cursor, the time of day (from 
 *        SECONDS()) and the browse object. (These arguments are the same as
 *        those passed with a "hot spot" except the value of the hot spot ID
 *        is NIL.) The value of <xDoubleAction> takes precedence over 
 *        <nReadInit>. 
 *
 *     <bKeyHandler> is a code block which calls a routine for handling any
 *        keys which are not already defined (setkey(), arrows, pgup/dn, etc)
 *        This routine receives two arguments the first is the INKEY() value 
 *        of the key and the second is the browse object. For example, this 
 *        block should look like: {|a,b| KeyReceiver(a,b)}. The routine should 
 *        return REFRESH_NONE, REFRESH_CURRENT, REFRESH_ALL, or REFRESH_EXIT 
 *        which will make no changes to screen, refresh the current row, 
 *        refresh the screen or exit the browse. This function can be useful
 *        in implimenting a picklist.
 *
 *  $RETURNS$
 *     <nRecno> is the number of the record selected by the <nExitKey> key or
 *        mouse if appropriate. 0 is returned if there are either no records 
 *        matching the WHILE condition or an <Esc> is pressed instead of an 
 *        <Enter> (or the defined exit key) or mouse double click if enabled.
 *  $DESCRIPTION$
 *     This is a full database browse which implements scoping using an indexed
 *     database and a WHILE condition. It can also work with an unindexed file
 *     if the entire database is being browsed. It also supplies mouse support 
 *     and active scroll bars which can be used with the mouse. Additionally, 
 *     the browse will support editing of current records and appending new 
 *     records.
 *
 *     Two options for determine the logical record number of the current 
 *     record are supplied. The first (default) method uses standard Clipper
 *     calls and does a binary search for find the record number of the 
 *     current and last records.  The second uses the function NTXPOS provided
 *     in NTX2.ZIP originally by Graham D. McKechnie and modified by 
 *     Brian Marasca. These routines interact directly with the index file
 *     to determine the location. They can be turned on by setting the 
 *     appropriate define at the top of the source file.
 *
 *  $EXAMPLES$
 *     * This example will only show those people with last name of "JONES"
 *     * in the TBNames.dbf which contains at least the fields:
 *     * Last, First, City AND is indexed on Last + First.
 *     LOCAL nRecSel    := 0
 *     LOCAL aFields    := {}
 *     LOCAL bWhile     := {||TBNames->Last = "JONES"}
 *     LOCAL cKey       := "JONES"
 *     LOCAL nFreeze    := 1
 *     LOCAL lSaveScrn  := .t.
 *     LOCAL cColorList := "N/W, N/BG, B/W, B/BG, B/W, B/BG, R/W, B/R"
 *     LOCAL bInitScreen:= {|| FancyScreen()}
 *     LOCAL nReadInit := GET_ON_RETURN+DOUBLE_CLICK_GET  // return and dbl clk
 *     LOCAL bAltReader := NIL  // no alternate reader
 *     LOCAL nExitKey := K_ALT_E // alt-E will get you out
 *     LOCAL lAppendMode := .T. // we will be daring and allow appends
 *     LOCAL bAppendReader := {|| MybrowseReader()} // The reader
 *
 *     USE TBNames INDEX TBNames NEW // indexed on Last + First
 *
 *     * Pass Heading as character and Field as Block including Alias
 *     * To eliminate the need to use FIELDWBLOCK() function in BROWSE()
 *     AADD(aFields, {"Last Name" , ;
 *          {|x| IIF(x=NIL,TBNames->Last, TBNames->Last:=x) },.T.  } )
 *     AADD(aFields, {"First Name", ;
 *          {|x| IIF(x=NIL, TBNames->First, TBNames->First:=x) },.T. } )
 *     AADD(aFields, {"City"      , ;
 *          {|x| IIF(x=NIL, TBNames->City, TBNames->City:=x},.T. } )
 *     AADD(aFields, {"Zip Code"  , ;
 *          {|x| IIF(x=NIL, TBNames->City, TBNames->City:=x},.T.,;
 *          "@R 99999-9999"})
 *
 *     IF MBROWSE( aFields, bWhile, cKey, nFreeze, lSaveScrn, ;
 *       cColorList, 3, 6, MaxRow() - 2, MaxCol() - 6, bInitScreen,;
 *       nReadInit, bAltReader, nExitKey, lAppendMode, bAppendReader) == 0
 *       ? "Sorry, NO Records Were Selected"
 *     ELSE
 *        ? "You Selected: " + TBNames->Last +" "+ ;
 *           TBNames->First +" "+ TBNames->City
 *     ENDIF
 *
 * Notes: This browse supplies scroll bars with active arrows for both columns
 *     and rows by default. See BRSetScroll() for how to turn them off.
 *     Mouse functions: left button on arrow moves one item. 
 *                      Right button on arrow pages up/down or pans left/right.
 *                      Both buttons on arrow moves to top/bottom
 *                      Click on data area moves to that item.
 *                      Double click on item allows edit if programmer desires.
 *     This system was written with reentrancy in mind but it has not been
 *          tested for such.
 *
 *     This browse system will handle calling functions set by SET KEY in the
 *     normal way. Three return values (defined in mbrowse.ch) are significant.
 *     REFRESH_CURRENT will cause browse:RefreshCurrent() to be called,
 *     REFRESH_ALL will cause browse:RefreshAll() to be called after your
 *     function is complete and REFRESH_EXIT will cause the browse to exit.
 *     The same return values are applicable to functions called but hot spots.
 *
 *     The information stored about the browse is stored in browse:cargo as an
 *     array. The file mbrowse.ch contains the defines which allow access to
 *     this information including the while code block.
 *
 *  $SEEALSO$
 *     BuildBrowse() BRHotSpot() BRSetScroll() CreateBrowse() RunBrowse() DestroyBrowse()
 *  $END$
 */


FUNCTION MBROWSE(aFields, bWhileCond, cKey, nFreeze, lSaveScrn, ;
				cColorList, nTop, nLeft, nBottom, nRight, ;
				bScreenInit, nReadInit, bAltReader, nExitKey, lAppendMode,;
				bAppendReader, bRecordNum, bRecordTot, bMovedFunction,;
				xDoubleAction, bKeyHandler )

LOCAL nPassRec:=0
LOCAL oTheBrowse     // Array holding browse information

// Create the browse

oTheBrowse=CreateBrowse(aFields, bWhileCond, cKey, nFreeze, lSaveScrn, ;
				cColorList, nTop, nLeft, nBottom, nRight, ;
				bScreenInit, nReadInit, bAltReader, nExitKey, lAppendMode,;
				bAppendReader, bRecordNum, bRecordTot, bMovedFunction,;
				xDoubleAction, bKeyHandler )

// Run the browse

IF oTheBrowse!=NIL
	nPassRec=RunBrowse(oTheBrowse,.F.)

// Clean up screen

	DestroyBrowse(oTheBrowse)
ENDIF

RETURN (nPassRec)

// End of MBrowse

/*  $DOC$
 *  $FUNCNAME$
 *     CREATEBROWSE()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     Creates a database Browse with by a while condition with mouse
 *  $SYNTAX$
 *     CREATEBROWSE( <aFields>, <bWhileCond>, <cKey>
 *                 [,<nFreeze>] [,<lSaveScrn>] [,<cColorList>]
 *                 [,<nTop>] [,<nLeft>]
 *                 [,<nBottom>] [,<nRight>] [,<nReadInit>]
 *                 [,<bAltReader>] [,<nExitKey>] [,<lAppendMode>]
 *                 [,<AppendReader>,] [,<bRecordNum>] [,<bRecordTotal>]
 *                 [,<bMovedFunction>] [,<xDoubleAction>] [,<bKeyHandler>] ) ;
 *           -> <oTheBrowse>
 *
 *                (Note: the arguments are the same as MBROWSE)
 *  $ARGUMENTS$
 *
 *     All arguments are the same as MBROWSE.
 *
 *  $RETURNS$
 *     <oTheBrowse> is the TBROWSE object. See MBROWSE.CH for contents of Cargo
 *
 *  $DESCRIPTION$
 *     This sets up a browse which implements scoping using an indexed
 *     database and a WHILE condition. It can also work with an unindexed file
 *     if the entire database is being browsed. It also supplies mouse support 
 *     and active scroll bars which can be used with the mouse. Additionally, 
 *     the browse will support editing of current records and appending new 
 *     records.
 *
 *     It differs from MBROWSE() in that the TBROWSE object is set up brought
 *     to stabilization and then returned. This can be useful in cases when 
 *     you want to keep the browse object for an extended period of time
 *     but not necessarily on the screen all of the time. You can save the 
 *     information in a STATIC, return it to the screen at some later date
 *     and call RUNBROWSE to start it up again.
 *
 *     Note: it is the responsibility of the caller to cancel any mouse hot 
 *     spots included by the creation the scroll bars. They are returned
 *     by the "meta functions": VERT_SCROLL(oBrowse)[SB_MOUSEID] and
 *     HORIZ_SCROLL(oBrowse)[SB_MOUSEID] which are defined in mbrowse.ch.
 *
 *     Two options for determine the logical record number of the current 
 *     record are supplied. The first (default) method uses standard Clipper
 *     calls and does a binary search for find the record number of the 
 *     current and last records.  The second uses the function NTXPOS provided
 *     in NTX2.ZIP originally by Graham D. McKechnie and modified by 
 *     Brian Marasca. These routines interact directly with the index file
 *     to determine the location. They can be turned on by setting the 
 *     appropriate define at the top of the source file.
 *
 *  $EXAMPLES$
 *     * This example will only show those people with last name of "JONES"
 *     * in the TBNames.dbf which contains at least the fields:
 *     * Last, First, City AND is indexed on Last + First.
 *     LOCAL nRecSel    := 0
 *     LOCAL aFields    := {}
 *     LOCAL bWhile     := {||TBNames->Last = "JONES"}
 *     LOCAL cKey       := "JONES"
 *     LOCAL nFreeze    := 1
 *     LOCAL lSaveScrn  := .t.
 *     LOCAL cColorList := "N/W, N/BG, B/W, B/BG, B/W, B/BG, R/W, B/R"
 *     LOCAL bInitScreen:= {|| FancyScreen()}
 *     LOCAL nReadInit := GET_ON_RETURN+DOUBLE_CLICK_GET  // return and dbl clk
 *     LOCAL bAltReader := NIL  // no alternate reader
 *     LOCAL nExitKey := K_ALT_E // alt-E will get you out
 *     LOCAL lAppendMode := .T. // we will be daring and allow appends
 *     LOCAL bAppendReader := {|| MybrowseReader()} // The reader
 *
 *     USE TBNames INDEX TBNames NEW // indexed on Last + First
 *
 *     * Pass Heading as character and Field as Block including Alias
 *     * To eliminate the need to use FIELDWBLOCK() function in BROWSE()
 *     AADD(aFields, {"Last Name" , ;
 *          {|x| IIF(x=NIL,TBNames->Last, TBNames->Last:=x) },.T.  } )
 *     AADD(aFields, {"First Name", ;
 *          {|x| IIF(x=NIL, TBNames->First, TBNames->First:=x) },.T. } )
 *     AADD(aFields, {"City"      , ;
 *          {|x| IIF(x=NIL, TBNames->City, TBNames->City:=x},.T. } )
 *     AADD(aFields, {"Zip Code"  , ;
 *          {|x| IIF(x=NIL, TBNames->City, TBNames->City:=x},.T.,;
 *          "@R 99999-9999"})
 *
 *     oNewBrowse=CreateBrowse( aFields, bWhile, cKey, nFreeze, lSaveScrn, ;
 *       cColorList, 3, 6, MaxRow() - 2, MaxCol() - 6, ;
 *       nReadInit, bAltReader, nExitKey, lAppendMode, bAppendReader)
 *
 *     RunBrowse(oNewBrowse,.F.)
 *
 * Notes: This browse sets up scroll bars with active arrows for both columns
 *     and rows by default. See BRSetScroll() for how to turn them off.
 *     To use these scroll bars, one needs to use the functions BRMouseFunc()
 *     to handle mouse input.
 *
 *     Mouse functions: left button on arrow moves one item. 
 *                      Right button on arrow pages up/down or pans left/right.
 *                      Both buttons on arrow moves to top/bottom
 *                      Click on data area moves to that item.
 *                      Double click on item allows edit if programmer desires.
 *     This system was written with reentrancy in mind but it has not been
 *          tested for such.
 *
 *     The information stored about the browse is stored in browse:cargo as an
 *     array. The file mbrowse.ch contains the defines which allow access to
 *     this information including the while code block.
 *
 *  $SEEALSO$
 *     MBrowse() BRMouseFunc() BRHotSpot() BRSetScroll() RunBrowse() DestroyBrowse()
 *  $END$
 */


FUNCTION CreateBrowse(aFields, bWhileCond, cKey, nFreeze, lSaveScrn, ;
				cColorList, nTop, nLeft, nBottom, nRight, ;
				bScreenInit, nReadInit, bAltReader, nExitKey, lAppendMode,;
				bAppendReader, bRecordNum, bRecordTot, bMovedFunction, ;
				xDoubleAction, bKeyHandler )

* Local variables

LOCAL aScreen[BSCREENATTR]         // Returned values
LOCAL lKeepScrn
LOCAL cColorBack         // Background color
LOCAL oBrowse            // Browse object

// Set default values

DEFAULT	nFreeze     TO 0, ;
		lSaveScrn   TO .t., ;
		cColorList  TO "N/W, N/BG, B/W, B/BG, B/W, B/BG, R/W, B/R", ;
		nTop        TO 2, ;
		nLeft       TO 2, ;
		nBottom     TO MaxRow(), ;
		nRight      TO MaxCol(), ;
		nReadInit   TO NO_GET, ;
		nExitKey    TO IIF(nReadInit%DOUBLE_CLICK_GET=NO_GET,K_RETURN,K_ESC), ;
		lAppendMode TO .F., ;
		bWhileCond  TO {|| .T.}

lKeepScrn := (PCOUNT() > 6)

   // save old screen and colors 
IF lSaveScrn
   aScreen[BSCREEN] = SAVESCREEN(0, 0, MAXROW(), MAXCOL())
   aScreen[BSCREENROW]=MAXROW()
   aScreen[BSCREENCOL]=MAXCOL()
ELSE
   aScreen[BSCREEN]=NIL
   aScreen[BSCREENROW]=-1
   aScreen[BSCREENCOL]=-1
ENDIF
aScreen[BCOLOROLD]=SetColor()
aScreen[BSCREENINIT]=bScreenInit

// Background Color Is Based On First Color In Passed cColorList
cColorBack := IF(',' $ cColorList, ;
   SUBSTR(cColorList, 1, AT(',', cColorList) - 1), cColorList )

IF .NOT. lKeepScrn
   SetColor(cColorBack)
   CLEAR SCREEN
ENDIF

// Now get the actual browse object

DISPBEGIN()
oBrowse=BuildBrowse(aFields, bWhileCond, cKey, nFreeze, lSaveScrn, ;
				cColorList, nTop, nLeft, nBottom, nRight, ;
				nReadInit, bAltReader, nExitKey, lAppendMode,;
				bAppendReader, bRecordNum, bRecordTot, bMovedFunction, ;
				xDoubleAction, bKeyHandler )

IF oBrowse!=NIL

// Execute caller's screen setup routine

	IF VALTYPE(bScreenInit)="B"
		EVAL(bScreenInit)
	ENDIF

	SAVE_PROC_NAME(oBrowse,PROCNAME(2))
	SAVE_PROC_LINE(oBrowse,PROCLINE(2))

	aScreen[BCURSOR]=SetCursor(SC_NONE)

	SAVE_SCREEN(oBrowse,ACLONE(aScreen))

// Stabilize prior to accepting input to take of of the fast typers

	oBrowse:ForceStable()
ENDIF

DISPEND()

RETURN oBrowse

* End of CreateBrowse



/*  $DOC$
 *  $FUNCNAME$
 *     RUNBROWSE()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     Runs a database Browse built with CreateBrowse 
 *  $SYNTAX$
 *     RUNBROWSE( <oBrowse> [, <lReDisplay>]) -> <aRecNo>
 *
 *  $ARGUMENTS$
 *     <oBrowse> - An array containing browse information created
 *           by CreateBrowse()
 *     <lReDisplay> - Redisplay the entire object.
 *
 *  $RETURNS$
 *     <nRecno> is the number of the record selected by the <nExitKey> key or
 *        mouse if appropriate. 0 is returned if there are either no records 
 *        matching the WHILE condition or an <Esc> is pressed instead of an 
 *        <Enter> (or the defined exit key) or mouse double click if enabled.
 *  $DESCRIPTION$
 *     This runs a browse which implements scoping using an indexed
 *     database and a WHILE condition. It can also work with an unindexed file
 *     if the entire database is being browsed. It also supplies mouse support 
 *     and active scroll bars which can be used with the mouse. Additionally, 
 *     the browse will support editing of current records and appending new 
 *     records.
 *
 *     If <lReDisplay> is .T. then this routine redisplays any scrollbars and
 *     redisplays the browse object by issuing a oBrowse:RefreshAll()
 *
 *     Note: this routine sets the currently active browse. On exit it
 *     restores the screen if requested, the cursor, colors and cools the
 *     scroll bar hot spots if active. Caller is responsible for cooling
 *     and rewarming other hot spots associated with browse.
 *  $EXAMPLES$
 *     oNewBrowse=CreateBrowse( aFields, bWhile, cKey, nFreeze, lSaveScrn, ;
 *       cColorList, 3, 6, MaxRow() - 2, MaxCol() - 6, ;
 *       nReadInit, bAltReader, nExitKey, lAppendMode, bAppendReader)
 *
 *     RunBrowse(oNewBrowse)
 *  $SEEALSO$
 *     MBrowse() BRMouseFunc() BRHotSpot() BRSetScroll() CreateBrowse() DestroyBrowse()
 *  $END$
*/
FUNCTION RunBrowse(oTheBrowse, lReDisplay)

* Local variables

LOCAL lUseVert:=(VERT_SCROLL(oTheBrowse)!=NIL) // Vertical scroll bar in use
LOCAL lUseHoriz:=(HORIZ_SCROLL(oTheBrowse)!=NIL) // Horizontal scroll bar in use
LOCAL lMore              // Continue operation flag
LOCAL nKey               // key hit
LOCAL nMouseKey          // Mouse button hit
LOCAL nAction            // Caller's requested action of movement
LOCAL lDone              // Done with loop flag
LOCAL nPassRec:=0        // Returned record number
LOCAL nTime              // Time mouse clicked
LOCAL nMouseRow:=0       // Row when mouse clicked
LOCAL nMouseCol:=0       // Col when mouse clicked
LOCAL oOldBrowse:=oBrowseCurrent  //Save current browse
LOCAL nRecord            // Current record number

* Put object into file wide static

oBrowseCurrent=oTheBrowse

* Reset cursor to expected

SetCursor(SC_NONE)


* If we need to redisplay the object then do so

IF lReDisplay

* If we need to save the screen again do so

	IF GET_SCREEN(oTheBrowse)[BSCREENROW]!=-1
		GET_SCREEN(oTheBrowse)[BSCREEN]=SAVESCREEN(0,0,MAXROW(),MAXCOL())
		GET_SCREEN(oTheBrowse)[BSCREENROW]=MAXROW()
		GET_SCREEN(oTheBrowse)[BSCREENCOL]=MAXCOL()
	ENDIF

* if we need to do another screen init do it

	IF VALTYPE(GET_SCREEN(oTheBrowse)[BSCREENINIT])="B"
		EVAL(GET_SCREEN(oTheBrowse)[BSCREENINIT])
	ENDIF

* Refresh the browse

	DISPBEGIN()

/* If we are at a record that fits the bill then we will assume the caller
   wants to start here */

	IF .NOT.EVAL(WHILE_BLOCK(oTheBrowse))
/* Find the top and make sure we fit the criteria */
		TbWhileTop(oTheBrowse)
		IF .NOT. FOUND() .OR. LASTREC() == 0 .OR. !EVAL(WHILE_BLOCK(oTheBrowse))
			RETURN(0)
			DISPEND()
		ENDIF

	ENDIF

	nRecord=RECNO()
	oTheBrowse:RefreshAll()
	oTheBrowse:ForceStable()

// Make sure we're still on the right record after stabilizing

	DO WHILE RECNO()<> nRecord .AND. .NOT. oTheBrowse:hitTop()
		oTheBrowse:up()
		oTheBrowse:ForceStable()
	ENDDO

* Get where we are

	SAVE_TOTAL_COUNT(oTheBrowse,CountRec(@nRecord,oTheBrowse,COUNT_BOTH))
	SAVE_RECORD_NUM(oTheBrowse,nRecord)

* Display any scroll bars

#ifndef NO_SIDE_SCROLL
	IF lUseVert
		ScrollBarDisplay(VERT_SCROLL(oTheBrowse))

		ScrollBarUpdate(VERT_SCROLL(oTheBrowse),;
			RECORD_NUM(oTheBrowse), TOTAL_COUNT(oTheBrowse), .T.)
* Warm hot spot
		BRWarmSpot(VERT_SCROLL(oTheBrowse)[SB_MOUSEID])
	ENDIF
#endif

#ifndef NO_BOTTOM_SCROLL
	IF lUseVert
		ScrollBarDisplay(HORIZ_SCROLL(oTheBrowse))

		ScrollBarUpdate(HORIZ_SCROLL(oTheBrowse),OTHEBROWSE:colpos,;
			OTHEBROWSE:colCount, .T.)
* Warm hot spot
		BRWarmSpot(HORIZ_SCROLL(oTheBrowse)[SB_MOUSEID])
	ENDIF
#endif

	DISPEND()

ENDIF

   // Handle input from user if there is any

lMore := EVAL(WHILE_BLOCK(oTheBrowse))

WHILE (lMore)
/* stabilize the display */
	nKey=0
	nMouseKey=0
	DO WHILE (nKey==0 .AND. nMouseKey ==0 .AND. .NOT. oTheBrowse:stable() )
		oTheBrowse:stabilize()
		nKey := INKEY()
#ifndef NO_MOUSE
		nMouseKey := IIF(lMouseUsed,FT_MGETPOS(@nMouseRow,@nMouseCol),0)
#endif

	ENDDO

	IF ( oTheBrowse:stable )

/* Update scroll bars */

#ifndef NO_SIDE_SCROLL
		IF lUseVert
			ScrollBarUpdate(VERT_SCROLL(oTheBrowse),RECORD_NUM(oTheBrowse),;
				TOTAL_COUNT(oTheBrowse), .F.)
		ENDIF
#endif
#ifndef NO_BOTTOM_SCROLL
		IF lUseHoriz
			ScrollBarUpdate(HORIZ_SCROLL(oTheBrowse),oTheBrowse:colpos,;
				oTheBrowse:colCount,.F.)
		ENDIF
#endif

 /* display is stable */
		IF ( oTheBrowse:hitBottom .AND. ALLOW_APPEND_MODE(oTheBrowse) )

/* banged against End Of Logical File; go into append mode */

			DoAppend(oTheBrowse)
			nKey=0
			nMouseKey=0
		ELSE

/* normal mode */
			IF ( oTheBrowse:hitTop .OR. oTheBrowse:hitBottom )
				Tone(125, 0)
			ENDIF

// Make sure that the current record is showing
// up-to-date data in case we are on a network.

			oTheBrowse:refreshCurrent()
			oTheBrowse:ForceStable()

* Everything is stable so call the user defined function which informs the
* caller  of the movement within the browse

			lDone=.F.

			DO WHILE !lDone.AND.lMore

				lDone=.T.
* If we have moved then call user function if defined

				IF KEY_FUNCTION(oTheBrowse)!=NIL.AND.nBrowseMoved!=0

					nAction=EVAL(KEY_FUNCTION(oTheBrowse),oTheBrowse,nBrowseMoved)
					nBrowseMoved=0
				ENDIF

* Refresh screen based upon either KEY_FUNCTION call or setkey function call

				IF nAction!=NIL.AND.VALTYPE(nAction)="N";
					.AND.nAction!=REFRESH_NONE

* Reset flag so we can loop
					lDone=.F.
					IF nAction==REFRESH_CURRENT
						oTheBrowse:refreshCurrent()
					ELSEIF nAction==REFRESH_ALL
						oTheBrowse:RefreshAll()
					ELSEIF nAction==REFRESH_EXIT
						lMore=.F.
					ENDIF
* Restabalize
					oTheBrowse:ForceStable()
					nAction=REFRESH_NONE
				ENDIF
			ENDDO

* everything's done; just wait for a key or mouse input
* Show and hide the cursor so that others won't have to worry
* about writing over it.


			IF lMouseUsed.AND.lMore
#ifndef NO_MOUSE
				FT_MSHOWCRS()

				DO WHILE (nKey=0).AND.(nMouseKey=0)
					nMousekey=FT_MGETPOS(@nMouseRow,@nMouseCol)
					nKey=INKEY()
				ENDDO

* if we have input from the mouse then convert the mouse coordinates
* Get time so we can time a double click if necessary

				IF  nMouseKey>0
					nTime=SECOND()
				ENDIF

				FT_MHIDECRS()
#endif
			ELSEIF lMore
				nMouseKey=0
				nKey=INKEY(0)
			ENDIF
		ENDIF

	ENDIF

 /* process input */

	DO CASE
		CASE nKey == K_ESC
 /* Exit so do it */
			lMore = .F.
			nPassRec=0

 /* Mouse input */
		CASE nMouseKey>0

#ifndef NO_MOUSE
			lMouseExit=.F.

			lMore = BRMouseFunc(oTheBrowse,nMouseKey,INT(nMouseRow/8);
					,INT(nMouseCol/8), nTime)
			IF lMouseExit
				nPassRec=RECNO()
			ELSE
				nPassRec=0
			ENDIF
#endif
 /* Special exit request key */

		CASE nKey == EXIT_KEY(oTheBrowse)
			lMore = .F.
			nPassRec = RECNO()

 /* Normal Key input */
		CASE nKey != 0
			lMore = ApplyKey(nKey,oTheBrowse)
			nPassRec=RECNO()  

 /* Falls through for case where we do an append */

	ENDCASE


ENDDO  // for WHILE (lmore)

/* restore old screen */
IF !EMPTY(GET_SCREEN(oTheBrowse)[BSCREEN])
	RESTSCREEN(0,0,GET_SCREEN(oTheBrowse)[BSCREENROW],;
		GET_SCREEN(oTheBrowse)[BSCREENCOL], GET_SCREEN(oTheBrowse)[BSCREEN])
	GET_SCREEN(oTheBrowse)[BSCREEN]=NIL
ENDIF

/* reset cursor and colors */

SetCursor(GET_SCREEN(oTheBrowse)[BCURSOR])
SetColor(GET_SCREEN(oTheBrowse)[BCOLOROLD])

* Cool the hotspots on scroll bars
 /* Remove hot spots for scroll bars */

#ifndef NO_MOUSE
 #ifndef NO_SIDE_SCROLL
IF VERT_SCROLL(oTheBrowse)!=NIL
	BRCoolSpot(VERT_SCROLL(oTheBrowse)[SB_MOUSEID])
ENDIF
 #endif

 #ifndef NO_BOTTOM_SCROLL
IF HORIZ_SCROLL(oTheBrowse)!=NIL
	BRCoolSpot(HORIZ_SCROLL(oTheBrowse)[SB_MOUSEID])
ENDIF
 #endif
#endif


* restore any previous browse

oBrowseCurrent=oOldBrowse

*Return selected record

RETURN nPassRec

* End of RunBrowse

/*  $DOC$
 *  $FUNCNAME$
 *     DESTROYBROWSE()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     Cleans up after a database Browse created with CreateBrowse
 *  $SYNTAX$
 *     BUILDBROWSE( <oBrowse> ) -> NIL
 *  $ARGUMENTS$
 *     <oBrowse> is the browse object
 *
 *  $EXAMPLES$
 *     oNewBrowse=CreateBrowse( aFields, bWhile, cKey, nFreeze, lSaveScrn, ;
 *       cColorList, 3, 6, MaxRow() - 2, MaxCol() - 6, ;
 *       nReadInit, bAltReader, nExitKey, lAppendMode, bAppendReader)
 *
 *     RunBrowse(oNewBrowse)
 *     DestroyBrowse(oNewBrowse)
 *  $SEEALSO$
 *     MBrowse() BRMouseFunc() BRHotSpot() BRSetScroll() CreateBrowse() RunBrowse()
 *  $END$
*/

FUNCTION DestroyBrowse(oBrowse)


 /* Remove hot spots for scroll bars */

#ifndef NO_MOUSE
 #ifndef NO_SIDE_SCROLL

IF VERT_SCROLL(oBrowse)!=NIL
	BRRemHotSpot(VERT_SCROLL(oBrowse)[SB_MOUSEID])
ENDIF

 #endif

 #ifndef NO_BOTTOM_SCROLL

IF HORIZ_SCROLL(oBrowse)!=NIL
	BRRemHotSpot(HORIZ_SCROLL(oBrowse)[SB_MOUSEID])
	ENDIF

 #endif
#endif


RETURN NIL

* End of DestroyBrowse

/*  $DOC$
 *  $FUNCNAME$
 *     BUILDBROWSE()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     Build a database Browse with by a while condition with mouse
 *  $SYNTAX$
 *     BUILDBROWSE( <aFields>, <bWhileCond>, <cKey>,
 *                 [,<nFreeze>] [,<lSaveScrn>] [,<cColorList>]
 *                 [,<nTop>] [,<nLeft>]
 *                 [,<nBottom>] [,<nRight>] [,<nReadInit>]
 *                 [,<bAltReader>] [,<nExitKey>] [,<lAppendMode>]
 *                 [,<AppendReader>] [,<bRecordNum>] [,<bRecordTotal>]
 *                 [,<bMovedFunction>] [,<xDoubleAction>] [,<bKeyHandler>] ) ;
 *           -> <oBrowse>
 *
 *                (Note: the arguments are the same as MBROWSE except that
 *                <bScreenInit> has been removed.)
 *  $ARGUMENTS$
 *
 *           See MBROWSE() for a list of argument definitions. All except
 *           <bScreenInit> are used.
 *
 *  $RETURNS$
 *     <oBrowse> is the TBROWSE object which is has been created. See
 *           MBROWSE.CH for contents of oBrowse:cargo. If there are no
 *           records which meet the criteria or no fields in aField, 
 *           NIL is returned
 *  $DESCRIPTION$
 *     This sets up a browse which implements scoping using an indexed
 *     database and a WHILE condition. It can also work with an unindexed file
 *     if the entire database is being browsed. It also supplies mouse support 
 *     and active scroll bars which can be used with the mouse. Additionally, 
 *     the browse will support editing of current records and appending new 
 *     records.
 *
 *     It differs from MBROWSE() in that the TBROWSE object is set up but
 *     is not run. This can be useful in cases when you want to have two
 *     browses running at the same time or possibly the GET/READ system
 *     running with a browse active.
 *
 *     Note: it is the responsibility of the caller to cancel any mouse hot 
 *     spots included by the creation the scroll bars. They are returned
 *     by the "meta functions": VERT_SCROLL(oBrowse)[SB_MOUSEID] and
 *     HORIZ_SCROLL(oBrowse)[SB_MOUSEID] which are defined in mbrowse.ch.
 *
 *     Two options for determine the logical record number of the current 
 *     record are supplied. The first (default) method uses standard Clipper
 *     calls and does a binary search for find the record number of the 
 *     current and last records.  The second uses the function NTXPOS provided
 *     in NTX2.ZIP originally by Graham D. McKechnie and modified by 
 *     Brian Marasca. These routines interact directly with the index file
 *     to determine the location. They can be turned on by setting the 
 *     appropriate define at the top of the source file.
 *
 *  $EXAMPLES$
 *     * This example will only show those people with last name of "JONES"
 *     * in the TBNames.dbf which contains at least the fields:
 *     * Last, First, City AND is indexed on Last + First.
 *     LOCAL nRecSel    := 0
 *     LOCAL aFields    := {}
 *     LOCAL bWhile     := {||TBNames->Last = "JONES"}
 *     LOCAL cKey       := "JONES"
 *     LOCAL nFreeze    := 1
 *     LOCAL lSaveScrn  := .t.
 *     LOCAL cColorList := "N/W, N/BG, B/W, B/BG, B/W, B/BG, R/W, B/R"
 *     LOCAL bInitScreen:= {|| FancyScreen()}
 *     LOCAL nReadInit := GET_ON_RETURN+DOUBLE_CLICK_GET  // return and dbl clk
 *     LOCAL bAltReader := NIL  // no alternate reader
 *     LOCAL nExitKey := K_ALT_E // alt-E will get you out
 *     LOCAL lAppendMode := .T. // we will be daring and allow appends
 *     LOCAL bAppendReader := {|| MybrowseReader()} // The reader
 *
 *     USE TBNames INDEX TBNames NEW // indexed on Last + First
 *
 *     * Pass Heading as character and Field as Block including Alias
 *     * To eliminate the need to use FIELDWBLOCK() function in BROWSE()
 *     AADD(aFields, {"Last Name" , ;
 *          {|x| IIF(x=NIL,TBNames->Last, TBNames->Last:=x) },.T.  } )
 *     AADD(aFields, {"First Name", ;
 *          {|x| IIF(x=NIL, TBNames->First, TBNames->First:=x) },.T. } )
 *     AADD(aFields, {"City"      , ;
 *          {|x| IIF(x=NIL, TBNames->City, TBNames->City:=x},.T. } )
 *     AADD(aFields, {"Zip Code"  , ;
 *          {|x| IIF(x=NIL, TBNames->City, TBNames->City:=x},.T.,;
 *          "@R 99999-9999"})
 *
 *     oNewBrowse=BuildBrowse( aFields, bWhile, cKey, nFreeze, lSaveScrn, ;
 *       cColorList, 3, 6, MaxRow() - 2, MaxCol() - 6, ;
 *       nReadInit, bAltReader, nExitKey, lAppendMode, bAppendReader)
 *
 * Notes: This browse sets up scroll bars with active arrows for both columns
 *     and rows by default. See BRSetScroll() for how to turn them off.
 *     To use these scroll bars, one needs to use the functions BRMouseFunc()
 *     to handle mouse input.
 *
 *     Mouse functions: left button on arrow moves one item. 
 *                      Right button on arrow pages up/down or pans left/right.
 *                      Both buttons on arrow moves to top/bottom
 *                      Click on data area moves to that item.
 *                      Double click on item allows edit if programmer desires.
 *     This system was written with reentrancy in mind but it has not been
 *          tested for such.
 *
 *     The information stored about the browse is stored in browse:cargo as an
 *     array. The file mbrowse.ch contains the defines which allow access to
 *     this information including the while code block.
 *
 *  $SEEALSO$
 *     MBrowse() BRMouseFunc() BRHotSpot() BRSetScroll() CreateBrowse() RunBrowse() 
 *  $END$
 */


FUNCTION BuildBrowse(aFields, bWhileCond, cKey, nFreeze, lSaveScrn, ;
				cColorList, nTop, nLeft, nBottom, nRight, ;
				nReadInit, bAltReader, nExitKey, lAppendMode,;
				bAppendReader, bRecordNum, bRecordTot, bMovedFunction, ;
				xDoubleAction, bKeyHandler )

LOCAL column, i
LOCAL nTotalWidth:=0, nFrozenWidth:=0
LOCAL nSepLen, nDepth
LOCAL nCur_Rec
LOCAL aColors
LOCAL oBrowse
LOCAL lContinue

/* local copies indicating use of scroll bars */
LOCAL lUseHoriz:=.F., lUseVert:=.F.

DEFAULT nFreeze     TO 0, ;
		lSaveScrn   TO .t., ;
		cColorList  TO "N/W, N/BG, B/W, B/BG, B/W, B/BG, R/W, B/R", ;
		nTop        TO 2, ;
		nLeft       TO 2, ;
		nBottom     TO MaxRow(), ;
		nRight      TO MaxCol(), ;
		nReadInit   TO NO_GET, ;
		nExitKey    TO IIF(nReadInit%DOUBLE_CLICK_GET=NO_GET,K_RETURN,K_ESC), ;
		lAppendMode TO .F., ;
		bWhileCond  TO {|| .T.}

lContinue=VALTYPE(aFields)="A".AND.LEN(aFields)>0

IF lContinue
/* make new browse object */
	oBrowse := TBrowseNew(nTop, nLeft, nBottom, nRight)

/* initialize the browse system */
	lContinue=init_browse( oBrowse, cKey, bWhileCond, nReadInit, bAltReader, ;
		lAppendMode, nExitKey, xDoubleAction )
ELSE
	ALERT("No Fields Specified in Browse.")
ENDIF

IF lContinue
/* default heading and column separators */
	oBrowse:headSep := ""
	oBrowse:colSep  := "  "
	oBrowse:footSep := ""

	nSepLen=LEN(oBrowse:colSep)

/* add custom 'TbSkipWhil' (to handle passed condition) */
	oBrowse:skipBlock := {|x| TbSkipWhil(x, oBrowse)}

/* Set up substitute goto top and goto bottom */
/* with While's top and bottom records        */
	oBrowse:goTopBlock    := {|| TbWhileTop(oBrowse)}
	oBrowse:goBottomBlock := {|| TbWhileBot(oBrowse)}

/* colors */
	oBrowse:colorSpec := cColorList

	IF cDefScrollColor==NIL
		cScrollBarColor=LEFT(cColorList,AT(",",cColorList)-1)
	ELSE
	     cScrollBarColor=cDefScrollColor
	ENDIF

	SAVE_HEADING_DEPTH(oBrowse,2)
	TOTAL_WIDTH(oBrowse)={}
	ASIZE(TOTAL_WIDTH(oBrowse),LEN(aFields)+1)
	SAVE_TOTAL_WIDTH(oBrowse,1,0)
	SAVE_APPEND_BLOCK(oBrowse, bAppendReader)
	SAVE_PROC_NAME(oBrowse,PROCNAME(1))
	SAVE_PROC_LINE(oBrowse,PROCLINE(1))
	SAVE_ALT_RECNO(oBrowse,bRecordNum)
	SAVE_ALT_RECTOT(oBrowse,bRecordTotal)
	SAVE_KEY_FUNCTION(oBrowse,bMovedFunction)
	SAVE_KEY_HANDLER(oBrowse,bKeyHandler)

/* add a column for each field in the current workarea */
	FOR i = 1 TO LEN(aFields)

   /* make the new column */
		column := TBColumnNew( aFields[i, 1], aFields[i, 2] )

 // If there is a picture and other get parameter variables then save them //

#ifndef NO_EDITS
		column:cargo=ARRAY(C_CARGO_SIZE)
		SAVE_ALLOW_EDIT(column,;
			IIF(LEN(aFields[i])>2, IIF(aFields[i,3]=NIL, .T. ,aFields[i,3]), .T.))
		SAVE_PICTURE(column, IIF(LEN(aFields[i])>3,aFields[i,4],NIL))
		SAVE_WHEN(column, IIF(LEN(aFields[i])>4,aFields[i,5],NIL))
		SAVE_VALID(column, IIF(LEN(aFields[i])>5,aFields[i,6],NIL))
		SAVE_GET_BLOCK(column, IIF(LEN(aFields[i])>7.AND.!EMPTY(aFields[i,8]),;
			aFields[i,8],aFields[i,2]))
#endif

// Check to see if heading for this column is larger than previous

		IF HEADING_DEPTH(oBrowse)<(nDepth:=2+FT_NOOCCUR(";",aFields[i, 1]))
			SAVE_HEADING_DEPTH(oBrowse,nDepth)
		ENDIF

// column default colors

		column:defColor := {3, 4}

// optional color block

		IF LEN(aFields[i])>6
			column:ColorBlock=aFields[i,7]
		ENDIF

		IF LEN(aFields[i])>8.AND..NOT.EMPTY(aFields[i,9])
			column:width=aFields[i,9]
		ENDIF

		oBrowse:addColumn(column)

// Now keep track of the running width of the columns 

		nTotalWidth+=oBrowse:colwidth(i)+nSepLen
		SAVE_TOTAL_WIDTH(oBrowse,i+1,nTotalWidth)

 // Add up width of frozen columns 

		IF i<=nFreeze
			nFrozenWidth+=oBrowse:colwidth(i)+nSepLen
		ENDIF

	NEXT

// freeze columns 
	IF nFreeze <> 0
		oBrowse:freeze := nFreeze
	ENDIF
	SAVE_FREEZE_WIDTH(oBrowse,nFrozenWidth)

// Get the number of records and the current record number
// This is used for scroll bars and updating screen

	SAVE_TOTAL_COUNT(oBrowse,CountRec(@nCur_Rec,oBrowse,COUNT_BOTH))
	SAVE_RECORD_NUM(oBrowse,nCur_Rec)

// If record is not the first then call correctscreen to get the record
// further down the screen

	IF nCur_rec!=1
		CorrectScreen(RECNO(), oBrowse)
	ENDIF

// Initialize the scroll bars 

#ifndef NO_SIDE_SCROLL

	IF lVertScroll

		RecScroll(oBrowse)
		lUseVert=.T.
	ENDIF
#endif

#ifndef NO_BOTTOM_SCROLL
	IF lHorizScroll
		ColScroll(oBrowse)
		lUseHoriz=.T.
	ENDIF
#endif
ENDIF

// Return to caller with the browse object

IF !lContinue
* Failure value
	oBrowse=NIL
ENDIF

RETURN oBrowse


****
* TbSkipWhil()
*
* Special skip function to handle while condition, append mode and mouse
*
* Calling parameters: n - The number of records to skip
*                     oBrowse - The current browse

STATIC FUNCTION TbSkipWhil(n, oBrowse)
LOCAL i

i := 0

* Don't move or no records

IF n == 0 .OR. LASTREC() == 0

// Skip 0 (significant on a network)
	SKIP 0

ELSEIF (n > 0 .and. RECNO() <> LASTREC() + 1)

  // Skip forward
	DO WHILE i < n
		SKIP 1
		IF ( EOF() .OR. .NOT. EVAL(WHILE_BLOCK(oBrowse)))
			SKIP -1
			EXIT
		ENDIF

		i++
	ENDDO

ELSEIF n < 0

 // Skip backward
	DO WHILE i > n
		SKIP -1
		IF .NOT. EVAL(WHILE_BLOCK(oBrowse))
			SKIP 1
			EXIT
		ELSEIF  BOF()
			EXIT
		ENDIF

		i--
	ENDDO

ENDIF

* update the record pointer for the scroll bars

SAVE_RECORD_NUM(oBrowse,RECORD_NUM(oBrowse)+i)

RETURN i

* End of TbSkipWhil()


*****
* TbWhileTop
*
* Function to go to the top of the active portion of the database
*
* Calling parameters: oBrowse - The current browse object
*

FUNCTION TbWhileTop(oBrowse)
LOCAL cSoftSave := SET(_SET_SOFTSEEK, .t.)

IF SEEK_KEY(oBrowse)!=NIL
	SEEK SEEK_KEY(oBrowse)
ELSE
	GOTO TOP
ENDIF

SET(_SET_SOFTSEEK, cSoftSave)

RETURN NIL

*****
*
*  TbWhileBot
*
* Function to go to the bottom of the active portion of the database
*
* Calling parameters: oBrowse - The current browse object
*

FUNCTION TbWhileBot(oBrowse)
   * SeekLast: Finds Last Record For Matching Key
   * Developed By Jon Cole
   * With softseek set on, seek the first record after condition.
   * This is accomplished by incrementing the right most character of the
   * string cKey by one ascii character.  After SEEKing the new string,
   * back up one record to get to the last record which matches cKey.


LOCAL cKey

cKey=SEEK_KEY(oBrowse)
IF cKey=NIL
	GOTO BOTTOM
ELSE
	SEEK LEFT(cKey, LEN(cKey) -1) + CHR( ASC( RIGHT(cKey,1) ) +1) SOFTSEEK
	SKIP -1
ENDIF

RETURN NIL

******
* Init_Browse()
*
* Initializes the browse system for this browse object includeing the mouse
*
* Calling Parameters:  oBrowse - The browse object to work on
*                      cKey - Character key for seek if browse in funtioning
*                              in "while" mode
*                      bWhileCond - Code block for evaluating the while 
*                              condition
*                      nReadInit - action of hitting return flag
*                      bAltReader - any alternate reader for gets
*                      lAppendMode - Logical indicating allow append during
*                              browse
*                      nExitKey - Key used to exit plus possibly 
*                                 DOUBLE_CLICK_EXIT for mouse exit
*
* Returns: .F. if no records are present or none fit while condition
*

STATIC FUNCTION init_browse( oBrowse, cKey, bWhileCond, nReadInit, bAltReader, ;
				        lAppendMode, nExitKey,xDoubleAction )

LOCAL cDoubleType       // Type of value of xDoubleAction

/* fill in the cargo variables */

oBrowse:cargo:=ARRAY(B_CARGO_SIZE)

/* default actions */

#ifndef NO_APPENDS

SET_ALLOW_APPEND(oBrowse,lAppendMode)

#else

SET_ALLOW_APPEND(oBrowse,.F.)

#endif

* Save what to do if the user hits return and what the exit key is

SAVE_RETURN_ACTION(oBrowse,nReadInit%DOUBLE_CLICK_GET)
SAVE_EXIT_KEY(oBrowse,nExitKey)

* Determine what will happen when a double click is detected within the browse

IF (cDoubleType:=VALTYPE(xDoubleAction))="B"

* Code block passed so it will be save for future execution

	SAVE_DOUBLE_BLOCK(oBrowse,xDoubleAction)
	SAVE_DOUBLE_CLICK(oBrowse,DOUBLE_BLOCK)

ELSEIF cDoubleType="N"

* Specifically what to do from caller

	SAVE_DOUBLE_CLICK(oBrowse,xDoubleAction)

ELSE
* Default value and compatability with passing through nReadInit
	
	SAVE_DOUBLE_CLICK(oBrowse,;
		IIF(nReadInit/DOUBLE_CLICK_GET>0.9,DOUBLE_CLICK_GET,DOUBLE_NONE))
ENDIF

* Save any alternate reader code

SAVE_ALT_READER(oBrowse,bAltReader)

/* If we are using the while portion then test for a record being present */

IF cKey!=NIL

/* Save the key and while block information */

	SAVE_SEEK_KEY(oBrowse,cKey)
	SAVE_WHILE_BLOCK(oBrowse,bWhileCond)

/* If we are at a record that fits the bill then we will assume the caller
   wants to start here */

	IF .NOT.EVAL(bWhileCond)
/* Find the top and make sure we fit the criteria */
		TbWhileTop(oBrowse)
		IF .NOT. FOUND() .OR. LASTREC() == 0 .OR. !EVAL(bWhileCond)
			RETURN(.F.)
		ENDIF

	ENDIF

ELSE

/* Use default values if while condition not being used */

	SAVE_SEEK_KEY(oBrowse,NIL)
	SAVE_WHILE_BLOCK(oBrowse,{|| .T.})
/* We will assume the caller wants to stay at this record */

ENDIF

/* Initialize mouse if present */

IF !lIgnoreMouse

#ifndef NO_MOUSE
	lMouseUsed=FT_MINIT()
#endif
ELSE
	lMouseUsed=.F.
ENDIF

/* Now return */

RETURN .T.

#ifndef NO_SIDE_SCROLL
*****
*
* RecScroll()
*
* Purpose: initialize scroll bars for side
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     5/30/91   LJL       Initial Version
*         V1.01    12/29/92   LJL       Added calling parameter
*
* Calling Parameters: oBrowse - Current browse object
*
* Returns: NIL
*
STATIC FUNCTION RecScroll(oBrowse)

LOCAL nT, nL, nB, nR
LOCAL aBar[SB_ELEMENTS]

#ifdef LEFT_SCROLL

* If we have no room then don't do any work

IF oBrowse:nLeft>0

* build the scroll bar and display it

	SAVE_VERT_SCROLL(oBrowse,ScrollBarNew(OBrowse:nTop,OBrowse:nLeft-1,;
		OBrowse:nBottom,cScrollBarColor,1,SB_VERTICAL))
#else

IF oBrowse:nRight<MAXCOL()

	SAVE_VERT_SCROLL(oBrowse,ScrollBarNew(OBrowse:nTop,OBrowse:nRight+1,;
		OBrowse:nBottom,cScrollBarColor,1,SB_VERTICAL))
#endif

	ScrollBarDisplay(VERT_SCROLL(oBrowse))

	ScrollBarUpdate(VERT_SCROLL(oBrowse),;
		RECORD_NUM(oBrowse), TOTAL_COUNT(oBrowse), .T.)

* Save the hot spot in the browse

#ifndef NO_MOUSE
	VERT_SCROLL(oBrowse)[SB_MOUSEID]=BRHotSpot(VERT_SCROLL(oBrowse)[SB_ROWTOP],;
		VERT_SCROLL(oBrowse)[SB_COLTOP],;
		VERT_SCROLL(oBrowse)[SB_ROWBOTTOM],;
		VERT_SCROLL(oBrowse)[SB_COLBOTTOM],;
		{|nB,nRow,nCol,nTime,oB| mousescroll(nB,nRow,nCol,nTime,oB)},;
		0,0.1,.F.)
#endif

ENDIF

RETURN NIL

* End of RecScrol

#endif

#ifndef NO_BOTTOM_SCROLL
*****
*
* ColScroll()
*
* Purpose: initialize scroll bars for bottom
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     6/1/91    LJL       Initial Version
*         V1.01    12/29/92   LJL       Added calling parameter
*
* Calling Parameters: oBrowse - Current browse object
*
* Returns: NIL
*
STATIC FUNCTION ColScroll(oBrowse)

* if we have room then do the work

IF oBrowse:nBottom<MAXROW()
* build the scroll bar and display it

	SAVE_HORIZ_SCROLL(oBrowse,ScrollBarNew(OBrowse:nBottom+1,OBrowse:nLeft,;
		OBrowse:nRight,cScrollBarColor,1,SB_HORIZONTAL))

	ScrollBarDisplay(HORIZ_SCROLL(oBrowse))

	ScrollBarUpdate(HORIZ_SCROLL(oBrowse),OBrowse:colpos,OBrowse:colCount, .T.)

* Add a mouse event for the scroll bar and save it in the browse

#ifndef NO_MOUSE
	HORIZ_SCROLL(oBrowse)[SB_MOUSEID]=BRHotSpot(HORIZ_SCROLL(oBrowse);
		[SB_ROWTOP],;
		HORIZ_SCROLL(oBrowse)[SB_COLTOP],;
		HORIZ_SCROLL(oBrowse)[SB_ROWBOTTOM],;
		HORIZ_SCROLL(oBrowse)[SB_COLBOTTOM],;
		{|nB,nRow,nCol,nTime,oB| mousescroll(nB,nRow,nCol,nTime,oB)},;
		0,0.1,.F.)
#endif

ENDIF

RETURN NIL

* End of ColScrol
#endif

*****
*
* ApplyKey
*
* This routine handles the appropriate action for each key entered
*
* Calling parameters: nKey - the inkey code of the pressed key
*
* Returns: .T. if the browse should continue, .F. if it should not. This
*           action is based upon the callers indication of action to be
*           taken on the user hitting return
*
STATIC FUNCTION ApplyKey(nKey, oBrowse)

* Local variables

LOCAL lMore:=.T.  // signal to continue browse - default is yes
LOCAL i, j, k, done
LOCAL nAction:=REFRESH_NONE
LOCAL bKeyBlock
LOCAL nCur_Rec           // current logical record number
LOCAL nRecNo             // current physical record number
LOCAL nColPos            // Current column in browse

* handle the movement keys

DO CASE

* Any user defined keys

CASE ((bKeyBlock:=SetKey(nKey)) <> NIL)

	nRecNo=RECNO()
	nColPos=oBrowse:ColPos
	nAction=EVAL(bKeyBlock,PROC_NAME(oBrowse), PROC_LINE(oBrowse))
* signal any changes

	IF RECNO()!=nRecNo
		nBrowseMoved=CHANGED_RECORD
	ENDIF
	IF nColPos!=oBrowse:ColPos
		nBrowseMoved+=PANNED_RECORD
	ENDIF
	IF nAction=NIL.OR.VALTYPE(nAction)!="N"

* Perform requested action

	ELSEIF nAction==REFRESH_CURRENT
		oBrowse:refreshCurrent()
	ELSEIF nAction==REFRESH_ALL
		CountRec(@nCur_Rec,oBrowse,RECNUM)
		SAVE_RECORD_NUM(oBrowse,nCur_Rec)
		CorrectScreen(RECNO(),oBrowse)
	ELSEIF nAction==REFRESH_EXIT
		lMore=.F.
	ENDIF

CASE ( nKey == K_DOWN )
	oBrowse:down()
	nBrowseMoved:=CHANGED_RECORD

CASE ( nKey == K_UP )
	oBrowse:up()
	nBrowseMoved:=CHANGED_RECORD

CASE ( nKey == K_PGDN )
	oBrowse:pageDown()
	nBrowseMoved:=CHANGED_RECORD

CASE ( nKey == K_PGUP )
	oBrowse:pageUp()
	nBrowseMoved:=CHANGED_RECORD

CASE ( nKey == K_CTRL_PGUP )
	SAVE_RECORD_NUM(oBrowse,1)
	oBrowse:goTop()
	nBrowseMoved:=CHANGED_RECORD

CASE ( nKey == K_CTRL_PGDN )

* Note: we must save the record number before calling oBrowse:goBottom since
* the browse object will call the skipper to move the display to the top of
* the screen page in the goBottom but waits until stabilization time to move
* back to the last record.

	SAVE_RECORD_NUM(oBrowse,TOTAL_COUNT(oBrowse))
	oBrowse:goBottom()
	nBrowseMoved:=CHANGED_RECORD

CASE ( nKey == K_RIGHT )
	oBrowse:right()
	nBrowseMoved:=PANNED_RECORD

	CASE ( nKey == K_LEFT )
	oBrowse:left()
	nBrowseMoved:=PANNED_RECORD

CASE ( nKey == K_HOME )
	oBrowse:home()
	nBrowseMoved:=PANNED_RECORD

CASE ( nKey == K_END )
	oBrowse:end()
	nBrowseMoved:=PANNED_RECORD

CASE ( nKey == K_CTRL_LEFT )
	oBrowse:panLeft()
	nBrowseMoved:=PANNED_RECORD

CASE ( nKey == K_CTRL_RIGHT )
	oBrowse:panRight()
	nBrowseMoved:=PANNED_RECORD

CASE ( nKey == K_CTRL_HOME )
	oBrowse:panHome()
	nBrowseMoved:=PANNED_RECORD

CASE ( nKey == K_CTRL_END )
	oBrowse:panEnd()
	nBrowseMoved:=PANNED_RECORD

CASE ( nKey == K_ESC )
	lMore := .f.

CASE ( nKey == K_RETURN )

* Action is based upon caller's directive

	IF RETURN_ACTION(oBrowse)==GET_ON_RETURN
		lMore := DoGet(oBrowse)
	ENDIF

OTHERWISE

* No action taken so ring the bell

	IF KEY_HANDLER(oBrowse)=NIL
		Tone(125, 0)
	ELSE
		nRecNo=RECNO()
		nColPos=oBrowse:ColPos

		nAction=EVAL(KEY_HANDLER(oBrowse),nKey,oBrowse)
* signal any changes

		IF RECNO()!=nRecNo
			nBrowseMoved=CHANGED_RECORD
		ENDIF
		IF nColPos!=oBrowse:ColPos
			nBrowseMoved+=PANNED_RECORD
		ENDIF

		IF nAction=NIL.OR.VALTYPE(nAction)!="N"

* Perform requested action

		ELSEIF nAction==REFRESH_CURRENT
			oBrowse:refreshCurrent()
		ELSEIF nAction==REFRESH_ALL
* Things have changed a great deal so recount and repaint
			CountRec(@nCur_Rec,oBrowse,RECNUM)
			SAVE_RECORD_NUM(oBrowse,nCur_Rec)
			CorrectScreen(RECNO(),oBrowse)
		ELSEIF nAction==REFRESH_EXIT
			lMore=.F.
		ENDIF
	ENDIF

ENDCASE

* Done so return

RETURN lMore

* End of ApplyKey

***
*   DoGet()
*   Do a GET for the current column in the browse.
*
* Calling parameters: oBrowse - Current browse object
*
* Returns: logical indicating if browse should continue. It is set to .F.
*          when the user has modified the only record showing and it no longer
*          fits the while condition
*
* Note: Assumse that the active index (if used) is not changed during the
*       read.
*
STATIC FUNCTION DoGet(oBrowse)

LOCAL lMore:=.T.
LOCAL xMore

#ifndef NO_EDITS

LOCAL bIns, lScore, lExit
LOCAL oCol, getList, nKey, new_rec := 0
LOCAL xOldKey, xNewRecord, nRecIncr, nRecNum, lForceUpdate:=.F.
LOCAL bIndexBlock
LOCAL cIndexKey
LOCAL cReturnType
LOCAL nRecNo         // physical record number
LOCAL nColPos        // column position

// Get the current column object from the oBrowse

oCol := oBrowse:getColumn(oBrowse:colPos)

// If the user is allowed to edit the field then do so

IF ALLOW_EDIT(oCol)

// Make sure screen is fully updated, dbf position is correct, etc.
	oBrowse:ForceStable()

// If the caller supplied an alternate reader code block then use it

	IF VALTYPE(ALT_READER(oBrowse))="B"

		nRecNo=RECNO()
		nColPos=oBrowse:ColPos
		cReturnType=VALTYPE(xMore:=EVAL(ALT_READER(oBrowse)))

* signal any changes

		IF RECNO()!=nRecNo
			nBrowseMoved=CHANGED_RECORD
		ENDIF
		IF nColPos!=oBrowse:ColPos
			nBrowseMoved+=PANNED_RECORD
		ENDIF

* Logical returned so don't translate

		IF cReturnType=="L"
			lMore=xMore
		ELSEIF cReturnType=="N"
* Standard refresh returns
			IF xMore==REFRESH_CURRENT
				oBrowse:refreshCurrent()
			ELSEIF xMore==REFRESH_ALL
				CountRec(@nRecNo,oBrowse,RECNUM)
				SAVE_RECORD_NUM(oBrowse,nRecNo)
				CorrectScreen(RECNO(),oBrowse)
			ELSEIF xMore==REFRESH_EXIT
				lMore=.F.
			ENDIF

		ENDIF

// otherwise use our own

	ELSE

// Save the current record's key value (or NIL)
// (for an explanation, refer to the rambling note below)

		cIndexKey=INDEXKEY()
		bIndexBlock=&("{|| "+IIF(EMPTY(cIndexKey),".T.",cIndexKey)+"}")
		xOldKey := EVAL(bIndexBlock)

// save alternate record in case we change the index value and will
// no longer fit into the while condition
		nLastRec=0
		IF RECORD_NUM(oBrowse)=1 .AND. SEEK_KEY(oBrowse)!=NIL
			SKIP 1
			IF EVAL(WHILE_BLOCK(oBrowse))
				nLastRec=RECNO()
				nRecIncr=1
			ENDIF
			SKIP -1
		ELSEIF SEEK_KEY(oBrowse)!=NIL
			SKIP -1
			IF EVAL(WHILE_BLOCK(oBrowse))
				nLastRec=RECNO()
				nRecIncr=-1
				SKIP 1
			ELSE
				SKIP 2
				IF EVAL(WHILE_BLOCK(oBrowse)).AND..NOT.EOF()
					nLastRec=RECNO()
					nRecIncr=1
				ENDIF
				SKIP -1
			ENDIF
		ENDIF

// Save global state
		lScore := Set(_SET_SCOREBOARD, .F.)
		lExit := Set(_SET_EXIT, .T.)
		bIns := SetKey(K_INS)

// Set insert key to toggle insert mode and cursor shape
		SetKey( K_INS, {|| InsToggle()} )

// Set initial cursor shape
		SetCursor( IF(ReadInsert(), SC_INSERT, SC_NORMAL) )

// Create a corresponding GET

		GetList := {GetNew(Row(),Col(),GET_BLOCK(oCol),oCol:heading;
			,GET_PICTURE(oCol);
			,oBrowse:colorSpec)}
		GetList[1]:preBlock=GET_WHEN(oCol)
		GetList[1]:postBlock=GET_VALID(oCol)

// Read it using the standard reader
// NOTE: for a shared database, an RLOCK() is required here

		READ

// If the record no longer fits the while condition then move to the
// alternate record
		IF .NOT. EVAL(WHILE_BLOCK(oBrowse))
			IF nLastRec>0
				GOTO nLastRec
* Reset scroll bar counters

				SAVE_TOTAL_COUNT(oBrowse,TOTAL_COUNT(oBrowse)-1)
				SAVE_RECORD_NUM(oBrowse,RECORD_NUM(oBrowse)+nRecIncr)
				lForceUpdate=.T.
			ELSE
				lMore=.F.
			ENDIF
* If the index value has changed then reset the counters

		ELSEIF EVAL(bIndexBlock)<> xOldKey
// save current record information

			CountRec(@nRecNum,oBrowse,RECNUM)
			SAVE_RECORD_NUM(oBrowse,nRecNum)
			lForceUpdate=.T.

		ENDIF

// Restore state

		SetCursor(0)
		Set(_SET_SCOREBOARD, lScore)
		Set(_SET_EXIT, lExit)
		SetKey(K_INS, bIns)

* Get the record's key value (or NIL) after the GET
		xNewRecord := RECNO() 

* If the key has changed (or if this is a new record)

		IF lMore .AND. lForceUpdate

* Try to force screen to correct position

			CorrectScreen(xNewRecord, oBrowse)

		ENDIF


    // Check exit key from get
		nKey := LASTKEY()
		IF nKey == K_UP .OR. nKey == K_DOWN .OR.;
			 nKey == K_PGUP .OR. nKey == K_PGDN

        // Ugh
			KEYBOARD( CHR(nKey) )
		ENDIF

	ENDIF
ENDIF

#endif

RETURN lMore

* End of DoGet


*******
*  DoAppend
*
* This routine will finish the handling of appended records
*
* Calling parameters: oBrowse - Current browse object
*
* Returns: NIL
*

STATIC FUNCTION DoAppend(oBrowse)

#ifndef NO_APPENDS

LOCAL xNewRecord, xOldRecord, nNewRec, nNew, nLastRec
LOCAL lScore, lExit, bIns, nOldRec


// Make sure screen is fully updated, dbf position is correct, etc.

oBrowse:ForceStable()

// Save the current record Number
// (for an explanation, refer to the rambling note above in DoGet)
xOldRecord := RECNO()

// Save global state
lScore := Set(_SET_SCOREBOARD, .F.)
lExit := Set(_SET_EXIT, .T.)
bIns := SetKey(K_INS)

// Set insert key to toggle insert mode and cursor shape
SetKey( K_INS, {|| InsToggle()} )

// Set initial cursor shape
SetCursor( IF(ReadInsert(), SC_INSERT, SC_NORMAL) )

// Now use the callers routine to get new records. The number returned is
// the number of new records. 

IF ((nNew:=EVAL(APPEND_BLOCK(oBrowse)))=0) .OR.;
	(.NOT.EVAL(WHILE_BLOCK(oBrowse)) )

* If we were not left on a valid record return to the bottom again
	SAVE_RECORD_NUM(oBrowse,TOTAL_COUNT(oBrowse))
	oBrowse:goBottom()
	oBrowse:forceStable()
ENDIF

* Restore state
SetCursor(0)
Set(_SET_SCOREBOARD, lScore)
Set(_SET_EXIT, lExit)
SetKey(K_INS, bIns)

* Reset the Browse based if things have changed. See Long winded explanation
* above in DOGET for what is going on here

// Get the record's key value (or NIL) after the GET
xNewRecord := RECNO() 

// If the key has changed (or if this is a new record)

IF  .NOT.(xNewRecord == xOldRecord) .OR. nNew > 0

	CorrectScreen(xNewRecord, oBrowse) // Fix the screen

ENDIF


* Reset scroll bar counters

SAVE_TOTAL_COUNT(oBrowse,TOTAL_COUNT(oBrowse)+nNew)
CountRec(@nNewRec,oBrowse,RECNUM)
SAVE_RECORD_NUM(oBrowse,nNewRec)

#endif

RETURN NIL

* End of DoAppend

******
*
*   CorrectScreen
*
* Purpose: correct the browse screen after edits and large moves
*
* Calling Parameters:  nRecord  - The current record number
*                      oBrowse - Current browse object
*
* Returns: NIL
*
* This was moved from TBDEMO.PRG. The explanation follows:
*
* What this next piece of code does:
*
* When a TBrowse stabilizes, it always tries to leave the
* same "cell" highlighted as was previously highlighted. That
* is, it always tries to keep the highlight at the same position
* within the browse window unless it is explicitly moved via an
* "up" or "down" message. The TBrowse positions the data source
* in a corresponding fashion. If there aren't enough rows left
* in the data source (i.e. EOF is encountered while trying to
* adjust the database to match the window), the TBrowse will
* relent and move the cursor upward, leaving it on the correct
* record but with part of the window unfilled.
*
* That works OK for logical EOF, but a problem can occur when
* a GET on a key field causes the current record to move so
* close to logical BOF that it is impossible to highlight the
* correct record while leaving the highlight at its previous
* position within the window. In this case, TBrowse opts to
* leave the highlight in the same position within the window,
* even though that position no longer corresponds with the same
* record as before. That is, it repositions the database as far
* as it will go, then leaves the highlight where it was. The
* result is that you end up with the highlight on a different
* record than the one you just edited.
*
* The following piece of code addresses this by checking to see
* if the current record's key value changed during the GET. If
* so (or if the record is a new record, just appended), the code
* below forces a complete refresh and a full stabilization. It
* then checks to see if this caused the TBrowse to position the
* database to a different record than before. If so, the old
* record is assumed to be somewhere "above" the current record,
* and a series of "up" messages are issued to the browse to get
* the highlight to move up to the proper position.

STATIC FUNCTION CorrectScreen(nRecord, oBrowse)

LOCAL nBack, nOldRec, i

DISPBEGIN()

// Do a complete refresh

oBrowse:refreshAll()
oBrowse:ForceStable()

// Make sure we're still on the right record after stabilizing

DO WHILE RECNO()<> nRecord .AND. .NOT. oBrowse:hitTop()
	oBrowse:up()
	oBrowse:ForceStable()
ENDDO

 // Check to see if we have a full screen. If not and we have a enough
 // records then move the display around to get a full screen.

nBack=TOTAL_COUNT(oBrowse)-RECORD_NUM(oBrowse)

IF  oBrowse:RowCount-oBrowse:RowPos> nBack

	nOldRec=RECORD_NUM(oBrowse)
	oBrowse:GoBottom()
	FOR i= 1 TO nBack
		oBrowse:up()
	NEXT
	oBrowse:ForceStable()
	SAVE_RECORD_NUM(oBrowse,nOldRec)
ENDIF

DISPEND()

RETURN NIL

* End of CorrectScreen


/*
***
*   ForceStable()
*   Force a complete stabilization of a TBrowse.
*

FUNCTION ForceStable(browse)

DISPBEGIN()

DO WHILE .NOT. browse:stabilize()
ENDDO

DISPEND()

RETURN NIL

* End of ForceStable
*/

#ifndef NEITHER_CHANGER

***
*   InsToggle()
*   Toggle the global insert mode and the cursor shape.
*

STATIC FUNCTION InsToggle()

IF READINSERT()
	READINSERT(.F.)
	SETCURSOR(SC_NORMAL)

ELSE
	READINSERT(.T.)
	SETCURSOR(SC_INSERT)

ENDIF

RETURN NIL

* end of insToggle

#endif

*****
*
* CountRec()
*
* Purpose: Counts the number of active records for use on scroll bars
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     5/10/91   LJL       Initial Version
*         V1.01    10/01/91   LJL       Fixed single record in DBF error
*         V1.02     4/17/92   LJL       Changed skip_amt calculation for
*                                       determining the logical record number
*                                       to use the number of logical records
*                                       instead of the number of physical 
*                                       records
*         V1.03    11/10/92   LJL       Added check for empty database. Added
*                                       support for NTXPOS
*
* Calling Parameters: nCur_rec - BY REFERENCE - the position of the
*                              current record
*                     browse - The current browse object
*                     nAction - The desired action - COUNT_BOTH or RECNUM
*
* Returns: number of records active
*
STATIC FUNCTION CountRec(nCur_rec, browse, nAction)

LOCAL nNumRecords:=NIL
LOCAL old_skip, skip_amt, done, nNumCur, save_rec
LOCAL last_rec, i, incr, Rec_cnt, first_record
LOCAL nNumRec:=LASTREC(), nTopRec
LOCAL nIndexOrd

* if we have only one record then just set the values since algorithm below
* needs at least two records

IF nNumRec=1
	nNumRecords=1
	nCur_rec=1

* Otherwise we need to look
ELSEIF nNumRec=0
	nNumRecords=0
	nCur_rec=0

ELSE

* Save current record number

	save_rec=RECNO()

	IF (nIndexOrd:=INDEXORD())>0

#ifdef USE_NTXPOS

* An index is active so use it to determine the position for NTXPOS()
* Clipper 5.2 requires adding 2 to the index order number!

		TbWhileTop(browse)
		nTopRec=NTXPOS(nIndexOrd,RECNO())
		nCur_rec=1+NTXPOS(nIndexOrd,save_rec)-nTopRec
		IF nAction=COUNT_BOTH
			TbWhileBot(browse)
			nNumRecords=1+NTXPOS(nIndexOrd,RECNO())-nTopRec
		ENDIF

#else

* Go to top record

		IF nAction=RECNUM
			incr=-1
			i=2
			nNumRecords=TOTAL_COUNT(browse)
		ELSE
			TbWhileTop(browse)
			first_record=RECNO()

			incr=1
			i=1
		ENDIF

		DO WHILE i<3

* i = 1 finds the total number of records
* i = 2 finds the sequential number of the current record

			done=.F.
			last_rec=RECNO()
			Rec_cnt=1

* if an alternate is given then use it

			IF i=1 .AND. VALTYPE(ALT_RECTOT(browse))="B"
				Rec_cnt=EVAL(ALT_RECTOT(browse))
				done=.T.
			ELSEIF i=2 .AND. VALTYPE(ALT_RECNO(browse))="B"
				Rec_cnt=EVAL(ALT_RECNO(browse))
				done=.T.
			ENDIF

* Initial amount to skip

			skip_amt=INT(IIF(i=1,LASTREC(),nNumRecords)/2)*incr

			DO WHILE !done

* skip and see where we are

				SKIP skip_amt
				IF (IIF(incr=-1,BOF(),EOF())) .OR. ;
					!EVAL(WHILE_BLOCK(browse))

*  hit the end so we really don't know what happened

					old_skip=Skip_amt
					skip_amt=INT((skip_amt+incr)/2)
* If we are skipping no more then we are done
					done=skip_amt==incr.AND.old_skip==incr
					GOTO last_rec
				ELSE

* Not to the end so add the amount we skipped

					Rec_cnt+=skip_amt*incr
					last_rec=RECNO()
					old_skip=skip_amt
					skip_amt=INT((skip_amt+incr)/2)
				ENDIF

			ENDDO

*move only one so we are maybe done

			IF i=1
				nNumRecords=Rec_cnt
				incr=-1
				GOTO save_rec
				i++
				IF save_rec=first_record
					nCur_rec=1
					i=4
				ENDIF
			ELSE
				nCur_rec=Rec_cnt
				i=4
			ENDIF

		ENDDO

#endif
	ELSE

* No index so just take the physical record number

		nCur_Rec=RECNO()
		TbWhileBot(browse)
		nNumRecords=RECNO()
	ENDIF

* Go back to original record

	GOTO save_rec

ENDIF


* Return number of records

RETURN nNumRecords

* End of CountRec

*****
*
* ActiveBrowse()
*
* Purpose: returns the currently active browse
/*  $DOC$
 *  $FUNCNAME$
 *     ACTIVEBROWSE()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     Returns the currently active browse object
 *  $SYNTAX$
 *     ActiveBrowse() -> <oBrowse>
 *  $ARGUMENTS$
 *     None
 *  $RETURNS$
 *     <oBrowse> - The currently active browse
 *  $DESCRIPTION$
 *     This routine returns the currently active browse so that one can
 *     get any of the exported instance variables.
 *  $EXAMPLES$
 *     IF nRow >ActiveBrowse():nTop
 *         <Do something>
 *     ENDIF
 *  $SEEALSO$
 *     MBrowse() RunBrowse()
 *  $INCLUDE$
 *     
 *  $END$
 */
*
FUNCTION ActiveBrowse

RETURN(oBrowseCurrent)

* End of ActiveBrowse

*****
*
* function BRSetScroll()
*
* Purpose: Turn off/on horizontal or vertical scroll bars for browse
*          
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     4/21/92   LJL       Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     BRSETSCROLL()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     Turn scroll bars on/off in MBROWSE 
 *  $SYNTAX$
 *     BRIGNOREMOUSE( <nType>, <lOn> ) -> <lCurStatus>
 *  $ARGUMENTS$
 *     <nType> - Either HORIZSCROLL or VERTSCROLL which are defined
 *               in MBROWSE.CH which signifies which scroll bar you are
 *               making the change or request about.
 *     <lOn> - logical for state of scroll bar .T. = scroll bar is to 
 *             be used, .F. = scroll bar is not to be used. If this argument
 *             is absent then the routine returns the current status.
 *  $RETURNS$
 *     <lCurStatus> - The current setting of use scroll bar flag.
 *  $DESCRIPTION$
 *     This routine allows one to turn on or off the use of either of the
 *     scroll bars used within MBROWSE. This routine must be called prior
 *     to running MBROWSE since it determines how the browse is built which
 *     in turn determines its behavior.
 *
 *     If the second argument is missing then this routine simply returns
 *     the current setting of the appropriate flag.
 *  $EXAMPLES$
 *     lCurHoriz=BRSetScroll(HORIZ_SCROLL,.T.)
 *  $SEEALSO$
 *     
 *  $INCLUDE$
 *     
 *  $END$
 */
*
FUNCTION BRSetScroll( nType, lOn )

* Local Parameters
LOCAL oldsetting

*Save old value
oldsetting=IIF(nType=HORIZSCROLL,lHorizScroll,lVertScroll)

IF lOn!=NIL
	IF nType=HORIZSCROLL
		lHorizScroll=lOn
	ELSEIF nType=VERTSCROLL
		lVertScroll=lOn
	ENDIF
ENDIF

RETURN oldsetting

* End of BRSetScroll


*****
*
* function BRUpdateScroll()
*
* Purpose: Update horizontal and vertical scroll bars for browse
*          
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     5/23/92   LJL       Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     BRUpdateScroll()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     Update scroll bars 
 *  $SYNTAX$
 *     BRUpdateScroll( <oBrowse> ) -> NIL
 *  $ARGUMENTS$
 *     <oBrowse> - The browse object containing the scroll bars to update
 *  $RETURNS$
 *     NIL
 *  $DESCRIPTION$
 *     This routine allows one to update the scroll bars while externally
 *     controling the browse having started the browse using BUILDBROWSE()
 *  $EXAMPLES$
 *     BRUpdateScroll(oBrowse)
 *  $SEEALSO$
 *     BUILDBROWSE()
 *  $INCLUDE$
 *     
 *  $END$
 */
*
FUNCTION BRUpdateScroll( oBrowse )


#ifndef NO_SIDE_SCROLL
IF VERT_SCROLL(oBrowse)!=NIL
	ScrollBarUpdate(VERT_SCROLL(oBrowse),RECORD_NUM(oBrowse),;
		TOTAL_COUNT(oBrowse), .F.)
ENDIF
#endif

#ifndef NO_BOTTOM_SCROLL

IF HORIZ_SCROLL(oBrowse)!=NIL
	ScrollBarUpdate(HORIZ_SCROLL(oBrowse),oBrowse:colpos,;
		oBrowse:colCount,.F.)
ENDIF
#endif

RETURN NIL

* End of BRUpdateScroll

#ifndef NO_MOUSE
*****
*
* BRMouseFunc
*
* This function determines if the mouse is in the browse area and if it is then
* action needs to be taken. 
* Additionally this routine processes any screen "hot spots" which are selected
* by the user. 
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     5/27/91   LJ Letendre   Initial Version
*         V1.01     4/21/92   LJL           Changed for external use
*         V1.02    11/10/92   LJL           Added REFRESH_EXIT option to
*                                           hotspot return code
* Returns: NIL
/*  $DOC$
 *  $FUNCNAME$
 *     BRMOUSEFUNC()
 *  $CATEGORY$
 *     BROWSE
 *  $ONELINER$
 *     Process mouse events within the given browse
 *  $SYNTAX$
 *     BRMouseFunc( <oBrowse>, <nButton>, <nRow>, <nCol>, <nTime>) -> <lMore>
 *  $ARGUMENTS$
 *     <oBrowse> - The browse to process
 *     <nButton> - Button which was hit 1=Left, 2=Right, 4=Center
 *     <nRow>    - Row coordinate of mouse pointer when button hit in screen
 *                 row/col coordinates
 *     <nCol>    - Col coordinate of mouse pointer
 *     <nTime>   - The system time that the button was clicked
 *     
 *  $RETURNS$
 *     <lMore> which signifies if the browse is to continue. It will only be
 *             .F. if operating under a while condition and the last record
 *             is no longer valid because the user modified a field which
 *             is part of the while key or if REFRESH_EXIT is returned from
 *             a hotspot.
 *  $DESCRIPTION$
 *     This routine processes mouse events. First hot spots are checked then
 *     a check for movement or activation of an edit within the browse is made.
 *  $EXAMPLES$
 *     FT_MSHOWCRS()
 *
 *     nKey=0
 *     nMouseKey=0
 *     DO WHILE (nKey=0).AND.(nMouseKey=0)
 *        nMousekey=FT_MGETPOS(@nMouseRow,@nMouseCol)
 *        nKey=INKEY()
 *     ENDDO
 *
 * if we have input from the mouse then convert the mouse coordinates
 * Get time so we can time a double click if necessary
 *
 *     IF  nMouseKey>0
 *        nTime=SECOND()
 *     ENDIF
 *
 *     FT_MHIDECRS()
 *
 *    IF nMouseKey>0
 *       BRMouseFunc(oBrowse,nMouseKey,INT(nMouseRow/8),INT(nMouseCol/8),nTime)
 *    ENDIF
 *
 *  $SEEALSO$
 *     MBrowse() BuildBrowse() BRHotSpot()
 *  $INCLUDE$
 *     
 *  $END$
 */
*
* Notes: The processing of "Hot Spots" occurs prior to checking for the
* cursor being in a the browse. Therefore, the code block will be executed
* without the check for moving the cursor to another cell.
*
*

FUNCTION BRMouseFunc(oBrowse,nButton, nRow, nCol, nTime)

*
* Local variables:
*
LOCAL j, working, nAction, done, col_right, k
LOCAL lMore:=.T.
LOCAL nCur_Rec

* if we have input from the mouse then convert the mouse coordinates
* Clear button press counts so call can determine double click easily

IF nButton%2=1 && left button 
	FT_MBUTPRS(0)
ENDIF

IF (INT(nButton/2)%2)=1 && right button
	FT_MBUTPRS(1)
ENDIF

IF (nButton>=4) && middle button
	FT_MBUTPRS(2)
ENDIF


* Check the general hot spots first

working=GeneralSpot(nButton, nRow, nCol, nTime)


* First check to see if any action hot spots were clicked on

j=1
DO WHILE (j<=LEN(aMouseSpot).AND.working)

* Check coordinates
	working=.NOT.(aMouseSpot[j,9].AND.;
				nRow>=aMouseSpot[j,1].AND.nRow<=aMouseSpot[j,3].AND.;
				nCol>=aMouseSpot[j,2].AND.nCol<=aMouseSpot[j,4].AND.;
				(aMouseSpot[j,8]=0.OR.aMouseSpot[j,8]=nButton))

* If we have a match then execute the code block

	IF .NOT.working

* do the request

		nAction=EVAL(aMouseSpot[j,5],nButton,nRow,nCol,nTime,oBrowse,j)

* Take the appropriate action if requested
          IF nAction=NIL.OR.VALTYPE(nAction)!="N"

		ELSEIF nAction=REFRESH_CURRENT
			oBrowse:refreshCurrent()
		ELSEIF nAction=REFRESH_ALL
* If screen changed this much then rework things entirely
			CountRec(@nCur_Rec,oBrowse,RECNUM)
			SAVE_RECORD_NUM(oBrowse,nCur_Rec)
			CorrectScreen(RECNO(),oBrowse)
		ELSEIF nAction=REFRESH_EXIT
			lMore=.F.
		ENDIF

* Wait for release if requested
		IF aMouseSpot[j,7]
			DO WHILE FT_MBUTREL(1)!=0
			ENDDO
		ENDIF

* Pause for the minimum amount of time

		sleep(aMouseSpot[j,6],nTime)
	ENDIF
* increment counter
	j++

ENDDO

* if we did not find a hit then continue on
IF working.AND.lMore

* move the cursor as requested
	lMore = BRMoveCursor(oBrowse,nRow, nCol, nButton, nTime)

* Wait for release so that scrolling or hot spots are not
* messed up by not releasing quickly enough

	DO WHILE FT_MBUTREL(0)!=0
	ENDDO

ENDIF

* Now return

RETURN lMore

* End of  MouseFunc

******
*
*  BRMoveCursor()
*
* Purpose: move the cursor to the specified location-initiate read on dbl click
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     5/30/91   LJ Letendre   Initial Version
*         V1.01     4/21/92   LJL           Added browse object to calling pars
*         V1.02    12/28/92   LJL           Added double click exit
*         V1.03     2/15/93   LJL           Added Double click code block call
 
* Calling parameters: oBrowse - The current browse
*                     nRow - Current mouse Row
*                     nCol - Current mouse Column
*                     nButton - Mouse Button hit
*                     nTime - Time the button was hit
*
* Returns: logical value indicating if the browse should continue. See DoGet
*          for when this gets changed.
*
* Note: This routine is not operational during an append READ therefore
*       the read can not be terminated with the mouse.

FUNCTION BRMoveCursor(oBrowse, nRow, nCol, nButton, nTime)

LOCAL j, k, col_right, nMove, done, nLeftOffset, nLeftSkew
LOCAL nRightSkew, nColSep, nLeftCol, nSkew, lMore:=.T.
LOCAL nAction, nCur_Rec

STATIC nOldRow, nOldCol, nOldTime

* Entry point

nAction=REFRESH_NONE

* Search for a match in coordinates

IF nRow>=oBrowse:nTop+HEADING_DEPTH(oBrowse);
		.AND.nRow<= oBrowse:nBottom-nBotDepth;
		.AND.nCol>=oBrowse:nLeft;
		.AND.nCol<=oBrowse:nRight

* check for double click first if necessary and if it was then edit the field
* or Exit

	IF DOUBLE_CLICK(oBrowse)!=DOUBLE_NONE .AND. nOldRow=nRow ;
		.AND. nOldCol=nCol .AND. (nTime-nOldTime)<=mDoubleClk()

		IF DOUBLE_CLICK(oBrowse)=DOUBLE_CLICK_EXIT.AND.nButton=1
* Exit
			lMore=.F.
			lMouseExit=.T.
		ELSEIF DOUBLE_CLICK(oBrowse)=DOUBLE_CLICK_GET.AND.nButton=1
* Do a get
			lMore:=DoGet(oBrowse)
		ELSEIF DOUBLE_CLICK(oBrowse)=DOUBLE_BLOCK
* Execute the specified code block

			nAction:=EVAL(GET_DOUBLE_BLOCK(oBrowse),;
				nButton,nRow,nCol,nTime,oBrowse)

			IF nAction=NIL.OR.VALTYPE(nAction)!="N"

* Perform requested action

			ELSEIF nAction==REFRESH_CURRENT
				oBrowse:refreshCurrent()
			ELSEIF nAction==REFRESH_ALL
* Repaint and recount since things have changed a great deal
				CountRec(@nCur_Rec,oBrowse,RECNUM)
				SAVE_RECORD_NUM(oBrowse,nCur_Rec)
				CorrectScreen(RECNO(),oBrowse)
			ELSEIF nAction==REFRESH_EXIT
				lMore=.F.
				lMouseExit=.T.
			ENDIF

		ENDIF

	ELSEIF nButton=1
* Left Button
		nMove=nRow-obrowse:nTop-oBrowse:rowpos+1-HEADING_DEPTH(oBrowse)
* move up or down
		IF nmove>0
			FOR j=1 TO nmove
				oBrowse:down()
			NEXT
			nBrowseMoved=CHANGED_RECORD
		ELSEIF nMove<0
			FOR j=1 TO -nmove
				oBrowse:up()
			NEXT
			nBrowseMoved=CHANGED_RECORD
		ENDIF

* Move left or right

* Figure if we are moving into a column
* First the empty space on the left is determined. Note if oB:leftvisible=0
*     then all of the columns are frozen

		nColSep=LEN(oBrowse:colSep)
		nLeftCol=oBrowse:nLeft  // just a feeling that this will be faster
		nSkew=1+oBrowse:nRight-nLeftCol-FREEZE_WIDTH(oBrowse);
			-IIF(oBrowse:leftvisible>0,;
			 (TOTAL_WIDTH(oBrowse)[oBrowse:rightvisible+1];
			 -TOTAL_WIDTH(oBrowse)[oBrowse:leftvisible]),0)+nColSep

		nLeftSkew=INT(nSkew/2)
          nRightSkew=nSkew-nLeftSkew+nColSep
		nLeftCol-=(nColSep/2)

* If we need to go to a frozen column do the work here

          IF nCol<(nLeftCol+FREEZE_WIDTH(oBrowse))

* Figure which column we need to go to
			done=.F.
			j=0
			DO WHILE .NOT.done .AND. j<oBrowse:freeze
				j++
				done=(nCol<(nLeftCol+TOTAL_WIDTH(oBrowse)[j+1]))
			ENDDO

* In the moveable columns

		ELSE
			done=.F.
			j=oBrowse:leftvisible
			nLeftSkew+=nLeftCol+FREEZE_WIDTH(oBrowse)-TOTAL_WIDTH(oBrowse)[j]
			j--
			DO WHILE .NOT. done .AND. j<oBrowse:rightvisible
				j++
				IF j=oBrowse:rightvisible
					nLeftSkew+=nRightSkew
				ENDIF
				done=(nCol<(nLeftSkew+TOTAL_WIDTH(oBrowse)[j+1]))
			ENDDO
		ENDIF
 
* Now move
		nMove=oBrowse:colpos-j
		IF nMove>0.AND.done
			FOR j=1 TO nMove
				oBrowse:left()
			NEXT
			nBrowseMoved+=PANNED_RECORD
		ELSEIF nMove<0.AND.done
			FOR j=1 TO (-nMove)
				oBrowse:right()
			NEXT
			nBrowseMoved+=PANNED_RECORD
		ENDIF

* Save old coordinates and time
		nOldRow=nRow
		nOldCol=nCol
		nOldTime=nTime
* Force stable

		oBrowse:ForceStable()

	ENDIF

ENDIF

RETURN lMore

* End of BRMoveCursor

*****
*
* MouseScroll()
*
* Purpose: process mouse scrolling requests
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     6/2/91    LJL       Initial Version
*         V1.01     10/18/91  LJL       Added logic to go TOP or Bottom
*                                       and then skip if it is less work
*                                       to do so.
*         V1.02     11/10/92  LJL       Changed return value to actually
*                                       return a legal value - REFRESH_NONE
*         V1.03     06/13/93  LJL       Changed to use NTXREC to move large
*                                       amounts
*
* Calling Parameters: nBut - The button number 1=Left 2=Right, 3=Both
*                     nRow - The row that the cursor is in when hit
*                     nCol - The column the mouse was in when hit
*                     nTime - The time the button was hit
*                     oBrowse - The current browse
* 
* Returns: REFRESH_NONE

STATIC FUNCTION MouseScroll(nBut,nRow,nCol,nTime,oBrowse)

LOCAL aScrollBar
LOCAL nScrollHeight, nNewRec, i, nSkip
LOCAL lCorrect
LOCAL nReturn:=REFRESH_NONE
LOCAL nRecPos       // desired record position
LOCAL nIndexOrd      // current index order number
LOCAL nRecNo         // record to go to

* Get the scroll bar information

aScrollBar=IIF(VERT_SCROLL(oBrowse)[SB_COLTOP]=nCol,;
		    VERT_SCROLL(oBrowse),HORIZ_SCROLL(oBrowse))

* Determine where we are

* On the left or top arrow

IF nRow=aScrollbar[SB_ROWTOP].AND.nCol=aScrollBar[SB_COLTOP]

* Left button
	IF nBut=1
		IF aScrollBar[SB_DIRECTION]=SB_VERTICAL
			oBrowse:up()
			nBrowseMoved=CHANGED_RECORD
		ELSE
			oBrowse:left()
			nBrowseMoved=PANNED_RECORD
		ENDIF

* Right Button
	ELSEIF nBut=2
		IF aScrollBar[SB_DIRECTION]=SB_VERTICAL

			oBrowse:pageUp()
			nBrowseMoved=CHANGED_RECORD

* Since we are moving a full page don't allow repeating

#ifndef FASTPAGE
			DO WHILE FT_MBUTREL(1)!=0
			ENDDO
#endif
		ELSE
			oBrowse:panLeft()
			nBrowseMoved=PANNED_RECORD
		ENDIF
* Both buttons
	ELSEIF nBut=3
		IF aScrollBar[SB_DIRECTION]=SB_VERTICAL
		     SAVE_RECORD_NUM(oBrowse,1)
			oBrowse:goTop()
			nBrowseMoved=CHANGED_RECORD
* force a wait for release
			DO WHILE FT_MBUTREL(1)!=0
			ENDDO

		ENDIF

		oBrowse:ForceStable()

	ENDIF
* Bottom or right arrow

ELSEIF nRow=aScrollBar[SB_ROWBOTTOM].AND.nCol=aScrollBar[SB_COLBOTTOM]

* Left button
	IF nBut=1
		IF aScrollBar[SB_DIRECTION]=SB_VERTICAL
			oBrowse:down()
			nBrowseMoved=CHANGED_RECORD
		ELSE
			oBrowse:right()
			nBrowseMoved=PANNED_RECORD
		ENDIF

* Right Button
	ELSEIF nBut=2
		IF aScrollBar[SB_DIRECTION]=SB_VERTICAL

			oBrowse:pageDown()
			nBrowseMoved=CHANGED_RECORD
* Since we are moving a full page don't allow repeating

#ifndef FASTPAGE
			DO WHILE FT_MBUTREL(1)!=0
			ENDDO
#endif
		ELSE
			oBrowse:panRight()
			nBrowseMoved=PANNED_RECORD
		ENDIF
* Both buttons?
	ELSEIF nBut=3
		IF aScrollBar[SB_DIRECTION]=SB_VERTICAL

* Note: we must save the record number before calling oBrowse:goBottom since
* the browse object will call the skipper to move the display to the top of
* the screen page in the goBottom but waits until stabilization time to move
* back to the last record.

		     SAVE_RECORD_NUM(oBrowse,TOTAL_COUNT(oBrowse))
			oBrowse:goBottom()
			nBrowseMoved=CHANGED_RECORD

* force a wait for release
			DO WHILE FT_MBUTREL(1)!=0
			ENDDO
		ENDIF
	ENDIF
ELSE

* move display according to where the cursor was clicked on the scroll bar
	IF nBut=1
		IF aScrollbar[SB_DIRECTION]=SB_VERTICAL
			DISPBEGIN()
			nScrollHeight=aScrollBar[SB_ROWBOTTOM]-aScrollBar[SB_ROWTOP] - 2
* Calculate new position
               nNewRec=ROUND(1+((nRow-aScrollBar[SB_ROWTOP]-1);
					*MAX(1,TOTAL_COUNT(oBrowse)-1))/nScrollHeight,0)

			IF nNewRec<1
				nNewRec=1
			ELSEIF nNewRec>TOTAL_COUNT(oBrowse)
				nNewRec=TOTAL_COUNT(oBrowse)
			ENDIF

* Move to new position

			lCorrect=.T.

* If the record is on the screen then move to it

			IF (nSkip:= nNewRec-RECORD_NUM(oBrowse))<0.AND.;
				(-nSkip)<oBrowse:RowPos

* Just scroll up
				FOR i = 1 TO (-nSkip)
					oBrowse:up()
				NEXT
				lCorrect=.F.
				nBrowseMoved=CHANGED_RECORD

			ELSEIF nSkip>0.AND.nSkip<=(oBrowse:nBottom-oBrowse:nTop;
					-HEADING_DEPTH(oBrowse)-oBrowse:RowPos)

* Just scroll down
				FOR i = 1 TO nSkip
					oBrowse:down()
				NEXT
				lCorrect=.F.
				nBrowseMoved=CHANGED_RECORD

* Top Record so do short cut

			ELSEIF nNewRec=1
				SAVE_RECORD_NUM(oBrowse,1)
				oBrowse:GoTop()
				lCorrect=.F.
				nBrowseMoved=CHANGED_RECORD

* Bottom Record so do short cut to bottom

			ELSEIF nNewRec=TOTAL_COUNT(oBrowse)
				SAVE_RECORD_NUM(oBrowse,TOTAL_COUNT(oBrowse))
				oBrowse:GoBottom()
				lCorrect=.F.
				nBrowseMoved=CHANGED_RECORD

* Must go the hard way by skipping

			ELSEIF nSkip<>0
#ifdef USE_NTXPOS
* If we are using ntxrec then get the record number we need to go to and
* do so
				IF (nIndexOrd:=INDEXORD())>0
					nRecPos=NTXPOS(nIndexOrd,RECNO())+nSkip
					nRecNo=NTXREC(nIndexOrd,nRecPos)
					GOTO nRecNo
* In case there is a filter in effect go to a legal record
					SKIP
					SKIP-1
				ELSE
					GOTO RECNO()+nSkip
				ENDIF
#else

				IF ABS(nSkip)<=oBrowse:rowCount

					SKIP nSkip

* or faster from bottom up

				ELSEIF nSkip>(TOTAL_COUNT(oBrowse)-nNewRec)

					oBrowse:GoBottom()
					SKIP nNewRec-TOTAL_COUNT(oBrowse)

* Test to see if it would be quicker to go from top and then skip

				ELSEIF (-nSkip)>nNewRec
					oBrowse:GoTop()
					SKIP nNewRec-1

* Or just plain skip
				ELSE
					SKIP nSkip
				ENDIF
#endif
				nBrowseMoved=CHANGED_RECORD

			ENDIF

			IF lCorrect
				SAVE_RECORD_NUM(oBrowse,nNewRec)
				CorrectScreen(RECNO(), oBrowse)
			ENDIF
*  DO columns now

		ELSE
			nScrollheight=aScrollBar[SB_COLBOTTOM]-aScrollBar[SB_COLTOP]-2
			nNewRec=ROUND(1+((ncol-aScrollBar[SB_COLTOP]-1);
					*MAX(1,oBrowse:colCount-1))/nScrollHeight,0)

* Move to new position

			nNewRec=oBrowse:colpos-nNewRec
			IF nNewRec>0
				FOR i=1 TO nNewRec
					oBrowse:left()
				NEXT
				nBrowseMoved=PANNED_RECORD
			ELSEIF nNewRec<0
				FOR i=1 TO (-nNewRec)
					oBrowse:right()
				NEXT
				nBrowseMoved=PANNED_RECORD
			ENDIF
			oBrowse:ForceStable()

		ENDIF
* Send out display no that we are done

		DISPEND()

* force a wait for release

		DO WHILE FT_MBUTREL(1)!=0
		ENDDO


	ENDIF

ENDIF

RETURN nReturn

* End of MouseScroll

******
*
* BRHotSpot()
*
* This function allows the caller to define a location on the screen which
* if clicked on with the mouse will cause an action to take place.
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     5/20/91   LJ Letendre   Initial Version
*                                            - taken from MOUSEGET
* 
/*  $DOC$
 *  $FUNCNAME$
 *     BRHOTSPOT()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     Defines Mouse Hot spots for MBROWSE
 *  $SYNTAX$
 *     BRHotSpot( <nTopRow>, <nLeftCol>, <nBotRow>, <nRightCol>, <bAction>, ;
 *                <nButton>, <nSleep>, <lRelease>) -> nId
 *
 *  $ARGUMENTS$
 *     <nTopRow> - the top row of the area 
 *     <nLeftCol> - the left column of the area
 *     <nBotRow> - the bottom row of the area
 *     <nRightCol> - the right column of the area 
 *     <bAction> - Code block which will be executed when
 *              mouse is clicked in the area
 *     <nButton> - Optional button number for action to occur. IF
 *              equal to 0 or NIL, the action occurs on 
 *              clicking anybutton (the code block can decide 
 *              what to do with based upon the button). If equal
 *              to 1, code block executes only on left click,
 *              if equal to 2 only on right click and if equal
 *              to 4(?) then the middle button.
 *     <nSleep> - Optional value of a minimum time (in seconds) to
 *              wait between servicing multiple button presses. 
 *              Prevents routine from operating too quickly and 
 *              reading the press of a button multiple times 
 *              when not intended. If =NIL then the default value
 *              is used (see MDefSleep()).
 *     <lRelease> - Optional Logical Value. If set to .T. the
 *              servicing routine will pause after the completion
 *              of bAction for the release of the mouse button(s)
 *              Useful for guaranteeing no multiple hits on
 *              an area. If =NIL then the default is used (see
 *              MDefRelease())
 *     
 *  $RETURNS$
 *     <nId> which is an ID to be used to remove the area with a call
 *              to BRRemHotSpot(nId)
 *  $DESCRIPTION$
 *     This routine defines a hot spot for MBROWSE, which will be activated 
 *     it the user clicks the mouse in the defined area. The action which is
 *     executed is defined by the code block bAction which is called with
 *     four arguments:
 *
 *                 nButNum: the number of the button pressed with
 *                          1=left, 2=right, 4=middle(?).
 *                 nRow: The row that the mouse cursor was in when it
 *                       was clicked
 *                 nCol: The column that the mouse cursor was in when it
 *                       was clicked
 *                 nTime: The time returned by SECOND() shortly after the
 *                       button was clicked.
 *                 nId:   The hot spot Id number.
 *
 *        Thus the code block should have a form similar to the following
 *        if one wishes to use the button/cursor information:
 *
 *      {|nButNum, nRow, nCol, nTime, nId| MyFunc(NButNum,nRow,nCol,nTime,nId)}
 *
 *      The hot spot should return REFRESH_NONE, REFRESH_CURRENT or 
 *      REFRESH_ALL which are defined in MBROWSE.CH. These values will cause
 *      nothing, a call to browse:RefreshCurrent() and browse:RefreshAll()
 *      to be executed at the completion of the code block.
 *
 *  $EXAMPLES$
 *      BRHotSpot(1,10,1,20,{|| ShowHelp()},1,,.T.) // hot spot shows help
 *  $SEEALSO$
 *      BRCOOLSPOT() BRWARMSPOT() BRREMHOTSPOT() nggenrl.ngo:MDEFSLEEP() nggenrl.ngo:MDEFRELEASE()
 *  $INCLUDE$
 *
 *  $END$
 */
*
FUNCTION BRHotSpot( nTopRow, nLeftCol, nBotRow, nRightCol, bAction, nButton,;
				 nSleep, lRelease)
*
* Local variables
*

* Entry point

RETURN AddHotSpot(aMouseSpot,@nFreeSpot,;
			{nTopRow, nLeftCol, nBotRow, nRightCol, bAction,;
			IIF(nSleep=NIL,MDefSleep(),nSleep),;
			IIF(lRelease=NIL,MDefRelease(),lRelease),;
			IIF(nButton=NIL,0,nButton),.T.})


* End of BRHotSpot

******
*
* BRRemHotSpot()
*
* This subroutine clears the specified Hotspot 
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     5/20/91   LJ Letendre   Initial Version
*         V1.01     10/25/91  LJ Letendre   Moved code to general service 
*                                           routine so it can be shared
*
/*  $DOC$
 *  $FUNCNAME$
 *     BRREMHOTSPOT()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     This subroutine clears the specified MBROWSE Hot Spot 
 *  $SYNTAX$
 *     BRRemHotSpot( <nId> ) -> NIL
 *  $ARGUMENTS$
 *     <nID> - the ID number of the region to remove from active duty. 
 *             It is given by BRHotSpot.
 *  $RETURNS$
 *      NIL
 *  $DESCRIPTION$
 *      This routine removes a mouse hot spot from the MBROWSE list of active 
 *      hot spots.
 *  $EXAMPLES$
 *      nHelpId=BRHotSpot(1,1,1,10,{|| ShowHelp()})
 *
 *     <MBROWSE code>
 *
 *      BRRemHotSpot(nHelpId)
 *  $SEEALSO$
 *      BRHOTSPOT() BRCOOLSPOT() BRWARMSPOT()
 *  $INCLUDE$
 *
 *  $END$
 */
*
FUNCTION BRRemHotSpot(nID)
*
* Local variables
*

* Call actual service routine

nFreeSpot=RemHotSpot(nId,aMouseSpot,nFreeSpot)

RETURN NIL

* End of BRRemHotSpot

******
*
* BRCoolSpot()
*
* This subroutine deactivates the specified HotSpot without deleting it
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     5/17/91   LJ Letendre   Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     BRCOOLSPOT()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     This subroutine deactivates the specified MBROWSE Hot Spot
 *  $SYNTAX$
 *     BRCOOLSPOT(<nId>) -> NIL
 *  $ARGUMENTS$
 *      <nID> - the ID number of the MBROWSE Hot Spot to remove from active 
 *           duty. It is given by BRHotSpot.
 *  $RETURNS$
 *      NIL
 *  $DESCRIPTION$
 *      This routine deactivates the specified hot spot without removing it
 *      from the list of hot spots. It can later be reactivated with
 *      BRWarmSpot()
 *
 *  $EXAMPLES$
 *      nHelpId=BRHotSpot(1,1,1,10,{|| ShowHelp()}
 *
 *      FUNCTION ShowHelp()
 *      BRCoolSpot(nHelpId)   // Cool off the help hot spot
 *      <BROWSE code>       // get type of help needed
 *      BRWarmSpot(nHelpId)   // reactivate before returning
 *      RETURN NIL
 *
 *  $SEEALSO$
 *      BRWARMSPOT() BRREMHOTSPOT() BRHOTSPOT()
 *  $INCLUDE$
 *
 *  $END$
 */
*
FUNCTION BRCoolSpot(nID)
*
* Local variables
*

aMouseSpot[nId,9]=.F.

RETURN NIL

* End of BRCoolSpot

******
*
* BRWarmSpot()
*
* This subroutine reactivates the specified HotSpot which was deactivated
* by RBCoolSpot
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     5/17/91   LJ Letendre   Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     BRWARMSPOT()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     This subroutine reactivates the specified MBROWSE Hot Spot
 *  $SYNTAX$
 *     BRWARMSPOT(<nId>) -> NIL
 *  $ARGUMENTS$
 *      <nID> - the ID number of the MBROWSE Hot Spot to return to active duty. 
 *           It is given by BRHotSpot and should have been deactivated by
 *           BRCoolSpot()
 *
 *  $RETURNS$
 *      NIL
 *  $DESCRIPTION$
 *      This routine reactivates the specified MBROWSE hot spot after having 
 *      been deactivated by BRCoolSpot(). 
 *
 *  $EXAMPLES$
 *      BRWarmSpot(nHelpId)   // Turn the help hot spot back on
 *
 *  $SEEALSO$
 *      BRCOOLSPOT() BRREMHOTSPOT() BRHOTSPOT()
 *  $INCLUDE$
 *
 *  $END$
 */
*
FUNCTION BRWarmSpot(nID)
*
* Local variables
*

aMouseSpot[nId,9]=.T.

RETURN NIL

* End of BRWarmSpot

*****
*
* function BRIgnoreMouse()
*
* Purpose: force the routines to ignore the mouse and perform the overhead
*          necessary for mouse support
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     5/10/91   LJL       Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     BRIGNOREMOUSE()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     Force MBROWSE to ignore the presence of the mouse
 *  $SYNTAX$
 *     BRIGNOREMOUSE(<lIgnore>) -> <lCurMouse>
 *  $ARGUMENTS$
 *     <lIgnore> - logical for ignoring mouse .T. = act as if mouse 
 *             is not present. If absent just returns current setting
 *  $RETURNS$
 *     <lCurMouse> - The current setting of mouse ignore flag.
 *  $DESCRIPTION$
 *     This routine allows one to force the MBROWSE system to ignore the
 *     presence of the mouse and to not incur the overhead associated with
 *     it.
 *  $EXAMPLES$
 *     lCurIgnore=BRIgnoreMouse(.T.)
 *  $SEEALSO$
 *     
 *  $INCLUDE$
 *     
 *  $END$
 */
*
FUNCTION BRIgnoreMouse(lIgnore)

* Local Parameters
LOCAL oldsetting

*Save old value
oldsetting=lIgnoreMouse

IF lIgnore!=NIL
	lIgnoreMouse=lIgnore
ENDIF

RETURN oldsetting

* End of BRIgnoreMouse

* End of conditional compilation for NO_MOUSE
#endif

*****
*
* function BRScrollColor()
*
* Purpose: Provide a color for scroll bars
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     7/14/91   LJL       Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     BRScrollColor()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     Provide a color for the scroll bars in BROWSE
 *  $SYNTAX$
 *     BRScrollColor(<cColor>) -> <cCurColor>
 *  $ARGUMENTS$
 *     <cColor> - A color string defining the scroll bar colors
 *  $RETURNS$
 *     <cCurColor> - The current setting of scroll bar colors.
 *  $DESCRIPTION$
 *     This routine allows one to set the scroll bar colors. If none is
 *     set when MBROWSE or BUILDBROWSE is called then the first color of 
 *     the color string passed to either of those routines is used as the
 *     scroll bar color. The initial value of cCurColor is NIL.
 *
 *     Calling with no value returns the current value. You can pass NIL
 *     to reset this function to the original behavior.
 *
 *  $EXAMPLES$
 *     cCurColor=BRScrollColor("R/W")
 *  $SEEALSO$
 *     
 *  $INCLUDE$
 *     
 *  $END$
 */
*
FUNCTION BRScrollColor(cScrollColor)

* Local Parameters
LOCAL oldsetting

*Save old value
oldsetting=cDefScrollColor

IF PCOUNT()>0
	cDefScrollColor=cScrollColor
ENDIF

RETURN oldsetting

*****
*
* function SetWhileBlock()
*
* Purpose: Set the while code block after browse created
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     2/16/93   LJL       Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     SetWhileBlock()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     Set/change the while code block in browse object
 *  $SYNTAX$
 *     SetWhileBlock(<oBrowse>, <bWhileBlock>) -> bOldBlock
 *  $ARGUMENTS$
 *     <oBrowse> - The browse object of interest
 *     <bWhileBlock> - The new while code block
 *  $RETURNS$
 *     <bOldBlock> - The current while code block.
 *  $DESCRIPTION$
 *     This routine allows one to change the while code block of a browse
 *     object. This can be useful if you are saving a browse object for
 *     reuse but want to change the effective filter. This routine is not
 *     intended for use while the browse is actually displayed.
 *
 *     Calling with no value returns the current value.
 *
 *  $EXAMPLES$
 *     bCurBlock=SetWhileBlock(oBrowse,{|| ACCTNUM>nNewAccounts)
 *  $SEEALSO$
 *     SetWhileKey()
 *  $INCLUDE$
 *     
 *  $END$
 */

FUNCTION SetWhileBlock(oBrowse,bWhileBlock)

* Local variables

LOCAL bOldBlock

* Entry point

bOldBlock=WHILE_BLOCK(oBrowse)

IF VALTYPE(bWhileBlock)="B"
	SAVE_WHILE_BLOCK(oBrowse,bWhileBlock)
ENDIF

RETURN bOldBlock

* End of SetWhileBlock

*****
*
* function SetWhileKey()
*
* Purpose: Set the while search key after browse created
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     2/16/93   LJL       Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     SetWhileKey()
 *  $CATEGORY$
 *     Browse
 *  $ONELINER$
 *     Set/change the while search key in browse object
 *  $SYNTAX$
 *     SetWhileKey(<oBrowse>, <xKey>) -> xOldKey
 *  $ARGUMENTS$
 *     <oBrowse> - The browse object of interest
 *     <bWhileKey> - The new while search key
 *  $RETURNS$
 *     <bOldKey> - The current while search key.
 *  $DESCRIPTION$
 *     This routine allows one to change the while search key of a browse
 *     object. This can be useful if you are saving a browse object for
 *     reuse but want to change the effective filter. This routine is not
 *     intended for use while the browse is actually displayed.
 *
 *     Calling with no value returns the current value.
 *
 *  $EXAMPLES$
 *     bCurBlock=SetWhileKey(oBrowse,cAcctNum)
 *  $SEEALSO$
 *     SetWhileBlock()
 *  $INCLUDE$
 *     
 *  $END$
 */

FUNCTION SetWhileKey(oBrowse,xKey)

* Local variables

LOCAL xOldKey

* Entry point

xOldKey=SEEK_KEY(oBrowse)

IF xKey!=NIL
	SAVE_SEEK_KEY(oBrowse,xKey)
ENDIF

RETURN xOldKey

* End of SetWhileKey