/*
 * File......: ACHOICE.PRG
 * Author....: Leo Letendre CIS: 73607,233
 * Date......: 6/23/93
 * Revision..: V2.0      // Update to Clipper 5.2 release
 * Log file..:
 * 
 *
 * This routine and all accompaning database structures are 
 * Copyright (C) 1991-1993 Leo J. Letendre. All rights reserved.
 *
 * Permission is automatically granted to those who wish to use these
 * routines in any application. Permission is not granted to anyone wishing
 * to include these in any third party shareware or commercial library.
 *
 * Modification history:
 * ---------------------
 *        Version    Date      Who       Notes
 *         V1.00     10/7/91   LJL       Initial Version
 *         V2.00     12/26/91  LJL       Started over. TBROWSE too much of
 *                                       a kludge
 *         V2.01     2/5/92    LJL       Changed testing of parameters to set
 *                                       defaults so that they are compatable
 *                                       with third party libraries
 *         V2.02     2/7/92    LJL       Fixed changes made in 2.01
 *         V2.03     2/28/92   LJL       Fixed single line display bugs
 *         V2.04     3/11/92   LJL       Fixed hang up when mouse not used
 *         V2.05     3/13/92   LJL       Added support for passing characters
 *                                       strings in <alSelectableItems> which
 *                                       is allowed in CLIPPER
 *         V2.06     5/22/92   LJL       Made sure the selected initial item
 *                                       is an integer
 *         V2.07     1/10/93   LJL       Added some more argument checks
 *
 *
 * $Log$
 *
 */

#include "achoice.ch"
#include "inkey.ch"
#include "scrolbr2.ch"

* if the scroll bar will never be used remove the comment from the following
* line and the scroll bar code will not be compiled.

*#define NO_SCROLL
#ifdef NO_SCROLL
	#stdout
	#stdout No Scroll bars included in ACHOICE
#else
	#stdout
	#stdout Scroll bars included in ACHOICE
#endif

#define AC_INDIV     -1
#define AC_FIRST     -2
#define AC_LAST      -3
#define AC_FIRSTDISP -4
#define AC_LASTDISP  -5
#define AC_PAGE      -6

#define TOPROW      State[1]      && Top of ACHOICE
#define LEFTCOL     State[2]      && Left of ACHOICE
#define BOTTOMROW   State[3]      && Bottom of ACHOICE
#define RIGHTCOL    State[4]      && Right of ACHOICE
#define ITEMS       State[5]      && Menu items of ACHOICE
#define NITEMS      State[6]      && Number of items in ACHOICE
#define LUSERFUN    State[7]      && User function included in call
#define USERFUN     State[8]      && Actual User function code block
#define LUSESCROLL  State[9]      && Scroll Bar requested
#define BSELECT     State[10]     && Code block used to determine selectivity
#define TOPITEM     State[11]     && Top item currently displayed in menu
#define DISPLINES   State[12]     && number of lines in menu display
#define CURLINE     State[13]     && Currently selected line in menu (0=top)
#define NORMCOLOR   State[14]     && Normal color
#define SELECTCOLOR State[15]     && Selected color
#define UNSELCOLOR  State[16]     && Unselectable color
#define OLDCOLOR    State[17]     && original screen color
#define WIDTH       State[18]     && width of the menu area
#define SCROLLBAR   State[19]     && scroll bar infomation storage

STATIC State[19]         && The current achoice information size=last element 
                         && defined above

STATIC nIndex            && The index into the array of choices
STATIC PROC_NAME         && name of calling routine
STATIC PROC_LINE         && line of calling routine
STATIC nLastMove         && value of last change of nIndex
STATIC aCMouseSpot:={}   && List of hot spots
STATIC ACFree_spot:=0    && Next free spot for hot spots
STATIC lMouseUsed:=.F.   && Mouse Used logical
STATIC lIgnoreMouse:=.F. && Ignore any mouse if present
STATIC lHitTop           && Hit top of available item in menu flag
STATIC lHitBottom        && Hit bottom of available item in menu flag
STATIC aReg[12]          && registers used in calling mouse interupts
STATIC cSaveDev          && Saved current SET DEVICE TO setting

*+
FUNCTION achoice(nTop, nLeft, nBottom, nRight, acMenuItems,;
			  alSelectableItems, cUserFunction, nInitialItem, nWindowRow,;
			  lUseScroll)
/*  $DOC$
 *  $FUNCNAME$
 *     ACHOICE()
 *  $CATEGORY$
 *     Achoice
 *  $ONELINER$
 *     Replaces achoice with mouse aware routine
 *  $SYNTAX$
 *     achoice(<nTop>, <nLeft>, <nBottom>, <nRight>, <acMenuItems>,;
 *             <alSelectableItems>, <cUserFunction>, <nInitialItem>,;
 *             <nWindowRow>, <lUseScroll>) -> nChoice
 *  $ARGUMENTS$
 *      <nTop>, <nLeft>, <nBottom>, <nRight> are the upper left and
 *           lower right window coordinates.
 *      <acMenuItems> is an array of character strings to display
 *           as menu choices
 *      <alSelectableItems> is a parallel array of logical values--one
 *           element for each item in <acMenuItems>--that specify the 
 *           selectable menu items.  Elements can either be logical values 
 *           or character strings.  If the element is a character string, 
 *           it is evaluated as a macro expression which should evaluate 
 *           to a logical data type.  In either case, a value of false 
 *           (.F.) means that the corresponding menu item is not 
 *           available, and a value of true (.T.) means that it is
 *           available.  If <lSelectableItems> is specified instead of 
 *           an array, false (.F.) makes all menu items unavailable and 
 *           true (.T.) makes all menu items available.  By default, all 
 *           menu items are available for selection.
 *
 *           NOTE: The way this is implemented, you can also pass a code
 *           block. This appears to be compatable with standard clipper.
 *      <cUserFunction> is the name of a user defined function
 *           for handling unrecognized keystrokes
 *      <nInitialItem> is the position within acMenuItems which
 *           is initially highlighted when the menu is displayed
 *      <nWindowRow> is the window row on which the initial
 *           menu item will appear
 *      <lUseScroll> is a logical indicating the display and use
 *           of a scroll bar on the right side. Defaults to .F.
 *      
 *  $RETURNS$
 *      The number of the item selected or 0 if aborted.
 *  $DESCRIPTION$
 *      All of the arguments are meant to follow the standard CLIPPER function
 *      so please see the CLIPPER documentation for a better discription.
 *      
 *      The arguments to the optional user function has one additional 
 *      argument (MouseKey). This argument is the inkey value of the 
 *      equivalent mouse function if the mouse was used to cause an exception.
 *      Currently this is only invoked when the mouse is clicked outside of 
 *      the achoice area which normally terminates the achoice.
 *      
 *  $EXAMPLES$
 *
 *  $SEEALSO$
 *
 *  $INCLUDE$
 *
 *  $END$
 */
*-
* LOCAL variables: 
LOCAL OldState, nOldIndex, OldProc_name, OldProc_line  && save state information
LOCAL nChoice, i, j, nMore, lCont
LOCAL nKey, nMouseKey, nMouseRow, nMouseCol, nTime && mouse or key input
LOCAL lUserFun
LOCAL SaveCursor, lHideMouse:=.F.
LOCAL cSpace, nStart, cTemp

*
* Entry Point
*
* Set default values for those passed without proper values

IF VALTYPE(alSelectableItems)<>"L" .AND. EMPTY(alSelectableItems)
	alSelectableItems=.T.
ENDIF

IF EMPTY(nInitialItem).OR.VALTYPE(nInitialItem)!="N"
	nInitialItem=1
ENDIF

IF EMPTY(nWindowRow).OR.VALTYPE(nWindowRow)!="N"
	nWindowRow=0
ENDIF

IF VALTYPE(lUseScroll)<>"L"
	lUseScroll=.F.
ENDIF

lUserFun:=VALTYPE(cUserFunction)="C".AND..NOT.EMPTY(cUserFunction)

* Save old values

nOldIndex=nIndex
OldProc_name=Proc_name
OldProc_line=Proc_line
OldState=ACLONE(State)

* Set up new values
SaveCursor=SETCURSOR(0)

nMore := AC_CONT
Proc_name=PROCNAME(1)
Proc_line=PROCLINE(1)
lHitTop:=lHitBottom:=.F.

* Setup other global values

TOPROW=nTop
LEFTCOL=nLeft
BOTTOMROW=nBottom
RIGHTCOL=nRight
NITEMS:=LEN(acMenuItems)
LUSERFUN=lUserFun
WIDTH=nRight-nLeft+1
DISPLINES=nBottom-nTop+1
LUSESCROLL=lUseScroll.AND.(DISPLINES>2)
DISPLINES=MIN(DISPLINES,NITEMS)

* Fix nWindowRow if it is larger than the number of lines displayed

IF nWindowRow>(DISPLINES-1)
	nWindowRow=DISPLINES-1
ENDIF

* Save the menu items padding out to the display width
ITEMS:={}
ASIZE(ITEMS,NITEMS)
cSpace=SPACE(WIDTH)

FOR i=1 TO NITEMS
	ITEMS[i]=LEFT(acMenuItems[i]+cSpace,WIDTH)
NEXT

* Colors
OLDCOLOR:=SETCOLOR()
NORMCOLOR=LEFT(OLDCOLOR,(nStart:=AT(",",OLDCOLOR))-1)

cTemp=SUBSTR(OLDCOLOR,nStart+1)
SELECTCOLOR:=LEFT(cTemp,AT(",",cTemp)-1)
UNSELCOLOR=SUBSTR(cTemp,RAT(",",cTemp)+1)

* Set up select code block for skipper

* First codify the user function if used
IF lUserFun
	USERFUN=&("{|action,item,row,MKey| "+cUserFunction+"(action,item,row,MKey)}")
ENDIF

* Set up the selectable code block

IF VALTYPE(alSelectableItems)="A"
	BSELECT={|j| IIF(VALTYPE(alSelectableItems[j])='L',alSelectableItems[j],;
			EVAL(alSelectableItems[j]))}
* convert all character strings to code blocks
	FOR i=1 TO NITEMS
		IF VALTYPE(alSelectableItems[i])="C"
			alSelectableItems[i]=&("{|| "+alSelectableItems[i]+"}")
		ENDIF
	NEXT
ELSE
	IF VALTYPE(alSelectableItems)="C"
		alSelectableItems=&("{|| "+alSelectableItems+"}")
	ENDIF
	BSELECT={|| IIF(VALTYPE(alSelectableItems)='L',alSelectableItems,;
			EVAL(alSelectableItems))}
ENDIF

* Now clear the area for the display

DISPBEGIN()

@ nTop, nLeft CLEAR TO nBottom, nRight

* Display initial list of items

IF nInitialItem<0 .OR. nInitialItem>NITEMS
	nInitialItem=1
ENDIF

* get where we will start

TOPITEM=MAX(1,nInitialItem-nWindowRow)

* correct if we will not have enough items

IF TOPITEM+DISPLINES-1 > NITEMS
		TOPITEM=MAX(1,NITEMS-DISPLINES+1)
ENDIF

* Set the current index

nIndex=INT(nInitialItem)

* now put up the display

cSaveDev=SET(_SET_DEVICE,"SCREEN")

j=0
FOR i=TOPITEM TO MIN(TOPITEM+DISPLINES-1,NITEMS)
	IF i=nIndex
		SETCOLOR(SELECTCOLOR)
		CURLINE=j
	ELSEIF EVAL(BSELECT,i)
		SETCOLOR(NORMCOLOR)
	ELSE
		SETCOLOR(UNSELCOLOR)
	ENDIF
	@ TOPROW+j, LEFTCOL SAY ITEMS[i]
	j++
NEXT

SETCOLOR(OLDCOLOR)

* Fix the display for unselected items

FixItem(1)

* Put up correct scroll bar

#ifndef NO_SCROLL
IF LUSESCROLL
	SCROLLBAR=ScrollBarNew(nTop,;
			nRight+1,nBottom,,1,SB_VERTICAL)
	ScrollBarDisplay(SCROLLBAR)
     ScrollBarUpdate(SCROLLBAR,nIndex,NITEMS, .T.)
ENDIF
#endif
SET(_SET_DEVICE,cSaveDev)

DISPEND()

* Check to see if we have a selectable item

IF VALTYPE(alSelectableItems)="A"
	lCont=.F.
	i=1
	DO WHILE i<=NITEMS.AND..NOT.lCont
		lCont=lCont.OR.EVAL(BSELECT,i)
		i++
	ENDDO
	IF lUserFun .AND. .NOT. lCont 
		nMore=EVAL(USERFUN,AC_NOITEM,nInitialItem,;
				IIF(nWindowRow<>NIL,nWindowRow,0))
	ELSEIF .NOT. lCont
		nMore=AC_ABORT
	ENDIF
ELSE
	IF lUserFun .AND. .NOT. alSelectableItems
		nMore=EVAL(USERFUN, AC_NOITEM,nInitialItem,;
				IIF(nWindowRow<>NIL,nWindowRow,0))
	ELSEIF .NOT. alSelectableItems
		nMore=AC_ABORT
	ENDIF
ENDIF


* If things continue to be OK then continue

IF nMore=AC_CONT

* Now initialize the mouse
	IF lIgnoreMouse
		lMouseUsed=.F.

	ELSEIF (lHideMouse:=lMouseUsed:=FT_MINIT())
		FT_MSHOWCRS()
* Wait for release of mouse button so that when we start this loop a relase
* will not terminate the achoice until the user has actually clicked the
* button since the achoice was presented

		DO WHILE FT_MBUTREL()<>0
		ENDDO
	ENDIF

ENDIF

* Now go into loop to accept input

DO WHILE nMore=AC_CONT

* If a scroll bar is in use then update it

#ifndef NO_SCROLL
	IF LUSESCROLL
         cSaveDev=SET(_SET_DEVICE,"SCREEN")
         ScrollBarUpdate(SCROLLBAR,nIndex,NITEMS, .F.)
         SET(_SET_DEVICE,cSaveDev)
	ENDIF
#endif

* everything's done; just wait for a key or mouse input
* Show and hide the cursor so that others won't have to worry
* about writing over it.

* Call any user function saying that we are in the idle mode

	IF lUserFun .AND. nMore=AC_CONT
		IF lMouseUsed
			FT_MHIDECRS()
		ENDIF
		nMore=EVAL(USERFUN,IIF(lHitTop, ;
			AC_HITTOP,IIF(lHitBottom,AC_HITBOTTOM,AC_IDLE)),nIndex,CURLINE)
		IF lMouseUsed
			FT_MSHOWCRS()
		ENDIF
	ENDIF

* Go into input loop

	IF nMore=AC_CONT

		IF lMouseUsed

			nKey=0
			nMouseKey=0
* Loop for input
			DO WHILE (nKey=0).AND.(nMouseKey=0)
				nKey=INKEY()
				nMousekey=FT_MGETPOS(@nMouseRow,@nMouseCol)
			ENDDO

* if we have input from the mouse then convert the mouse coordinates
* Get time so we can time a double click if necessary

			IF  nMouseKey>0
			   nTime=SECOND()
			ENDIF
* Clear any final release information for subsequent use

			FT_MBUTREL(0)

		ELSE
			nMouseKey=0
			nKey=INKEY(0)
		ENDIF

	ENDIF

 /* process input */

	DO CASE
	CASE nMore<>AC_CONT

 /* Mouse input */
	CASE nMouseKey>0

		nMore = MouseFunc(nMouseKey,nMouseRow,nMouseCol,nTime)

 /* Normal Key input */
	CASE nKey != 0
		nMore = ApplyKey(nKey)

	ENDCASE


ENDDO  // for WHILE (nMore=AC_CONT)

IF nMore=AC_SELECT .AND. EVAL(BSELECT,nIndex)   /* Takes care of all*/
	nChoice=nIndex                            /* items = .F. */ 
ELSE
	nChoice=0
ENDIF

* Restore old values

State={}
State=ACLONE(OldState)
nIndex=nOldIndex
Proc_name=OldProc_name
Proc_line=OldProc_line
SETCURSOR(SaveCursor)

IF lHideMouse
	FT_MHIDECRS()
ENDIF

* Return to caller


RETURN nChoice

* End of achoice

*******
*
* FixItem
*
* Purpose: This routine fixes the display in case we are on a non-selectable
*          item. Only called when needed.
*
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     10/20/91  LJ Letendre   Initial Version
*         V2.00     12/27/91  LJ Letendre   A few changes for the V2 scheme
*
* Calling Parameters: nLastMove - The number of items last moved. Used to get
*                                 the direction of the last move
*
* Returns: NIL
*
*
STATIC FUNCTION FixItem(nLastMove)
LOCAL lDone, lDown, i:=0, nCount:=0

*
* Entry Point
*

lDown=(nLastMove>0)
lDone=EVAL(BSELECT,nIndex)

DO WHILE .NOT. lDone .AND. nCount<2

* If we are moving down then look for the next selectable item
	IF lDown
		i:=nIndex
		DO WHILE (.NOT. lDone) .AND. (i < NITEMS)
* find it
			i++
			lDone=EVAL(BSELECT,i)
		ENDDO

	ELSE
* same for going up but just the opposite direction

		i=nIndex
		DO WHILE (.NOT. lDone) .AND. (i > 1)

			i--
			lDone=EVAL(BSELECT,i)

		ENDDO
	ENDIF

* Way to loop so we can cover both directions if necessary

	IF .NOT. lDone
		nCount++
		lDown=.NOT.lDown
	ENDIF
ENDDO

* If we found one then move to it

IF lDone .AND. i<> 0
	ScrollDisplay(AC_INDIV,i-nIndex)
ENDIF

RETURN NIL

* End of FixItem


*****
*
* ApplyKey
*
* This routine handles the appropriate action for each key entered
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     10/20/91  LJ Letendre   Initial Version
*         V2.00     12/27/91  LJ Letendre   Second go round
*
* Calling parameters: nKey - the inkey code of the pressed key
*
* Returns: AC_ABORT, AC_SELECT or AC_CONT based upon user input
*
STATIC FUNCTION ApplyKey(nKey)

* Local variables

LOCAL nMore:=AC_CONT  && signal to continue ACHOICE - default is yes
LOCAL i, j, k, lDone, nAction
LOCAL bKeyBlock


* handle the movement keys

  DO CASE
     CASE ( nKey == K_DOWN )
        ScrollDisplay(AC_INDIV,1)

     CASE ( nKey == K_UP )
        ScrollDisplay(AC_INDIV,-1)

     CASE ( nKey == K_PGDN )
        ScrollDisplay(AC_PAGE,1)

     CASE ( nKey == K_PGUP )
        ScrollDisplay(AC_PAGE,-1)

     CASE ( nKey == K_CTRL_PGUP )
        ScrollDisplay(AC_FIRST)

     CASE ( nKey == K_CTRL_PGDN )
        ScrollDisplay(AC_LAST)

     CASE ( nKey == K_CTRL_HOME )
        ScrollDisplay(AC_FIRSTDISP)

     CASE ( nKey == K_CTRL_END )
        ScrollDisplay(AC_LASTDISP)

     CASE ( nKey == K_RIGHT ) .AND. !LUSERFUN
        nMore=AC_ABORT

	CASE ( nKey == K_LEFT ) .AND. !LUSERFUN
        nMore=AC_ABORT

     CASE ( nKey == K_HOME ) .AND. !LUSERFUN
        ScrollDisplay(AC_FIRST)

     CASE ( nKey == K_END ) .AND. !LUSERFUN
        ScrollDisplay(AC_LAST)

     CASE ( nKey == K_ESC ) .AND. !LUSERFUN
         nMore := AC_ABORT

     CASE ( nKey == K_RETURN ) .AND. !LUSERFUN
         nMore := AC_SELECT


     OTHERWISE

* Now handle any function keys specified by caller

         IF (bKeyBlock:=SetKey(nKey)) <> NIL

              IF lMouseUsed
                   FT_MHIDECRS()
              ENDIF
              cSaveDev=SET(_SET_DEVICE,"SCREEN")
              EVAL(bKeyBlock,PROC_NAME, PROC_LINE)
              SET(_SET_DEVICE,cSaveDev)
              IF lMouseUsed
                   FT_MSHOWCRS()
              ENDIF
         ELSE

* If we have a user function then we had better call it

              IF LUSERFUN
                  IF lMouseUsed
                      FT_MHIDECRS()
                  ENDIF
                  cSaveDev=SET(_SET_DEVICE,"SCREEN")

                  nMore := EVAL(USERFUN,AC_EXCEPT,nIndex,CURLINE)

                  SET(_SET_DEVICE,cSaveDev)
                  IF lMouseUsed
                       FT_MSHOWCRS()
                  ENDIF
              ENDIF

* If the user function tells us to go to the next one or we have no user 
* function then search for the next item with the indicated first letter

              IF (nMore=AC_GOTO) .OR. .NOT. LUSERFUN .AND. nKey>0

* Change exit code so that we will later continue

                  nMore=AC_CONT
* Search for an item begining with the letter given bu the user.

                  lDone=.F.
                  i=nIndex
                  nKey=UPPER(CHR(nKey))

                  DO WHILE !lDone .AND. i< NITEMS
* search for next item
                       i++
                       lDone=(nKey==SUBSTR(ITEMS[i],1,1)).AND.(EVAL(BSELECT,i))
                  ENDDO

* If not found then start from the begining and search

                  IF !lDone
                       i=0
                       DO WHILE !lDone .AND. i<nIndex
* search for next item
                            i++
                            lDone=(nKey==SUBSTR(ITEMS[i],1,1));
                                    .AND.(EVAL(BSELECT,i))
                       ENDDO
                  ENDIF

* If we found a match then move to it

                  IF lDone
                       i=i-nIndex
                       ScrollDisplay(AC_INDIV,i)
                  ENDIF

              ENDIF
         ENDIF

     ENDCASE


* Done so return

RETURN nMore

* End of ApplyKey


*+
STATIC FUNCTION ScrollDisplay(nMode, nMove)
*
* This routine and all accompaning database structures are 
* Copyright (C) 1991 Leo J. Letendre. All rights reserved.
*
* Purpose: Scroll the ACHOICE type display
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     12/27/91   LJL       Initial Version
*
* Calling parameters: nMove - The number of items to move
*                     lPageMode - The screen should be moved in page mode
*
* Returns: NIL
*
* Notes: 
*-
* LOCAL variables: 
LOCAL nScroll, j, i, nOldIndex:=nIndex
LOCAL nDirection:=0, nStartWrite, nEndWrite

*
* Entry Point
*
* We Will be updating so turn off the mouse

IF lMouseUsed
     FT_MHIDECRS()
ENDIF
cSaveDev=SET(_SET_DEVICE,"SCREEN")

* Try to make things look instant

DISPBEGIN()

* Set the old item back to its unselected color. We can have Unselected
* Items rewritten when this routine is called by FixItem so we must include
* That possibility

IF EVAL(BSELECT,nIndex)
	SETCOLOR(NORMCOLOR)
ELSE
	SETCOLOR(UNSELCOLOR)
ENDIF
@ TOPROW+CURLINE, LEFTCOL SAY ITEMS[nIndex]


IF nMode=AC_INDIV

	nDirection=nMove

* See if the move stays within the bonds of the items

	IF nIndex+nMove<=0
		nMove=1-nIndex
	ELSEIF nIndex+nMove > NITEMS
		nMove=NITEMS-nIndex
	ENDIF

* default start write for the move only within the already displayed area

	nStartWrite:=nEndWrite:=nIndex:=nIndex+nMove

* If the move requires scrolling then do so

	nScroll:=nMove+CURLINE

* if we scroll do it
	IF nScroll<0
* Scroll down to put in new on top
		TOPITEM+=nScroll
		SCROLL(TOPROW,LEFTCOL,BOTTOMROW,RIGHTCOL,nScroll)
		nStartWrite=TOPITEM
		nEndWrite=TOPITEM+MIN(DISPLINES,-nScroll)-1
	ELSEIF nScroll>=DISPLINES

* Scroll up to place new on the bottom

		nScroll=nScroll-DISPLINES+1
		SCROLL(TOPROW,LEFTCOL,BOTTOMROW,RIGHTCOL,nScroll)
		TOPITEM+=nScroll
		nStartWrite=TOPITEM+MAX(0,DISPLINES-nScroll)
		nEndWrite=nStartWrite+MIN(DISPLINES,nScroll)-1
	ENDIF
* now setup the display of the new items

	CURLINE=nIndex-TOPITEM
	j=nStartWrite-TOPITEM

ELSEIF nMode=AC_FIRST

	nDirection=-1
* first item in the list

	IF TOPITEM=1

		nEndWrite:=1

* not on the screen so repaint
	ELSE

		nEndWrite=DISPLINES
	ENDIF

* Set common parameters

	nStartWrite=1
	nIndex=1
	CURLINE=0
	TOPITEM=1
	j=0

* last item in list

ELSEIF nMode=AC_LAST

	nDirection=1

* See if it is on the screen

	IF TOPITEM+DISPLINES-1=NITEMS

		nStartWrite:=NITEMS
		j:=CURLINE:=DISPLINES-1
		
* not on the screen so repaint
	ELSE

		nStartWrite:=TOPITEM:=NITEMS-DISPLINES+1
		CURLINE=DISPLINES-1
		j=0

	ENDIF

	nIndex=NITEMS
	nEndWrite=nIndex

* First item on current display

ELSEIF nMode=AC_FIRSTDISP

	nDirection=-1
	j:=CURLINE:=0
	nIndex:=nStartWrite:=nEndWrite:=TOPITEM

ELSEIF nMode=AC_LASTDISP

	nDirection=1
	nStartWrite:=nEndWrite:=nIndex:=TOPITEM+DISPLINES-1
	j:=CURLINE:=DISPLINES-1
	
* if we need to move in page mode then do so

ELSEIF nMode=AC_PAGE

	nDirection:=nMove:=IIF(nMove<0, -DISPLINES+1, DISPLINES-1)

* Scroll up the necessary number of lines

	nScroll=IIF(nMove<0, MAX(nMove,1-TOPITEM), ;
			MIN(nMove,NITEMS-TOPITEM-DISPLINES+1))
	TOPITEM=TOPITEM+nScroll

* Handle the move to the first or last item

	IF nScroll<>0
		nIndex+=nScroll
	ELSEIF nMove<0
		nIndex=1
	ELSEIF nMove>0
		nIndex=NITEMS
	ENDIF

* just redisplay the entire display

	CURLINE=nIndex-TOPITEM
	j=0
	nStartWrite=TOPITEM
	nEndWrite=nStartWrite+DISPLINES-1

ENDIF

* Now make the changes

FOR i= nStartWrite TO nEndWrite
	IF i=nIndex
		SETCOLOR(SELECTCOLOR)
	ELSEIF EVAL(BSELECT,i)
		SETCOLOR(NORMCOLOR)
	ELSE
		SETCOLOR(UNSELCOLOR)
	ENDIF
	@ TOPROW+j, LEFTCOL SAY ITEMS[i]
	j++
NEXT

SET(_SET_DEVICE,cSaveDev)

* Now fix for unselected

FixItem(nDirection)

DISPEND()


* Final test for hitting top or bottom

lHitTop=(nOldIndex=nIndex.AND.nDirection<0)
lHitBottom=(nOldIndex=nIndex.AND.nDirection>0)

* return color to normal and mouse cursor back on

SETCOLOR(OLDCOLOR)

IF lMouseUsed
     FT_MSHOWCRS()
ENDIF


RETURN NIL

* End of ScrollDisplay

*****
*
* MouseFunc()
*
* This function determines if the mouse is in the ACHOICE area and if it is then
* action needs to be taken. 
* Additionally this routine processes any screen "hot spots" which are selected
* by the user. 
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     12/27/91  LJ Letendre   Initial Version
*         V1.01     3/5/92    LJL           Fixed right button on scroll
*                                           arrow bug.
*         V1.02     4/4/92    LJL           Fixed bug with scroll bar when
*                                           the number of items was fewer than
*                                           the number of lines in the ACHOICE
*         V1.03     5/12/92   LJL           Fixed bug which prevented click
*                                           outside of ACHOICE from terminating
*                                           it
*
* Calling Parameters:
*                    nButton - Button which was hit - reserved for future
*                    nRow - Row coordinate of mouse pointer when button hit
*                    nCol - Col coordinate of mouse pointer
*                    nTime - The system time that the button was clicked
*
* Returns: One of the Return codes AC_* found in achoice.ch
*
* Notes: The processing of "Hot Spots" occurs prior to checking for the
* cursor being in the ACHOICE. Therefore, the code block will be executed
* without the check for moving the cursor to another portion of the Achoice.
*
* The calls to the Forum mouse functions have been replaced by the equivalent
* code in the time critical loops in order to speed them up as much as possible.
* The original calls are still present for reference but are commented out.
*

STATIC FUNCTION MouseFunc(nButton, nRow, nCol, nTime, bSelect, nItems)

*
* Local variables:
*
LOCAL j, working, working2:=.T.
LOCAL done, col_right, i, nButNum
LOCAL result:=AC_CONT, lMoved:=.F.
LOCAL nMove, nRel
LOCAL lOutBox, cSaveDev
LOCAL nRel2, nRow2, nCol2
LOCAL nTop, nBottom, nLeft, nRight
LOCAL nRightScroll, nLastMove

* if we have input from the mouse then convert the mouse coordinates
* Clear button release counts so call can determine button is still down

IF nButton%2=1 && left button 
    FT_MBUTREL(0,@nRel)
    nButNum=0
ENDIF
IF (INT(nButton/2)%2)=1 && right button
    FT_MBUTREL(1)
    nButNum=1
ENDIF
IF (nButton>=4) && middle button
    FT_MBUTREL(2)
    nButNum=2
ENDIF

nRow=INT(nRow/8)
nCol=INT(nCol/8)

* Load into scalars - Probably faster than the actual array values

nTop=TOPROW
nLeft=LEFTCOL
nBottom=MIN(BOTTOMROW,nTop+NITEMS-1)
nRight=RIGHTCOL

#ifndef NO_SCROLL

nRightScroll:=nRight+IIF(LUSESCROLL,1,0)
#else

nRightScroll:=nRight
#endif
* Set flag to denote completion

cSaveDev=SET(_SET_DEVICE,"SCREEN")


* Check the general hot spots first

FT_MHIDECRS()
working=GeneralSpot(nButton, nRow, nCol, nTime)
FT_MSHOWCRS()

* First check to see if any action hot spots were clicked on

j=LEN(aCMouseSpot)
DO WHILE (j>=1.AND.working)

* Check coordinates
	working=.NOT.(aCMouseSpot[j,9].AND.;
				nRow>=aCMouseSpot[j,1].AND.nRow<=aCMouseSpot[j,3].AND.;
				nCol>=aCMouseSpot[j,2].AND.nCol<=aCMouseSpot[j,4].AND.;
				(aCMouseSpot[j,8]=0.OR.aCMouseSpot[j,8]=nButton))

* If we have a match then execute the code block

	IF .NOT.working

* do the request

		FT_MHIDECRS()
		EVAL(aCMouseSpot[j,5],nButton,nRow,nCol,nTime,j)
		FT_MSHOWCRS()

* Wait for release if requested
		IF aCMouseSpot[j,7]
			DO WHILE FT_MBUTREL(0)!=0
			ENDDO
		ENDIF

* Pause for the minimum amount of time
		IF aCMouseSpot[j,6]>0
			sleep(aCMouseSpot[j,6],nTime)
		ENDIF
* fake a release for the following routines
		nRel=1

	ENDIF
* decrement counter
	j--

ENDDO

* Look to see if the button was pressed and released outside of the area of 
* interest to indicate that the achoice should be closed.

IF working .AND. nButNum=0 .AND. ;
		((nRow<nTop .OR. nRow>nBottom .OR.;
		 nCol<nLeft .OR. nCol>nRight) .AND.;
		(!LUSESCROLL .OR.(LUSESCROLL.AND.(nRow<nTop .OR. nRow>BOTTOMROW.OR.;
		 nCol!=nRightScroll))))
		working2=.T.


	DO WHILE working2 

		FT_MBUTREL(0,@nRel)
		FT_MGETPOS(@nRow,@nCol)
		nRow=INT(nRow/8)
		nCol=INT(nCol/8)

* If we have not released the button and are still outside of the
* ACHOICE then keep waiting

		lOutBox=(nRow<nTop .OR. nRow>nBottom;
		.OR. nCol<nLeft .OR. nCol>nRightScroll)

		working2=(nRel=0) .AND. lOutBox

	ENDDO

* Totally exit if the button was released

	IF nRel>0 .AND. lOutBox
		working=.F.
		IF LUSERFUN
			result=EVAL(USERFUN,AC_EXCEPT,nIndex,CURLINE,K_ESC)
		ELSE
			result=AC_ABORT
		ENDIF
	ENDIF

ENDIF

* if we did not find a hit then continue on

IF working .AND. nButNum=0

	working2=.T.
	DO WHILE nRel=0

* Check to see if the button is still down and in the current ACHOICE region.
* if it is then see if we need to scroll


		DO WHILE nRel=0 .AND.;
			 nRow>=nTop .AND. nRow<=nBottom .AND.;
			 nCol>=nLeft .AND. nCol<=nRight

* See if we need to move
			IF (nMove:=nRow-nTop-CURLINE)<>0

* we do, so set the old one back to the unselected color if it was selected

				aReg[1] := 2          // set mouse function call 2
				FT_INT86( 51, aReg )  // execute mouse interrupt
*				FT_MHIDECRS()
				IF EVAL(BSELECT,nIndex)
					SETCOLOR(NORMCOLOR)
					@ nTop+CURLINE, nLeft SAY ITEMS[nIndex]
				ENDIF
* Now highlight the item if is it selectable otherwise do nothing

				nIndex+=nMove
				CURLINE+=nMove
				IF EVAL(BSELECT,nIndex)
					SETCOLOR(SELECTCOLOR)
					@ nTop+CURLINE, nLeft SAY ITEMS[nIndex]
				ENDIF
			
#ifndef NO_SCROLL
				IF LUSESCROLL
				    ScrollBarUpdate(SCROLLBAR,nIndex,NITEMS, .F.)
				ENDIF
#endif
				aReg[1] := 1          // set mouse function call 1
				FT_INT86( 51, aReg )  // execute mouse interrupt
*				FT_MSHOWCRS()
				nLastMove=nMove
			ENDIF


			aReg[1] := 6                // set mouse function call 6
			aReg[2] := 0                // pass parameter for left
			FT_INT86( 51, aReg )        // execute mouse interrupt
			nRel := aReg[2]             // store updated release count 
*			FT_MBUTREL(0,@nRel)

* Need to do this since mouse driver returns the row and col of the last
* release 
			aReg[1] := 3                // set mouse function call 3
			FT_INT86( 51, aReg )        // execute mouse interrupt
			nRow=INT(aReg[4]/8)
			nCol=INT(aReg[3]/8)

*			FT_MGETPOS(@nRow,@nCol)
*			nRow=INT(nRow/8)
*			nCol=INT(nCol/8)
			working=.F.

		ENDDO

* fix for not selectables

		IF !EVAL(BSELECT,nIndex)
			FixItem(nLastMove)
		ENDIF		

* Check to see if we are in the scroll area for moving one item at a time

#ifndef NO_SCROLL
		IF working .AND. LUSESCROLL .AND. nCol=nRightScroll .AND.;
			nRow>=nTop .AND. nRow<=BOTTOMROW

			MouseScroll(nButton, nRow, nCol)
			sleep(mDefSleep(),nTime)
			working2=.F.
		ENDIF
#endif

* If the cursor is in the box then select the item

		IF nRel>0 .AND. nRow>=nTop .AND. nRow<=nBottom .AND.;
			 nCol>=nLeft .AND. nCol<=nRight

* This is a separate test below since the above will move the selected item 
* around if the mouse is sitting on an unselectable item. However, we don't
* want to select an item that is not available so we must always do the test
* as a separate step.

			IF EVAL(BSELECT,nIndex).AND. nRow == nTop+CURLINE
* selectable so do so
				result=AC_SELECT
			ENDIF

			working=.F.

		ENDIF

* Clear any final release information for subsequent use

		FT_MBUTREL(0,@nRel2)
		FT_MGETPOS(@nRow,@nCol)
		nRow=INT(nRow/8)
		nCol=INT(nCol/8)
		nRel=nRel+nRel2

	ENDDO

* restore old color and device in case we changed it

	SET(_SET_DEVICE,cSaveDev)
	SETCOLOR(OLDCOLOR)

ENDIF

* Check to see if we are in the scroll area for moving one page at a time
* The positioning here requires that the button be released between each
* page change.

#ifndef NO_SCROLL
IF working .AND. working2 .AND. LUSESCROLL .AND. nCol=nRightScroll .AND.;
	nRow>=nTop .AND. nRow<=BOTTOMROW

	MouseScroll(nButton, nRow, nCol)
	sleep(mDefSleep(),nTime)
ENDIF
#endif

* Now return

RETURN result

* End of  MouseFunc

#ifndef NO_SCROLL

*****
*
* MouseScroll()
*
* Purpose: process mouse scrolling requests
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     10/20/91  LJL           Initial Version
*         V2.00     12/28/91  LJL           Changes for new scheme
* 
* Calling Parameters: nBut - The button number 1=Left 2=Right, 3=Both
*                     nRow - The row that the cursor is in when hit
*                     nCol - The column the mouse was in when hit
* 

STATIC FUNCTION MouseScroll(nBut,nRow,nCol)

LOCAL aScrollBar, nLastMove
LOCAL nScrollHeight, nNewRec

* Get the scroll bar information

aScrollBar=SCROLLBAR

* Determine where we are

* On the Top arrow

IF nRow=aScrollbar[SB_ROWTOP].AND.nCol=aScrollBar[SB_COLTOP]

nLastMove=-1
* Left button
	IF nBut=1
		ScrollDisplay(AC_INDIV,-1)

* Right Button
	ELSEIF nBut=2
		ScrollDisplay(AC_PAGE,-1)
* Both buttons
	ELSEIF nBut=3
		ScrollDisplay(AC_FIRST)
* force a wait for release
		DO WHILE FT_MBUTREL(0)!=0
		ENDDO

	ENDIF
* Bottom or right arrow

ELSEIF nRow=aScrollBar[SB_ROWBOTTOM].AND.nCol=aScrollBar[SB_COLBOTTOM]

nLastMove=1

* Left button
	IF nBut=1
		ScrollDisplay(AC_INDIV,1)

* Right Button
	ELSEIF nBut=2
		ScrollDisplay(AC_PAGE,1)
* Both buttons?
	ELSEIF nBut=3
		ScrollDisplay(AC_LAST)

* force a wait for release
		DO WHILE FT_MBUTREL(0)!=0
		ENDDO
	ENDIF
ELSE

* move display according to where the cursor was clicked on the scroll bar
	IF nBut=1
		nScrollHeight=aScrollBar[SB_ROWBOTTOM]-aScrollBar[SB_ROWTOP] - 2
* Calculate new position
          nNewRec=INT(ROUND(1+((nRow-aScrollBar[SB_ROWTOP]-1);
				*MAX(1,NITEMS-1))/nScrollHeight,0))


		nLastMove=nNewRec-nIndex
		ScrollDisplay(AC_INDIV,nLastMove)

* force a wait for release

		DO WHILE FT_MBUTREL(1)!=0
		ENDDO


	ENDIF
ENDIF

IF !EVAL(BSELECT,nIndex)
	fixitem(nLastMove)
ENDIF

RETURN NIL

* End of MouseScroll
#endif

******
*
* ACHotSpot()
*
* This function allows the caller to define a location on the screen which
* if clicked on with the mouse will cause an action to take place.
*
* Modification History:
*        Version    Date      Who         Notes
*         V1.00     10/25/91  LJL         Initial Version
*         V1.01     12/12/91  LJL         Added Hot Spot ID number   
* 
/*  $DOC$
 *  $FUNCNAME$
 *     ACHOTSPOT()
 *  $CATEGORY$
 *     Achoice
 *  $ONELINER$
 *     Defines Mouse Hot spots for achoice
 *  $SYNTAX$
 *     ACHotSpot( <nTopRow>, <nLeftCol>, <nBotRow>, <nRightCol>, <bAction>, ;
 *                <nButton>, <nSleep>, <lRelease>) -> nId
 *
 *  $ARGUMENTS$
 *     <nTopRow> - the top row of the area 
 *     <nLeftCol> - the left column of the area
 *     <nBotRow> - the bottom row of the area
 *     <nRightCol> - the right column of the area 
 *     <bAction> - Code block which will be executed when
 *              mouse is clicked in the area
 *     <nButton> - Optional button number for action to occur. IF
 *              equal to 0 or NIL, the action occurs on 
 *              clicking anybutton (the code block can decide 
 *              what to do with based upon the button). If equal
 *              to 1, code block executes only on left click,
 *              if equal to 2 only on right click and if equal
 *              to 4(?) then the middle button.
 *     <nSleep> - Optional value of a minimum time (in seconds) to
 *              wait between servicing multiple button presses. 
 *              Prevents routine from operating too quickly and 
 *              reading the press of a button multiple times 
 *              when not intended. If =NIL then the default value
 *              is used (see MDefSleep()).
 *     <lRelease> - Optional Logical Value. If set to .T. the
 *              servicing routine will pause after the completion
 *              of bAction for the release of the mouse button(s)
 *              Useful for guaranteeing no multiple hits on
 *              an area. If =NIL then the default is used (see
 *              MDefRelease())
 *     
 *  $RETURNS$
 *     nId which is an ID to be used to remove the area with a call
 *              to ACRemHotSpot(nId)
 *  $DESCRIPTION$
 *     This routine defines a hot spot for ACHOICE, which will be activated 
 *     it the user clicks the mouse in the defined area. The action which is
 *     executed is defined by the code block bAction which is called with
 *     five arguments:
 *
 *                 nButNum: the number of the button pressed with
 *                          1=left, 2=right, 4=middle(?).
 *                 nRow: The row that the mouse cursor was in when it
 *                       was clicked
 *                 nCol: The column that the mouse cursor was in when it
 *                       was clicked
 *                 nTime: The time returned by SECOND() shortly after the
 *                       button was clicked.
 *                 nId:   The hot spot Id number.
 *
 *        Thus the code block should have a form similar to the following
 *        if one wishes to use the button/cursor information:
 *
 *      {|nButNum, nRow, nCol, nTime, nId| MyFunc(NButNum,nRow,nCol,nTime,nId)}
 *
 *  $EXAMPLES$
 *      ACHotSpot(1,10,1,20,{|| ShowHelp()},1,,.T.) // hot spot shows help
 *  $SEEALSO$
 *      ACCOOLSPOT() ACWARMSPOT() ACREMHOTSPOT()
 *  $INCLUDE$
 *
 *  $END$
 */
*-

FUNCTION ACHotSpot( nTopRow, nLeftCol, nBotRow, nRightCol, bAction, nButton,;
				 nSleep, lRelease)
*
* Local variables
*

* Entry point

* Now add the coordinates

RETURN AddHotSpot(ACMouseSpot,@ACfree_spot,;
                 {nTopRow, nLeftCol, nBotRow, nRightCol, bAction,;
                  IIF(nSleep=NIL,MDefSleep(),nSleep),;
                  IIF(lRelease=NIL,MDefRelease(),lRelease),;
                  IIF(nButton=NIL,0,nButton),.T.})

* End of ACHotSpot

******
*
* ACRemHotSpot()
*
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     10/25/91  LJL           Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     ACREMHOTSPOT()
 *  $CATEGORY$
 *     Achoice
 *  $ONELINER$
 *     This subroutine clears the specified Hotspot 
 *  $SYNTAX$
 *     ACRemHotSpot( <nId> ) -> NIL
 *  $ARGUMENTS$
 *     <nID> - the ID number of the region to remove from active duty. 
 *             It is given by ACHotSpot.
 *  $RETURNS$
 *      NIL
 *  $DESCRIPTION$
 *      This routine removes a mouse hot spot from the ACHOICE list of active 
 *      hot spots.
 *  $EXAMPLES$
 *      ACRemHotSpot(nHelpId)
 *  $SEEALSO$
 *      ACHOTSPOT() ACCOOLSPOT() ACWARMSPOT()
 *  $INCLUDE$
 *
 *  $END$
 */
*
* Returns: NIL
*
FUNCTION ACRemHotSpot(nID)
*
* Local variables
*

* Call service routine

ACFree_Spot=RemHotSpot(nId, ACMouseSpot, ACFree_Spot)

RETURN NIL

* End of ACRemHotSpot

******
*
* ACCoolSpot()
*
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     10/25/91  LJ Letendre   Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     ACCOOLSPOT()
 *  $CATEGORY$
 *     Achoice
 *  $ONELINER$
 *     This subroutine deactivates the specified ACHOICE HotSpot
 *  $SYNTAX$
 *     ACCOOLSPOT(<nId>) -> NIL
 *  $ARGUMENTS$
 *      <nID> - the ID number of the ACHOICE Hot Spot to remove from active 
 *           duty. It is given by ACHotSpot.
 *  $RETURNS$
 *      NIL
 *  $DESCRIPTION$
 *      This routine deactivates the specified hot spot without removing it
 *      from the list of hot spots. It can later be reactivated with
 *      ACWarmSpot()
 *  $EXAMPLES$
 *      ACCoolSpot(nHelpId)   // Cool off the help hot spot
 *  $SEEALSO$
 *      ACWARMSPOT() ACREMHOTSPOT() ACHOTSPOT()
 *  $INCLUDE$
 *
 *  $END$
 */
*-

FUNCTION ACCoolSpot(nID)
*
* Local variables
*

ACMouseSpot[nid,9]=.F.

RETURN NIL

* End of ACCoolSpot

******
*
* ACWarmSpot()
*
* This subroutine reactivates the specified HotSpot which was deactivated
* by ACCoolSpot
*
* Modification History:
*        Version    Date      Who           Notes
*         V1.00     10/25/91   LJ Letendre   Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     ACWARMSPOT()
 *  $CATEGORY$
 *     Achoice
 *  $ONELINER$
 *     This subroutine reactivates the specified ACHOICE Hot Spot
 *  $SYNTAX$
 *     ACWARMSPOT(<nId>) -> NIL
 *  $ARGUMENTS$
 *      <nID> - the ID number of the ACHOICE Hot Spot to return to active duty. 
 *           It is given by ACHotSpot and should have been deactivated by
 *           ACCoolSpot()
 *  $RETURNS$
 *      NIL
 *  $DESCRIPTION$
 *      This routine reactivates the specified ACHOICE hot spot after having 
 *      been deactivated by ACCoolSpot(). 
 *  $EXAMPLES$
 *      ACWarmSpot(nHelpId)   // Turn the help hot spot back on
 *  $SEEALSO$
 *      ACCOOLSPOT() ACREMHOTSPOT() ACHOTSPOT()
 *  $INCLUDE$
 *
 *  $END$
 */
*
FUNCTION ACWarmSpot(nID)
*
* Local variables
*

ACMouseSpot[nId,9]=.T.

RETURN NIL

* End of ACWarmSpot

*****
*
* function ACIgnoreMouse()
*
* Modification History:
*        Version    Date      Who       Notes
*         V1.00     10/25/91  LJL       Initial Version
*
/*  $DOC$
 *  $FUNCNAME$
 *     ACIGNOREMOUSE()
 *  $CATEGORY$
 *      Achoice
 *  $ONELINER$
 *     Ignore the mouse if present when executing ACHOICE
 *  $SYNTAX$
 *     ACIGNOREMOUSE( <lIgnore> ) -> lCurrent
 *  $ARGUMENTS$
 *     <lIgnore> - logical for ignoring mouse .T. = act as if mouse
 *               is not present. If absent just returns current setting
 *
 *  $RETURNS$
 *     Setting in effect prior to call as a logical.
 *  $DESCRIPTION$
 *     This routine causes the ACHOICE routine to ignore the mouse if present
 *     based upon the passed parameter. the current (prior to the call) 
 *     value is returned. If not parameter is passed, only the current setting
 *     is returned.
 *  $EXAMPLES$
 *     lOldSetting := ACIgnoreMouse(.T.)
 *  $SEEALSO$
 *     ACHOICE()
 *  $INCLUDE$
 *
 *  $END$
 */


* Calling Parameters: 
* Returns: 
*
FUNCTION ACIgnoreMouse(lIgnore)

* Local Parameters
LOCAL oldsetting

*Save old value
oldsetting=lIgnoreMouse

IF lIgnore!=NIL
	lIgnoreMouse=lIgnore
ENDIF

RETURN oldsetting

* End of ACIgnoreMouse


