/***
    DBFMEMO.PRG - allows the storage of "memo" fields in standard DBF files

    Requires a DBF file with the following structure:
      NAME        TYPE LEN DEC
      ----------- ---- --- ---
      TYPE        C      2
      CODE        C     10
      SEQUENCE    C      1
      TEXTLINE    C     64
    ...and an index on TYPE+CODE+SEQUENCE

    The files have to be open for the function to work (duh!)

    Comments:
       This function currently supports memos up to 256 64-byte lines long.
       This limitation can be overcome by simply increasing the size of
       SEQUENCE to 2 bytes and modifying the corresponding code to use 12BIN()
       and BIN2I() to store the line counter.  Another way to increase the
       maximum memo size would be to increase TEXTLINE.  You can optimize the
       memos any way you want like this.

       The cType parameter to B_DBFMEMO() allows you to segregate memos from
       different files, fields, etc..  This way you can use a SINGLE memo file
       to service multiple DBF files.  For example suppose file A contains 3
       memo fields and file B contains 2, you could Type them like this:
          "A1", "A2", "A3", "B1", "B2"

          To store a memo in file A for memo field 2 for record ID "000123":
             lSuccess:=B_DBFMEMO( MEMOSAVE, "A2", "000123", "THIS IS A MEMO" )

          To retrieve the same memo:
             cMemo:=B_DBFMEMO( MEMOGRAB, "A2", "000123" )

          To delete the same memo:
             lSuccess:=B_DBFMEMO( MEMODELETE, "A2", "000123" )

       If you have a file that does NOT contain a unique key field, you will
       probably have to use some sort of memo-id scheme.  Go to bottom of memo
       file, increment last memo->code used or maintain a last_memo field in a
       configuration file and increment it whenever a memo is added.  Either
       way you will probably want to write a function for adding memos that
       does this work for you.

       Please note, if your application allows the user to change the key field,
       the corresponding memo->code field will also have to be changed for all
       memo records associated with that key field.
*/



#include "common.ch"

// DBF-MEMO DEFINES - Place these in an application-wide header file
#define MEMOSAVE    1
#define MEMODELETE  2
#define MEMOGRAB    3
#define MEMOALIAS   memo



FUNCTION B_DBFMEMO(nMode,cType,cCode,cText)
/* B_DBFMEMO(nMode,cType,cCode,[cText]) --> ?
   nMode is MEMOSAVE, MEMODELETE, MEMOGRAB
   cType is a 2 char attribute that allows you to seperate types of memos
   cCode is a 10 char attribute that points to the individual memo

   Returns: if MEMOSAVE or MEMODELETE lSuccess
            if MEMOGRAB cMemoText

*/
   LOCAL xReturn ,;
         lOk := .T.,;
         nSequence := 1

   cType := PADR(cType,2)
   cCode := PADR(cCode,10)

   DO CASE
   CASE nMode == MEMOSAVE
      IF (cText <> NIL)
         // Find first record of existing memo and replace as long as
         // slots exist for this type and code
         MEMOALIAS->(DBSEEK(cType+cCode+CHR(1)))
         WHILE MEMOALIAS->type == cType .AND.;
               MEMOALIAS->code == cCode .AND.;
               .NOT. MEMOALIAS->(EOF()) .AND.;
               .NOT. EMPTY(cText) .AND. lOk

            IF ( lOk := MEMOALIAS->(B_RLOCK()) )
               MEMOALIAS->sequence := CHR(nSequence)
               MEMOALIAS->textline := SUBSTR(cText,1,64)
               lOk := (++nSequence <= 254)
               cText := SUBSTR(cText,65)
               MEMOALIAS->(DBUNLOCK())
               MEMOALIAS->(DBSKIP())
            END
         END

         // We are either going to add new slots or delete existing
         // unused slots.
         IF .NOT. EMPTY(cText) .AND. lOk
            // Add new slots for this type and code
            WHILE (.NOT. EMPTY(cText)) .AND. lOk
               IF ( lOk := MEMOALIAS->(B_ADDREC()) )
                  MEMOALIAS->type := cType
                  MEMOALIAS->code := cCode
                  MEMOALIAS->sequence := CHR(nSequence)
                  MEMOALIAS->textline := SUBSTR(cText,1,64)
                  lOk := (++nSequence <= 254)
                  cText := SUBSTR(cText,65)
                  MEMOALIAS->(DBUNLOCK())
               END
            END
         ELSE
            // Delete old unused slots for this type and code
            lOk := MEMOALIAS->(B_DELREC(,{|| MEMOALIAS->type == cType .AND.;
                                 MEMOALIAS->code == cCode } ))
         END

      ELSE
         lOk := .F.
      ENDIF
      xReturn := lOk

   CASE nMode == MEMODELETE
      MEMOALIAS->(DBSEEK(cType+cCode))
      lOk := MEMOALIAS->(B_DELREC(,{|| MEMOALIAS->type == cType .AND.;
                           MEMOALIAS->code == cCode } ))
      xReturn := lOk

   CASE nMode == MEMOGRAB
      xReturn := ""
      MEMOALIAS->(DBSEEK(cType+cCode))
      WHILE MEMOALIAS->type == cType .AND.;
            MEMOALIAS->code == cCode .AND.;
            .NOT. MEMOALIAS->(EOF())
         xReturn += MEMOALIAS->textline
         MEMOALIAS->(DBSKIP())
      END

   END
RETURN xReturn


//EOF
