/* 
*  MKSHADOW.CMD
*
*******************************************************************************
* created:  6-23-93                     Frank Yannetta          Vancouver, BC
* public domain release: 11-18-93
* This script is strictly use-at-your-own-risk.  The author assumes no
* responsibilities or liabilities.  Nothing is asked for it; nothing should be 
* expected from it or the author.
*
* That being said.., I am releasing this code so that people who are learning
* REXX have something new to disect.  This script works but its function is
* better performed by the 'Find' utility on each folder menu (thunk, went head
* on desk when I finally noticed Find).  I thought to convert it to command
* line utility that could be used in a batch file, but I couldn't think of
* any batch operations that, using MKSHADOW, would prove to be a benefit 
* rather than a novelty.  Rexx is still new to me so expect to find all sorts
* of odd approaches and questionable control flow.  I programmed the best part
* of this script using the time-worn by-the-seat-of-your-pants method, so it's
* not as procedural as it should have been.  In closing, I've beefed up the 
* documentation a bit to make it more clear as to what this script can and
* cannot do.  Good luck.
*
* P.S. The following ideas for reusing the MKSHADOW code were considered:
* 
*	- can be modified to create shadows in specific folders on the Desktop
*	  by selecting the target folder via the file system. This gets around
*	  the WP ObjectID limitation, noted below in the original Usage descr.
*       - rebuild as a simpler, generic command-line File-Finder; add filters
*         ((why not? - it's been done))
*       - rebuild as a general find routine 
*         - command-line switches re <G>oto, <D>etails, <S>hadow (to Desktop 
*           of folder), <O>pen, file <T>ype, <C>reation date, <M>odification 
*           date;  
*         - there should also be a negation switch for any of the above; 
*         - 'no-prompt' switch for batch mode operations - in this case need
*           to define default unattended actions
*         - switch to bypass all WPS features
*         ((why not? - too bulky; I like simple and direct apps))
*       - rebuild as a 'Work Area' generator - i.e. use to build a work area
*         folder and record the object elements added to a file; recreate a
*         work area from an existing record file; consider using a VREXX
*	  interface
*         ((why not? - object Copy and templates cover most of this))
*
*******************************************************************************
*
*       REXX script to create a shadow of a drive object (file or directory) 
*       and place this shadow on the Desktop.
*
*
*       << skip this section, see 'Correction re Usage', below >>
*
*       Usage:
*
*           MKSHADOW.CMD <filespec>     - when one or more full filename specificatios 
*           (only a single filespec       (i.e. drive, path and filename) is provided
*           is accepted - paths need      the procedure will verify that the target
*           not be full)                  objects exist and if so will immediately
*                                         create a shadow of each object on the Desktop.
*                                         Wildcard specifications are not permitted.
*
*       Note:   It is not practical to provide for a command line argument to specify an
*               alternate location in which to place shadows created by this procedure.
*               In order to place a shadow in, say, a subfolder on the Desktop, the user
*               would need to provide the WPS object ID for that folder.  Few user's
*               would know the object ID's of their folders (fewer would care) and there
*               is no means in REXX to query this string.
*
*
*****************************************************************************
* Correction re usage:
*
*       The usage description, above, is somewhat murky and inaccurate.  It 
*       should read:
*               - <filespec> is any file system file/path specification that 
*                 includes any of the following forms:
*                       <drive:><path><name>
*                       <path><name>
*                       <drive:><name>    
*                       <name>
*
*                 where <drive:> is a drive specification (e.g. C:);  
*                       <path> is a path specification (e.g. \WP\REPORTS\);
*                       <name> is the name of a file or directory (e.g. MYFILE.RPT)
*
*                 - *IMPORTANT*:  only <name> may contain wildcards
*                   Example:  D:\WORK\MY*.*  or  C:TEMP.*
*
*                 - to specify a directory as the target, don't include the 
*                   trailing backslash 
*                   Note:  if the trailing backslash is used then *.* is 
*                   implied for <name>.
*                   Example:  C:\TEMP  <-- will locate file\dir C:\TEMP
*                             C:\TEMP\ <-- will locate files\dirs C:\TEMP\*.*
*                 - if <name> is specified then <drive:> is optional (in this
*                   case, all drives are searched), and <path> is optional 
*                   (in this case, all directories searched recursively from 
*                   the root)
*                   EXAMPLE: TEMP       <-- search all drives, all dirs
*                            D:TEMP     <-- search all dirs on drive D:
*                            \WORK\TEMP <-- search all drives
*
*                 - when a match list is displayed for selecting the target
*                   files/dirs to shadow, any invalid entries are ignored
*                   without prompting or error recovery
*
****************************************************************************
* Modifications:
*
*  1) command line options have now been added; usage is now of the form:
*
*       MKSHADOW <filespec> <options>
*
*       - *IMPORTANT*: the order of <filespec> <options> is crucial; do *not*
*         enter these reversed or mixed
*       - the <filespec> argument is required; no changes in any of the
*         specification formats (see description above), but target matching
*         may be modified by <options>
*       - the <options> argument is optional; the options that can be 
*         specified include:
*               /S      force recursive search of subdirectories
*               /F      search for files only
*               /D      search for directories only
*
*         NOTES:
*               Use /S with discretion.  If too many matches are found for
*               the name spec, MKSHADOW will abort and ask the user to be more
*               specific.  Note that /S is already implied when the path
*               specification is omitted from the <filespec> argument.
*
*               If neither /F or /D are specifed, the default is to search
*               for both files and directories (this is the same as 
*               specifying both /F and /D).  
*
*               All specified options must be space seperated, as in '/S /F'.
*               Options are not case sensitive.
*
*****************************************************************************
* lIMITATIONS:
*
*       Well, there are many. The following are just some of the highlights:
*
*       1. There is user-defined limit to the number of file names that will
*          be displayed for matches to the filespec argument.  This limit
*          was essential in order to keep the match list down to a single
*          screen - rather than fuss about trying to build a paging routine.
*          The session window will be resized (using the MODE command) when 
*          the number of matches exceeds the number of screen lines.  The 
*          'maximum matches' value entered currently (see assignment of var
*          maxmatchlist, below) may not be practical for some displays.
*          The variable maxmatchlist should be set to 8 less than the max
*          number of lines you can view in an OS/2 window.
*       2. Surprise!  You can't create a shadow of an object if the name of 
*          that object matches that of any object currently on the Desktop.  
*          This is a limitation imposed by the REXX function SysCreateObject
*          (you and I can do it using the mouse, so what gives here?).  So, 
*          while you can locate two or more files named TEMP (for example), 
*          only one of these can be shadowed to the Desktop using MKSHADOW.
*          [I suspect the answer might lie in specifying the TITLE= option
*          for SysCreateObject, say as in TITLE=<full_filespec> - I also  
*          suspect that using TITLE thusly might be a fine way of trashing 
*          the real name of a file, so for now I'm going to leave it alone.]
*       3. Repeated calls to MKSHADOW will result in shadow icons being 
*          overlaid on the Desktop.  
*       4. Pretty certain that I haven't found all the bugs yet!
*
***************************************************************************
***************************************************************************/

/*-----------------------*/
/*  user-defined limits  */
/*-----------------------*/

	/* maximum number of matched names to display - too many will cause abort */
maxmatchlist = 90

	/* maximum number of shadows to create without confirmation if the user   */
	/* enters ALL when selecting from the match list                          */
maxselectwarning = 5


/****************************** Error Messages ************************************/

	/* ERROR:  No input argument                                            */
ErrNoArg   = ''

	/* ERROR:  Argument was drive specification only, without path or name  */
ErrDrvOnly = 'You must specify a path and/or name in addition to a drive.'

	/* ERROR:  The path portion of the target identifier contains wildcards */
ErrBadPath = 'Wildcards are not permitted in path portion of identifier.'  

	/* ERROR:  The drive specification of the identifier is invalid         */
ErrBadDrv  = 'Drive specified does not exist or is not accessible.'

	/* ERROR:  No file or directory was found to match the given identifier */
ErrNoMatch = 'No files/dirs were found to match the specified identifier.'

	/* ERROR:  Too many matches were found for identifier                   */
ErrMatchMax = 'Too many matches were found for identifier.  Be more specific.' 

	/* User elected to abort:  Nothing was selected from match list         */
MsgUserCancel = 'No objects were selected.  Nothing was done'

/************************** end of Error Messages *******************************/


/*--------------------------------------------------*/
/*                      Start                       */
/*--------------------------------------------------*/

call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
call SysLoadFuncs


	/* If no cmd line arg then exit, otherwise parse the arg   */
	/* Note: var cloptions not used again until line:          */
	/*      searchopt = Parse_Options cloptions ... (below)    */
ARG fullname cloptions
fullname = STRIP(fullname)
IF fullname = "" THEN CALL AbortNoRecover ErrNoArg

	/* check for common typo: where ; is substituted for : in drive spec */
IF ( pos(';',fullname) = 2 ) THEN  fullname = overlay(':',fullname,2)

	/* atomize the filespec */
targdrive = FILESPEC('drive', fullname)
targpath  = FILESPEC( 'path', fullname)
targname  = FILESPEC( 'name', fullname)

	/* Establish the state of each element of the full name specification   */
	/* ---------------------------------------------------------------------*/
	/* Cases:       driveOK pathOK  nameOK          action                  */
	/*              ------- ------  ------  ------------------------------- */
	/*   1             N      N       N     exit: AbortNoRecover(NoArg)     */
	/*   2             N      N       Y     search all drives for NAME      */
	/*   3             N      Y       N     search all drives for PATH\*.*  */
	/*   4             N      Y       Y     search all drives for PATH\NAME */
	/*   5             Y      N       N     exit: AbortNoRecover(DrvOnly)   */ 
	/*   6             Y      N       Y     search DRIVE for NAME           */
	/*   7             Y      Y       N     search DRIVE for PATH\*.*       */
	/*   8             Y      Y       Y     search DRIVE for PATH\NAME      */
	/*                                                                      */
	/* Note:  Wildcards are permitted in names but not paths                */

driveOK = \(targdrive = "")
pathOK  = \(targpath = "")
nameOK  = \(targname = "")

	/* if only a drive spec is given then exit */
IF ( driveOK & \(pathOK | nameOK) ) THEN CALL AbortNoRecover ErrDrvOnly

	/* if a path was provided with a trailing backslash then make filespec *.* */
IF ( pathOK ) THEN DO
		/* if path has wildcards then exit */
	IF VERIFY(targpath, '*?', 'M') > 0 THEN CALL AbortNoRecover ErrBadPath
		/* if no name was provided (i.e. path was followed by trailing  */
		/* backslash (\) then assume name to be *.* (i.e. all files)    */
	ELSE IF ( \nameOK ) THEN DO
		targname = '*.*'
		nameOK   = 1
		END
	END

/*********************************************************************************/

	/* get list of accessible drives and break down into stem var */
drivelist = SysDriveMap('C:','USED') 
drivelist = STRIP(drivelist)

i = 0
DO UNTIL drivelist = ""
	i = i + 1
	PARSE VAR drivelist drive.i drivelist
	drivelist = STRIP(drivelist, 'L')
	END
drive.0 = i

/*********************************************************************************/

	/* if drive is known (path: don't care) then check if drive exists; if  */
	/* so then limit search to drive                                        */
IF driveOK THEN DO
	driveOK = 0
	DO i = 1 TO drive.0 UNTIL driveOK
		IF targdrive = drive.i THEN driveOK = 1
		END
	IF \driveOK THEN CALL AbortNoRecover ErrBadDrv
	ELSE DO
		drive.0 = 1
		drive.1 = targdrive
		END
	END

/*********************************************************************************/

	/* Proceed with name matching on targdrive (where drive.0 = 1) or on all */
	/* drives (where drive.0 > 1) - Note: the name of the stem for the list  */
	/* of matched names is 'object'; the search is for physical file and     */
	/* directory 'objects', it does not involve logical Desktop objects such */
	/* as existing shadows, program reference objects or miscellaneous       */
	/* objects (e.g. shredder)                                               */
object.0 = 0
targspec = targpath""targname


	/* Parse command line options, if any, to set SysFileTree */
	/* option string.  Where no command line options were     */
	/* entered, the state of var pathOK will be used to set   */
	/* options                                                */
searchopt = Parse_Options( cloptions, pathOK )


/*----------------------------------------------*/
/* Commence searching on each the target drives */
/*----------------------------------------------*/

DO i = 1 TO drive.0
	SAY 'Searching drive' drive.i
	rc = SysFileTree(drive.i""targspec,temp,searchopt)

	IF rc THEN SAY "Can't search drive" drive.i
	ELSE DO j = 1 to temp.0
		object.0 = object.0 + 1
		k = object.0
		object.k = temp.j
		END
	END

/*---------------*/
/* end of search */
/*---------------*/


	/* if too many matches for name (or none) then exit, otherwise  */
	/* display the list of matches as numbered lines and get the    */
	/* number(s) of the matched name(s) to shadow                   */

IF object.0 = 0 THEN CALL AbortNoRecover ErrNoMatch

IF object.0 > maxmatchlist THEN CALL AbortNoRecover ErrMatchMax

	/* if only one match then set pick list (target selection) to the first */
	/* (and only) entry in object match list; bypass the match list display */
IF object.0 = 1 THEN DO
	pick.0 = 1
	pick.1 = 1
	SIGNAL CreateShadow
	END

	/* for longer lists, increase the screen length */
IF object.0 > 18 THEN DO
	scrlen = object.0 + 8
	'mode co80,'scrlen
	END


/*--------------------------------------------------------------*/
/* display the list of objects located by the name match search */
/*--------------------------------------------------------------*/

DisplayList:
cls
SAY '          L O C A T E D   F I L E S   A N D   D I R E C T O R I E S'
SAY ''
SAY
DO i = 1 to object.0
	SAY format(i,2) ' ' object.i
	END

SAY
SAY

cpos = SysCurPos()
promptpos = overlay(',',cpos,pos(' ',cpos))
SAY 'Enter line number of objects to shadow: (A:all ENTER:none or, # ... #)'
cpos = SysCurPos()
responsepos = overlay(',',cpos,pos(' ',cpos))


/*-----------------------------------------*/
/* Process target selection via match list */
/*-----------------------------------------*/

	/* Parse the response string entered by the user; if not 'A'    */
	/* (all) or 'Enter' (none) then keep only valid picks:  i.e.    */
	/* any number in range of the matched object list entries       */

GetPickList:

PULL response

picklist = STRIP(response)
PARSE UPPER VAR picklist pick.1 picklist

SELECT
	WHEN pick.1 = "" THEN DO
		CALL AbortNoRecover MsgUserCancel
		END
	WHEN ABBREV(pick.1,"A") THEN DO
		IF object.0 > maxselectwarning THEN DO
			CLS
			SAY 'About to create shadows of ALL objects in the match list'
			SAY 'OK to proceed?  (Y/n)'
			response = SysGetKey('NOECHO')
				/* if not then go back to redisplay the match list */
			IF response = "N" THEN SIGNAL DisplayList
			END
		DO i = 1 to object.0
			pick.i = i
			END
		pick.0 = object.0
		END
	WHEN DATATYPE(pick.1) = 'NUM' THEN DO 
			/* check if pick # is in range - i.e. in the pick list */
			/* if not then discard this selection                  */
		IF (pick.1 > 0) & (pick.1 <= object.0) THEN i = 1
		ELSE i = 0

		DO UNTIL picklist = ""
			i = i + 1
			PARSE VAR picklist pick.i picklist
				/* if pick entry  is not a number or pick # is  */
				/* out of range, then discard this entry        */
			IF (DATATYPE(pick.1) = 'CHAR') | (pick.i <= 0) | (pick.i > object.0) THEN i = i - 1
			END

			/* if we have valid selections then assign pick counter */
			/* From here we proceed to LABEL: CreateShadow          */
		IF i > 0 THEN pick.0 = i

			/* otherwise no valid numbers were entered - get some more */
		ELSE DO
				/* overwite the previous prompt and response lines on */
				/* the screen and get the user to reenter their picks */
			INTERPRET 'cpos = SysCurPos('promptpos')'
			SAY 'Reenter line # of objects to shadow - seperate each with a space              '
			INTERPRET 'cpos = SysCurPos('responsepos')'
			SAY '                                                                              '
			INTERPRET 'cpos = SysCurPos('responsepos')'
			SIGNAL GetPickList
			END
		END
	OTHERWISE DO
		INTERPRET 'cpos = SysCurPos('responsepos')'
		SAY '                                                                              '
		INTERPRET 'cpos = SysCurPos('responsepos')'
		SIGNAL GetPickList
		END             
	END




/**************************************************************************************/
CreateShadow:

DO i = 1 to pick.0

	j = pick.i
		/* x,y screen position where icon of shadow object will be      */
		/* placed;  these icons will be located on the Desktop in rows  */
		/* ascending from the bottom of screen, in any one of seven     */
		/* columns (10 unit intervals and 8 unit intervals respectively)*/
	iconRT = (( pick.i - ( trunc((pick.i-1)/7) * 7 ) ) + 3) * 8
	iconUP = (trunc((pick.i-1)/7) * 10) + 5
	name = FILESPEC("name", object.j)
	options = 'SHADOWID='object.j';ICONPOS='iconRT','iconUP

	IF SysCreateObject("WPShadow",name,"<WP_DESKTOP>",options) THEN
		SAY object.j': shadowed OK' 
	ELSE 
		SAY object.j': shadow creation FAILED!'

	END

SIGNAL AllDone


/*========================   F U N C T I O N S   =========================*/

/* Function: Parse_Options                                      */
/* Args:     var cloptions;string (command line options)        */
/*           var pathOK;int,boolean (path specification state) */
/* Returns:  string                                             */
/*--------------------------------------------------------------*/
/* Parse any command line options entered by the user which     */
/* will set course of action for function SysFileTree.          */
/* If var pathOK is false (i.e. path specification was not      */
/* specified, then force recursive searching of directories.    */
/* Note: the testing for CL options is not exhaustive - just    */
/* see if certain option strings can be located in the entry    */

Parse_Options:

enable. = 0
enable.recursion = (pos( '/S', cloptions ) > 0) 
enable.files     = (pos( '/F', cloptions ) > 0)
enable.dirs      = (pos( '/D', cloptions ) > 0)

	/* if no path spec'ed (only drive+name) then force dir recursion */
IF enable.recursion | \pathOK THEN optionlist = 'OS'
ELSE                               optionlist = 'O'

SELECT
	WHEN enable.files & \enable.dirs THEN optionlist = optionlist||'F'
	WHEN enable.dirs & \enable.files THEN optionlist = optionlist||'D'
		/* else, if neither (or both) 'F'iles or 'D'irectories */
		/* are specified then set search to find both          */
	OTHERWISE                             optionlist = optionlist||'B'

END
DROP enable 
RETURN optionlist

	

/****************** Start of Error Terminate Routines *********************/

/* Function: AbortNoRecover                                     */
/* Args:     string (error msg string)                          */
/* Returns:  nothing                                            */
/*--------------------------------------------------------------*/
/* Report an error message to user.  Prompt is displayed for    */
/* user to select further usage info.  This routine will        */
/* terminate the current script - control is not returned to    */
/* the caller.  This routine is a temporary stub provided until */
/* an error recovery mechanism can be implemented (if required) */

AbortNoRecover:
CLS
IF ARG(1) = "" THEN SIGNAL UsageMsg
ELSE DO
	SAY
	SAY '**MKSHADOW**  ' ARG(1)
	SAY
	SAY 'Press 1 for description of MKSHADOW usage, any other key to continue.'
	response = SysGetKey('NOECHO')
	IF response = '1' THEN SIGNAL UsageMsg
	ELSE SIGNAL AllDone     /* immediate exit; skip usage msg output */
	END
/* end of AbortNoRecover - dummy return */
RETURN


UsageMsg:
	CLS
	SAY
	SAY 'USAGE:  MKSHADOW.CMD <drive:><path><name> <options>'
	SAY '    where <drive:> is a drive specification (e.g. C:);'  
	SAY '          <path> is a path specification (e.g. \WP\REPORTS\);'
	SAY '          <name> is the name of a file or directory (e.g. MYFILE.RPT)'
	SAY '          <options> are /S (recursive search), /F (files only), /D (dirs only)'
	SAY
	SAY 'NOTES:'
	SAY '  1) only <name> may contain wildcards'
	SAY "  2) to specify a directory as the target, don't include the trailing backslash"
	SAY '     Note:  if the trailing backslash is used then *.* is implied for <name>.'
	SAY '  3) if <name> is specified then <drive:> is optional (in this case, all drives'
	SAY '     are searched, and <path> is optional (all directories searched recursively'
	SAY
	SAY 'EXAMPLES:'
	SAY '  MKSHADOW C:\WP\REPORTS\MYFILE.RPT'
	SAY '    - a shadow of this file will be created, if it exists'
	SAY '  MKSHADOW \WORK\TEMP /d /s'
	SAY '    - all drives searched for directories named TEMP in directory \WORK or its'
	SAY '      subdirectories; target(s) to be shadowed selected from list of matches'
	SAY '  MKSHADOW D:TEMP.* /f     [ /S is implicit in this case ]'
	SAY '    - all directories on drive D: searched for filenames matching TEMP.*;'
	SAY '      target(s) to be shadowed are selected from a list of matches'
	say '______________________________________________________________________________'
	'pause'        


/*********************************************************************************/

AllDone:

	EXIT
