/*
 * WMAIL        MicroWalt Extended Mail Agent.
 *              Local message delivery, V6-style, which uses "From_" postmarks
 *              to separate messages in the mailbox.  The mailbox file is some-
 *              where in a spool directory, and contains only lines of ASCII
 *              text.  Forwarding is implemented by putting a line containing
 *              the text "Forward to <xxxx>" in a mailbox file.
 *
 * Authors:     Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 *              Peter S. Housel, <housel@en.ecn.purdue.edu>
 *		Copyright 1988-1992 MicroWalt Corporation
 */
#include "wmail.h"
#include <sys/stat.h>
#include <signal.h>
#ifdef MSDOS
#   include <dos.h>
#   include <io.h>
#   include <process.h>
#else
#   include <sys/wait.h>
#endif


static char lockname[PATH_MAX];             /* path name of mailbox lock    */
static MBOX *mbox = (MBOX *)NULL;           /* pointer to active mailbox    */


static _PROTOTYPE( char *hdr_clear, (char *)                                );
static _PROTOTYPE( void hdr_date, (char *, char *)                          );
static _PROTOTYPE( char *hdr_field, (MSG *, char *)                         );
static _PROTOTYPE( void hdr_fetch, (MSG *, char *)                          );
static _PROTOTYPE( int  mb_lock, (char *, int)                              );
static _PROTOTYPE( MBOX *mb_open, (char *)                                  );


/*
 * Clear out any in the given address line.
 * Some mailers add text which shouldn't be in a V6/V7 "From user date"
 * address line.  This routine removes all that junk, and returns the
 * address of the new string.
 */
static char *hdr_clear(old)
char *old;
{
  char *buf;
  register char *bp, *sp;
  int parstack, litstack;

  if ((buf = (char *) malloc(strlen(old) + 1)) == (char *)NULL) {
        fprintf(stderr, "hdr_clear: out of memory: \"%s\"\n", old);
        return((char *)NULL);
  }
  bp = buf; sp = old;
  parstack = 0; litstack = 0;
  while (*sp != '\0') switch(*sp) {
        case '(':
                parstack++;
                sp++;
                break;
        case ')':
                parstack--;
                if (parstack < 0) {
                        fprintf(stderr, "hdr_clear: bad header: \"%s\"\n", old);
                        return((char *)NULL);
                } else sp++;
                break;
        case '<':
                litstack++;
                sp++;
                break;
        case '>':
                litstack--;
                if (litstack < 0) {
                        fprintf(stderr, "hdr_clear: bad header: \"%s\"\n", old);
                        return((char *)NULL);
                } else sp++;
                break;
        default:
                if (parstack == 0 && litstack == 0) *bp++ = *sp;
                sp++;
  }
  *bp = '\0';
  return(buf);
}


/*
 * Compress the date format in cp[]
 *
 * from   Mon, 12 May 90 12:34:56  or  Mon May 12 12:34:56 1990
 * to     12 May 12:34             or  May 12 12:34
 *
 * This function needs a lot of work.  Perhaps we should simply chop the
 * entire string into known parts, and then re-compose a new string? - FvK
 */
static void hdr_date(str, bufp)
register char *str;
char *bufp;
{
  if (strchr(str, ',')) {
        strcpy(bufp, str + 5);
        strcpy(bufp + 7, str + 15);
  } else strcpy(bufp, str + 4);
  if ((str = strrchr(bufp, ':')) != (char *)NULL) *str = '\0';
}


/*
 * Find the given entry in the mail-header
 * Search for the first occurence of string 'text' in the header.
 * Copy the text following it into the 'let' structure.
 * Return buffer if found, else NULL.
 */
static char *hdr_field(msg, text)
MSG *msg;
char *text;
{
  static char buff[1024];

  msg->curr = msg->start;
  do {
        if (mb_rdline(msg, buff) <= 0) break;
        if (buff[0] == '\0') return((char *)NULL);
        if (!strncmp(buff, text, strlen(text))) return(buff + strlen(text));
  } while(1);
  return((char *)NULL);
}


/*
 * Decode an old-style (V6/V7) mail header.
 * This is a line like:
 *
 *      From <user> <date> [remote from <host>]
 *
 * We want to find out the <user>, <date> and possible <remote> fields.
 * Warning: some mailers (especially some configurations of the
 *          SendMail program) allow comments (like (RealName)) to be
 *          placed in this field.  These comments are now removed.
 */
static void hdr_fetch(msg, text)
MSG *msg;
char *text;
{
  register char *bp, *sp;
  char *buf, *cp;
  int i;

  /* First of all, clear out any junk. */
  if ((buf = hdr_clear(text)) == (char *)NULL) return;
  sp = buf;

  /* Then, mark the end of the 'user' field. Skip until <date> field. */
  while ((*sp != ' ') && (*sp != '\t')) sp++;
  *sp++ = '\0';
  while ((*sp == ' ') || (*sp == '\t')) sp++;

  /* SP now contains <date> and (possible) <remote> fields. */
  cp = sp;
  while (1) {
        if ((bp = strchr(sp++, 'r')) != (char *)NULL) {
                if (!strncmp(bp, "remote from ", 12)) break;
        } else break;
  }

  if (bp != (char *)NULL) {
        sp = bp + 12;
        *(bp - 1) = '\0';
        bp = strchr(sp, ' ');
        if (bp != (char *)NULL)
                while ((*bp == ' ') || (*bp == '\t')) *bp++ = '\0';
        i = strlen(sp);
  } else i = 0;

  /* Decode the `Date' field in the header. */
  hdr_date(cp, msg->date);

  /*
   * Find the return-addresses.
   * First, check for a From: field.  If it fails, use the
   * current "host" and "user" names, and create a bang.
   */
  if ((cp = hdr_field(msg, "From: ")) == (char *)NULL) {
        if ((msg->from = (char *)malloc(strlen(buf) + i + 4)) == (char *)NULL) {
                fprintf(stderr, "hdr_fetch: out of memory.\n");
                exit(-1);
        }
        /* Create the (dangerous) return-address. */
        if (i > 0) sprintf(msg->from, "%s!%s", sp, buf);
          else strcpy(msg->from, buf);
  } else {
        /* First find a possible <address> field. */
        if ((sp = strchr(cp, '<')) != (char *)NULL) {
                cp = ++sp;
                /* Remove the <address> trailing > */
                if ((sp = strchr(cp, '>')) != (char *)NULL) *sp = '\0';
        } else {
                /* Remove possible (Real Name) field. */
                if ((sp = strchr(cp, '(')) != (char *)NULL) {
                        /* Also strip trailing spaces. */
                        while ( *(sp - 1) == ' ' || *(sp - 1) == '\t') sp--;
                        *sp = '\0';
                }
        }

        if ((msg->from = (char *)malloc(strlen(cp) + 2)) == (char *)NULL) {
                fprintf(stderr, "hdr_fetch: out of memory.\n");
                exit(-1);
        }
        strcpy(msg->from, cp);
  }

  /*
   * The default Reply-address is usually simpler, since most mailers
   * generate a Reply-To: or Return-Path: field.  If either of these
   * (in this order) is is present, use it.  Otherwise, use the previously
   * decoded address.
   */
  cp = hdr_field(msg, "Reply-To: ");
  if (cp == (char *)NULL) cp = hdr_field(msg, "Return-Path: ");
  if (cp != (char *)NULL) {
        /* First find a possible <address> field. */
        if ((sp = strchr(cp, '<')) != (char *)NULL) {
                cp = ++sp;
                /* Remove the <address> trailing > */
                if ((sp = strchr(cp, '>')) != (char *)NULL) *sp = '\0';
        } else {
                /* Remove possible (Real Name) field. */
                if ((sp = strchr(cp, '(')) != (char *)NULL) {
                        /* Also strip trailing spaces. */
                        while ((*(sp - 1) == ' ') || (*(sp - 1) == '\t')) sp--;
                        *sp = '\0';
                }
        }

        msg->reply = (char *)malloc(strlen(cp) + 2);
        if (msg->reply == (char *)NULL) {
                fprintf(stderr, "get_hdr: out of memory.\n");
                exit(-1);
        }
        strcpy(msg->reply, cp);
  } else msg->reply = msg->from;

  /* Release the allocated memory. */
  free(buf);
}


/* Lock a mailbox file. */
static int mb_lock(path, mode)
register char *path;
int mode;
{
#ifndef MSDOS
  char buff[PATH_MAX];
  register char *sp;
#endif
  register int i;
#ifdef MSDOS
  int fd;
#endif

  /* We only need to lock SYSTEM mailboxes. */
  if (opt_f == 1) return(0);

  /* Do we have to UNLOCK a mailbox? */
  if (mode == 0) {
        (void) unlink(lockname);
        return(0);
  }

  i = 0;
#ifdef MSDOS
  sprintf(lockname, "%s.LCK", path);
  while((fd = creatnew(lockname, 0)) < 0) {
        (void) sleep(5);
        if (++i > 5) return(-1);
  }
  (void) close(fd);
#else
  strncpy(buff, path, PATH_MAX);
  if ((sp = strrchr(buff, '/')) != (char *)NULL) {
	*sp++ = '\0';
	sprintf(lockname, "%s/LCK..%s", buff, sp);
  } else sprintf(lockname, "LCK..%s", path);
  while(link(path, lockname) < 0) {
        (void) sleep(5);
        if (++i > 5) return(-1);
  }
#endif
  return(0);
}


/* Open a mailbox and create a handle for it. */
static MBOX *mb_open(mailbox)
char *mailbox;
{
  MBOX *mb;

  /* Create a mailbox handle. */
  if ((mb = (MBOX *)malloc(sizeof(MBOX))) == (MBOX *)NULL) {
        fprintf(stderr, "mb_open: out of memory!\n");
        return((MBOX *)NULL);
  }

  /* Generate the path name for a mailbox. */
  if (mailbox == (char *)NULL) sprintf(mb->path, MAILBOX, u_name);
    else strcpy(mb->path, mailbox);

  /* Try opening the mailbox. */
#ifdef MSDOS
  mb->mbox = fopen(mb->path, "rb");
#else
  mb->mbox = fopen(mb->path, "r");
#endif

  mb->update = 0;
  return(mb);
}


/* Get address of some message in mailbox. */
MSG *mb_select(msgnum)
register int msgnum;
{
  register MSG *msg;

  msg = mbox->first;
  while (msg != (MSG *)NULL) {
        if (msg->seq == msgnum) return(msg);
        msg = msg->next;
  }
  return((MSG *)NULL);
}


/* Fetch one line of the contents of a message. */
int mb_rdline(msg, bufp)
MSG *msg;
char *bufp;
{
  off_t opos, max;
  register off_t pos;
  register char *sp;
  register int c;
  int count;

  if (mbox == (MBOX *)NULL) {
        fprintf(stderr, "MBOX: mailbox not open!\n");
        return(-1);
  }
  if (msg == (MSG *)NULL || msg->status == MS_DELETED) {
        fprintf(stderr, "Message %d: inapropriate message!\n", msg->seq);
        return(-1);
  }
  sp = bufp;
  count = 0;
  opos = ftell(mbox->mbox);
  pos = msg->curr;
  if (msg->chars == (off_t) -1) max = (off_t) -1;
    else max = (msg->start + msg->chars);
  (void) fseek(mbox->mbox, pos, SEEK_SET);
  do {
	if (feof(mbox->mbox) || ((max > 0L) && (pos++ >= max))) break;
        if ((c = fgetc(mbox->mbox)) == EOF) break;
        if (c == '\r') continue;
        *sp++ = c;
        count++;
	if (c == '\n') break;
  } while(1);
  *sp = '\0';
#ifdef MSDOS
  if ((sp = strchr(bufp, '\r')) != (char *)NULL) *sp = '\0';
#endif
  if ((sp = strchr(bufp, '\n')) != (char *)NULL) *sp = '\0';
  msg->curr = ftell(mbox->mbox);
  (void) fseek(mbox->mbox, opos, SEEK_SET);
  return(count);
}


/* Check if there is any mail for the calling user. */
int mb_check(mailbox)
char *mailbox;
{
  char buff[1024];
  MBOX *mb;
  register char *sp;

  /* Open the mailbox. */
  mb = mb_open(mailbox);
  if ((mb == (MBOX *)NULL) || (mb->mbox == (FILE *)NULL)) {
        if (opt_e == 0) printf("No mail for %s.\n", u_name);
        if (mb != (MBOX *)NULL) free(mb);
        return(1);
  }

  /*
   * Check a mailbox "forward" request.  This means, that we have
   * to see if the named mailbox has a first line containing:
   *
   *      Forward to XXXX
   *
   * If this is true, the username part of the line is put into a
   * (static) buffer, and returned.  Otherwise, NULL is returned.
   */
  if (fgets(buff, 1024, mb->mbox) == (char *)NULL) {
        if (opt_e == 0) printf("No mail for %s.\n", u_name);
        (void) fclose(mb->mbox);
        free(mb);
        return(1);
  }
#ifdef MSDOS
  if ((sp = strchr(buff, '\r')) != (char *)NULL) *sp = '\0';
#endif
  if ((sp = strchr(buff, '\n')) != (char *)NULL) *sp = '\0';

  if (!strncmp(buff, "Forward to ", 11)) {
        sp = &buff[11];
        while ((*sp == ' ') || (*sp == '\t')) sp++;
        if (opt_e == 0) printf("Your mail is being forwarded to %s.\n", sp);
        (void) fclose(mb->mbox);
        free(mb);
        mb = (MBOX *)NULL;
        return(1);
  }
  (void) fclose(mb->mbox);
  free(mb);
  mb = (MBOX *)NULL;
  return(0);
}


/* Read the contents of the Mail-Box into memory. */
MBOX *mb_read(mailbox)
char *mailbox;
{
  char buff[1024];
  register MSG *msg;
  register char *sp, *cp;
  off_t curr;
  int lines, seq, state;

  /* Check if our mail is being forwarded. */
  if (mb_check(mailbox) != 0) return((MBOX *)NULL);

  /* Open the mailbox. */
  mbox = mb_open(mailbox);
  if ((mbox == (MBOX *)NULL) || (mbox->mbox == (FILE *)NULL)) {
        if (opt_e == 0) printf("No mail for %s.\n", u_name);
        if (mbox != (MBOX *)NULL) free(mbox);
        mbox = (MBOX *)NULL;
        return((MBOX *)NULL);
  }

  /* Determine where all messages start and measure the messages. */
  mbox->first = (MSG *)NULL;
  mbox->last = (MSG *)NULL;
  mbox->entries = 0;
  seq = 1;
  lines = 1;
  state = 1;
  do {
        curr = ftell(mbox->mbox);
        if (fgets(buff, 1024, mbox->mbox) == (char *)NULL) break;
#ifdef MSDOS
        if ((sp = strchr(buff, '\r')) != (char *)NULL) *sp = '\0';
#endif
        if ((sp = strchr(buff, '\n')) != (char *)NULL) *sp = '\0';
        /*
         * The first message starts with "From ", all others start
         * with "\nFrom ".  Check here if we just found another
         * letter...
         */
        if (!strncmp(buff, "From ", 5) && (state > 0)) {
                if ((msg = (MSG *)malloc(sizeof(MSG))) == (MSG *)NULL) {
                        fprintf(stderr, "mb_read: out of memory!\n");
                        (void) fclose(mbox->mbox);
                        free(mbox);
                        mbox = (MBOX *)NULL;
                        return((MBOX *)NULL);
                }
                if (mbox->last == (MSG *)NULL) {
                        mbox->first = msg;
                        msg->prev = (MSG *)NULL;
                } else {
                        msg->prev = mbox->last;
                        mbox->last->next = msg;
                        mbox->last->chars = curr - mbox->last->start;
                        mbox->last->lines = lines;
                        lines = 1;
                }
                mbox->last = msg;
                msg->next = (MSG *)NULL;
                msg->status = MS_NEW;
                msg->start = curr;
                msg->curr = msg->start;
                msg->seq = seq++;
                mbox->entries++;
                state = 0;
        } else {
                lines++;
                if (buff[0] == '\0') state++;
                  else state = 0;
        }
  } while(1);
  if (mbox->last != (MSG *)NULL) {
        mbox->last->chars = curr - mbox->last->start;
        mbox->last->lines = lines;
  }

  /* We now know where the messages are, read message headers. */
  msg = mbox->first;
  while (msg != (MSG *)NULL) {
        if ((sp = hdr_field(msg, "From ")) == (char *)NULL) {
                fprintf(stderr, "No FROM postmark in letter %d (%ld)\n",
                                        msg->seq, (long) msg->start);
                msg = msg->next;
                continue;
        }
        hdr_fetch(msg, sp);

        /* Find the "Subject" field, if any... */
        if ((sp = hdr_field(msg, "Subject: ")) == (char *)NULL) sp = "<none>";
        msg->subject = (char *)malloc(strlen(sp) + 1);
        if (msg->subject == (char *)NULL) {
                fprintf(stderr, "mb_read: out of memory!\n");
                (void) fclose(mbox->mbox);
                free(mbox);
                mbox = (MBOX *)NULL;
                return((MBOX *)NULL);
        } else strcpy(msg->subject, sp);

        /* Find the real sender name in the From: field, if any... */
        sp = hdr_field(msg, "From: ");
        if (sp != (char *)NULL) {
                if ((cp = strchr(sp, '<')) != (char *)NULL) *cp = '\0';
                  else if ((cp = strchr(sp, '(')) != (char *)NULL) {
                        sp = cp + 1;
                        if ((cp = strchr(sp, ')')) != (char *)NULL) *cp = '\0';
                }
        } else {
                sp = hdr_field(msg, "From ");
                if (sp != (char *) NULL) {
                        if ((cp = strchr(sp, ' ')) != (char *)NULL) *cp = '\0';
                        if ((cp = strrchr(sp, '!')) != (char *)NULL) sp = cp +1;
#if _UNIX
                          else sp = getpwnam(sp)->pw_gecos;
#endif
                } else sp = "Unknown";
        }

        /* Stuff the real name in memory. */
        msg->realname = (char *)malloc(strlen(sp) + 1);
        if (msg->realname == (char *)NULL) {
                fprintf(stderr, "mb_read: out of memory!\n");
                (void) fclose(mbox->mbox);
                free(mbox);
                mbox = (MBOX *)NULL;
                return((MBOX *)NULL);
        } else strcpy(msg->realname, sp);

        msg = msg->next;
  }
  return(mbox);
}


/* Purge the mailbox file. */
int mb_purge(mailbox)
MBOX *mailbox;
{
  char temp[PATH_MAX], buff[1024];
  MSG nmsg, *msg;
  register FILE *fp;
  register int new, c;
#ifdef MSDOS
  time_t now;
#endif

  /* Is this really needed? */
  if (mailbox->update == 0) {
        (void) fflush(mailbox->mbox);
        (void) fclose(mailbox->mbox);
        return(0);
  }

  /* Generate path name for mailbox temp file. */
#if _UNIX
  sprintf(temp, CP_MTEMP, getpid());
#else
  (void) time(&now);
  sprintf(temp, CP_MTEMP, (int) (now & 0xFFFFL));
#endif

  /* Create temporary data file. */
#ifdef MSDOS
  if ((fp = fopen(temp, "wb")) == (FILE *)NULL) {
#else
  if ((fp = fopen(temp, "w")) == (FILE *)NULL) {
#endif
        perror(temp);
        return(-1);
  }

  /* Copy letters from old file to new file. */
  msg = mailbox->first;
  while(msg != (MSG *)NULL) {
        if (msg->status != MS_DELETED) {
                msg->curr = msg->start;
                while(mb_rdline(msg, buff) > 0) {
#ifdef MSDOS
			fprintf(fp, "%s\r\n", buff);
#else
                        fprintf(fp, "%s\n", buff);
#endif
                }
        }
        msg = msg->next;
  }

  /*
   * We now copied all "not deleted" letters from the old mailbox file.
   * Create a "fake" message which starts at the byte after the last
   * message in the box, and which continues up to the physical end of
   * the mailbox file (as indicated by "-1").  Check if there is any
   * data after the last letter, which otherwise would be lost.
   */
  nmsg.curr = (off_t) (mailbox->last->start + mailbox->last->chars);
  nmsg.chars = (off_t) -1;
  nmsg.status = MS_NEW;
  new = 0;
  while(mb_rdline(&nmsg, buff) > 0) {
	new++;
#ifdef MSDOS
        fprintf(fp, "%s\r\n", buff);
#else
        fprintf(fp, "%s\n", buff);
#endif
  }

  /* Re-open the new mailbox file for reading. */
#ifdef MSDOS
  if ((fp = freopen(temp, "rb", fp)) == (FILE *)NULL) {
#else
  if ((fp = freopen(temp, "r", fp)) == (FILE *)NULL) {
#endif
        perror(temp);
        (void) unlink(temp);
        return(-1);
  }

  /* Shut off signals during the update. */
  (void) signal(SIGINT, SIG_IGN);
#ifdef SIGHUP
  (void) signal(SIGHUP, SIG_IGN);
#endif
#ifdef SIGQUIT
  (void) signal(SIGQUIT, SIG_IGN);
#endif

  /* Lock the current mailbox. */
  if (mb_lock(mailbox->path, 1) < 0) {
        fprintf(stderr, "Cannot lock %s for UPDATE.\n", mailbox->path);
        return(-1);
  }

  /* Open the mailbox for writing, truncating it if needed. */
  (void) fclose(mailbox->mbox);
#ifdef MSDOS
  if ((mailbox->mbox = fopen(mailbox->path, "wb")) == (FILE *)NULL) {
#else
  if ((mailbox->mbox = fopen(mailbox->path, "w")) == (FILE *)NULL) {
#endif
        perror(mailbox->path);
        (void) mb_lock(mailbox->path, 0);
        return(-1);
  }

  /*
   * Copy temporary data file to new mailbox.  We use a single-character
   * loop, to provide for a binary copy of the mailbox files, rather than
   * a "string" based copy in which the CR/LF stuff gets translated.
   */
  while (1) {
	if (feof(fp) || ferror(mailbox->mbox)) break;
	if ((c = fgetc(fp)) == EOF) break;
	(void) fputc(c, mailbox->mbox);
  }
  (void) fflush(fp); (void) fflush(mailbox->mbox);

  /* Remove temporary data file and unlock mailbox. */
  (void) unlink(temp);
  (void) mb_lock(mailbox->path, 0);

  if (new > 1) printf("New mail has arrived.\n");
  return(0);
}
