/* [DOC] File ****************************************************
** MODULE INFORMATION*
**********************
**      FILE    NAME:       PWGINP.C
**      SYSTEM  NAME:       POWER
**      VERSION NUMBER      3.00
**
** Descr:       Text Input function for
**              Programmer Oriented Window Environment Routines.
**              (NoC) July 1989 - Ling Thio Software Productions.
******************************************************************
** ADMINISTRATIVE INFORMATION*
******************************
**      ORIGINAL AUTHOR:    Ling Thio
**      CREATION DATE:      89/07/10
**      CHANGES:            none
*****************************************************************/

#include <string.h>                         /* for str*() */
#include <ctype.h>                          /* for toupper() */
#include "pwinc.h"                          /* Main include file */

static int insert = 1;
static PWKEY pw_input_mou[7] = {
    PWK_LEFT,                               /* horizontal mouse movement*/
    PWK_RIGHT,
    0,                                      /* vertical mouse movement  */
    0,      
    PWK_ENTER,                              /* left mouse button        */
    0,                                      /* middle button disabled   */
    PWK_ESC,                                /* right mouse button       */
};

/* [DOC] Function *********************************************
** Name:        pw_input                                  [API]
** SYNOPSIS:    PWKEY pw_input(win, tsrow, tscol, text, 
**                             srow, scol, ncol, var, size, 
**                             symbols, mode);
**              PWWIN *win          Pointer to window handle
**              int   tsrow         Row. pos. of 'text'
**              int   tscol         Col. pos. of 'text'
**              char  *text         Input description
**              int   srow          Row. pos. of input area
**              int   scol          Col. pos. of input area
**              int   ncol          Size of input area
**              char  *buf          Input buffer
**              int   bufsize       Size of input buffer
**              char  *symbols      Allowed input symbols
**              int   mode          Input mode
** Descr:       Gets an input string.
**              The 'pw_input' function displays the description
**                'text' at the position 'tscol', 'tsrow'.
**              Depending on the input mode 'mode', it will then
**                display the first 'ncol characters of 'buf',
**                or series of dots at the position
**                'scol', 'srow'.
**              Depending on 'mode' again, this function
**                terminates, or asks the user to key in a new
**                value for the variable 'buf'.
**              If the character string 'symbols' is a non-zero 
**                pointer,  only the characters in this string
**                will be accepted as valid input letters.
**              Note that the window input bufsize 'ncol' may be
**                smaller than the actual input buffer size.
**                The input will scroll when needed.
**              If 'ncol' == 0, 'ncol' will be set to the
**                buffer size 'bufsize' - 1.
**              Note also that a buffer of size 'bufsize' can
**                contain a maximum of 'bufsize'-1 characters,
**                allowing space for the '\0' delimiter.
**              Valid modes are:
**                  PWM_EMPTY       Display dots.
**                  PWM_DISPLAY     Display contents of 'buf'.
**                  PWM_PROMPT      Display dots and prompts.
**                  PWM_UEMPTY      Upper case of 'EMPTY'.
**                  PWM_UDISPLAY    Upper case of 'DISPLAY'.
**                  PWM_UPROMPT     Upper case of 'PROMPT'.
** RETURNS:     Returns the last key pressed, which is one of
**                the following key codes:
**                  PWK_ENTER, PWK_ESC, PWK_UP, PWK_DOWN.
**                or 0 in PWM_EMPTY or PWM_DISPLAY mode.
**************************************************************/
PWKEY pw_input(PWWIN * win, int tsrow, int tscol, 
               char *text,  int srow,  int scol, int ncol,
               char *buf,   unsigned int bufsize, char *symbols, int mode)
{
    char *tmp;
    char *start;
    char *cur;
    char *end;
    int i;
    PWKEY key;
    int stay = 0;
    int update;
    int upper = 0;
    char *keyleft;
    char *keyright;
    PWKEY *prev_mou;

    _pwg_status = _pwg_status ^ PW_IN_INPUT;
    if ((ncol == 0) || ((unsigned int)(ncol + 1) > bufsize))
        ncol = bufsize - 1;
    win->ccol = scol;
    win->crow = srow;
    if (mode > 4)
    {
        upper++;
        mode = mode + PWM_EMPTY - PWM_UEMPTY;
    }
    cur = start = tmp = malloc(bufsize);
    if (mode == PWM_EMPTY)
    {
        *tmp = 0;
    }
    else
    {
        pwg_strncpy(tmp, buf, bufsize);
    }
    if (upper)
        strupr(tmp);
    end = tmp + strlen(tmp);
    if (mode == PWM_DISPLAY)
        memset(end, ' ', bufsize-1 - strlen(tmp));
    else
        memset(end, _pwcf_backgr[INPCHAR], bufsize-1 - strlen(tmp));
    tmp[bufsize-1] = '\0';
    pw_putsat(win, tsrow, tscol, text);

    /* display visible portion of string */
    i = start[ncol];
    start[ncol] = '\0';
    pw_putsat(win, srow, scol, start);
    start[ncol] = (char)i;

    if ((mode == PWM_EMPTY) || (mode == PWM_DISPLAY))
    {
        free(tmp);
        return 0;
    }

    /* temporarily replace window key info line */
    keyleft = win->keyleft;
    keyright = win->keyright;
    pw_keyinfo(win, _pw_f1f9esc, "F6=Restore  Ins=Ins/Ovw  Cursor  Enter");
    pw_block(win, tsrow, tscol, 1, strlen(text), PWT_HIGHLIGHT);
    pw_block(win, srow,  scol,  1, ncol,         PWT_HIGHLIGHT);
    pws_cursor(win->srow+srow, win->scol+scol);
    prev_mou = pwl_GetMouDef();
    pwl_SetMouDef(pw_input_mou);
    while ( ((key = pw_getkey()) != PWK_ESC) && (key != PWK_ENTER) &&
            (key != PWK_UP) && (key != PWK_DOWN) ) 
    {
        update = 0;
        switch(key) 
        {
        case PWK_LEFT:
            stay++;
            if (cur > tmp)
                cur--;
            break;
        case PWK_CLEFT:
            stay++;
            while ((cur > tmp) && (*(cur-1) == ' '))
                cur--;
            while ((cur > tmp) && (*(cur-1) != ' '))
                cur--;
            break;
        case PWK_RIGHT:
            stay++;
            if (cur < end)
                cur++;
            break;
        case PWK_CRIGHT:
            stay++;
            while ((cur < end) && (*cur != ' '))
                cur++;
            while ((cur < end) && (*cur == ' '))
                cur++;
            break;
        case PWK_HOME:
            stay++;
            cur = tmp;
            break;
        case PWK_END:
            stay++;
            cur = end;
            break;
        case PWK_CHOME:                     /* delete entire string */
            cur = tmp;
        case PWK_CEND:                      /* delete from cur to end */
        case PWK_CDEL:
            stay++;
            memset(cur, _pwcf_backgr[INPCHAR], (int)(end-cur));
            end = cur;
            update++;
            break;
        case PWK_RESTORE:
            stay = 0;
            cur = tmp;
            pwg_strncpy(tmp, buf, bufsize);
            if ((mode >= PWM_UEMPTY) && (key < 256))
                strupr(tmp);
            end = tmp + strlen(tmp);
            memset(end, _pwcf_backgr[INPCHAR], bufsize-1 - strlen(tmp));
            update++;
            break;
        case PWK_INSERT:
            stay++;
            insert = (insert) ? 0 : 1;
            if ((win->status & PWM_NOBORDER) == 0)
            {
                if (insert) 
                    for (i=0; i<8; i++)
                        pw_putcat(win, win->nrow, i, _pwcf_border[PWCF_H]);
                else
                    pw_putsat(win, win->nrow, 0, "Typeover");
            } 
            break;
        case PWK_BACKSPACE:
            stay++;
            if (cur > tmp)
            {
                cur--;
                memcpy(cur, cur+1, (int)(end-cur));
                end--;
                *end = _pwcf_backgr[INPCHAR];
                update++;
            }
            break;
        case PWK_DEL:
            stay++;
            if (cur < end) 
            {
                memcpy(cur, cur+1, (int)(end-cur));
                end--;
                *end = _pwcf_backgr[INPCHAR];
                update++;
            }
            break;
        }
        if ((key>=32) && (key<=126) && ((unsigned int)(cur-tmp)+1 < bufsize) &&
            (!symbols || strchr(symbols, key))) 
        {                                   /* handle alphanum keys */
            if (upper)
                key = toupper(key);
            if (!stay)
            {                               /* clear if first key=alphanum */
                stay++;
                *tmp = 0;
                cur = end = tmp;
                memset(cur, _pwcf_backgr[INPCHAR], bufsize-1);
            }
            if (insert)
            {                               /* Insert mode-->move */
                memmove(cur, cur-1, end-cur+1);
                if ((unsigned int)(end-tmp)+1 < bufsize)
                    end++;
                else
                    *end = '\0';
            }
            *cur = (char)key;
            if (++cur > end)
                end = cur;
            update++;
        }
        if (cur > tmp+1)
        {
            if ((int)(cur-start) < 2)
            {                               /* update input window */
                start = cur-2;
                update++;
            }
        }
        else
        {
            if (start > cur)
            {                               /* update input window */
                start = cur;
                update++;
            }
        }
        if (cur < tmp+bufsize-2)
        {
            if ((int)(cur-start) > ncol-2)
            {
                start = cur-ncol+2;
                update++;
            }
        }
        else
        {
            if ((int)(cur-start) > ncol)
            {
                start = cur-ncol;
                update++;
            }
        }
        if (update)
        {
            /* display visible portion of string */
            i = start[ncol];
            start[ncol] = '\0';
            pw_putsat(win, srow, scol, start);
            start[ncol] = (char)i;
        }
        pws_cursor(win->srow+srow, win->scol+scol+(int)(cur-start));
    }
    pwl_SetMouDef(prev_mou);
    pwg_CursorOff();
    if (key == PWK_ESC) 
        pwg_strncpy(tmp, buf, bufsize);
    else
    {
        *end = 0;
        strcpy(buf, tmp);
    } 
    for (i=strlen(tmp); i<ncol; i++)
        tmp[i] = ' ';
    tmp[ncol] = 0;

    /* display visible portion of string */
    tmp[ncol] = '\0';
    pw_putsat(win, srow, scol, tmp);
    pw_block(win, tsrow, tscol, 1, strlen(text), PWT_NORMAL);
    free(tmp);

    /* restore previous window key info line */
    pw_keyinfo(win, keyleft, keyright);

    _pwg_status = _pwg_status ^ PW_IN_INPUT;
    return key;
}



